﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using NW4F.LayoutBinaryConverter;
using System.Diagnostics;
using System.IO;
using System.Threading;

namespace ArchiveShaderCombiner
{
    class Program
    {
        // 詳細コンバイナのバリエーションテーブル用識別コード
        const int DetailedCombinerVariationTableCode = 110;
        const int CombinerUserShaderVariationTableCode = 111;

        // プリプロセッサマクロの数
        const int ShaderPreprocessorMacroCount = 12;

        const int ShaderVariationKey0 = 10; // _INTERNAL_SHADER_VARIATION_KEY0 の PreprocessorDefinitionValue 配列内でのインデックス
        const int ShaderVariationKey1 = 11; // _INTERNAL_SHADER_VARIATION_KEY1 の PreprocessorDefinitionValue 配列内でのインデックス

        static void Main(string[] args)
        {
            /*
                * args[0] bgsh ディレクトリ名
                * args[1] apiTypeName
                * args[2] codeTypeName
                * args[3] (--keep-shader-variation-xml)
                * args[4] (--shader-cache-path)
                * args[5] (path)
                */

            bool keepShaderVariationXml = false;
            string shaderCachePath = string.Empty;
            if (args.Length < 3 || args.Length > 6)
            {
                Console.WriteLine(Resources.ErrorArgCount);
                return;
            }
            if (args.Length > 3)
            {
                if (args.Contains("--keep-shader-variation-xml"))
                {
                    keepShaderVariationXml = true;
                }

                int index = Array.IndexOf(args, "--shader-cache-path");
                if (index != -1 && index + 1 < args.Length)
                {
                    shaderCachePath = args[index + 1];
                }
            }

            const string variationFileName = "ArchiveShaderVariation.xml";

            // xml ファイルの列挙
            string[] xmlFiles = System.IO.Directory.GetFiles(args[0], "*.xml", System.IO.SearchOption.AllDirectories);
            GfxShaderVariation merged = null;
            System.Xml.Serialization.XmlSerializer xmlSerializer = new System.Xml.Serialization.XmlSerializer(typeof(GfxShaderVariation));
            foreach (string xmlFile in xmlFiles)
            {
                // 出力ファイルは入力しない
                if (xmlFile.Substring(xmlFile.Length - variationFileName.Length) == variationFileName)
                {
                    continue;
                }

                GfxShaderVariation variation;
                {
                    System.IO.StreamReader file = new System.IO.StreamReader(xmlFile);
                    variation = (GfxShaderVariation)xmlSerializer.Deserialize(file);
                    file.Close();
                }

                if (merged == null)
                {
                    // 1 つ目のバリエーションファイル
                    merged = variation;
                }
                else
                {
                    // 2 つ目以降のバリエーションファイル
                    merged = Merge(merged, variation);
                    if (merged == null)
                    {
                        Console.WriteLine(Resources.ErrorFormat);
                        return;
                    }
                }
            }

            if (merged == null)
            {
                // xml ファイルが 1 つも無い
                return;
            }

            // コンバイナユーザーシェーダ設定
            if (IsShaderKeyExistInPixelShaderVariationTable(merged, CombinerUserShaderVariationTableCode))
            {
                List<string> glslFiles = GetCombinerUserShaderGlslFilesFromPixelShaderVariationTable(merged);
                if (glslFiles != null && glslFiles.Any())
                {
                    // glsl ファイルの列挙
                    const string combinerVariationFilesName = "CombinerUserShaderVariation.glsl";

                    // glsl のファイルリストから、それらをインクルードし #define で分割されたファイルを生成します。
                    CreateCombinerVariationFile(args[0], glslFiles, combinerVariationFilesName);

                    // CombinerVariationFile.glsl に対応する値を PreprocessorDefinitionValue の NW_COMBINERUSERSHADER_TYPE に設定します。
                    ChangeVariationTableWithCombinerVariationFiles(merged, glslFiles);

                    // コンバイナユーザーシェーダ用の個別バリエーションテーブルを作成
                    bool bCombinerUserShaderVariationTable = CreateIndividualVariationTable(args[0], merged, CombinerUserShaderVariationTableCode);
                    Debug.Assert(bCombinerUserShaderVariationTable); // glsl のファイルリストがありながら、バリエーションファイルの出力が無いケースは無い。
                }
            }

            string mergedXmlPath = args[0] + "\\" + variationFileName;
            string outputShaderPath = args[0] + "\\__ArchiveShader.bnsh";
            {
                System.IO.StreamWriter file = new System.IO.StreamWriter(mergedXmlPath, false, new UTF8Encoding(true));
                xmlSerializer.Serialize(file, merged);
                file.Close();
            }

            if (!CallShaderConverter(outputShaderPath, mergedXmlPath, args[1], args[2], shaderCachePath))
            {
                return;
            }

            string outputTablePath = args[0] + "\\__ArchiveShader.bushvt";
            if (!CreateVariationTable(outputTablePath, merged))
            {
                return;
            }

            // 詳細コンバイナ用の個別バリエーションテーブルを作成
            if (!CreateIndividualVariationTable(args[0], merged, DetailedCombinerVariationTableCode))
            {
                return;
            }

            // 一時ファイルの削除
            if (!keepShaderVariationXml)
            {
                if (System.IO.File.Exists(mergedXmlPath))
                {
                    System.IO.File.Delete(mergedXmlPath);
                }
                foreach (string xmlFile in xmlFiles)
                {
                    if (System.IO.File.Exists(xmlFile))
                    {
                        System.IO.File.Delete(xmlFile);
                    }
                }
            }
        }

        static bool CompareVariationConstantDefinition(GfxShaderVariation a, GfxShaderVariation b)
        {
            // VariationConstantBuffer
            GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantBufferClass aConstant =
                a.ShaderVariationDefinition.PixelShaderVariationDefinition.VariationConstantBuffer;
            GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantBufferClass bConstant =
                b.ShaderVariationDefinition.PixelShaderVariationDefinition.VariationConstantBuffer;
            if (aConstant.name != bConstant.name)
            {
                return false;
            }

            // VariationConstantDefinitionArray
            GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantDefinitionArrayClass.VariationConstantDefinitionClass[] aVariationArray =
                a.ShaderVariationDefinition.PixelShaderVariationDefinition.VariationConstantDefinitionArray.VariationConstantDefinition;
            GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantDefinitionArrayClass.VariationConstantDefinitionClass[] bVariationArray =
                b.ShaderVariationDefinition.PixelShaderVariationDefinition.VariationConstantDefinitionArray.VariationConstantDefinition;

            if(aVariationArray.Length != bVariationArray.Length)
            {
                return false;
            }

            int num = aVariationArray.Length;
            for (int i = 0; i < num; i++)
            {
                if (aVariationArray[i].index != bVariationArray[i].index ||
                    aVariationArray[i].name != bVariationArray[i].name ||
                    aVariationArray[i].type != bVariationArray[i].type)
                {
                    return false;
                }
            }
            return true;
        }

        static GfxShaderVariation Merge(GfxShaderVariation a, GfxShaderVariation b)
        {
            // ShaderVariationDefinition は一致していなければならない
            {
                if (a.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray.PreprocessorDefinitionDefinition.Length !=
                    b.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray.PreprocessorDefinitionDefinition.Length)
                {
                    return null;
                }
                int num = a.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray.PreprocessorDefinitionDefinition.Length;
                for (int i = 0; i < num; i++)
                {
                    if (a.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray.PreprocessorDefinitionDefinition[i].index !=
                        b.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray.PreprocessorDefinitionDefinition[i].index)
                    {
                        return null;
                    }
                    if (a.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray.PreprocessorDefinitionDefinition[i].name !=
                        b.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray.PreprocessorDefinitionDefinition[i].name)
                    {
                        return null;
                    }
                }
            }

            // 詳細コンバイナ用のパリエーション定義は一致していなければならない
            if (!CompareVariationConstantDefinition(a, b))
            {
                return null;
            }

            List<string> variationList = new List<string>();

            // a のバリエーションを辞書に登録
            foreach (GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass variationValue in a.ShaderVariationValueArray.ShaderVariationValue)
            {
                string s = VariationValueToString(variationValue);
                if (s == null)
                {
                    return null;
                }
                // 既に含まれていればスキップ
                if (variationList.Contains(s))
                {
                    continue;
                }
                variationList.Add(s);
            }

            // b のバリエーションを辞書に登録
            foreach (GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass variationValue in b.ShaderVariationValueArray.ShaderVariationValue)
            {
                string s = VariationValueToString(variationValue);
                if (s == null)
                {
                    return null;
                }
                // 既に含まれていればスキップ
                if (variationList.Contains(s))
                {
                    continue;
                }
                variationList.Add(s);
            }

            string[] variations = variationList.ToArray();

            // バリエーションを再構築
            {
                int existence_length = variations.Length;
                a.ShaderVariationValueArray.length = existence_length;
                a.ShaderVariationValueArray.ShaderVariationValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass[existence_length];
                for (int i = 0; i < existence_length; i++)
                {
                    int[] values;
                    {
                        string[] strings = variations[i].Split(',');
                        values = new int[strings.Length];
                        for (int j = 0; j < strings.Length; j++)
                        {
                            values[j] = int.Parse(strings[j]);
                        }
                    }
                    GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass valueArray = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass();
                    valueArray.length = ShaderPreprocessorMacroCount;
                    valueArray.PreprocessorDefinitionValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass.PreprocessorDefinitionValueClass[ShaderPreprocessorMacroCount];
                    for (int j = 0; j < ShaderPreprocessorMacroCount; j++)
                    {
                        valueArray.PreprocessorDefinitionValue[j] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass.PreprocessorDefinitionValueClass();
                        valueArray.PreprocessorDefinitionValue[j].index = j;
                        valueArray.PreprocessorDefinitionValue[j].value = values[j];
                    }
                    a.ShaderVariationValueArray.ShaderVariationValue[i] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass();
                    a.ShaderVariationValueArray.ShaderVariationValue[i].index = i;
                    a.ShaderVariationValueArray.ShaderVariationValue[i].VertexShaderVariationValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VertexShaderVariationValueClass();
                    a.ShaderVariationValueArray.ShaderVariationValue[i].VertexShaderVariationValue.PreprocessorDefinitionValueArray = valueArray;
                    a.ShaderVariationValueArray.ShaderVariationValue[i].PixelShaderVariationValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PixelShaderVariationValueClass();
                    a.ShaderVariationValueArray.ShaderVariationValue[i].PixelShaderVariationValue.PreprocessorDefinitionValueArray = valueArray;

                    // 詳細コンバイナ用バリエーション
                    {
                        bool isDetailedCombinerVariationTableCode = values[ShaderVariationKey0] == DetailedCombinerVariationTableCode;
                        bool isCombinerUserShaderVariationTableCode = values[ShaderVariationKey0] == CombinerUserShaderVariationTableCode;
                        GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass detailedCombinerValueArray = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass();
                        detailedCombinerValueArray.length = 7;
                        detailedCombinerValueArray.VariationConstantValueArray = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass[detailedCombinerValueArray.length];

                        // ステージ数
                        detailedCombinerValueArray.VariationConstantValueArray[0] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass();
                        detailedCombinerValueArray.VariationConstantValueArray[0].index = 0;
                        detailedCombinerValueArray.VariationConstantValueArray[0].value = isDetailedCombinerVariationTableCode ? String.Format("{0}", values[ShaderPreprocessorMacroCount]) : "0";

                        // ステージ
                        for (int j = 0; j < detailedCombinerValueArray.length - 1; j++)
                        {
                            // glsl 内のバリエーション定数の値です。
                            // 利用されるステージ数、及び ivec4 の出力を行います。
                            detailedCombinerValueArray.VariationConstantValueArray[1 + j] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass();
                            detailedCombinerValueArray.VariationConstantValueArray[1 + j].index = 1 + j;
                            if (isDetailedCombinerVariationTableCode ||
                                isCombinerUserShaderVariationTableCode)
                            {
                                int baseIndex = (ShaderPreprocessorMacroCount+ 1) + j * 4;
                                detailedCombinerValueArray.VariationConstantValueArray[1 + j].value = String.Format("{0} {1} {2} {3}",
                                    values[baseIndex + 0],
                                    values[baseIndex + 1],
                                    values[baseIndex + 2],
                                    values[baseIndex + 3]);
                            }
                            else
                            {
                                // ダミー
                                detailedCombinerValueArray.VariationConstantValueArray[1 + j].value = String.Format("{0} {1} {2} {3}",
                                    0,
                                    0,
                                    0,
                                    0);
                            }
                        }
                        a.ShaderVariationValueArray.ShaderVariationValue[i].PixelShaderVariationValue.VariationConstantValueArray = detailedCombinerValueArray;
                    }
                }
            }

            return a;
        }

        static string VariationValueToString(GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass variationValue)
        {
            // VertexShaderVariationValue と PixelShaderVariationValue は一致していなければならない
            if (variationValue.VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue.Length !=
                variationValue.PixelShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue.Length)
            {
                return null;
            }

            int num = variationValue.VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue.Length;
            string s = "";
            for (int i = 0; i < num; i++)
            {
                var valVS = variationValue.VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[i];
                var valPS = variationValue.VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[i];
                Debug.Assert(valVS != null && valPS != null);

                if (valVS.value != valPS.value)
                {
                    return null;
                }
                if (i != 0)
                {
                    s += ",";
                }
                s += valVS.value.ToString();
            }

            // 詳細コンバイナ用のキーコード追加（バリエーション変数を扱う詳細コンバイナの場合は全て '110' が入力されます）
            int key0 = variationValue.PixelShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[ShaderVariationKey0].value;
            if (key0 == DetailedCombinerVariationTableCode||
                key0 == CombinerUserShaderVariationTableCode )
            {
                const int firstDetailedCombinerStageNum = 1;        // '0' 番目は利用ステージ数
                int varietionValue_num = variationValue.PixelShaderVariationValue.VariationConstantValueArray.VariationConstantValueArray.Length;

                s += ",";
                s += variationValue.PixelShaderVariationValue.VariationConstantValueArray.VariationConstantValueArray[0].value;

                for (int i = 0; i < varietionValue_num - 1; i++)
                {
                    // ２つ目以降のバリエーション変数に詳細コンバイナ用のビットフィールド値が４づつ６ステージ分入っている為、数値化する
                    string[] strArray = variationValue.PixelShaderVariationValue.VariationConstantValueArray.VariationConstantValueArray[firstDetailedCombinerStageNum + i].value.Split(' ');
                    for(int j = 0; j < strArray.Count(); j++)
                    {
                        s += ",";
                        s += strArray[j];
                    }
                }
            }

            return s;
        }

        /// <summary>
        /// 子プロセスの標準出力に書き込まれた内容をコンソールに出力
        /// </summary>
        /// <param name="state"></param>
        static void ConsoleOutFromChildStdOut(object state)
        {
            StreamReader stdout = (StreamReader)state;

            string line;
            while (null != (line = stdout.ReadLine()))
            {
                Debug.WriteLine(line);
            }
        }

        /// <summary>
        /// 外部プロセスのエラーを読む
        /// </summary>
        public class ProccesErrorReader
        {
            public readonly StreamReader stderr;
            public readonly EventWaitHandle waitHandle = new EventWaitHandle(false, EventResetMode.ManualReset);
            public volatile string errStr;

            /// <summary>
            /// 構築
            /// </summary>
            public ProccesErrorReader(StreamReader procStdError)
            {
                this.stderr = procStdError;
                ThreadPool.QueueUserWorkItem(new WaitCallback(GenerateStdErrString_), null);
            }

            /// <summary>
            ///
            /// </summary>
            public string ReadErrorMsg()
            {
                bool isErrMsg = this.waitHandle.WaitOne(5000, false);
                return isErrMsg ? this.errStr : "";
            }

            /// <summary>
            /// コンバータの標準エラー出力を文字列として取得。
            /// </summary>
            void GenerateStdErrString_(object state)
            {
                StringBuilder sb = new StringBuilder();
                try
                {
                    string line;
                    while (null != (line = this.stderr.ReadLine()))
                    {
                        sb.AppendLine(line);
                    }
                }
                catch (Exception ex)
                {
                    // ログの書き出し失敗
                    Debug.WriteLine(ex.ToString());
                }

                this.errStr = sb.ToString();

                if (!this.waitHandle.Set())
                {
                    Debug.WriteLine("StdErr EventHandle fail \"Set()\"");
                }
            }
        }

        static string GetArgForShaderCacheDir_(string shaderCachePath)
        {
            return !string.IsNullOrEmpty(shaderCachePath) ? string.Format(" --shader-cache-directory=\"{0}\"", shaderCachePath) : "";
        }

        static bool CallShaderConverter(string outputPath, string variationPath, string apiTypeName, string codeTypeName, string shaderCachePath)
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            System.Diagnostics.ProcessStartInfo info = new System.Diagnostics.ProcessStartInfo();
            string path = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetEntryAssembly().Location);
            info.FileName = path + "\\..\\GraphicsTools\\ShaderConverter.exe";
            if (!System.IO.File.Exists(info.FileName))
            {
                info.FileName = path + "\\..\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\GraphicsTools\\ShaderConverter.exe";
            }
            string includeDirectoryWithCombinerUserShader = Path.GetDirectoryName(outputPath);

            info.Arguments =
                " --vertex-shader=\"" + path + "\\shaders\\ui2d_BuildinVertexShader.glsl\"" +
                " --pixel-shader=\"" + path + "\\shaders\\ui2d_BuildinPixelShader.glsl\"" +
                " --include-directory=\"" + path + "\\shaders\"" +
                " --include-directory=\"" + includeDirectoryWithCombinerUserShader + "\"" +
                " --variation=\"" + variationPath + "\"" +
                " -o \"" + outputPath + "\"" +
                " -sGlsl" +
                " -a" + apiTypeName +
                " -c" + codeTypeName +
                (codeTypeName != "Source" ? " --reflection" : "") +
                " --glsl-version=430" +
                GetArgForShaderCacheDir_(shaderCachePath) +
               " --silent" +
                " --preprocess" +
                " --preprocessor-definition NEED_330_EXTENSION=0";
            if (apiTypeName == "Vk")
            {
                info.Arguments += " --preprocessor-definition NN_GFX_VULKAN=1";
            }

            info.UseShellExecute = false;
            info.CreateNoWindow = true;
            info.RedirectStandardError = true;
            info.StandardErrorEncoding = Encoding.Default;
            info.RedirectStandardOutput = true;
            info.StandardOutputEncoding = Encoding.Default;
            process.StartInfo = info;
            StringBuilder errorBuilder = new StringBuilder();
            process.OutputDataReceived += new DataReceivedEventHandler((s, e) => { errorBuilder.Append((e != null && e.Data != null) ? e.Data : ""); });
            process.Start();
            ProccesErrorReader procErrorReader = new ProccesErrorReader(process.StandardError);
            process.WaitForExit();
            string error = procErrorReader.ReadErrorMsg();
            if (error != "")
            {
                Console.WriteLine(error);
                return false;
            }
            return true;
        }

        static bool CreateVariationTable(string outputPath, GfxShaderVariation variation)
        {
            int num = variation.ShaderVariationValueArray.length;
            List<ushort> table = new List<ushort>();
            table.Add((ushort)(num / 3)); // 先頭にバリエーション数を記録
            // 頂点カラー有効・無効、メッシュ用のバリエーションがセットになっているため 3 つずつ飛ばしながら記録する
            for (int i = 0; i < num; i += 3)
            {
                int key0 = variation.ShaderVariationValueArray.ShaderVariationValue[i].VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[ShaderVariationKey0].value;
                int key1 = variation.ShaderVariationValueArray.ShaderVariationValue[i].VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[ShaderVariationKey1].value;
                table.Add((ushort)key0);
                table.Add((ushort)key1);
                table.Add((ushort)i);
            }
            try
            {
                System.IO.BinaryWriter file = new System.IO.BinaryWriter(new System.IO.FileStream(outputPath, System.IO.FileMode.Create));
                foreach (ushort n in table)
                {
                    file.Write(n);
                }
                file.Close();
            }
            catch
            {
                return false;
            }
            return true;
        }

        static bool CreateIndividualVariationTable(string outputPath, GfxShaderVariation variation, int individualCode)
        {
            int num = variation.ShaderVariationValueArray.length;
            List<uint> table = new List<uint>();
            table.Add((uint)(num / 3)); // 先頭にバリエーション数を記録
            bool useIndividualCode = false;
            // 頂点カラー有効・無効、メッシュ用のバリエーションがセットになっているため 3 つずつ飛ばしながら記録する
            for (int i = 0; i < num; i += 3)
            {
                int key0 = variation.ShaderVariationValueArray.ShaderVariationValue[i].VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[ShaderVariationKey0].value;
                if (key0 == individualCode)
                {
                    useIndividualCode = true;
                    table.Add((uint)(6 * 4));   // 利用する値の数
                    table.Add((uint)i);         // インデックス

                    uint[] variationConstantArray = GetVariationConstantStageArray(variation.ShaderVariationValueArray.ShaderVariationValue[i].PixelShaderVariationValue.VariationConstantValueArray.VariationConstantValueArray);
                    variationConstantArray = variationConstantArray.Skip(1).ToArray(); // ステージ数は除外します。
                    foreach(var code in variationConstantArray)
                    {
                        table.Add(code);
                    }
                }
                else
                {
                    // 詳細コンバイナではないテーブルは '0' を入れる
                    table.Add((ushort)0);
                }
            }

            // 詳細コンバイナが利用されていない場合は出力をしない
            if (!useIndividualCode)
            {
                return true;
            }

            string outputTablePath = MakeIndividualArchiveShaderVariationTableName(outputPath, individualCode, 0);
            try
            {
                System.IO.BinaryWriter file = new System.IO.BinaryWriter(new System.IO.FileStream(outputTablePath, System.IO.FileMode.Create));
                foreach (uint n in table)
                {
                    file.Write(n);
                }
                file.Close();
            }
            catch
            {
                return false;
            }
            return true;
        }

        static string MakeIndividualArchiveShaderVariationTableName(string outputPath, int variationKey0, int variationKey1)
        {
            return outputPath + "\\__ArchiveShader_" + variationKey0.ToString() + ".bushvt";
        }

        static void CreateCombinerVariationFile(string outputPath, List<string> glslFiles, string fileName)
        {
            if(glslFiles.Count() <= 0)
            {
                return;
            }

            Encoding sjisEnc = Encoding.GetEncoding("utf-8");
            string outputFileName = Path.Combine(outputPath, fileName);
            StreamWriter writer = new StreamWriter(outputFileName, false, sjisEnc);

            for (int i = 0; i < glslFiles.Count(); i++)
            {
                if (i == 0)
                {
                    writer.WriteLine("#if NW_COMBINERUSERSHADER_TYPE == {0}", i + 1);
                }
                else
                {
                    writer.WriteLine("#elif NW_COMBINERUSERSHADER_TYPE == {0}", i + 1);
                }
                writer.WriteLine("#include \"{0}\"", glslFiles[i]);
            }
            writer.WriteLine("#endif");

            writer.Close();
        }

        /// <summary>
        /// バリエーション定数を int 型の配列で得ます。
        /// </summary>
        /// <param name="variationConstantValueArray"></param>
        /// <returns></returns>
        /// <remarks>
        /// int の配列で要素数は 25 の固定です。並び順はステージ数 ＋ （６ステージ×４要素）になります。
        /// </remarks>
        static uint[] GetVariationConstantStageArray(GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass[] variationConstantValueArray)
        {
            const int stageVecCount = 1 + 24; // ステージ数 ＋ （６ステージ×４要素）
            uint[] stageArray = new uint[stageVecCount];
            int varietionValue_num = variationConstantValueArray.Length;
            Debug.Assert(varietionValue_num == 7, "incorrect number of elements"); // 要素数が正しくありません。

            // １つ目はステージ数が入っているため、数値化する。
            string strInt = variationConstantValueArray[0].value;
            uint stageCount = uint.Parse(strInt);
            stageArray[0] = stageCount;

            const int variationConstantIVec4StartIndex = 1;
            for (int i = variationConstantIVec4StartIndex, count = 0; i < varietionValue_num; i++, count++)
            {
                // ２つ目以降のバリエーション変数に詳細コンバイナ用のビットフィールド値が４づつ６ステージ分入っている為、数値化する。
                string strIvec4 = variationConstantValueArray[i].value;
                string[] strArray = strIvec4.Split(' ');
                Debug.Assert(strArray.Count() == 4, "incorrect number of array");
                uint[] bitArray = strArray.Select(s => uint.Parse(s)).ToArray();
                bitArray.CopyTo(stageArray, variationConstantIVec4StartIndex + (count * 4));
            }
            return stageArray;
        }

        /// <summary>
        /// バリエーションテーブル内のコンバイナユーザーシェーダのプリプロセッサ定義を変更します。
        /// </summary>
        /// <param name="variation"></param>
        /// <param name="glslFiles"></param>
        static void ChangeVariationTableWithCombinerVariationFiles(GfxShaderVariation variation, List<string> glslFiles)
        {
            foreach (GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass variationValue in variation.ShaderVariationValueArray.ShaderVariationValue)
            {
                int key0 = variationValue.PixelShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[ShaderVariationKey0].value;
                if (key0 == CombinerUserShaderVariationTableCode)
                {
                    uint[] variationConstantArray = GetVariationConstantStageArray(variationValue.PixelShaderVariationValue.VariationConstantValueArray.VariationConstantValueArray);
                    variationConstantArray = variationConstantArray.Skip(1).ToArray(); // ステージ数は除外します。

                    string fileName = "";
                    foreach(uint code in variationConstantArray)
                    {
                        fileName += System.Text.Encoding.ASCII.GetString(BitConverter.GetBytes(code));
                    }
                    // '\0' の要素が文字列に複数入る事がある為、必要のない要素を除きます。
                    int eofIndex = fileName.IndexOf('\0');
                    if (eofIndex != -1)
                    {
                        fileName = fileName.Substring(0, eofIndex);
                    }

                    // 拡張子を抜いた、ファイル名で比較を行います。
                    string[] glslFilesWithoutExt = glslFiles.Select(s => Path.GetFileNameWithoutExtension(s)).ToArray();
                    int index = Array.IndexOf(glslFilesWithoutExt, fileName);
                    Debug.Assert(index != -1, "\"" + fileName.ToString() + ".glsl\" file does not exist");

                    // NW_COMBINERUSERSHADER_TYPE の値を反映します。
                    Debug.Assert(variationValue.PixelShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue.Length >= 10, "format is irregularity");
                    const int combinerUserShaderVariationFirstCount = 1;
                    const int combinerUserShaderType = 7;
                    variationValue.PixelShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[combinerUserShaderType].value = index + combinerUserShaderVariationFirstCount;
                    variationValue.VertexShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[combinerUserShaderType].value = index + combinerUserShaderVariationFirstCount;
                }
            }
        }

        /// <summary>
        /// 特定の key0 を持つシェーダーがピクセルシェーダバリエーションテーブルに含まれるかを調べる。
        /// </summary>
        /// <param name="variation"></param>
        /// <param name="shaderKey0"></param>
        /// <returns></returns>
        static bool IsShaderKeyExistInPixelShaderVariationTable(GfxShaderVariation variation, int shaderKey0)
        {
            foreach (GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass variationValue in variation.ShaderVariationValueArray.ShaderVariationValue)
            {
                if (shaderKey0 == variationValue.PixelShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[ShaderVariationKey0].value)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// ピクセルシェーダバリエーションテーブルからコンバイナユーザーシェーダ用の glsl ファイルリストを得る。
        /// </summary>
        static List<string> GetCombinerUserShaderGlslFilesFromPixelShaderVariationTable(GfxShaderVariation variation)
        {
            List<string> glslFiles = new List<string>();

            foreach (GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass variationValue in variation.ShaderVariationValueArray.ShaderVariationValue)
            {
                int shaderKey0 = variationValue.PixelShaderVariationValue.PreprocessorDefinitionValueArray.PreprocessorDefinitionValue[ShaderVariationKey0].value;
                if (shaderKey0 == CombinerUserShaderVariationTableCode)
                {
                    uint[] variationConstantArray = GetVariationConstantStageArray(variationValue.PixelShaderVariationValue.VariationConstantValueArray.VariationConstantValueArray);
                    variationConstantArray = variationConstantArray.Skip(1).ToArray(); // ステージ数は除外します。

                    string fileName = "";
                    foreach(uint code in variationConstantArray)
                    {
                        fileName += System.Text.Encoding.ASCII.GetString(BitConverter.GetBytes(code));
                    }
                    int eofIndex = fileName.IndexOf('\0');
                    if (eofIndex != -1)
                    {
                        // '\0' の要素が文字列に複数入る事がある為、必要のない要素を除きます。
                        fileName = fileName.Substring(0, eofIndex);
                    }
                    fileName += ".glsl";
                    if (!glslFiles.Contains(fileName))
                    {
                        glslFiles.Add(fileName);
                    }
                }
            }
            return glslFiles;
        }

    }
}
