﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using LayoutEditor.PluginSample.Properties;
using LayoutEditor.Plugin;

namespace LayoutEditor.PluginSample
{
    /// <summary>
    /// ボタンプラグインのサンプルです。
    /// </summary>
    public class ButtonPluginSample : ButtonPluginBase
    {
        /// <summary>
        /// ツールバーに追加するボタンです。
        /// </summary>
        private List<ToolStripItem> buttons;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ButtonPluginSample()
        {
            // LayoutEditorのソリューションでdllのソースコードを開くためのブレークポイント
            // System.Diagnostics.Debugger.Break();
        }

        /// <summary>
        /// ツールバーに追加するボタンを取得します。
        /// </summary>
        public override IEnumerable<ToolStripItem> EnumerateButtons()
        {
            return this.buttons;
        }

        /// <summary>
        /// ボタンプラグインを初期化します。
        /// </summary>
        /// <param name="appInfo">アプリケーション情報</param>
        public override bool Initialize(ButtonPluginParam appInfo)
        {
            base.Initialize(appInfo);

            this.buttons = new List<ToolStripItem>();

            // セパレータを作成
            this.buttons.Add(new ToolStripSeparator());

            // ボタン1を追加
            ToolStripButton button1 = new ToolStripButton();
            button1.AutoSize = false;
            button1.Size = new Size(28, 28);
            button1.Margin = new Padding(1, 0, 0, 1);
            button1.Padding = new Padding(0, 0, 0, 0);
            button1.DisplayStyle = ToolStripItemDisplayStyle.Image;
            button1.ImageScaling = ToolStripItemImageScaling.None;
            button1.Image = Resources.Icon_01;
            button1.ToolTipText = "ボタン1";

            // クリックしたときにメッセージボックスを表示する
            button1.Click += (ss, ee) =>
            {
                this.AppInfo.ViewManager.ShowMessageBox(this.AppInfo.AppForm, "button1", "ButtonPlugin", MessageBoxButtons.OK);
            };

            this.buttons.Add(button1);

            // ボタン2を作成
            ToolStripButton button2 = new ToolStripButton();
            button2.AutoSize = false;
            button2.Size = new Size(28, 28);
            button2.Enabled = false;
            button2.Margin = new Padding(1, 0, 0, 1);
            button2.Padding = new Padding(0, 0, 0, 0);
            button2.DisplayStyle = ToolStripItemDisplayStyle.Image;
            button2.ImageScaling = ToolStripItemImageScaling.None;
            button2.Image = Resources.Icon_02;
            button2.ToolTipText = "ボタン2";

            // クリックしたときにメッセージボックスを表示する
            button2.Click += (ss, ee) =>
            {
                this.AppInfo.ViewManager.ShowMessageBox(this.AppInfo.AppForm, "button2", "ButtonPlugin", MessageBoxButtons.OK);
            };

            // アクティブなレイアウトのペインが存在するときだけボタンを押せるようにする
            LECore.LayoutEditorCore.Scene.OnSceneModify += ( ss, ee ) =>
            {
                if (LECore.LayoutEditorCore.Scene.CurrentISubScene == null)
                {
                    button2.Enabled = false;
                }
                else
                {
                    button2.Enabled = LECore.LayoutEditorCore.Scene.CurrentISubScene.IPaneArray.Any();
                }
            };

            this.buttons.Add(button2);

            // ボタン3を作成
            ToolStripButton button3 = new ToolStripButton();
            button3.AutoSize = false;
            button3.Size = new Size(28, 28);
            button3.Margin = new Padding(1, 0, 0, 1);
            button3.Padding = new Padding(0, 0, 0, 0);
            button3.DisplayStyle = ToolStripItemDisplayStyle.Image;
            button3.ImageScaling = ToolStripItemImageScaling.None;
            button3.Image = Resources.Icon_03;
            button3.ToolTipText = "ボタン3";

            // クリックしたときに新しくレイアウトを作る
            button3.Click += (ss, ee) =>
            {
                var result = LECore.LayoutEditorCore.CreateAsNewFile();
                System.Diagnostics.Debug.Assert(result != null);

                LECore.Structures.ISubScene newScene = result.ISubScene;
                newScene.BackGround.ScreenSize = new LECore.Structures.FVec2(this.AppInfo.ViewManager.AppSetting.LastSetScreenSize);

                LayoutEditor.Forms.ToolWindows.LayoutWindow.LayoutWindow lw = this.AppInfo.ViewManager.AddNewLayoutWindowAsNewDocument(result);
            };

            this.buttons.Add(button3);

            return true;
        }
    }
}
