﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#include <stdio.h>
#include <dshow.h>

#include "MovieToTexture.h"

// qedit.h にかつて存在していた定義
EXTERN_C const IID IID_ISampleGrabberCB;
MIDL_INTERFACE("0579154A-2B53-4994-B0D0-E773148EFF85")
ISampleGrabberCB : public IUnknown
{
public:
    virtual HRESULT STDMETHODCALLTYPE SampleCB(double SampleTime, IMediaSample *pSample) = 0;
    virtual HRESULT STDMETHODCALLTYPE BufferCB(double SampleTime, BYTE *pBuffer, long BufferLen) = 0;
};
EXTERN_C const IID IID_ISampleGrabber;
MIDL_INTERFACE("6B652FFF-11FE-4fce-92AD-0266B5D7C78F")
ISampleGrabber : public IUnknown
{
public:
    virtual HRESULT STDMETHODCALLTYPE SetOneShot(BOOL OneShot) = 0;
    virtual HRESULT STDMETHODCALLTYPE SetMediaType(const AM_MEDIA_TYPE *pType) = 0;
    virtual HRESULT STDMETHODCALLTYPE GetConnectedMediaType(AM_MEDIA_TYPE *pType) = 0;
    virtual HRESULT STDMETHODCALLTYPE SetBufferSamples(BOOL BufferThem) = 0;
    virtual HRESULT STDMETHODCALLTYPE GetCurrentBuffer(long *pBufferSize, long *pBuffer) = 0;
    virtual HRESULT STDMETHODCALLTYPE GetCurrentSample(IMediaSample **ppSample) = 0;
    virtual HRESULT STDMETHODCALLTYPE SetCallback(ISampleGrabberCB *pCallback, long WhichMethodToCallback) = 0;
};
EXTERN_C const CLSID CLSID_SampleGrabber;

// キャプチャ処理を行うクラス
class CaptureCallback : public ISampleGrabberCB
{
public:
    CaptureCallback()
        : m_FrameIndex(0)
        , m_Ref(1)
        , m_BeginFrame(0)
        , m_EndFrame(0)
    {
    }

    // 初期化
    void Initialize(int beginFrame, int endFrame)
    {
        m_BeginFrame = beginFrame;
        m_EndFrame = endFrame;
    }

    // ISampleGrabberCB の実装
    virtual HRESULT STDMETHODCALLTYPE SampleCB(double SampleTime, IMediaSample *pSample)
    {
        return E_NOTIMPL;
    }
    virtual HRESULT STDMETHODCALLTYPE BufferCB(double SampleTime, BYTE *pBuffer, long BufferLen)
    {
        if (m_BeginFrame <= m_FrameIndex && m_FrameIndex <= m_EndFrame)
        {
            ConvertToTga(pBuffer);
        }
        m_FrameIndex++;
        return S_OK;
    }

    // IUnknown の実装
    HRESULT STDMETHODCALLTYPE QueryInterface(REFIID riid, LPVOID* ppv)
    {
        *ppv = NULL;
        if (riid == IID_IUnknown || riid == IID_ISampleGrabberCB)
        {
            *ppv = static_cast<ISampleGrabberCB*>(this);
            AddRef();
            return S_OK;
        }
        return E_NOINTERFACE;
    }
    ULONG STDMETHODCALLTYPE AddRef()
    {
        return ++m_Ref;
    }
    ULONG STDMETHODCALLTYPE Release()
    {
        if (--m_Ref == 0)
        {
            delete this;
            return 0;
        }
        return m_Ref;
    }

    // コールバックを登録するメソッド
    HRESULT SetCallback(ISampleGrabber* pSampleGrabber)
    {
        m_FrameIndex = 0;

        const long whichMethodToCallback = 1; // Call ISampleGrabberCB::BufferCB method.
        return pSampleGrabber->SetCallback(this, whichMethodToCallback);
    }

    // フレームのインデックスを取得するメソッド
    int GetFrameIndex() const
    {
        return m_FrameIndex;
    }

    HRESULT ConvertToTga(BYTE* pData);

private:
    int m_FrameIndex;
    LONG m_Ref;
    int m_BeginFrame;
    int m_EndFrame;
};

// 関数定義
static HRESULT Initialize();
static void Finalize();
static HRESULT Capture();

// 変数定義
static const wchar_t* g_pInputFileName = NULL;
static const wchar_t* g_pOutputDirName = NULL;
static const wchar_t* g_pTextureNamePrefix = NULL;
static bool g_Alpha;
static int g_PaddingNum;
static bool g_UnderscoreEnabled;
static IGraphBuilder* g_pGraphBuilder = NULL;
static IMediaControl* g_pMediaControl = NULL;
static IMediaEventEx* g_pEvent = NULL;
static IBaseFilter* g_pSampleGrabberFilter = NULL;
static ISampleGrabber* g_pSampleGrabber = NULL;
static CaptureCallback g_CaptureCallback;
static BITMAPINFOHEADER g_BmpInfo;

// DLL メイン
BOOL WINAPI DllMain(HINSTANCE, DWORD, LPVOID)
{
    return TRUE;
}

// 動画をテクスチャに変換する関数
extern "C" __declspec(dllexport) int ConvertMovieToTexture(
    int* pFrameCount,
    int* pWidth,
    int* pHeight,
    const wchar_t* pInputFileName,
    const wchar_t* pOutputDirName,
    const wchar_t* pTextureNamePrefix,
    int beginFrame,
    int endFrame,
    bool alpha,
    int paddingNum,
    bool underscoreEnabled)
{
    int result = 1;
    g_pInputFileName = pInputFileName;
    g_pOutputDirName = pOutputDirName;
    g_pTextureNamePrefix = pTextureNamePrefix;
    g_Alpha = alpha;
    g_PaddingNum = paddingNum;
    g_UnderscoreEnabled = underscoreEnabled;
    g_CaptureCallback.Initialize(beginFrame, endFrame);

    HRESULT hr = S_OK;

    if (SUCCEEDED(hr))
    {
        hr = Initialize();
    }
    if (SUCCEEDED(hr))
    {
        hr = Capture();
    }
    if (SUCCEEDED(hr))
    {
        // 動画情報を返す
        *pFrameCount = g_CaptureCallback.GetFrameIndex();
        *pWidth = static_cast<int>(g_BmpInfo.biWidth);
        *pHeight = static_cast<int>(g_BmpInfo.biHeight);
    }
    Finalize();
    return hr;
}

// 初期化処理
static HRESULT Initialize()
{
    HRESULT hr = S_OK;

    if (SUCCEEDED(hr))
    {
        hr = CoInitializeEx(NULL, COINIT_APARTMENTTHREADED | COINIT_DISABLE_OLE1DDE);
    }
    if (SUCCEEDED(hr))
    {
        hr = CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_PPV_ARGS(&g_pGraphBuilder));
    }
    if (SUCCEEDED(hr))
    {
        hr = g_pGraphBuilder->QueryInterface(IID_PPV_ARGS(&g_pMediaControl));
    }
    if (SUCCEEDED(hr))
    {
        hr = g_pGraphBuilder->QueryInterface(IID_PPV_ARGS(&g_pEvent));
    }
    if (SUCCEEDED(hr))
    {
        hr = CoCreateInstance(CLSID_SampleGrabber, NULL, CLSCTX_INPROC_SERVER, IID_PPV_ARGS(&g_pSampleGrabberFilter));
    }
    if (SUCCEEDED(hr))
    {
        hr = g_pGraphBuilder->AddFilter(g_pSampleGrabberFilter, L"Sample Grabber");
    }
    if (SUCCEEDED(hr))
    {
        hr = g_pSampleGrabberFilter->QueryInterface(IID_PPV_ARGS(&g_pSampleGrabber));
    }
    if (SUCCEEDED(hr))
    {
        AM_MEDIA_TYPE mediaType;
        ZeroMemory(&mediaType, sizeof(mediaType));
        mediaType.majortype = MEDIATYPE_Video;
        mediaType.subtype = g_Alpha ? MEDIASUBTYPE_ARGB32 : MEDIASUBTYPE_RGB24;
        hr = g_pSampleGrabber->SetMediaType(&mediaType);
    }
    if (SUCCEEDED(hr))
    {
        BSTR pInputFileName = SysAllocString(g_pInputFileName);
        if (pInputFileName != NULL)
        {
            hr = g_pMediaControl->RenderFile(pInputFileName);
            SysFreeString(pInputFileName);
        }
        else
        {
            hr = E_FAIL;
        }
    }
    if (SUCCEEDED(hr))
    {
        IMediaFilter* pMediaFilter = NULL;
        hr = g_pGraphBuilder->QueryInterface(IID_IMediaFilter, reinterpret_cast<LPVOID*>(&pMediaFilter));
        if (SUCCEEDED(hr))
        {
            hr = pMediaFilter->SetSyncSource(NULL);
        }
        if (pMediaFilter != NULL)
        {
            pMediaFilter->Release();
            pMediaFilter = NULL;
        }
    }
    if (SUCCEEDED(hr))
    {
        AM_MEDIA_TYPE mediaType;
        hr = g_pSampleGrabber->GetConnectedMediaType(&mediaType);
        if (SUCCEEDED(hr))
        {
            if (mediaType.formattype != FORMAT_VideoInfo || mediaType.cbFormat < sizeof(VIDEOINFOHEADER) || mediaType.pbFormat == NULL)
            {
                hr = E_FAIL;
            }
        }
        if (SUCCEEDED(hr))
        {
            g_BmpInfo = reinterpret_cast<VIDEOINFOHEADER*>(mediaType.pbFormat)->bmiHeader;
        }
    }
    if (SUCCEEDED(hr))
    {
        hr = g_pSampleGrabber->SetBufferSamples(FALSE);
    }
    if (SUCCEEDED(hr))
    {
        hr = g_CaptureCallback.SetCallback(g_pSampleGrabber);
    }

    return hr;
}

// 終了処理
static void Finalize()
{
    if (g_pSampleGrabber != NULL)
    {
        g_pSampleGrabber->Release();
        g_pSampleGrabber = NULL;
    }
    if (g_pSampleGrabberFilter != NULL)
    {
        g_pSampleGrabberFilter->Release();
        g_pSampleGrabberFilter = NULL;
    }
    if (g_pEvent != NULL)
    {
        g_pEvent->Release();
        g_pEvent = NULL;
    }
    if (g_pMediaControl != NULL)
    {
        g_pMediaControl->Release();
        g_pMediaControl = NULL;
    }
    if (g_pGraphBuilder != NULL)
    {
        g_pGraphBuilder->Release();
        g_pGraphBuilder = NULL;
    }
    CoUninitialize();
}

// キャプチャ処理
static HRESULT Capture()
{
    HRESULT hr = S_OK;
    if (SUCCEEDED(hr))
    {
        hr = g_pMediaControl->Run();
    }
    if (SUCCEEDED(hr))
    {
        long evCode;
        g_pEvent->WaitForCompletion(INFINITE, &evCode);
    }
    return hr;
}

// TGA への変換処理
HRESULT CaptureCallback::ConvertToTga(BYTE* pData)
{
    // TGA で書き出し
    FILE* pFile;
    {
        wchar_t format[256];
        wchar_t path[MAX_PATH];
        swprintf_s(format, L"%%s\\%%s%s%%0%dd.tga", g_UnderscoreEnabled ? L"_" : L"", g_PaddingNum);
        swprintf_s(path, format, g_pOutputDirName, g_pTextureNamePrefix, m_FrameIndex);
        pFile = _wfopen(path, L"wb");
        if (pFile == NULL)
        {
            return E_FAIL;
        }
    }

    // TGA ヘッダ
    unsigned char header[12] = {
        0x00, // ID フィールド長=0
        0x00, // カラーマップの有無=なし
        0x02, // 画像形式=フルカラー
        0x00, 0x00, // カラーマップの原点
        0x00, 0x00, // カラーマップの長さ
        0x00, // カラーマップエントリーのサイズ
        0x00, 0x00, // 画像の X 座標
        0x00, 0x00, // 画像の Y 座標
    };
    fwrite(header, 1, sizeof(header), pFile);

    // 画像サイズと色深度
    unsigned short width = static_cast<unsigned short>(g_BmpInfo.biWidth);
    unsigned short height = static_cast<unsigned short>(g_BmpInfo.biHeight);
    unsigned char depth = static_cast<char>(g_BmpInfo.biBitCount);
    unsigned char descriptor = 0x08; // デスクリプタ=下から上に格納、アルファチャンネル深度は 8bit
    fwrite(&width, sizeof(width), 1, pFile);
    fwrite(&height, sizeof(height), 1, pFile);
    fwrite(&depth, sizeof(depth), 1, pFile);
    fwrite(&descriptor, sizeof(descriptor), 1, pFile);

    // 画像データ書き出し
    const size_t size = static_cast<size_t>(g_BmpInfo.biWidth * g_BmpInfo.biHeight * (g_BmpInfo.biBitCount / 8));
    fwrite(pData, sizeof(BYTE), size, pFile);

    // TGA フッタ
    unsigned char footer[8] = {
        0x00, 0x00, 0x00, 0x00, // ファイル位置
        0x00, 0x00, 0x00, 0x00, // デベロッパーディレクトリファイル位置
    };
    fwrite(footer, 1, sizeof(footer), pFile);
    char footerText[] = "TRUEVISION-XFILE.";
    fwrite(footerText, 1, sizeof(footerText), pFile);

    fclose(pFile);

    return S_OK;
}
