﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.IO;
using System.Windows.Forms;

namespace LayoutEditor
{
    using Structures.SerializableObject;
    using Forms.ToolWindows;
    using Forms.ToolWindows.AnimSectionTagWindow;
    using Forms.ToolWindows.CurveEditWindow;
    using Forms.ToolWindows.LayoutWindow;
    using Forms.ToolWindows.TextureMgrWindow;
    using Forms.ToolWindows.GroupWindow;
    using Forms.ToolWindows.HierarchyWindow;
    using Forms.ToolWindows.PanePasteControlWindow;
    using Forms.ToolWindows.ColorSetWindow;
    using Forms.ToolWindows.DebugWindow;
    using Forms.ToolWindows.AnimShareWindow;
    using Forms.ToolWindows.PropertyEditWindow;


    using LECore;
    using LECore.Structures;
    using LECore.Manipulator;
    using LECore.Structures.Core;

    using SysForms = System.Windows.Forms;
    using Forms.ToolWindows.PartsPalletWindow;
    using Forms.ToolWindows.PartsWindow;
    using LECore.Structures.Core.Command;
    using LayoutEditor.Forms.ToolWindows.LayoutSettingWindow;
    using System.Xml.Linq;
    using Forms.ToolWindows.StateWindow;


    /// <summary>
    /// 他のビュークラスが、ViewManagerを参照するときに使用するインタフェース
    /// 最小限の情報のみを公開します。
    ///
    /// 各ビュークラスのOwnerフィールドとして利用されます。
    /// </summary>
    public interface IViewManager
    {
        /// <summary>
        /// 選択アイテムの先頭のペインに対して、プロパティダイアログを新規に生成します。
        /// </summary>
        PropertyWindow MakeNewPropertyDlg(bool allowEmpty, bool show);
        LayoutWindow FindLayoutWindowBySubScene(ISubScene subScene);
        void Show(string viewName);

        AppSetting AppSetting { get; }

        AppMessageReporter MessageReporter { get; }

        DialogResult ShowMessageBox(IWin32Window owner, string text, string caption, MessageBoxButtons buttons);

        /// <summary>
        /// LayotWinowがあるか？
        /// </summary>
        bool IsLayoutWindowExsit { get; }

        WeifenLuo.WinFormsUI.Docking.DockPanel DockPanel { get; }
    }

    /// <summary>
    /// シーンからの色々なメッセージを適切なビューへ転送する役割を担います。
    /// シングルトンではありませんが、アプリケーションクラスが唯一の実体を持っています。
    ///
    /// 原則的には、すべてのメッセージは本クラスが受け取り、
    /// 適切なViewクラスに本クラスがディスパッチします。
    ///
    /// ディスパッチといいますが、実際には、
    /// ほとんど、すべてのメッセージは本クラスにおいて、
    /// 単純にダイアログの再描画実行へと再解釈されています。
    ///
    /// 現在はデバックコンソールに、メッセージ受信を出力しています。
    /// 将来は、グラフィカルにメッセージの伝播を確認できる機能が実装されるかもしれません。
    ///
    ///
    /// Gof. Mediatior パターンの Mediatior役に相当するクラスを意図して作成されています。
    /// (各Viewクラスが、colleague役 となります。)
    /// 現状は、インタフェースを使用した抽象化が行われていない状態です（TODO:）。
    ///
    ///
    /// </summary>
    public class ViewManager :
        IViewManager,                        // ViewManagerが実行できる命令を実装します
        ITimeChageEventListener,             // グローバル時間変更イベントの通知を受け取ります。
        ISceneModifyListener,                // シーンの変更イベントを受け取ります
        IAppEventListener                    // アプリケーションイベントリスナ
    {

        #region フィールド
        // メインフォーム：MDI 親ウインドウ
        readonly AppForm _theAppForm;

        //-------------- 各種ビュー
        readonly CurveEditWindow _curveEditWindow;
        readonly CmdQueueDebugWindow _cmdQueueDbgDlg;
        readonly PropertyWindowMgr _propertyWindowMgr;
        readonly TextureMgrWindow _textureMgrWindow;
        readonly GroupWindow _groupWindow;
        readonly HierarchyWindow _hierarchyWindow;
        readonly LayoutSettingWindow _layoutSettingWindow;
        readonly AnimSectionTagWindow _animSectionTagWindow;
        readonly PanePasteControlWindow _panePasteControlWindow;
        readonly PaneSearchWindow _paneSearchWindow;
        readonly ColorSetWindow _colorSetWindow;
        readonly AnimShareWindow _animationShareWindow;
        readonly PartsWindow _partsWindow;
        readonly PartsPalletWindow _partsPalletWindow;
        readonly StateSettingWindow _stateSettingWindow;


        readonly List<LayoutWindow> _layoutViewSet = new List<LayoutWindow>();
        readonly List<ILEToolWindow> _viewSet = new List<ILEToolWindow>();
        readonly List<LEToolWindow> _toolSet = new List<LEToolWindow>();

        // レイアウトウインドウ設定、ウインドウを閉じるタイミングで更新されます。
        readonly List<LEToolFormSetting> _layoutSettingSet = new List<LEToolFormSetting>();

        //-------------- 各種イベントハンドラ
        readonly EventHandler _OnLayoutWindowClosingHandler;
        readonly Action<ViewManagerMessage> _OnSendMessageHandler;

        readonly AppSetting _appSetting;
        readonly AppMessageReporter _messageReporter = null;

        int _currentRedo = 0;
        int _currentUndo = 0;

        private bool UpdateViewerRequested = false;

        #endregion フィールド


        #region レイアウトビューの追加・削除
        public LayoutWindow FindLayoutWindow(string dataFilePath)
        {
            foreach (LayoutWindow lw in _layoutViewSet)
            {
                // 拡張子以外が完全に一致するなら...
                if (Path.GetDirectoryName(lw.FilePath) ==
                    Path.GetDirectoryName(dataFilePath))
                {
                    if (Path.GetFileNameWithoutExtension(lw.FilePath) ==
                        Path.GetFileNameWithoutExtension(dataFilePath))
                    {
                        return lw;
                    }
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        bool IsLayoutToolFormSetting_(LEToolFormSetting formSetting)
        {
            // システム特殊情報を検索します
            string userData;
            if (formSetting.TryToFindUserDataStrByName(USERDATA_FOR_SYSTEM, out userData))
            {
                // LayoutWindow
                if (userData == USERDATA_VAL_FOR_LAYOUT)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        bool TryToLoadLayoutWindowSetting_(LayoutWindow lw, string dataPath)
        {
            foreach (LEToolFormSetting formSetting in _layoutSettingSet)
            {
                // システム特殊情報を検索します
                if (IsLayoutToolFormSetting_(formSetting))
                {
                    var lwSetting = formSetting.FindUserDataAllByName("RlytInputPath").FirstOrDefault((userData) => userData.Value == dataPath);
                    if (lwSetting != null)
                    {
                        lw.StartPosition = System.Windows.Forms.FormStartPosition.Manual;
                        lw.LoadSetting(formSetting, new LoadSettingOption { AlsoLoadOtherThanWorkspace = true });

                        // 最小化された状態で開いてもしょうがない
                        // メインウインドウ表示前に Minimized で Show すると D3DRenderer の初期化に失敗する
                        if (lw.WindowState == FormWindowState.Minimized)
                        {
                            lw.WindowState = FormWindowState.Normal;
                        }
                        return true;
                    }
                }
            }

            return false;
        }

        internal enum LoadResult
        {
            Loaded,
            Failed,
            Skipped,
        }

        /// <summary>
        /// ファイルを読み込みます。
        /// ただしく読み込めれば、trueを返します。
        /// </summary>
        internal LoadResult LoadFileAndRegisterLayoutWindow(
            string inputPath, LayoutEditorCore.LoadOption option)
        {
            LayoutWindow lw = this.FindLayoutWindow(inputPath);
            if (lw != null)
            {
                lw.Focus();
                return LoadResult.Skipped;
            }
            else
            {
                var result = LayoutEditorCore.LoadLayoutFile(inputPath, option);
                if (result != null)
                {
                    AddNewLayoutWindow(result);
                    return LoadResult.Loaded;
                }
            }
            return LoadResult.Failed;
        }

        /// <summary>
        /// 対象になるサブシーンを指定して、新規にレイアウトビューを追加します。
        /// </summary>
        LayoutWindow AddNewLayoutWindow_(LayoutDocument lytDoc)
        {
            LayoutWindow newLayoutWindow = new LayoutWindow(AppSetting);

            newLayoutWindow.SendMessageToViewManager += _OnSendMessageHandler;

            // 仮のサブシーンセットを作成します。

            // 新規作成したViewの対象として設定します。
            newLayoutWindow.AddTargetSubScene(lytDoc);
            //newLayoutWindow.MdiParent = _theAppForm;
            newLayoutWindow.ViewManager = this;
            newLayoutWindow.Closed += this._OnLayoutWindowClosingHandler;

            _layoutViewSet.Add(newLayoutWindow);
            _viewSet.Add(newLayoutWindow);

            _theAppForm.AddLayoutWindowMenuItem(newLayoutWindow);

            return newLayoutWindow;
        }

        /// <summary>
        /// 対象になるサブシーンを指定して、新規にレイアウトビューを追加します。
        /// </summary>
        public LayoutWindow AddNewLayoutWindowAsNewDocument(LayoutDocument lytDoc)
        {
            LayoutWindow lw = AddNewLayoutWindow_(lytDoc);

            // 表示
            lw.SetViewDefalut();
            if (_theAppForm.DockPanel.DocumentStyle == WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi)
            {
                lw.MdiParent = _theAppForm;
                lw.Show();
            }
            else
            {
                lw.Show(_theAppForm.DockPanel, WeifenLuo.WinFormsUI.Docking.DockState.Document);
            }

            return lw;
        }

        /// <summary>
        /// 対象になるサブシーンを指定して、新規にレイアウトビューを追加します。
        /// </summary>
        public LayoutWindow AddNewLayoutWindow(LayoutDocument lytDoc)
        {
            ISubScene targetSubScene = lytDoc.ISubScene;

            string dataFilePath = lytDoc.LastSavedFilePath;
            Debug.Assert(targetSubScene != null);
            Debug.Assert(dataFilePath != null && dataFilePath != string.Empty);
            Debug.Assert(Path.IsPathRooted(dataFilePath));

            if (dataFilePath != null && dataFilePath != string.Empty)
            {
                LayoutWindow lw = this.FindLayoutWindow(dataFilePath);
                if (lw == null)
                {
                    if (Path.GetExtension(dataFilePath).CompareTo(AppConstants.LayoutFileExt) != 0)
                    {
                        dataFilePath = Path.ChangeExtension(dataFilePath, AppConstants.LayoutFileExt);
                    }

                    // サブシーンを表示するViewをViewマネージャに登録します。
                    lw = this.AddNewLayoutWindow_(lytDoc);

                    // 設定を読み込みます。
                    bool settingLoaded = TryToLoadLayoutWindowSetting_(lw, dataFilePath);

                    if (!settingLoaded)
                    {
                        // 表示
                        lw.SetViewDefalut();
                    }

                    // MDI クライアントに収まるようにサイズを調整(最小化時は処理しません)
                    Size clientSize = _theAppForm.GetMdiCliantSize();
                    if (_theAppForm.WindowState != FormWindowState.Minimized && _theAppForm.IsLoaded)
                    {
                        Size boundedSize = new Size(
                            Math.Max(lw.MinimumSize.Width, Math.Min((int)(clientSize.Width * 0.9f), lw.Size.Width)),
                            Math.Max(lw.MinimumSize.Height, Math.Min((int)(clientSize.Height * 0.9f), lw.Size.Height)));

                        bool sizeAdjusted = lw.Size != boundedSize;
                        if (sizeAdjusted)
                        {
                            lw.Size = boundedSize;
                            lw.SetViewCenter();
                        }
                    }
                    else
                    {
                        Size boundedSize = new Size(
                            Math.Max(lw.MinimumSize.Width, lw.Size.Width),
                            Math.Max(lw.MinimumSize.Height, lw.Size.Height));

                        bool sizeAdjusted = lw.Size != boundedSize;
                        if (sizeAdjusted)
                        {
                            lw.Size = boundedSize;
                            lw.SetViewCenter();
                        }
                    }

                    // 表示
                    if (_theAppForm.DockPanel.DocumentStyle == WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi)
                    {
                        lw.MdiParent = _theAppForm;
                        lw.Show();
                    }
                    else
                    {
                        lw.Show(_theAppForm.DockPanel, WeifenLuo.WinFormsUI.Docking.DockState.Document);
                    }

                    // 位置が収まるように（表示のあとに調整する）
                    lw.Location = new Point(
                        lw.Right > clientSize.Width ? 0 : lw.Location.X,
                        lw.Bottom > clientSize.Height ? 0 : lw.Location.Y);

                }
                return lw;
            }
            return null;
        }


        /// <summary>
        /// 指定したサブシーンがいづれかの、
        /// ビューから参照されているか確認します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <returns></returns>
        bool CheckSubSceneIsReferencedBySomeView_(ISubScene subScene)
        {
            foreach (LayoutWindow ld in _layoutViewSet)
            {
                if (ld.Targets(subScene))
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 参照されていないサブシーンをシーンから取り除きます。
        /// </summary>
        /// <param name="removedReference"></param>
        void RemoveUnReferencedSubSceneFromScene_(ISubScene[] removedReference)
        {
            SceneManipulator sceneMnp = new SceneManipulator();
            sceneMnp.BindTarget(_CoreScene);

            foreach (ISubScene removedScene in removedReference)
            {
                if (!CheckSubSceneIsReferencedBySomeView_(removedScene))
                {
                    // すでに、どのビューからも参照されていないサブシーンが発見されたので、
                    // 消去します。
                    sceneMnp.RemoveSubScene(removedScene);
                }
                else
                {
                    // 他のビューから参照されることはないはず
                    Debug.Assert(false);
                }
            }
        }

        /// <summary>
        /// レイアウトビューを削除します。
        /// </summary>
        public void RemoveLayoutWindow(LayoutWindow layoutWindow)
        {
            Debug.Assert(_layoutViewSet.Contains(layoutWindow));
            Debug.Assert(_viewSet.Contains(layoutWindow));

            layoutWindow.Hide();
            layoutWindow.Closed -= this._OnLayoutWindowClosingHandler;

            // SystemMdi 以外のときのみ Close をよぶ。SystemMdi のときは閉じられるので大丈夫
            if (DockPanel.DocumentStyle != WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi)
            {
                layoutWindow.MdiParent = null;
                layoutWindow.Close();
            }

            _layoutViewSet.Remove(layoutWindow);
            _viewSet.Remove(layoutWindow);

            RemoveUnReferencedSubSceneFromScene_(layoutWindow.LoadedSubScenes.ToArray());

            _theAppForm.RemoveUnusedLayoutWindowMenuItem();

            if (_layoutViewSet.Count <= 0) { GC.Collect(0); GC.Collect(1); GC.Collect(2); }
        }

        /// <summary>
        /// フォーム閉じイベントハンドラ
        /// LayoutWindowが閉じられたときに呼び出されます。
        /// </summary>
        void OnLayoutWindowClosingHandler_(object sender, EventArgs args)
        {
            LayoutWindow lw = sender as LayoutWindow;

            /// 再生中のアニメーションを停止します。
            _curveEditWindow.Stop();

            //--------------------------------------------------------------
            // 設定データを保存します。
            // すでに情報が存在している場合は、古いデータを消去します。
            _layoutSettingSet.RemoveAll(delegate (LEToolFormSetting setting)
           {
               return lw.IsSelfSetting(setting);
           });

            // 新規作成
            LEToolFormSetting lws = new LEToolFormSetting(lw, false);
            lws.AddUserData(USERDATA_FOR_SYSTEM, USERDATA_VAL_FOR_LAYOUT);
            _layoutSettingSet.Add(lws);


            // 登録を削除します。
            RemoveLayoutWindow(lw);
        }


        #endregion レイアウトビューの追加・削除

        #region -------------------- プロパティ
        IScene _CoreScene
        {
            get { return LayoutEditorCore.Scene; }
        }

        /// <summary>
        /// ビューセット
        /// (
        /// TODO:独自のインタフェースを作る
        /// プラグインのロード時にViewセット情報の初期化をおこなう。
        /// )
        /// </summary>
        public ILEToolWindow[] ToolWindowSet
        {
            get { return _viewSet.ToArray(); }
        }

        /// <summary>
        /// 可視変更可能ウインドウ
        /// </summary>
        public ILEToolWindow[] VisibleToolWindowSet
        {
            get
            {
                return _viewSet.FindAll(
                    delegate (ILEToolWindow wnd)
                    {
                        return wnd.AllowToChangeVisible;
                    }).ToArray();
            }
        }

        /// <summary>
        /// アプリケーション共通の設定を取得します。
        /// </summary>
        public AppSetting AppSetting
        {
            get { return _appSetting; }
        }

        /// <summary>
        /// メッセージ表示
        /// </summary>
        public AppMessageReporter MessageReporter
        {
            get { return _messageReporter; }
        }

        #endregion -------------------- プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ViewManager(AppForm theAppForm, AppSetting setting, AppMessageReporter messageReporter)
        {
            // -------------- ハンドラの初期化
            _OnLayoutWindowClosingHandler = OnLayoutWindowClosingHandler_;

            _OnSendMessageHandler = OnSendMessageToViewManagerHandler_;

            _appSetting = setting;

            // -------------- GUIに関連する、静的な設定を行います。
            Controls.CompatibilityAlertBox.InitializeGetVisiblityFlagHandler(() => _appSetting.ShowCompatibilityAlertIcon);

            // -------------- メインフォームを設定します。
            _theAppForm = theAppForm;
            _theAppForm.SendMessageToViewManager += _OnSendMessageHandler;

            // TODO: 経路がショートカット気味なので別の手段を使う
            _theAppForm.PlatformChanged += (s, e) =>
                OnSceneModifyHandler(s, new SceneModifyEventArgs(SceneModifyEventArgs.Kind.PlatformChanged, null));

            // -------------------------------------------------------------------------------------------------------
            // 注意: ツールウインドウの増減や順番の入れ替えを行ったら、AppForm.InitToolbar_() も更新すること
            // -------------------------------------------------------------------------------------------------------

            // -------------- アニメーションカーブエディタ(暫定版)
            _curveEditWindow = new CurveEditWindow();
            _curveEditWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_curveEditWindow);

            // -------------- テクスチャマネージャダイアログ
            _textureMgrWindow = new TextureMgrWindow();
            _textureMgrWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_textureMgrWindow);

            // -------------- グループダイアログ
            _groupWindow = new GroupWindow();
            _groupWindow.SendMessageToViewManager += _OnSendMessageHandler;
            _groupWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_groupWindow);

            // -------------- 親子階層ダイアログ
            _hierarchyWindow = new HierarchyWindow();
            _hierarchyWindow.SendMessageToViewManager += _OnSendMessageHandler;
            _hierarchyWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_hierarchyWindow);

            // -------------- ペイン貼り付けコントロール
            _panePasteControlWindow = new PanePasteControlWindow();
            _panePasteControlWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_panePasteControlWindow);

            // -------------- フレーム区間タグ編集コントロール
            _animSectionTagWindow = new AnimSectionTagWindow();
            _animSectionTagWindow.Activated += Event_Form_Activated;
            _animSectionTagWindow.SendMessageToViewManager += _OnSendMessageHandler;
            _viewSet.Add(_animSectionTagWindow);

            // -------------- ペイン検索
            _paneSearchWindow = new PaneSearchWindow();
            _paneSearchWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_paneSearchWindow);

            // -------------- プロパティダイアログマネージャ
            _propertyWindowMgr = new PropertyWindowMgr(_theAppForm, _OnSendMessageHandler);
            _viewSet.Add(_propertyWindowMgr);

            // -------------- レイアウト設定ウインドウ
            _layoutSettingWindow = new LayoutSettingWindow();
            _layoutSettingWindow.Activated += Event_Form_Activated;
            _layoutSettingWindow.SendMessageToViewManager += _OnSendMessageHandler;
            _viewSet.Add(_layoutSettingWindow);

            // -------------- 色見本ウインドウ
            _colorSetWindow = new ColorSetWindow();
            _colorSetWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_colorSetWindow);

            // -------------- アニメーション共有情報ウインドウ
            _animationShareWindow = new AnimShareWindow(setting);
            _animationShareWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_animationShareWindow);

            // -------------- 部品設定ウインドウ
            _partsWindow = new PartsWindow();
            _partsWindow.Activated += Event_Form_Activated;
            _viewSet.Add(_partsWindow);
            _partsWindow.IsMultiControlMode = AppSetting.ProjectSettings.MultiControlEnabled;

            // -------------- 部品パレットウインドウ
            _partsPalletWindow = new PartsPalletWindow();
            _partsPalletWindow.Activated += Event_Form_Activated;
            _partsPalletWindow.SendMessageToViewManager += _OnSendMessageHandler;
            _viewSet.Add(_partsPalletWindow);

            // -------------- ステート設定ウインドウ
            _stateSettingWindow = new StateSettingWindow();
            _stateSettingWindow.Activated += Event_Form_Activated;
            _stateSettingWindow.SendMessageToViewManager += _OnSendMessageHandler;
            _viewSet.Add(_stateSettingWindow);

            // 参照の設定。
            _viewSet.ForEach((view) => view.ViewManager = this);

            // ------------- 時間変更イベントリスナ登録
            // 時間変更イベント、時間範囲変更イベントリスナに登録
            GlobalTime.Inst.OnTimeChanged += OnTimeChangedHandler;

            // シーン変更ハンドラを登録します。
            _CoreScene.OnSceneModify += OnSceneModifyHandler;

            // アプリケーション設定変更イベントハンドラを設定します。
            _appSetting.Modified += OnAppSettingModifiedHandler_;

            // -------------- アプリケーション設定を設定します。
            // 複数のシーンを生成しシーン変更イベントを
            // 処理する必要があるため、イベントハンドラ設定後に実行します。
            LoadSetting_(setting.ViewManagerSetting);

            // -------------- アンドゥーコマンドキュー・デバック用のダイアログ
            _cmdQueueDbgDlg = new CmdQueueDebugWindow();

            // 転送関係の設定
            ViewerExecuter.ErrorOnViewerThread += ViewerExecuter_ErrorOnViewerThread;
            Application.Idle += Application_Idle;
#if DEBUG
            //_cmdQueueDbgDlg.Owner = _theAppForm;
            _cmdQueueDbgDlg.Activated += Event_Form_Activated;
            _viewSet.Add(_cmdQueueDbgDlg);
            _cmdQueueDbgDlg.Show();
#endif
            foreach (var view in _viewSet)
            {
                view.OwnerCmdKeyAcceptor = _theAppForm;
            }

            foreach (var view in _viewSet.OfType<WeifenLuo.WinFormsUI.Docking.DockContent>())
            {
                // ウインドウを隠したときにメインウインドウが他のウインドウに隠れないようにする
                // この現象はDockPanel.DocumentStyle にも依存しているかもしれない
                view.DockStateChanged += (s, e) =>
                {
                    if (view.DockState == WeifenLuo.WinFormsUI.Docking.DockState.Hidden)
                    {
                        if (!_theAppForm.IsDisposed)
                        {
                            _theAppForm.Activate();
                        }
                    }
                };

            }
        }

        private void ViewerExecuter_ErrorOnViewerThread(object sender, EventArgs e)
        {
            if (_theAppForm.IsHandleCreated)
            {
                _theAppForm.Invoke((Action)(() => {
                    ViewerExecuter.CancelPreview();
                    _theAppForm.AutoUpdateViewer = false;
                }));
            }
        }

        private void Application_Idle(object sender, EventArgs e)
        {
            // UI の処理を重くしないためにアイドル時に転送処理を開始する
            if (UpdateViewerRequested && AppSetting.AutoUpdateViewer)
            {
                DbgConsole.WriteLine("UpdateRequested");
                foreach (LayoutWindow layoutView in _layoutViewSet)
                {
                    if (layoutView.CurrentSubScene == LECore.LayoutEditorCore.Scene.CurrentISubScene)
                    {
                        layoutView.RunPreviewer(
                            AppSetting.SendViewerAnimation ?
                            ViewerPreviewParam.PreviewMode.Animation :
                            ViewerPreviewParam.PreviewMode.Control,
                            true);
                    }
                }
                UpdateViewerRequested = false;
            }
        }

        public void RunPreviewer(ViewerPreviewParam.PreviewMode mode)
        {
            foreach (LayoutWindow layoutView in _layoutViewSet)
            {
                if (layoutView.CurrentSubScene == LECore.LayoutEditorCore.Scene.CurrentISubScene)
                {
                    layoutView.RunPreviewer(mode, false);
                }
            }
            UpdateViewerRequested = false;
        }

        #region アプリケーション設定変更
        /// <summary>
        /// アプリケーション設定変更イベントハンドラ
        /// </summary>
        void OnAppSettingModifiedHandler_(object sender, EventArgs args)
        {
            // すべてのレイアウトビューを再描画します。
            // UpdateAllLayoutWindow_();
            AppEventArgs appArgs = new AppEventArgs(AppEventKind.AppSettingChanged);
            foreach (ILEToolWindow viewWnd in _viewSet)
            {
                viewWnd.OnAppEvent(sender, appArgs);
            }

            // ビューアへの更新通知
            UpdateViewerRequested = true;
        }

        #endregion アプリケーション設定変更

        #region 設定保存、読み込み関連

        /// <summary>
        /// 設定を保存します。
        /// </summary>
        /// <param name="viewManagerSetting"></param>
        /// <param name="name"></param>
        public ViewManagerSetting SaveSettingCommon(ViewManagerSetting viewManagerSetting, string name)
        {
            ViewManagerSetting setting = new ViewManagerSetting(viewManagerSetting);

            // アプリケーションフォームの情報を保存します。
            if (_theAppForm.WindowState != SysForms.FormWindowState.Minimized)
            {
                setting.AppFormLocation = _theAppForm.DesktopLocation;
                setting.AppFormSize = _theAppForm.Size;
            }

            // その他の設定を保存します。
            {
                var viewManagerSettings = new LEToolFormSetting();
                viewManagerSettings.Name = name;
                viewManagerSettings.AddUserData("LastFileIODirectoryName", SaveLoadUIHelper.LastFileIODirectoryName);

                setting.ToolFormSettings.Add(viewManagerSettings);
            }



            return setting;
        }

        /// <summary>
        /// 設定を保存します。
        /// </summary>
        /// <param name="setting"></param>
        public ViewManagerSetting SaveSetting()
        {
            var setting = SaveSettingCommon(this.AppSetting.ViewManagerSetting, "ViewManager");
            setting.WindowZOrder.Clear();

            // すべてのツールビューの設定を保存します。
            foreach (ILEToolWindow toolView in ToolWindowSet)
            {
                setting.ToolFormSettings.Add(new LEToolFormSetting(toolView, false));
            }

            // 最近使ったファイルの最大数 + 10 個のレイアウトウインドウ設定を追加します。
            foreach (var lws in _layoutSettingSet.Skip(_layoutSettingSet.Count - (AppSetting.MaxRecentlyUsedFiles + 10)))
            {
                setting.ToolFormSettings.Add(lws);
            }

            // 現在の設定を上書き
            setting.SetDockPanelSetting(setting.WorkspaceName, DockPanelToXml());

            return setting;
        }

        /// <summary>
        /// ユーザー設定を保存します。
        /// </summary>
        /// <param name="setting"></param>
        public ViewManagerSetting SaveUserSetting()
        {
            // スナップ設定をViewManagerSettingから読み込む。
            // SnapToolWindowsはViewManagerSettingに設定値が保存されているため、SaveUserSettingを行う際に
            // ViewManagerUserSetting側にコピーする必要があります。
            this.AppSetting.ViewManagerUserSetting.SnapToolWindows = this.AppSetting.ViewManagerSetting.SnapToolWindows;

            var setting = SaveSettingCommon(this.AppSetting.ViewManagerUserSetting, "UserViewManager");
            setting.WindowZOrder.Clear();
            AppSetting.ViewManagerUserSetting.WindowZOrder.Clear();

            // LayoutWindow以外のツールビューの設定を保存します。
            foreach (var toolView in ToolWindowSet)
            {
                if (!(toolView is LayoutWindow))
                {
                    setting.ToolFormSettings.Add(new LEToolFormSetting(toolView, true));
                }
            }

            // 描画順を追加
            foreach (var ts in _toolSet)
            {
                setting.WindowZOrder.Add(ts.Name);
            }

            setting.DockPanelSettings = AppSetting.ViewManagerSetting.DockPanelSettings.Select(x => new ViewManagerSetting.DockPanelSetting()
            {
                id = x.id,
                label = x.label,
                DockPanel = x.DockPanel,
            }).ToList();
            setting.WorkspaceName = AppSetting.ViewManagerSetting.WorkspaceName;
            setting.SetDockPanelSetting(setting.WorkspaceName, DockPanelToXml());

            return setting;
        }


        /// <summary>
        /// ツールフォームの設定読み込みを試みます。
        /// </summary>
        LEToolFormSetting FindLoadToolFormSetting_(ViewManagerSetting setting, LEToolWindow form)
        {
            foreach (LEToolFormSetting formSetting in setting.ToolFormSettings)
            {
                // 自分の設定を発見したら読み込みます。
                if (formSetting.Name == form.Name)
                {
                    return formSetting;
                }
            }
            return null;
        }

        public const string USERDATA_FOR_SYSTEM = "SYSTEM_DATA";
        public const string USERDATA_VAL_FOR_LAYOUT = "LayoutWindowSetting";

        /// <summary>
        /// 設定を読み込みます。
        /// </summary>
        public void LoadSettingCommon(ViewManagerSetting setting, string name, bool visible, bool moveProperty, bool loadWorkspace)
        {
            Debug.Assert(setting != null);

            ILEToolWindow[] toolWindowSet = this.ToolWindowSet;

            //string[] recentlyUsedFileSet = SaveLoadUIHelper.GetRecentlyUsedFiles(10);

            // その他の設定を読み込みます。
            {
                var viewManagerSettings = setting.ToolFormSettings.Find((item) => item.Name == name);
                if (viewManagerSettings != null)
                {
                    string result;
                    if (viewManagerSettings.TryToFindUserDataStrByName("LastFileIODirectoryName", out result))
                    {
                        SaveLoadUIHelper.LastFileIODirectoryName = result;
                    }
                }
            }

            // すべてのツールビューの設定を読み込みます。
            foreach (LEToolFormSetting formSetting in setting.ToolFormSettings)
            {
                if (IsLayoutToolFormSetting_(formSetting))
                {
                    if (!loadWorkspace)
                    {
                        foreach (var inputFile in formSetting.FindUserDataAllByName("RlytInputPath"))
                        {
                            if (File.Exists(inputFile.Value))
                            {
                                _layoutSettingSet.Add(formSetting);
                            }
                        }
                    }
                }
                else
                {
                    // 通常の処理
                    foreach (ILEToolWindow wnd in toolWindowSet)
                    {
                        if (!moveProperty)
                        {
                            var propertyWindowMgr = wnd as PropertyWindowMgr;
                            if (propertyWindowMgr != null)
                            {
                                continue;
                            }
                        }

                        if (formSetting.Name == wnd.LEWindowName)
                        {
                            wnd.LoadSetting(
                                formSetting,
                                new LoadSettingOption() { AlsoLoadOtherThanWorkspace = !loadWorkspace });
                        }
                    }
                }
            }

            var dockPanelSetting = setting.DockPanelSettings.FirstOrDefault(x => x.id == setting.WorkspaceName);
            LoadDockPanels(dockPanelSetting?.DockPanel ?? GetDefaultDockPanelSetting(setting.WorkspaceName));
        }

        /// <summary>
        /// ツールウインドウの設定とドキュメントウインドウの設定をマージする
        /// </summary>
        private string MergeSettingXml(string tool, string document)
        {
            DbgConsole.WriteLine(tool);
            DbgConsole.WriteLine(document);
            var toolElem = XElement.Parse(tool);
            var docElem = XElement.Parse(document);

            #region ツールウインドウ
            // toolElem の Contents から LayoutWindow の項目を削除
            {
                var removeContents = from element in toolElem.Element("Contents").Elements()
                                     where element.Attribute("PersistString").Value.StartsWith(typeof(LayoutWindow).FullName)
                                     select element;
                removeContents.Remove();
            }

            // Content の ID を置換
            var toolContentsTable = new Dictionary<string, string>();
            foreach (var element in toolElem.Element("Contents").Elements())
            {
                var oldID = element.Attribute("ID").Value;
                var newID = $"{toolContentsTable.Count}";
                toolContentsTable.Add(oldID, newID);
                element.SetAttributeValue("ID", newID);
            }

            // Panes から Document の項目を削除
            {
                var removePanes = from element in toolElem.Element("Panes").Elements()
                                  where element.Attribute("DockState").Value == "Document"
                                  select element;
                removePanes.Remove();
            }

            // Pane の ActiveContent を置換
            foreach (var pane in toolElem.Elements("Panes").Elements("Pane"))
            {
                string newValue;
                if (toolContentsTable.TryGetValue(pane.Attribute("ActiveContent").Value, out newValue))
                {
                    pane.SetAttributeValue("ActiveContent", newValue);
                }
            }

            // Content の RefID を置換
            foreach (var content in toolElem.Element("Panes").Descendants("Content"))
            {
                string newValue;
                if (toolContentsTable.TryGetValue(content.Attribute("RefID").Value, out newValue))
                {
                    content.SetAttributeValue("RefID", newValue);
                }
            }

            // Pane の ID を置換
            var toolPaneContentsTable = new Dictionary<string, string>();
            foreach (var element in toolElem.Element("Panes").Elements())
            {
                var oldID = element.Attribute("ID").Value;
                var newID = $"{toolPaneContentsTable.Count}";
                toolPaneContentsTable.Add(oldID, newID);
                element.SetAttributeValue("ID", newID);
            }

            // DockWindow 内の RefID を置換
            foreach (var pane in toolElem.Element("DockWindows").Descendants("Pane"))
            {
                string newValue;
                if (toolPaneContentsTable.TryGetValue(pane.Attribute("RefID").Value, out newValue))
                {
                    pane.SetAttributeValue("RefID", newValue);
                }

                if (toolPaneContentsTable.TryGetValue(pane.Attribute("PrevPane").Value, out newValue))
                {
                    pane.SetAttributeValue("PrevPane", newValue);
                }
            }

            // FloatWindow 内の RefID を置換
            foreach (var pane in toolElem.Element("FloatWindows").Descendants("Pane"))
            {
                string newValue;
                if (toolPaneContentsTable.TryGetValue(pane.Attribute("RefID").Value, out newValue))
                {
                    pane.SetAttributeValue("RefID", newValue);
                }

                if (toolPaneContentsTable.TryGetValue(pane.Attribute("PrevPane").Value, out newValue))
                {
                    pane.SetAttributeValue("PrevPane", newValue);
                }
            }
            #endregion

            #region ドキュメントウインドウ
            // docElem の Contents から LayoutWindow の項目を削除
            {
                var removeContents = from element in docElem.Element("Contents").Elements()
                                     where !element.Attribute("PersistString").Value.StartsWith(typeof(LayoutWindow).FullName)
                                     select element;
                removeContents.Remove();
            }

            // Content の ID を置換
            var docContentsTable = new Dictionary<string, string>();
            foreach (var element in docElem.Element("Contents").Elements())
            {
                var oldID = element.Attribute("ID").Value;
                var newID = $"{toolContentsTable.Count + docContentsTable.Count}";
                docContentsTable.Add(oldID, newID);
                element.SetAttributeValue("ID", newID);
            }

            // Panes から Document 以外の項目を削除
            {
                var removePanes = from element in docElem.Element("Panes").Elements()
                                  where element.Attribute("DockState").Value != "Document"
                                  select element;
                removePanes.Remove();
            }

            // Pane の ActiveContent を置換
            foreach (var pane in docElem.Elements("Panes").Elements("Pane"))
            {
                string newValue;
                if (docContentsTable.TryGetValue(pane.Attribute("ActiveContent").Value, out newValue))
                {
                    pane.SetAttributeValue("ActiveContent", newValue);
                }
            }

            // Content の RefID を置換
            foreach (var content in docElem.Element("Panes").Descendants("Content"))
            {
                string newValue;
                if (docContentsTable.TryGetValue(content.Attribute("RefID").Value, out newValue))
                {
                    content.SetAttributeValue("RefID", newValue);
                }
            }

            // Pane の ID を置換
            var docPaneContentsTable = new Dictionary<string, string>();
            foreach (var element in docElem.Element("Panes").Elements())
            {
                var oldID = element.Attribute("ID").Value;
                var newID = $"{toolPaneContentsTable.Count + docPaneContentsTable.Count}";
                docPaneContentsTable.Add(oldID, newID);
                element.SetAttributeValue("ID", newID);
            }

            var docDockWindow = docElem.Element("DockWindows").Elements("DockWindow").First(x => x.Attribute("DockState").Value == "Document");

            // DockWindow 内の RefID, PrevPane を置換
            foreach (var pane in docDockWindow.Descendants("Pane"))
            {
                string newValue;
                if (docPaneContentsTable.TryGetValue(pane.Attribute("RefID").Value, out newValue))
                {
                    pane.SetAttributeValue("RefID", newValue);
                }

                if (docPaneContentsTable.TryGetValue(pane.Attribute("PrevPane").Value, out newValue))
                {
                    pane.SetAttributeValue("PrevPane", newValue);
                }
            }
            #endregion

            #region マージ
            // Content の追加
            var toolContents = toolElem.Element("Contents");
            foreach (var content in docElem.Element("Contents").Elements())
            {
                toolContents.Add(content);
            }
            toolContents.SetAttributeValue("Count", toolContents.Elements().Count());

            // Pane の追加
            var toolPanes = toolElem.Element("Panes");
            foreach (var pane in docElem.Elements("Panes").Elements())
            {
                toolPanes.Add(pane);
            }
            toolPanes.SetAttributeValue("Count", toolPanes.Elements().Count());


            // DockWindow を置換
            var toolDockWindow = toolElem.Element("DockWindows").Elements("DockWindow").First(x => x.Attribute("DockState").Value == "Document");
            toolDockWindow.ReplaceNodes(docDockWindow.Element("NestedPanes"));
            #endregion

            DbgConsole.WriteLine(toolElem.ToString());
            return toolElem.ToString();
        }

        /// <summary>
        /// ドッキングウインドウの配置切り替え
        /// </summary>
        public void LoadDockPanels(string settings)
        {
            DockPanel.SuspendLayout(true);

            // ウインドウの状態を初期化
            string beforeClearXml;
            var layoutWindows = ClearDockPanel(out beforeClearXml);

            // 設定をマージ
            settings = MergeSettingXml(settings, beforeClearXml);

            // 復元
            if (!string.IsNullOrEmpty(settings))
            {
                try
                {
                    using (MemoryStream stream =
                        new MemoryStream(System.Text.Encoding.UTF8.GetBytes(settings)))
                    {
                        DockPanel.LoadFromXml(stream, persistString =>
                        {
                            foreach (LEToolWindow toolWindow in ToolWindowSet.OfType<LEToolWindow>())
                            {
                                if (persistString == toolWindow.GetType().FullName)
                                {
                                    return toolWindow;
                                }
                            }

                            if (persistString == typeof(PropertyWindow).FullName)
                            {
                                return MakeNewPropertyDlg(true, false);
                            }

                            LayoutWindow layoutWindow;
                            if (layoutWindows.TryGetValue(persistString, out layoutWindow))
                            {
                                return layoutWindow;
                            }

                            return null;
                        }
                        );
                    }
                }
                catch (Exception e)
                {
                    Debug.Assert(false, e.ToString());

                    ShowMessageBox(this._theAppForm, StringResMgr.Get("ERROR_LOADDOCKPANEL"), _theAppForm.Text, MessageBoxButtons.OK);

                    // TODO: 安全な状態にする

                    // 開くだけ開く
                    foreach (var layoutWindow in layoutWindows.OrderBy(x => x.Key).Select(x => x.Value))
                    {
                        layoutWindow.Show(DockPanel);
                    }
                }
            }

            DockPanel.ResumeLayout(true, true);
        }

        public void ChangeDockPanelMode(string mode)
        {
            var setting = AppSetting.ViewManagerSetting;
            setting.SetDockPanelSetting(AppSetting.ViewManagerSetting.WorkspaceName, DockPanelToXml());
            setting.WorkspaceName = mode;
            var newSetting = setting.DockPanelSettings.FirstOrDefault(x => x.id == setting.WorkspaceName);
            var dockPanelSetting = newSetting?.DockPanel ?? GetDefaultDockPanelSetting(setting.WorkspaceName);

            LoadDockPanels(dockPanelSetting);
        }

        private string GetDefaultDockPanelSetting(string mode)
        {
            switch (mode)
            {
                case ViewManagerSetting.DockPanelModeLayout:
                    return Properties.Resources.DockPanel_Layout;
                case ViewManagerSetting.DockPanelModeAnimation:
                    return Properties.Resources.DockPanel_Animation;
                case ViewManagerSetting.DockPanelModelParts:
                    return Properties.Resources.DockPanel_Parts;
            }

            return null;
        }

        /// <summary>
        /// 初期状態を復元
        /// </summary>
        public void LoadDefaultDockPanelSetting()
        {
            // デフォルトの状態にする
            AppSetting.ViewManagerSetting.WorkspaceName = ViewManagerSetting.DockPanelModeLayout;
            LoadDockPanels(GetDefaultDockPanelSetting(ViewManagerSetting.DockPanelModeLayout));

            // 空にする
            AppSetting.ViewManagerSetting.DockPanelSettings = new List<ViewManagerSetting.DockPanelSetting>();
        }

        public string DockPanelToXml()
        {
            using (var stream = new MemoryStream())
            {
                DockPanel.SaveAsXml(stream, System.Text.Encoding.UTF8);

                var xml = System.Text.Encoding.UTF8.GetString(stream.ToArray());
                if (xml.Length > 0 && xml[0] == (char)0xFEFF)
                {
                    xml = xml.Substring(1);
                }

                return xml;
            }
        }

        /// <summary>
        /// 設定を読み込みます。
        /// </summary>
        public void LoadSetting_(ViewManagerSetting setting)
        {
            Debug.Assert(setting != null);

            LoadSettingCommon(setting, "ViewManager", true, true, false);
        }

        /// <summary>
        /// ユーザー設定を読み込みます。
        /// </summary>
        public void LoadUserSetting_(ViewManagerSetting setting)
        {
            Debug.Assert(setting != null);

            LoadSettingCommon(setting, "ViewManagerUser", false, false, true);

            AppSetting.ViewManagerSetting.WorkspaceName = setting.WorkspaceName;
            AppSetting.ViewManagerSetting.DockPanelSettings = setting.DockPanelSettings.Select(x =>
            new ViewManagerSetting.DockPanelSetting()
            {
                id = x.id,
                label = x.label,
                DockPanel = x.DockPanel

            }).ToList();
        }


        /// <summary>
        /// 設定データ読み込み後に行う初期化処理です。
        /// </summary>
        public void InitializeAfterLoadSetting()
        {
            foreach (var toolWindow in ToolWindowSet.OfType<IInitializeAfterLoadSetting>())
            {
                toolWindow.InitializeAfterLoadSetting();
            }
        }


        #endregion 設定保存、読み込み関連

        #region IViewManagerMessageSender からのメッセージへの応答
        /// <summary>
        /// ToolWindow からのメッセージに対応します。
        /// </summary>
        private void OnSendMessageToViewManagerHandler_(ViewManagerMessage args)
        {
            if (args != null)
            {
                switch (args.MessageKind)
                {
                    case ViewManagerMessageKind.FindLayoutPathBySubScene:
                        {
                            foreach (LayoutWindow lw in _layoutViewSet)
                            {
                                if (object.ReferenceEquals(lw.CurrentSubScene, args.MessageParam))
                                {
                                    args.Result = lw.FilePath;
                                }
                            }
                        }
                        break;
                    case ViewManagerMessageKind.ShowProperty:
                        {
                            // プロパティウインドウを表示します。
                            this.MakeNewPropertyDlg(true, true);
                            break;
                        }
                    case ViewManagerMessageKind.ShowCurveEditor:
                        {
                            if (!_curveEditWindow.Visible)
                            {
                                Show(_curveEditWindow.Text);
                            }
                            _curveEditWindow.Activate();
                            _curveEditWindow.ShowTargetAttributeCurve(args.MessageParam as IAnmAttribute[]);
                            break;
                        }
                    case ViewManagerMessageKind.OpenLayout:
                        {

                            if (LoadFileAndRegisterLayoutWindow(
                                args.MessageParam as string,
                                LayoutEditorCore.LoadOption.TryToOpenRlan) == LoadResult.Failed)
                            {
                                LayoutEditorCore.MsgReporter.ReportError(
                                    StringResMgr.Get("ERROR_FAILED_TO_LOAD_TITLE"),
                                    StringResMgr.Get("ERROR_FAILED_TO_LOAD", args.MessageParam as string)
                                    );
                            }
                            break;
                        }
                    case ViewManagerMessageKind.ForceCloseLayout:
                        {
                            LayoutWindow lw = this.FindLayoutWindow(args.MessageParam as string);
                            lw.CloseIgnoreSaving();
                            break;
                        }
                    case ViewManagerMessageKind.ForceCloseLayoutAll:
                        {
                            List<LayoutWindow> currentLytWnd = new List<LayoutWindow>(_layoutViewSet);
                            foreach (LayoutWindow lw in currentLytWnd)
                            {
                                lw.CloseIgnoreSaving();
                            }
                            break;
                        }
                    case ViewManagerMessageKind.CreateDerivativeLayout:
                        {
                            var inputPath = args.MessageParam as string;
                            var option = LayoutEditorCore.LoadOption.TryToOpenRlan | LayoutEditorCore.LoadOption.LoadAsDerivativeBase;

                            var result = LayoutEditorCore.LoadLayoutFileAsNewFile(inputPath, option);
                            if (result != null)
                            {
                                AddNewLayoutWindowAsNewDocument(result);
                            }

                            break;
                        }
                    case ViewManagerMessageKind.ReloadPartsAll:
                        {
                            foreach (LayoutWindow layoutView in _layoutViewSet)
                            {
                                ISubScene subScene = layoutView.InitialSubScene;
                                if (subScene != null)
                                {
                                    SubSceneHelper.ReloadAllPartsPanes(subScene);
                                }
                            }

                            break;
                        }
                    case ViewManagerMessageKind.WriteMessage:
                        {
                            if (args.MessageParam is string)
                            {
                                _theAppForm.WriteMessage(args.MessageParam as string);
                            }
                            break;
                        }
                    case ViewManagerMessageKind.WriteWarning:
                        {
                            if (args.MessageParam is string)
                            {
                                _theAppForm.WriteWarning(args.MessageParam as string);
                            }
                            break;
                        }
                    case ViewManagerMessageKind.SaveStateChanged:
                        {
                            _layoutSettingWindow.UpdateLayoutFilePath((LayoutWindow.TargetSubSceneSaveState)args.MessageParam);
                            _theAppForm.UpdateText();
                            _textureMgrWindow.UpdateState();
                            break;
                        }
                    case ViewManagerMessageKind.SendViewer:
                        {
                            RunPreviewer(AppSetting.SendViewerAnimation ? ViewerPreviewParam.PreviewMode.Animation : ViewerPreviewParam.PreviewMode.Control);
                            break;
                        }
                    case ViewManagerMessageKind.CreateKeyFrameOnTargetCurve:
                        {
                            // カーブエディタが表示されているときだけキーを打つ
                            if (_curveEditWindow.Visible)
                            {
                                _curveEditWindow.CreateKeyFrameOnTargetCurve();
                            }
                            break;
                        }
                    case ViewManagerMessageKind.UpdateWindowTitle:
                        _theAppForm.UpdateText();
                        break;
                    case ViewManagerMessageKind.PreviewCurveInTargetTag:
                        {
                            if (!_curveEditWindow.Visible)
                            {
                                Show(_curveEditWindow.Text);
                            }
                            _curveEditWindow.PreviewKeyAll(args.MessageParam);
                        }
                        break;
                    case ViewManagerMessageKind.UpdatePanePasteControlWindow:
                        {
                            if (_panePasteControlWindow.Visible)
                            {
                                _panePasteControlWindow.UpdateProperty();
                            }
                        }
                        break;
                    case ViewManagerMessageKind.DuplicatePanes:
                        {
                            var lw = GetActiveLayoutWindow_();
                            lw.DuplicateSelectedPanes(args.MessageParam as IPane);
                        }
                        break;
                    case ViewManagerMessageKind.KeyCmd:
                        {
                            var lw = GetActiveLayoutWindow_();

                            KeyCmdViewManagerMessage keyCmdArgs = args as KeyCmdViewManagerMessage;
                            Keys key = keyCmdArgs.KeyData;
                            switch (key)
                            {
                                case Keys.A:
                                    {
                                        lw.SetViewDefalut();
                                    }
                                    break;
                                case Keys.F:
                                    {
                                        lw.SetViewFocus();
                                    }
                                    break;
                                case Keys.Shift | Keys.P:
                                    {
                                        _hierarchyWindow.ReleaseHierarchy();
                                    }
                                    break;
                                case Keys.P:
                                    {
                                        _hierarchyWindow.SetHierarchy();
                                    }
                                    break;
                                case Keys.G:
                                    {
                                        lw.AddPaneAsNullPaneChild(keyCmdArgs.PaneSet as IEnumerable<IPane>);
                                    }
                                    break;
                                case Keys.Up:
                                case Keys.Down:
                                case Keys.Left:
                                case Keys.Right:
                                case Keys.Shift | Keys.Up:
                                case Keys.Shift | Keys.Down:
                                case Keys.Shift | Keys.Left:
                                case Keys.Shift | Keys.Right:
                                    {
                                        lw.MainView.Event_MainView_KeyDown(this, new KeyEventArgs(key));
                                    }
                                    break;
                            }
                        }
                        break;
                    case ViewManagerMessageKind.ProjectSettingChanged:
                        _textureMgrWindow.UpdateState();
                        break;
                    default:
                        Debug.Assert(false);
                        break;
                }
            }
            else
            {
                // 不正な引数によるメッセージです。
                Debug.Assert(false);
            }
        }

        #endregion ToolWindowからのメッセージへの応答

        #region ツールウインドウ表示・非表示切り替え
        /// <summary>
        /// 隠されたツールをもとの状態にもどします。
        /// </summary>
        public void RestoreLocked()
        {
            // TODO:いずれは本関数のみになる。
        }

        /// <summary>
        /// ビューの名前を指定して、表示状態を設定します。
        /// </summary>
        public void Show(string viewName)
        {
            ILEToolWindow[] views = this.VisibleToolWindowSet;
            foreach (ILEToolWindow view in views)
            {
                if (view.LEWindowName == viewName)
                {
                    if (view is LEToolWindow)
                    {
                        var toolWindow = (LEToolWindow)view;
                        if (toolWindow.DockPanel == null)
                        {
                            Size size = ((LEToolWindow)view).Size;
                            var tmp = DockPanel.DefaultFloatWindowSize;
                            DockPanel.DefaultFloatWindowSize = size;
                            toolWindow.Show(DockPanel);
                            DockPanel.DefaultFloatWindowSize = tmp;
                            if (toolWindow.FloatPane == null)
                            {
                                toolWindow.FloatPane = DockPanel.Theme.Extender.DockPaneFactory.CreateDockPane(
                                    toolWindow,
                                    WeifenLuo.WinFormsUI.Docking.DockState.Float, false);
                            }
                            toolWindow.FloatPane.FloatWindow.ClientSize = size;
                        }
                        else
                        {
                            toolWindow.Show();
                        }
                    }
                    else if (view == _propertyWindowMgr)
                    {
                        _propertyWindowMgr.ShowAll();
                    }
                }
            }
        }
        #endregion ツールウインドウ表示・非表示切り替え

        #region IViewManagerメンバ

        /// <summary>
        /// アクティブなLayoutWindowを取得します。
        /// </summary>
        public LayoutWindow GetActiveLayoutWindow_()
        {
            if (DockPanel.DocumentStyle == WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi)
            {
                return _theAppForm.ActiveMdiChild as LayoutWindow;
            }
            else
            {
                return DockPanel.ActiveDocument as LayoutWindow;
            }
        }

        /// <summary>
        /// 選択アイテムの先頭のペインに対して、プロパティダイアログを新規に生成します。
        /// </summary>
        public PropertyWindow MakeNewPropertyDlg(bool allowEmpty, bool show)
        {
            // 選択アイテムの先頭のペインに対して、プロパティダイアログを新規に生成します。
            IScene globalScene = LECore.LayoutEditorCore.Scene;
            IPane[] panes = globalScene.CurrentISubScene != null ? globalScene.CurrentISubScene.ISelectedSet.IPaneArray : new IPane[0];
            if (panes.Length != 0 || allowEmpty)
            {
                LayoutWindow activeView = GetActiveLayoutWindow_();
                if (activeView != null || allowEmpty)
                {
                    return _propertyWindowMgr.RegisterNewDialog(panes, show);
                }
            }

            return null;
        }

        /// <summary>
        /// 指定したサブシーンを表示するLayotWinowを検索します。
        /// </summary>
        public LayoutWindow FindLayoutWindowBySubScene(ISubScene subScene)
        {
            foreach (LayoutWindow lWnd in _layoutViewSet)
            {
                if (lWnd.Targets(subScene))
                {
                    return lWnd;
                }
            }
            return null;
        }

        /// <summary>
        /// LayotWinowがあるか？
        /// </summary>
        public bool IsLayoutWindowExsit
        {
            get { return _layoutViewSet.Count > 0; }
        }

        /// <summary>
        /// メッセージボックスを表示する。
        /// </summary>
        public DialogResult ShowMessageBox(IWin32Window owner, string text, string caption, MessageBoxButtons buttons)
        {
            // TODO : ダメなときはダミーの Box を返す。
            return MessageBox.Show(owner, text, caption, buttons);
        }

        public WeifenLuo.WinFormsUI.Docking.DockPanel DockPanel { get { return _theAppForm.DockPanel; } }

        #endregion IViewManagerメンバ

        #region ITimeChageEventListener メンバ

        /// <summary>
        /// デバックメッセージを表示します。
        /// </summary>
        void ReportDebugMessage_( int time, TimeChageEventType type )
        {
            switch( type )
            {
                case TimeChageEventType.Play:
                // アニメーション再生
                DbgConsole.WriteLine( string.Format( "Event---->> TimePlayPlay[ time = {0}]", time.ToString() ) );
                break;
                case TimeChageEventType.Stop:
                // アニメーション停止
                DbgConsole.WriteLine( string.Format( "Event---->> TimePlayStop[ time = {0}]", time.ToString() ) );
                break;
                case TimeChageEventType.Tick:
                // グローバル時間の変更
                DbgConsole.WriteLine( string.Format( "Event---->> TimeTick[ time = {0}]", time.ToString() ) );
                break;
            }
        }

        /// <summary>
        /// 時間変更イベントハンドラ
        /// </summary>
        /// <param name="time"></param>
        public void OnTimeChangedHandler( int time, TimeChageEventType type )
        {
            ReportDebugMessage_( time, type );

            //
            // すべてのビューに通知します。
            //
            foreach( ILEToolWindow toolWindow in _viewSet )
            {
                // 時間変更イベント
                toolWindow.OnTimeChangedHandler( time, type );
            }

            DbgConsole.WriteLine( $"<<---- Event {Enum.GetName(typeof(TimeChageEventType), type)}" );

            // ビューアへの更新要求の処理
            switch (type)
            {
                case TimeChageEventType.Stop:
                case TimeChageEventType.RangeChanged:
                    UpdateViewerRequested = true;
                    break;
            }
        }

        #endregion

        #region ISceneModifyListener メンバ


        /// <summary>
        /// シーン変更ハンドラの記述
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void OnSceneModifyHandler( object sender, SceneModifyEventArgs e )
        {
            // 強制的に時間を書き換えているときは通知しない。
            if (GlobalTime.Inst.SettingTimeForcibly)
            {
                return;
            }

            //
            // アニメーション再生中は変更メッセージに対応した
            // シーンの更新を行いません。
            //
            // アニメーション再生時には大量の更新メッセージが発生するため、
            // パフォーマンスへの影響が大きいからです。
            //
            // 時間更新イベントハンドラにViewの更新を記述してください。
            //

            if( GlobalTime.Inst.IsPlaying && GlobalTime.Inst.CallingOnTimeChanged)
            {
                return;
            }

            // 分割モードの区間タグ切り替え時は通知しません。
            //
            // アニメーション再生中の区間タグ切り替えは大量の更新メッセージが発生するため、
            // パフォーマンスへの影響が大きいからです。
            //
            ISubScene isubScene = sender as ISubScene;
            if (isubScene != null)
            {
                if (isubScene.EvaluateMultiAnimation)
                {
                    return;
                }
            }

            //-------------- デバック表示
            string msg =
                string.Format( "Event---->> SceneModified[ sender = {0} target = {1} ]",
                sender.ToString(), e.Target.ToString() );
            DbgConsole.WriteLine( msg );

            //----------------------------------------
            // すべてのビューに通知します。
            foreach( ILEToolWindow toolWindow in _viewSet )
            {
                // シーン変更イベント
                toolWindow.OnSceneModifyHandler( sender, e );
            }

            //-------------- デバック表示
            DbgConsole.WriteLine( "<<---- Event" );


            ISubScene newCurrentScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            if (newCurrentScene != null)
            {
                // 対象が変更されていれば...
                ISubSceneCmdQueue cmdQueue = newCurrentScene.ISubSceneCmdQueue;
                if (cmdQueue.NumRedoCommands != _currentRedo ||
                    cmdQueue.NumUndoCommands != _currentUndo)
                {
                    _currentRedo = cmdQueue.NumRedoCommands;
                    _currentUndo = cmdQueue.NumUndoCommands;

                    _theAppForm.WriteMessage(string.Format("---"));
                }
            }

            // ビューアへの更新要求の処理
            switch (e.Target)
            {
                // ビューアへ影響しない変更に対しては何もしない
                case SceneModifyEventArgs.Kind.SelectedSetModify:
                case SceneModifyEventArgs.Kind.SelectedSetClicked:
                case SceneModifyEventArgs.Kind.ClipBoardModify:
                case SceneModifyEventArgs.Kind.CurrentPaneModify:
                    break;
                default:
                    UpdateViewerRequested = true;
                    break;
            }

            switch (e.Target)
            {
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                    _theAppForm.UpdateText();
                    break;
            }
        }

        #endregion ISceneModifyListener メンバ

        #region IAppEventListener メンバ

        public void OnAppEvent( object sender, AppEventArgs args )
        {
            foreach( ILEToolWindow toolWindow in _viewSet )
            {
                toolWindow.OnAppEvent( sender, args );
            }
        }

        #endregion

        /// <summary>
        /// フォーカス時イベント
        /// </summary>
        public void Event_Form_Activated(Object sender, EventArgs e)
        {
            var leToolWindow = sender as LEToolWindow;
            if (leToolWindow == null)
            {
                return;
            }

            if (_toolSet.Contains(leToolWindow))
            {
                _toolSet.Remove(leToolWindow);
            }
            _toolSet.Add(leToolWindow);
        }

        public Dictionary<string, LayoutWindow> ClearDockPanel(out string xml)
        {
            var layoutWindows = new Dictionary<string, LayoutWindow>();
            foreach (var document in DockPanel.DocumentsToArray().OfType<LayoutWindow>())
            {
                document.PersistString = $"{typeof(LayoutWindow).FullName} {layoutWindows.Count}";
                layoutWindows.Add(document.PersistString, document);
            }

            xml = DockPanelToXml();

            foreach (var document in DockPanel.DocumentsToArray())
            {
                // SystemMdi のときは Document としては列挙されない
                Debug.Assert(_theAppForm.DockPanel.DocumentStyle != WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi);

                document.DockHandler.DockPanel = null;
            }

            foreach (var view in ToolWindowSet.OfType<LEToolWindow>())
            {
                //view.DockState = WeifenLuo.WinFormsUI.Docking.DockState.Unknown;
                if (view.DockPanel != null)
                {
                    // Float しているウインドウが復元できないので一旦ドックする
                    if (view.IsFloat)
                    {
                        view.DockTo(DockPanel, DockStyle.Right);
                    }

                    // 隠されている状態で、DockPanel を null にした後で、Show を呼び出すと例外が起きるので一旦表示状態にする
                    if (view.IsHidden)
                    {
                        // 上の DockTo の処理の後にしないと Float の状態が一瞬表示される。
                        view.Show();
                    }

                    view.DockPanel = null;
                }
            }

            _propertyWindowMgr.ClearAll();

            foreach (var floatingWindow in DockPanel.FloatWindows.ToArray())
            {
                floatingWindow.Dispose();
            }

            System.Diagnostics.Debug.Assert(DockPanel.Panes.Count == 0);
            System.Diagnostics.Debug.Assert(DockPanel.Contents.Count == 0);
            System.Diagnostics.Debug.Assert(DockPanel.FloatWindows.Count == 0);

            return layoutWindows;
        }
    }
}
