﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace LayoutEditor.Utility
{
    /// <summary>
    /// 指定した ToolStrip のショートカットを実行する。
    /// Form.ProcessCmdKey の既定の処理を差し替えるために使うことを想定している。
    /// Form.ProcessCmdKey の既定の処理では、親の Form があるときに、
    /// 親の Form の ToolStrip のショートカットが処理される。
    /// </summary>
    public class ShortcutHandler
    {
        public ShortcutHandler(ToolStrip menuStrip)
        {
            foreach (var item in menuStrip.Items.OfType<ToolStripItem>())
            {
                Register(item);
            }
        }

        /// <summary>
        /// コマンドキーの処理
        /// </summary>
        public bool ProcessCmdKey(Keys keys)
        {
            ToolStripMenuItem item;
            if (keysToItem.TryGetValue(keys, out item))
            {
                // メニューが無効なときに PerofrmClick しても何も実行されないが、
                // PerformClick は結果を返さないので判定が必要
                if (item.Enabled && item.Available)
                {
                    item.PerformClick();
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 同一として扱うキー
        /// </summary>
        static private Keys[][] SimilarKeyTable =
        {
            new [] { Keys.NumPad0, Keys.D0},
            new [] { Keys.NumPad1, Keys.D1},
            new [] { Keys.NumPad2, Keys.D2},
            new [] { Keys.NumPad3, Keys.D3},
            new [] { Keys.NumPad4, Keys.D4},
            new [] { Keys.NumPad5, Keys.D5},
            new [] { Keys.NumPad6, Keys.D6},
            new [] { Keys.NumPad7, Keys.D7},
            new [] { Keys.NumPad8, Keys.D8},
            new [] { Keys.NumPad9, Keys.D9},
            new [] { Keys.Add, Keys.Oemplus, Keys.Oemplus | Keys.Shift},
            new [] { Keys.Subtract, Keys.OemMinus, Keys.OemMinus | Keys.Shift},
            new [] {Keys.Delete, Keys.Back}
        };

        /// <summary>
        /// 同一として扱うキーの組み合わせの列挙
        /// </summary>
        private IEnumerable<Keys> SimilarKeys(Keys key)
        {
            Keys modifiers = key & (Keys.Shift | Keys.Control | Keys.Alt);
            Keys mainKey = key ^ modifiers;
            return SimilarKeyTable
                .Where(x => x.Contains(mainKey))
                .SelectMany(x => x).Select(x => x | modifiers)
                .Where(x => x != key);
        }

        /// <summary>
        /// キーとメニューの対応
        /// </summary>
        private Dictionary<Keys, ToolStripMenuItem> keysToItem = new Dictionary<Keys, ToolStripMenuItem>();

        /// <summary>
        /// キーとメニューの対応の登録
        /// </summary>
        private void Register(ToolStripItem item)
        {
            if (item is ToolStripMenuItem)
            {
                var menuItem = (ToolStripMenuItem)item;
                var keys = menuItem.ShortcutKeys;

                if (keys != Keys.None)
                {
                    keysToItem[keys] = menuItem;
                    foreach (var similarKeys in SimilarKeys(keys))
                    {
                        keysToItem[similarKeys] = menuItem;
                    }

                    // もともとのショートカットは無効にする
                    Debug.Assert(!string.IsNullOrEmpty(menuItem.ShortcutKeyDisplayString));
                    menuItem.ShortcutKeys = Keys.None;
                }
            }

            if (item is ToolStripDropDownItem)
            {
                foreach (var child in ((ToolStripDropDownItem)item).DropDownItems.OfType<ToolStripItem>())
                {
                    Register(child);
                }
            }
        }
    }
}
