﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Runtime.InteropServices;
using System.IO;
using System.Reflection;




namespace LayoutEditor.Utility
{
    /// <summary>
    /// マテリアルカラーシェーダクラス。
    /// </summary>
    public sealed class MaterialColorShader
    {
        private static readonly Bitmap	_diffuseSrc		= null;
        private static readonly Bitmap	_ambientSrc		= null;
        private static readonly Bitmap	_specularSrc	= null;
        private static readonly Bitmap	_backSrc		= null;
        private static Bitmap	_buffer0		= null;
        private static Bitmap	_buffer1		= null;

        /// <summary>
        ///
        /// </summary>
        static Bitmap GetBitmapResource_( string resName )
        {
            string resourceName   = "LayoutEditor.res.bmp." + resName;
            Assembly  assembly = Assembly.GetExecutingAssembly();
            Stream resourceStream = assembly.GetManifestResourceStream(resourceName);
            Debug.Assert(resourceStream != null, string.Format("マニフェストリソース '{0}' が見つかりません。", resourceName));

            return new Bitmap( resourceStream );
        }


        /// <summary>
        /// タイプコンストラクタ
        /// </summary>
        static MaterialColorShader()
        {
           _diffuseSrc   = GetBitmapResource_("MS_Diffuse.bmp");
           _ambientSrc	= GetBitmapResource_("MS_Ambient.bmp");
           _specularSrc	= GetBitmapResource_("MS_Specular.bmp");

            _buffer0		= new Bitmap(_diffuseSrc.Width, _diffuseSrc.Height, PixelFormat.Format32bppArgb);
            _buffer1		= new Bitmap(_diffuseSrc.Width, _diffuseSrc.Height, PixelFormat.Format32bppArgb);
            //	格子模様作成
            _backSrc		= new Bitmap( 128, 128, PixelFormat.Format1bppIndexed );
            BitmapData	bitmapData	= _backSrc.LockBits( new Rectangle(0, 0, 128, 128),
                ImageLockMode.WriteOnly, PixelFormat.Format1bppIndexed);
            IntPtr	pData	= bitmapData.Scan0;
            int		total	= 0;
            short	col1	= -1;
            short	col2	= 0;
            for( int x = 0; x < 128; x += 16, total += 2 )
            {
                short		col3	= col2;
                col2	= col1;
                col1	= col3;
                int		top	= total;
                for( int z = 0; z < 4; z ++ )
                {
                    for( int y = 0; y < 16; y ++, top += bitmapData.Stride )
                        Marshal.WriteInt16( pData, top, col1 );
                    for( int y = 0; y < 16; y ++, top += bitmapData.Stride )
                        Marshal.WriteInt16( pData, top, col2 );
                }
            }
            _backSrc.UnlockBits( bitmapData );
            ColorPalette	cl	= _backSrc.Palette;
            cl.Entries[0]	= Color.DarkGray;
            cl.Entries[1]	= Color.Gray;
            _backSrc.Palette	= cl;
        }

        /// <summary>
        /// マテリアルカラーの描画
        /// </summary>
        static public void DrawColor(Graphics gfx, Rectangle rect, MaterialColorShadingInfo si)
        {
            Rectangle	buffer_rect	= new Rectangle( 0, 0, _buffer1.Width, _buffer1.Height );
            using( Graphics g_dst = Graphics.FromImage( _buffer1 ) )
            {
                g_dst.Clear(Color.Transparent);
                //	背景描画
                switch( si.Background )
                {
                    //	格子背景
                    case MaterialShadingInfoBase.BackgroundType.Cross:
                    {
                        int		width	= (int)Math.Round( (float)rect.Width / 12 ) * 16;
                        width	= Math.Max( width, 96 );
                        width	= Math.Min( width, 128 );
                        g_dst.DrawImage(_backSrc, buffer_rect, 0, 0, width, width, GraphicsUnit.Pixel);
                        break;
                    }
                }

                using( Graphics g = Graphics.FromImage( _buffer0 ) )
                {
                    //	ディフューズBitmap作成
                    g.Clear(Color.Transparent);
                    ColorMatrix	cm	= new ColorMatrix();

                    cm.Matrix00 = 0.0f;
                    cm.Matrix10	= (float)si.Diffuse.R / 255;
                    cm.Matrix11 = (float)si.Diffuse.G / 255;
                    cm.Matrix22 = (float)si.Diffuse.B / 255;

                    cm.Matrix03 = 1.0f;
                    cm.Matrix33 = 0.0f;
                    cm.Matrix44 = 1.0f;

                    using( ImageAttributes ia = new ImageAttributes() )
                    {
                        ia.SetColorMatrix( cm );
                        g.DrawImage(_diffuseSrc, buffer_rect, 0, 0,
                            _diffuseSrc.Width, _diffuseSrc.Height, GraphicsUnit.Pixel, ia);
                    }
                }
                g_dst.DrawImage(_buffer0, buffer_rect);
                if( si.LightingDiffuse )
                {
                    using( Graphics g = Graphics.FromImage( _buffer0 ) )
                    {
                        //	アンビエントBitmap作成
                        g.Clear(Color.Transparent);
                        ColorMatrix	cm	= new ColorMatrix();

                        cm.Matrix00 = 0.0f;
                        cm.Matrix10	= (float)si.Diffuse.R * si.Ambient.R / (255 * 255);
                        cm.Matrix11 = (float)si.Diffuse.G * si.Ambient.G / (255 * 255);
                        cm.Matrix22 = (float)si.Diffuse.B * si.Ambient.B / (255 * 255);

                        cm.Matrix03 = 1.0f;
                        cm.Matrix33 = 0.0f;
                        cm.Matrix44 = 1.0f;

                        using( ImageAttributes ia = new ImageAttributes() )
                        {
                            ia.SetColorMatrix( cm );
                            g.DrawImage(_ambientSrc, buffer_rect, 0, 0,
                                _ambientSrc.Width, _ambientSrc.Height, GraphicsUnit.Pixel, ia);
                        }
                    }
                    g_dst.DrawImage(_buffer0, buffer_rect);
                }
                if( si.LightingSpecular )
                {
                    using( Graphics g = Graphics.FromImage( _buffer0 ) )
                    {
                        //	スペキュラーBitmap作成
                        g.Clear(Color.Transparent);
                        ColorMatrix	cm	= new ColorMatrix();

                        cm.Matrix00 = 0.0f;
                        cm.Matrix10	= (float)(si.Diffuse.R + ((float)si.Specular.R / 255) * (255-si.Diffuse.R)) / 255;
                        cm.Matrix11 = (float)(si.Diffuse.G + ((float)si.Specular.G / 255) * (255-si.Diffuse.G)) / 255;
                        cm.Matrix22 = (float)(si.Diffuse.B + ((float)si.Specular.B / 255) * (255-si.Diffuse.B)) / 255;

                        cm.Matrix03 = 1.0f;
                        cm.Matrix33 = 0.0f;
                        cm.Matrix44 = 1.0f;

                        using( ImageAttributes ia = new ImageAttributes() )
                        {
                            ia.SetColorMatrix( cm );
                            g.DrawImage(_specularSrc, buffer_rect, 0, 0,
                                _specularSrc.Width, _specularSrc.Height, GraphicsUnit.Pixel, ia);
                        }
                    }
                    g_dst.DrawImage(_buffer0, buffer_rect);
                }
            }
            gfx.DrawImage( _buffer1, rect );
        }
        /// <summary>
        /// マテリアルアルファの描画
        /// </summary>
        static public void DrawAlpha( Graphics gfx, Rectangle rect, MaterialAlphaShadingInfo si)
        {
            //	背景描画
            switch( si.Background )
            {
                //	格子背景
                case MaterialShadingInfoBase.BackgroundType.Cross:
                {
                    int		width	= (int)Math.Round( (float)rect.Width / 12 ) * 16;
                    width	= Math.Max( width, 96 );
                    width	= Math.Min( width, 128 );
                    gfx.DrawImage(_backSrc, rect, 0, 0, width, width, GraphicsUnit.Pixel);
                    break;
                }
            }

            ColorMatrix	cm	= new ColorMatrix();
            //	ライトありなら加工用に反転
            cm.Matrix00 = ( si.Lighting ) ? -1.0f : 1.0f;

            using( Graphics g = Graphics.FromImage( _buffer0 ) )
            {
                g.Clear(Color.Transparent);

                using( ImageAttributes ia = new ImageAttributes() )
                {
                    ia.SetColorMatrix( cm );
                    g.DrawImage(_diffuseSrc, new Rectangle( 0, 0, _buffer0.Width, _buffer0.Height ), 0, 0,
                        _diffuseSrc.Width, _diffuseSrc.Height, GraphicsUnit.Pixel, ia);
                    if( si.Lighting )
                    {
                        //	アルファ用に2枚合成
                        g.DrawImage(_ambientSrc, new Rectangle( 0, 0, _buffer0.Width, _buffer0.Height ), 0, 0,
                            _ambientSrc.Width, _ambientSrc.Height, GraphicsUnit.Pixel);
                        //	反転
                        using( Graphics gg = Graphics.FromImage( _buffer1 ) )
                        {
                            gg.Clear(Color.Transparent);
                            gg.DrawImage(_buffer0, new Rectangle( 0, 0, _buffer1.Width, _buffer1.Height ), 0, 0,
                                _buffer0.Width, _buffer0.Height, GraphicsUnit.Pixel, ia);
                        }
                    }
                }
                cm.Matrix00 = 0.0f;
                cm.Matrix10	= 1.0f;

                cm.Matrix03 = (float)si.Alpha / 255;
                cm.Matrix33 = 0.0f;

                using( ImageAttributes ia = new ImageAttributes() )
                {
                    ia.SetColorMatrix( cm );
                    gfx.DrawImage(( si.Lighting ) ? _buffer1 : _buffer0, rect, 0, 0, _diffuseSrc.Width, _diffuseSrc.Height, GraphicsUnit.Pixel, ia);
                }
            }
        }
    }

    #region MaterialShadingInfoBase
    /// <summary>
    /// マテリアルシェーディング情報ベースクラス。
    /// </summary>
    public class MaterialShadingInfoBase
    {
        private	BackgroundType _background	= BackgroundType.Cross;

        /// <summary>
        /// 背景タイプ。
        /// </summary>
        public enum BackgroundType
        {
            /// <summary>なし</summary>
            None,
            /// <summary>格子模様</summary>
            Cross
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialShadingInfoBase() {}

        /// <summary>
        /// 背景タイプ。
        /// </summary>
        public BackgroundType Background
        {
            get { return _background;  }
            set { _background = value; }
        }
    }

    #endregion

    #region MaterialColorShadingInfo
    /// <summary>
    /// マテリアルカラーシェーディング情報クラス。
    /// </summary>
    public sealed class MaterialColorShadingInfo : MaterialShadingInfoBase
    {
        private Color _diffuse          = Color.Empty;
        private Color _ambient          = Color.Empty;
        private Color _specular         = Color.Empty;
        private bool  _lightingDiffuse  = false;
        private bool  _lightingSpecular = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialColorShadingInfo() {}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialColorShadingInfo(Color diffuse, Color ambient, Color specular, bool lightingDiffuse, bool lightingSpecular)
        {
            _diffuse          = diffuse;
            _ambient          = ambient;
            _specular         = specular;
            _lightingDiffuse  = lightingDiffuse;
            _lightingSpecular = lightingSpecular;
        }

        /// <summary>
        /// ディフューズ。
        /// </summary>
        public Color Diffuse
        {
            get { return _diffuse;  }
            set { _diffuse = value; }
        }

        /// <summary>
        /// アンビエント。
        /// </summary>
        public Color Ambient
        {
            get { return _ambient;  }
            set { _ambient = value; }
        }

        /// <summary>
        /// スペキュラ。
        /// </summary>
        public Color Specular
        {
            get { return _specular;  }
            set { _specular = value; }
        }

        /// <summary>
        /// ディフューズライト計算フラグ。
        /// </summary>
        public bool LightingDiffuse
        {
            get { return _lightingDiffuse;  }
            set { _lightingDiffuse = value; }
        }

        /// <summary>
        /// スペキュラライト計算フラグ。
        /// </summary>
        public bool LightingSpecular
        {
            get { return _lightingSpecular;  }
            set { _lightingSpecular = value; }
        }
    }
    #endregion

    #region MaterialAlphaShadingInfo
    /// <summary>
    /// マテリアルアルファシェーディング情報クラス。
    /// </summary>
    public sealed class MaterialAlphaShadingInfo : MaterialShadingInfoBase
    {
        private int  _alpha    = 0;
        private bool _lighting = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialAlphaShadingInfo() {}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialAlphaShadingInfo(int alpha, bool lighting)
        {
            _alpha    = alpha;
            _lighting = lighting;
        }

        /// <summary>
        /// アルファ。
        /// </summary>
        public int Alpha
        {
            get { return _alpha;  }
            set { _alpha = value; }
        }

        /// <summary>
        /// ライト計算フラグ。
        /// </summary>
        public bool Lighting
        {
            get { return _lighting;  }
            set { _lighting = value; }
        }
    }
    #endregion
}
