﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Utility
{
    using Win32 = LECore.Win32;
    using System.Drawing;

    /// <summary>
    /// ListView用のヘルパーです。
    /// </summary>
    public static class ListViewHelper
    {
        /// <summary>
        /// 選択状態のキャッシュ
        /// </summary>
        public class SelectItemCache
        {
            readonly List<string> _selectedItems = new List<string>();
            int _itemCount = 0;
            int _scrollPos;

            /// <summary>
            /// 生成
            /// </summary>
            public SelectItemCache(ListView lv)
            {
                foreach (ListViewItem item in lv.SelectedItems)
                {
                    _selectedItems.Add(item.Text);
                }

                _itemCount = lv.Items.Count;
                _scrollPos = LECore.Win32.User32.GetScrollPos(lv.Handle, LECore.Win32.SB.SB_VERT);
            }

            /// <summary>
            /// 反映
            /// </summary>
            public void Apply(ListView lv)
            {
                ListViewItem lastSelectedItem = null;
                foreach (ListViewItem item in lv.Items)
                {
                    if (_selectedItems.Contains(item.Text))
                    {
                        item.Selected = true;
                        lastSelectedItem = item;
                    }
                }

                // 要素数が全く同じなら、スクロール位置をそのまま復元する。
                if (_itemCount == lv.Items.Count)
                {
                    int itemHeight = lv.TopItem != null ? lv.TopItem.Bounds.Height : 16;
                    LECore.Win32.User32.SendMessage(lv.Handle, LECore.Win32.LVM.LVM_SCROLL, (int)0, (int)_scrollPos * itemHeight);
                }
                else
                {
                    // 最後に選択したアイテムが表示されるようにスクロールする。
                    if (lastSelectedItem != null)
                    {
                        lastSelectedItem.EnsureVisible();
                    }
                }
            }
        }

        /// <summary>
        /// LVM_GETHEADER。
        /// </summary>
        public static IntPtr GetHeader(ListView lv)
        {
            return Win32.User32.SendMessage(lv.Handle, Win32.LVM.LVM_GETHEADER, IntPtr.Zero, IntPtr.Zero);
        }

        /// <summary>
        /// 選択状態をリセットします。
        /// </summary>
        /// <param name="lv">リストビューです。</param>
        static public void ResetSelectedItems(this ListView lv)
        {
            foreach (ListViewItem item in lv.SelectedItems)
            {
                item.Selected = false;
            }
        }

        /// <summary>
        /// 指定位置における、ListViewItem と サブアイテム番号 を 取得します。
        /// </summary>
        static public int GetSubItemAt(this ListView lvw, int x, int y, out ListViewItem listViewItem)
        {
            listViewItem = lvw.GetItemAt(x, y);
            if (listViewItem != null)
            {
                // サブアイテム行番号を判定します。
                int subItemX = listViewItem.GetBounds(ItemBoundsPortion.Entire).Left;
                int[] order = GetColumnOrderArray(lvw);
                for (int i = 0; i < order.Length; i++)
                {
                    ColumnHeader header = lvw.Columns[order[i]];
                    if (x < subItemX + header.Width)
                    {
                        return header.Index;
                    }
                    subItemX += header.Width;
                }
            }

            return -1;
        }

        /// <summary>
        /// 列順序取得。
        /// </summary>
        public static int GetColumnOrder(this ListView lv, int subItem)
        {
            if (lv.AllowColumnReorder)
            {
                int[] order = GetColumnOrderArray(lv);
                for (int i = 0; i < order.Length; i++)
                {
                    if (order[i] == subItem)
                    {
                        return i;
                    }
                }
            }
            return subItem;
        }

        /// <summary>
        /// ListView_GetSubItemRect()に相当。
        /// </summary>
        public static Rectangle GetSubItemRect(this ListView lv, int item, int subItem, int lvir)
        {
            Win32.RECT rect = new Win32.RECT();
            rect.top  = subItem;
            rect.left = lvir;
            Win32.User32.SendMessage(lv.Handle, Win32.LVM.LVM_GETSUBITEMRECT, (IntPtr)item, ref rect);
            return rect.ToRectangle();
        }

        /// <summary>
        /// ListView_SetColumnOrderArray()に相当。
        /// </summary>
        public static void SetColumnOrderArray(this ListView lv, int[] array)
        {
            Win32.User32.SendMessage(lv.Handle, Win32.LVM.LVM_SETCOLUMNORDERARRAY, (IntPtr)array.Length, array);
        }

        /// <summary>
        /// ListView_GetColumnOrderArray()に相当。
        /// </summary>
        public static int[] GetColumnOrderArray(this ListView lv)
        {
            int[] array = new int[lv.Columns.Count];
            Win32.User32.SendMessage(lv.Handle, Win32.LVM.LVM_GETCOLUMNORDERARRAY, (IntPtr)array.Length, array);
            return array;
        }
    }
}
