﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using System.Drawing;

using LECore.Manipulator;
using LECore.Structures;

namespace LayoutEditor.Structures.SerializableObject
{
    using LECore.Structures;
    using LECore;
    using System.Xml.Serialization;
    using System.IO;
    using LECore.Save_Load;

    /// <summary>
    /// 拡張ユーザデータのプリセットです。
    /// </summary>
    public struct ExUserDataPreset
    {
        public string Name { get; set; }
        public string Type { get; set; }

        /// <summary>
        /// 項目を複数選択可能かどうか
        /// </summary>
        public bool MultiSelect { get; set; }

        public string Descriptions { get; set; }



        public List<PaneKind> PaneKindLimitations { get; set; }
        public List<string> ControlNameLimitations { get; set; }
        public bool ApplicableToAnimTag { get; set; }
        public bool ApplicableToLayout { get; set; }

        [XmlArrayItem("EnumValue")]
        public List<EnumValue> EnumValues { get; set; }

        public class EnumValue
        {
            [XmlText]
            public string Text;

            [XmlAttribute]
            public string Descriptions;

            public override string ToString()
            {
                return Text;
            }

            public static string CalcFormattedString(EnumValue e)
            {
                if(e == null)
                {
                    return string.Empty;
                }

                return string.Format("{0}:{1}", e.Text, e.Descriptions);
            }
        }

        private bool HasPaneLimitations { get { return this.PaneKindLimitations != null && this.PaneKindLimitations.Count > 0; } }
        private bool HasControlLimitations { get { return this.ControlNameLimitations != null && this.ControlNameLimitations.Count > 0; } }

        /// <summary>
        /// 指定種類が対象か判定します。
        /// </summary>
        public bool IsTragetsPane(PaneKind kind)
        {
            // 対象種類リストが未設定なら
            if (!this.HasPaneLimitations)
            {
                // コントロールに対象を限定しているなら（ペインを対象にはしたくないはずなので）不合格。
                if (this.HasControlLimitations)
                {
                    return false;
                }

                // 合格
                return true;
            }

            return this.PaneKindLimitations.Exists((targetKind) => ((uint)targetKind & (uint)kind) != 0);
        }

        /// <summary>
        /// 指定種類が対象か判定します。
        /// </summary>
        public bool IsTragetsContorol(string controlName)
        {
            // 対象種類リストが未設定なら
            if (!this.HasControlLimitations)
            {
                // ペイン種類に対象を限定しているなら（コントロールを対象にはしたくないはずなので）不合格。
                if (this.HasPaneLimitations)
                {
                    return false;
                }

                // 合格
                return true;
            }

            return this.ControlNameLimitations.Exists((name) => name == controlName);
        }

        public bool IsTargetAnimTag()
        {
            return ApplicableToAnimTag;
        }

        public bool IsTargetSubScene()
        {
            return ApplicableToLayout;
        }
    }

    /// <summary>
    /// スケーラブルフォント設定
    /// </summary>
    public struct ScalableFontDesc
    {
        public string FontName { get; set; }
        public string FilePath { get; set; }
        public int FontSize { get; set; }
        public float BoldWeight { get; set; }
        public int BorderWidth { get; set; }
    }

    /// <summary>
    /// 新規ペインプレフィクス設定
    /// </summary>
    public struct DefaultPaneNames
    {
        public string Null { get; set; }
        public string Picture { get; set; }
        public string Text { get; set; }
        public string Window { get; set; }
        public string Bounding { get; set; }
        public string Alignment { get; set; }
        public string Capture { get; set; }
        public string Scissor { get; set; }
        public string Parts { get; set; }
        public string Group { get; set; }
        public bool UseTextureNameAsPicturePaneName { get; set; }
    }

    /// <summary>
    /// ユーザーコマンド
    /// </summary>
    public class UserCommand
    {
        [XmlAttribute]
        public string Name { get; set; }

        [XmlAttribute]
        public string Args { get; set; }

        [XmlAttribute]
        public bool ConfirmSave { get; set; }

        [XmlAttribute]
        public string ShortCut { get; set; }

        [XmlText]
        public string PathXml;

        public UserCommand()
        {
        }

        private string fullPath_ = null;
        public string FullPath
        {
            get { return fullPath_ ?? (fullPath_ = Environment.ExpandEnvironmentVariables(PathXml)); }
            set { fullPath_ = value; }
        }

        public string GetBaseDirectoryName()
        {
            return Path.GetDirectoryName(FullPath);
        }

        public IEnumerable<string> Commands
        {
            get
            {
                return !FullPath.Contains("*")
                    ? new[] { FullPath }
                    : Directory.EnumerateFiles(
                        Path.GetDirectoryName(FullPath),
                        Path.GetFileName(FullPath),
                        SearchOption.AllDirectories);
            }
        }
    }

    public class UserCombinerShaderSettings
    {
        public UserCombinerShaderSettings()
        {
            CombinerBlockDefinitionDirectories = new List<string>();
        }

        /// <summary>
        ///  コンバイナエディタ用の EftDefinitions ディレクトリのパス
        /// </summary>
        [XmlArrayItem("Directory")]
        public List<string> CombinerBlockDefinitionDirectories { get; set; }

        /// <summary>
        ///  コンバイナエディタのファイルパス
        /// </summary>
        public string CombinerEditorPath { get; set; }
    }

    /// <summary>
    /// テクスチャウインドウの表示項目
    /// </summary>
    public class TextureDirectory
    {
        /// <summary>
        /// サブディレクトリも対象とするか？
        /// </summary>
        [XmlAttribute]
        public bool AllDirectories { get; set; }

        /// <summary>
        /// サブディレクトリごとの表示を行うか？
        /// </summary>
        [XmlAttribute]
        public bool ShowDirectories { get; set; }

        /// <summary>
        /// UI 上の表示名
        /// </summary>
        [XmlAttribute]
        public string Label { get; set; }

        public enum BaseType
        {
            Absolute,
            Project,
            Layout,
        }

        /// <summary>
        /// パスの基準ディレクトリ
        /// </summary>
        [XmlAttribute]
        public BaseType Base { get; set; }

        /// <summary>
        /// ディレクトリパス
        /// 環境変数も指定可能
        /// </summary>
        [XmlText]
        public string Path { get; set; }
    }

    /// <summary>
    /// プロジェクト設定
    /// (ガンマ設定など、アプリケーション設定の一部は、コンテンツ側にプロジェクト設定という形で保存することになりました)
    /// </summary>
    public class ProjectSettings
    {
        const string _Version_1_0_0 = "100";
        const string _Version_1_0_1 = "101";
        const string _Version_1_0_2 = "102";
        const string _Version_1_0_3 = "103";
        const string _Version_1_0_4 = "104";
        const string _Version_1_0_5 = "105";
        const string _Version_1_0_6 = "106";
        const string _Version_1_0_7 = "107";
        const string _Version_1_0_8 = "108";

        /// <summary>
        /// 読み込んだファイルパス（シリアライズされません）
        /// </summary>
        [XmlIgnore]
        public string FilePath { get; set; }

        public string FileVersionString { get; set; }
        public bool SRGBWriteEnabled { get; set; }
        public bool FloatColorWriteEnabled { get; set; }
        public bool AstcTextureEnabled { get; set; }
        public bool DegammaTextureConversionEnabled { get; set; }
        public bool DegammaParameterConversionEnabled { get; set; }

        public bool CaptureTextureFunctionEnabled { get; set; }

        public bool AlignmentPaneEnabled { get; set; }

        public int PreviewDisplayWidth { get; set; }
        public int PreviewDisplayHeight { get; set; }

        public string ViewMode { get; set; }

        public bool IsPerspectiveDefault { get; set; }
        public float PerspectiveFOV { get; set; }
        public float PerspectiveNear { get; set; }
        public float PerspectiveFar { get; set; }
        public bool DepthTestEnabled { get; set; }

        public bool Is30Fps { get; set; }

        public string CustomPCViewerWindowTitleHint { get; set; }

        // 非公開オプション デバッグ用
        public bool SearchPCViewerByTitle { get; set; }

        public List<ExUserDataPreset> ExUserDataPresets { get; set; }

        public bool MultiControlEnabled { get; set; }

        public bool AutoFilteredFontGenerationEnabled { get; set; }
        public bool AutoFilteredFontGenerationFromSameNameFfntEnabled { get; set; }

        public bool PreviewConverterOptionOmitNoKeyAllEnabled { get; set; }

        public bool AllowNoGroupAnimationTag { get; set; }

        public bool ExtendedTagEnabled { get; set; }

        public bool MovieTextureEnabled { get; set; }
        public string MovieTextureOutputDirectory { get; set; }

        public bool KeepingSizeScaleEnabled { get; set; }

        /// <summary>
        /// ヘルプ等のドキュメントのパス
        /// </summary>
        public string DocumentPath { get; set; }

        public string PreSaveCommandPath{ get; set; }
        public string PostSaveCommandPath { get; set; }

        public string PreDeleteCommandPath { get; set; }
        public string PostDeleteCommandPath { get; set; }

        public string PreLoadCommandPath { get; set; }

        /// <summary>現状は、必要になっていないので、シリアライズしません。</summary>
        [XmlIgnore]
        public string PostCloseCommandPath { get; set; }
        /// <summary>現状は、必要になっていないので、シリアライズしません。</summary>
        [XmlIgnore]
        public int UserCommandTimeOutMilliSec { get; set; }

        public string PreArchiverCommandPath { get; set; }

        public string TextureSourceFileSearchRootPath { get; set; }

        public List<ScalableFontDesc> ScalableFontDescSet { get; set; }
        public List<UserCommand> UserCommands { get; set; }

        public string TileOptimize { get; set; }
        public int TileSizeThreshold { get; set; }

        public string IsGpuEncodingEnabled { get; set; }

        public Size DefaultInitialPaneSize { get; set; }

        public const float DefaultPerspectiveFOV = 45.0f;
        public const float DefaultPerspectiveNear = 1.0f;
        public const float DefaultPerspectiveFar = 10000.0f;

        public DefaultPaneNames DefaultPaneNames;
        public LECore.Structures.LECoreInterface.DefaultTextureFormat DefaultTextureFormat;

        public string DefaultTextBoxPaneTextID;

        /// <summary>
        /// スケーラブルフォント用のテクスチャキャッシュのサイズです。
        /// </summary>
        public System.Drawing.Size TextureCacheSizeForScalableFont { get; set; }

        /// <summary>
        ///  コンバイナエディタの設定です
        /// </summary>
        public UserCombinerShaderSettings UserCombinerShaderSettings { get; set; }

        /// <summary>
        /// コンバイナユーザーシェーダーが利用可能か判定します。
        /// </summary>
        public bool IsCombinerUserShaderEnabled
        {
            get
            {
                return this.UserCombinerShaderSettings.CombinerBlockDefinitionDirectories != null && this.UserCombinerShaderSettings.CombinerBlockDefinitionDirectories.Any();
            }
        }

        /// <summary>
        /// ステート機能が有効か
        /// </summary>
        public bool IsStateMachineEnalbed { get; set; }

        /// <summary>
        /// サムネイルの設定
        /// </summary>
        public bool SaveThumbnailImage { get; set; } = false;
        public int ThumbnailMaximumWidth { get; set; } = 96;
        public int ThumbnailMaximumHeight { get; set; } = 96;

        /// <summary>
        /// テクスチャウインドウに表示するディレクトリの設定
        /// </summary>
        public TextureDirectory[] TextureDirectories { get; set; } = null;

        /// <summary>
        ///
        /// </summary>
        public ProjectSettings()
        {
            this.FilePath = string.Empty;
            this.FileVersionString = _Version_1_0_8;
            this.ExUserDataPresets = new List<ExUserDataPreset>();
            this.CustomPCViewerWindowTitleHint = string.Empty;
            this.MultiControlEnabled = false;
            this.AutoFilteredFontGenerationEnabled = true;
            this.PreviewConverterOptionOmitNoKeyAllEnabled = false;
            this.AllowNoGroupAnimationTag = false;
            this.ExtendedTagEnabled = false;
            this.MovieTextureEnabled = false;
            this.MovieTextureOutputDirectory = null;
            this.KeepingSizeScaleEnabled = false;
            this.UserCommandTimeOutMilliSec = 10000;
            this.ScalableFontDescSet = new List<ScalableFontDesc>();
            this.UserCommands = new List<UserCommand>();
            this.DefaultPaneNames = new DefaultPaneNames();
            this.DefaultTextureFormat = new LECore.Structures.LECoreInterface.DefaultTextureFormat();
            this.TextureCacheSizeForScalableFont = new Size(1024, 2048);
            this.ViewMode = "original";
            this.FloatColorWriteEnabled = false;
            this.AstcTextureEnabled = false;
            this.CaptureTextureFunctionEnabled = false;
            this.TileOptimize = "performance";
            this.TileSizeThreshold = 75;
            this.IsGpuEncodingEnabled = "true";
            this.PreviewDisplayWidth = 1280;
            this.PreviewDisplayHeight = 720;
            this.UserCombinerShaderSettings = new UserCombinerShaderSettings();
            this.DefaultInitialPaneSize = SubSceneManipulator.DefaultInitialPaneSize;
            this.AlignmentPaneEnabled = true;
            this.IsStateMachineEnalbed = false;

            InitPerspective_();

            if (LayoutEditorCore.PlatformDetail != null && LayoutEditorCore.PlatformDetail.IsLinearGammaSettingEnabled)
            {
                SRGBWriteEnabled = true;
                DegammaTextureConversionEnabled = true;
                DegammaParameterConversionEnabled = false;
            }
            else
            {
                SRGBWriteEnabled = false;
                DegammaTextureConversionEnabled = false;
                DegammaParameterConversionEnabled = false;
            }
        }

        private void InitPerspective_()
        {
            this.IsPerspectiveDefault = false;
            this.PerspectiveFOV = DefaultPerspectiveFOV;
            this.PerspectiveNear = DefaultPerspectiveNear;
            this.PerspectiveFar = DefaultPerspectiveFar;
            this.DepthTestEnabled = false;
        }

        public void Set(ProjectSettings other)
        {
            this.FilePath = other.FilePath;

            this.SRGBWriteEnabled = other.SRGBWriteEnabled;
            this.FloatColorWriteEnabled = other.FloatColorWriteEnabled;
            this.AstcTextureEnabled = other.AstcTextureEnabled;
            this.DegammaTextureConversionEnabled = other.DegammaTextureConversionEnabled;
            this.DegammaParameterConversionEnabled = other.DegammaParameterConversionEnabled && other.DegammaTextureConversionEnabled;
            this.CaptureTextureFunctionEnabled = other.CaptureTextureFunctionEnabled;
            this.AlignmentPaneEnabled = other.AlignmentPaneEnabled;

            this.PreviewDisplayWidth = other.PreviewDisplayWidth;
            this.PreviewDisplayHeight = other.PreviewDisplayHeight;

            this.ViewMode = other.ViewMode;

            this.IsPerspectiveDefault = other.IsPerspectiveDefault;
            this.PerspectiveFOV = other.PerspectiveFOV;
            this.PerspectiveNear = other.PerspectiveNear;
            this.PerspectiveFar = other.PerspectiveFar;
            this.DepthTestEnabled = other.DepthTestEnabled;
            this.Is30Fps = other.Is30Fps;

            this.CustomPCViewerWindowTitleHint = other.CustomPCViewerWindowTitleHint;
            this.SearchPCViewerByTitle = other.SearchPCViewerByTitle;

            this.ExUserDataPresets.Clear();
            this.ExUserDataPresets.AddRange(other.ExUserDataPresets);
            this.MultiControlEnabled = other.MultiControlEnabled;
            this.AutoFilteredFontGenerationEnabled = other.AutoFilteredFontGenerationEnabled;
            this.AutoFilteredFontGenerationFromSameNameFfntEnabled = other.AutoFilteredFontGenerationFromSameNameFfntEnabled;
            this.PreviewConverterOptionOmitNoKeyAllEnabled = other.PreviewConverterOptionOmitNoKeyAllEnabled;
            this.AllowNoGroupAnimationTag = other.AllowNoGroupAnimationTag;
            this.ExtendedTagEnabled = other.ExtendedTagEnabled;
            this.MovieTextureEnabled = other.MovieTextureEnabled;
            this.MovieTextureOutputDirectory = other.MovieTextureOutputDirectory;
            this.KeepingSizeScaleEnabled = other.KeepingSizeScaleEnabled;

            this.DocumentPath = other.DocumentPath;

            this.PreSaveCommandPath = other.PreSaveCommandPath;
            this.PostSaveCommandPath = other.PostSaveCommandPath;
            this.PreDeleteCommandPath = other.PreDeleteCommandPath;
            this.PostDeleteCommandPath = other.PostDeleteCommandPath;

            this.PreLoadCommandPath = other.PreLoadCommandPath;
            this.PostCloseCommandPath = other.PostCloseCommandPath;

            this.PreArchiverCommandPath = other.PreArchiverCommandPath;

            this.TextureSourceFileSearchRootPath = other.TextureSourceFileSearchRootPath;

            this.UserCommandTimeOutMilliSec = other.UserCommandTimeOutMilliSec;

            this.ScalableFontDescSet.Clear();
            this.ScalableFontDescSet.AddRange(other.ScalableFontDescSet);

            this.UserCommands.Clear();
            this.UserCommands.AddRange(other.UserCommands);

            this.DefaultPaneNames = other.DefaultPaneNames;
            this.DefaultTextureFormat = other.DefaultTextureFormat;
            this.DefaultTextBoxPaneTextID = other.DefaultTextBoxPaneTextID;
            this.TextureCacheSizeForScalableFont = other.TextureCacheSizeForScalableFont;

            this.TileOptimize = other.TileOptimize;
            this.TileSizeThreshold = other.TileSizeThreshold;

            this.IsGpuEncodingEnabled = other.IsGpuEncodingEnabled;

            this.DefaultInitialPaneSize = other.DefaultInitialPaneSize;

            this.UserCombinerShaderSettings.CombinerBlockDefinitionDirectories = other.UserCombinerShaderSettings.CombinerBlockDefinitionDirectories;
            this.UserCombinerShaderSettings.CombinerEditorPath = other.UserCombinerShaderSettings.CombinerEditorPath;

            this.SaveThumbnailImage = other.SaveThumbnailImage;
            this.ThumbnailMaximumWidth = other.ThumbnailMaximumWidth;
            this.ThumbnailMaximumHeight = other.ThumbnailMaximumHeight;

            this.TextureDirectories = other.TextureDirectories;

            this.IsStateMachineEnalbed = other.IsStateMachineEnalbed;
        }

        private void Verify_()
        {
            // _Version_1_0_0 では、this.DegammaTextureConversionEnabled と this.SRGBWriteEnabled は同じ値にしておく。
            if (this.FileVersionString == _Version_1_0_0)
            {
                this.SRGBWriteEnabled = this.DegammaTextureConversionEnabled;
            }

            if (this.PerspectiveFOV == 0.0f)
            {
                // 古いプロジェクトやまったく新規のプロジェクトを読んだ場合。
                // たぶん不正な状態なので初期化する。
                InitPerspective_();
            }

            if (TextureDirectories == null)
            {
                // 明示的に指定がなかった場合の初期値
                TextureDirectories = new TextureDirectory[]
                {
                    new TextureDirectory()
                    {
                        AllDirectories = true,
                        Base = TextureDirectory.BaseType.Project,
                        Label = StringResMgr.Get("UNDER_PROJECT"),
                        Path = null,
                        ShowDirectories = true,
                    }
                };
            }
        }

        /// <summary>
        /// 変更判定（UIで変更されうるメンバーだけ比較する）
        /// </summary>
        public bool IsSettingChanged(ProjectSettings other)
        {
            return
                this.DegammaParameterConversionEnabled != other.DegammaParameterConversionEnabled ||
                this.DegammaTextureConversionEnabled != other.DegammaTextureConversionEnabled ||
                this.SRGBWriteEnabled != other.SRGBWriteEnabled ||
                this.ViewMode != other.ViewMode ||
                this.IsPerspectiveDefault != other.IsPerspectiveDefault ||
                this.PerspectiveFOV != other.PerspectiveFOV ||
                this.PerspectiveNear != other.PerspectiveNear ||
                this.PerspectiveFar != other.PerspectiveFar ||
                this.DepthTestEnabled != other.DepthTestEnabled ||
                this.Is30Fps != other.Is30Fps ||
                this.CustomPCViewerWindowTitleHint != other.CustomPCViewerWindowTitleHint ||
                this.MultiControlEnabled != other.MultiControlEnabled ||
                this.AutoFilteredFontGenerationEnabled != other.AutoFilteredFontGenerationEnabled ||
                this.PreviewConverterOptionOmitNoKeyAllEnabled != other.PreviewConverterOptionOmitNoKeyAllEnabled ||
                this.PreSaveCommandPath != other.PreSaveCommandPath ||
                this.PostSaveCommandPath != other.PostSaveCommandPath ||
                this.PreDeleteCommandPath != other.PreDeleteCommandPath ||
                this.PostDeleteCommandPath != other.PostDeleteCommandPath;
        }

        /// <summary>
        /// プロジェクト設定ファイルを書き出します。
        /// </summary>
        public static void WriteProjectSettingsFromPartsRoot(string partsRootPath, ProjectSettings newProjSettings)
        {
            // 有効な部品ルートフォルダが指定されていない場合は、何も行いません。
            if (!Directory.Exists(partsRootPath))
            {
                return;
            }

            string projectFilePath = Path.Combine(partsRootPath, AppConstants.ProjectFileName);

            try
            {
                bool bNotAlreadyExsist = !File.Exists(projectFilePath);

                // すでにファイルがある場合、現在の設定を読んで変更が無ければファイル更新をスキップします。
                if (!bNotAlreadyExsist)
                {
                    using (FileStream fs = new FileStream(projectFilePath, FileMode.Open, FileAccess.Read))
                    {
                        XmlSerializer xs = XmlSerializerCache.GetXmlSerializer(typeof(ProjectSettings));
                        ProjectSettings savedSettings = (ProjectSettings)XmlSerializerCache.Deserialize(xs, fs);

                        if (!savedSettings.IsSettingChanged(newProjSettings))
                        {
                            return;
                        }
                    }
                }

                bool saveResult = LayoutEditorCore.FileSave(projectFilePath, (outFile) =>
                {
                    try
                    {
                        using (FileStream fs = new FileStream(outFile, FileMode.Create))
                        {
                            using (StreamWriter sw = new StreamWriter(fs, Encoding.UTF8))
                            {
                                XmlSerializer xs = XmlSerializerCache.GetXmlSerializer(typeof(ProjectSettings));
                                xs.Serialize(sw, newProjSettings);
                            }
                        }

                        return true;
                    }
                    catch (Exception ex)
                    {
                        LayoutEditorCore.MsgReporter.ReportError(StringResMgr.Get("PARTS_PROJECT_TITLE"), StringResMgr.Get("PARTS_PROJECT_CANTSAVE", ex.Message));
                        return false;
                    }
                });

                // 新規作成の場合は通知します。
                if (saveResult && bNotAlreadyExsist)
                {
                    LayoutEditorCore.MsgReporter.ReportError(StringResMgr.Get("PARTS_PROJECT_TITLE"), StringResMgr.Get("PARTS_PROJECT_NEW", projectFilePath));
                }
            }
            catch (Exception e)
            {
                LayoutEditorCore.MsgReporter.ReportError(StringResMgr.Get("PARTS_PROJECT_TITLE"), StringResMgr.Get("PARTS_PROJECT_CANTSAVE", e.Message));
            }
        }

        /// <summary>
        /// プロジェクト設定ファイルの読み込みを試行します。
        /// 無い場合は新規に作ります。
        /// </summary>
        public static bool TryReadProjectSettingsFromPartsRoot(string partsRootPath, ProjectSettings projSettings)
        {
            // 有効な部品ルートフォルダが指定されていない場合は、何も行いません。
            if (!Directory.Exists(partsRootPath))
            {
                return false;
            }

            // プロジェクト設定ファイルを探す
            string projectFilePath = Path.Combine(partsRootPath, AppConstants.ProjectFileName);
            if (!File.Exists(projectFilePath))
            {
                // 無ければ現在の設定で自動生成する(生成したことをユーザに通知する)
                WriteProjectSettingsFromPartsRoot(partsRootPath, projSettings);
            }
            else
            {
                // 読み込んで設定して返す。
                try
                {
                    using (FileStream fs = new FileStream(projectFilePath, FileMode.Open, FileAccess.Read))
                    {
                        XmlSerializer xs = XmlSerializerCache.GetXmlSerializer(typeof(ProjectSettings));
                        var newSettings = XmlSerializerCache.Deserialize(xs, fs) as ProjectSettings;
                        newSettings.Verify_();

                        projSettings.Set(newSettings);
                    }

                    // スケーラブルフォントを登録します。
                    {
                        LayoutEditorCore.Scene.ClearScalableFont();
                        foreach (ScalableFontDesc scDesc in projSettings.ScalableFontDescSet)
                        {
                            LayoutEditorCore.Scene.RegisterScalableFont(scDesc.FontName, scDesc.FilePath, scDesc.FontSize, scDesc.BoldWeight, scDesc.BorderWidth);
                        }
                    }

                    // プリセット定義ファイルを探して読み込みます。
                    try
                    {
                        LayoutEditorCore.MsgReporter.BeginPacking(StringResMgr.Get("PARTS_PROJECT_TITLE"));
                        var userDataPresetFilesOrderedByName = Directory.GetFiles(partsRootPath, AppConstants.UserDataPresetFileExtPattern).OrderBy((file) => file.ToLowerInvariant());
                        foreach (var userDataPresetFile in userDataPresetFilesOrderedByName)
                        {
                            using (FileStream fs = new FileStream(userDataPresetFile, FileMode.Open, FileAccess.Read))
                            {
                                var presetSettings = XmlSerializerCache.Deserialize(XmlSerializerCache.GetXmlSerializer(typeof(ProjectSettings)), fs) as ProjectSettings;
                                presetSettings.Verify_();

                                foreach (ExUserDataPreset preset in presetSettings.ExUserDataPresets)
                                {
                                    // 同名の設定があった場合は、後から発見された定義で上書きします。
                                    int presetIndex = projSettings.ExUserDataPresets.FindIndex((p) => p.Name == preset.Name);
                                    if (presetIndex != -1)
                                    {
                                        projSettings.ExUserDataPresets.RemoveAt(presetIndex);
                                        projSettings.ExUserDataPresets.Insert(presetIndex, preset);
                                    }else
                                    {
                                        projSettings.ExUserDataPresets.Add(preset);
                                    }
                                }
                            }
                        }
                    }
                    finally
                    {
                        LayoutEditorCore.MsgReporter.EndPacking();
                    }

                    LayoutEditorCore.MsgReporter.OutLog(StringResMgr.Get("PARTS_PROJECT_TITLE"), StringResMgr.Get("PARTS_PROJECT_LOAD", projectFilePath));
                }
                catch (Exception e)
                {
                    LayoutEditorCore.MsgReporter.ReportError(StringResMgr.Get("PARTS_PROJECT_TITLE"), StringResMgr.Get("PARTS_PROJECT_CANTLOAD", e.Message));
                }
            }

            projSettings.FilePath = projectFilePath;

            return true;
        }

        private string SelectNewPaneName(string settingName, string defaultName)
        {
            if (string.IsNullOrEmpty(settingName))
            {
                return defaultName;
            }
            else
            {
                return settingName + (settingName.EndsWith("_") ? "00" : "_00");
            }
        }

        public string GetNewNullPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Null, SubSceneManipulator.DefaultNullPaneName);
        }

        public string GetNewPicturePaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Picture, SubSceneManipulator.DefaultPicturePaneName);
        }

        public string GetNewTextBoxPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Text, SubSceneManipulator.DefaultTextBoxPaneName);
        }

        public string GetNewBoundingPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Bounding, SubSceneManipulator.DefaultBoundingPaneName);
        }

        public string GetNewAlignmentPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Alignment, SubSceneManipulator.DefaultAlignmentPaneName);
        }

        public string GetNewScissorPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Scissor, SubSceneManipulator.DefaultScissorPaneName);
        }

        public string GetNewGroupPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Group, SubSceneManipulator.DefaultGroupName);
        }

        public string GetNewPartsPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Parts, SubSceneManipulator.DefaultPartsPaneName);
        }

        public string GetNewWindowPaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Window, SubSceneManipulator.DefaultWindowPaneName);
        }

        public string GetNewCapturePaneName()
        {
            return SelectNewPaneName(DefaultPaneNames.Capture, SubSceneManipulator.DefaultCapturePaneName);
        }

    }

    #region AppSetting
    /// <summary>
    /// アプリケーション設定パラメータ
    /// </summary>
    public class AppSetting
    {
        /// <summary>
        /// 描画に使用するレンダラの種類
        /// </summary>
        [Flags]
        public enum RendererType
        {
            None	= 0x0,
            GDI		= 0x1,
            D3D		= 0x2
        }

        /// <summary>
        ///
        /// </summary>
        [Flags]
        public enum TargetViewerType
        {
            PCViewer = 0x0,
            AEViewer = 0x1,
        }

        /// <summary>
        /// キー固定処理の種類
        /// </summary>
        [Flags]
        public enum BakeKeyMode
        {
            None = 0x0,
            Tool = 0x1,
            Converter = 0x2
        }

        /// <summary>
        /// 転送先の種類
        /// </summary>
        [Flags]
        public enum TargetScreen
        {
            Auto,
            Upper,
            Lower
        }

        #region 定数
        /// <summary>
        /// 規定の安全フレーム
        /// </summary>
        static SafeFrame _DefaultSafeFrame = new SafeFrame( 10, 10, 10, 10, false, Color.FromArgb( 64, Color.Black ) );
        /// <summary>
        /// バイナリコンバート・タイムアウト時間
        /// </summary>
        static int _DefaultBinaryConvertTimeOutMiliSec = 60 * 1000;

        #endregion 定数

        #region 型宣言
        /// <summary>
        /// アプリケーションの座標系種類
        /// </summary>
        public enum CoordinateType
        {
            Origin_Center,
            Origin_TopLeft
        }

        /// <summary>
        /// プラットフォーム毎の接続先
        /// </summary>
        public class TargetForUiPlatformName
        {
            [XmlAttribute]
            public string UiPlatformName { get; set; }

            [XmlAttribute]
            public TargetViewerType Target { get; set; }
        }
        #endregion 型宣言

        #region フィールド
        // 安全フレーム
        SafeFrame _safeFrame = _DefaultSafeFrame;
        CoordinateType _coordinateType = CoordinateType.Origin_Center;

        /// <summary>
        /// レンダラの種類
        /// </summary>
        RendererType _rendererType				= RendererType.D3D;
        RendererType _AvailableRrendererType	= RendererType.GDI;

        /// <summary>
        /// キー固定処理の種類
        /// </summary>
        BakeKeyMode _bakeKeyType = BakeKeyMode.Converter;

        /// <summary>
        ///
        /// </summary>
        TargetScreen _TargetScreen = TargetScreen.Auto;

        /// <summary>
        /// プレビュー関連設定
        /// </summary>
        bool _drawHiddenPanesAsNull = true;
        bool _convertTextboxControlCode = true;

        /// <summary>
        /// ペイン関連設定
        /// </summary>
        bool _ConveyTransparencyInfluenceOfNull = true;

        /// <summary>
        /// 平行移動操作の影響
        /// </summary>
        bool _transAffectHierarchy = true;
        bool _transAffectsChildrenFlip = false;

        /// <summary>
        /// ドラッグ操作でサイズを変更するか？
        /// (falseの場合は、スケールを変更します。)
        /// </summary>
        bool _paneDragModifierChangesSize = true;

        /// <summary>
        /// バイナリコンバート・タイムアウト時間
        /// </summary>
        int _binaryConvertTimeOutMiliSec = _DefaultBinaryConvertTimeOutMiliSec;

        /// <summary>
        /// アニメーション情報を貼り付けるかどうか
        /// </summary>
        bool _bPasteAnimation = true;

        /// <summary>
        /// 前回よまれた、フォントパス
        /// </summary>
        string _lastLoadedFontFilePath = string.Empty;

        /// <summary>
        /// 前回設定されたスクリーンサイズ
        /// </summary>
        FVec2 _lastSetScreenSize = new FVec2(AppConstants.DefaultScreenSize);

        /// <summary>
        /// 前回バイナリファイルが出力されたパス
        /// </summary>
        string _lastExportedBinaryFilePath = string.Empty;

        /// <summary>
        /// 前回レイアウトファイルが再出力されたフォルダパス
        /// </summary>
        string _lastOutputtedLayoutFolderPath = string.Empty;

        /// <summary>
        /// 前回一括同期されたフォルダパス
        /// </summary>
        string _lastUpdatedLayoutFolderPath = string.Empty;

        /// <summary>
        /// 前回参照したフォントのパス
        /// </summary>
        string _lastLoadedFontPath = string.Empty;

        /// <summary>
        /// 互換性にかんする警告アイコンを表示するか？
        /// </summary>
        bool _bShowCompatibilityAlertIcon = true;

        /// <summary>
        /// 警告ダイアログを無視するか？
        /// </summary>
        List<Forms.MessageReportDlg.MessageTypes> _ignoreSimilarMessage = new List<Forms.MessageReportDlg.MessageTypes>();

        /// <summary>
        /// 親子階層ウインドウで矢印キーによるペイン操作を行うか？
        /// </summary>
        bool _bHierarchyWindowPaneControl = false;

        /// <summary>
        /// ViewerManagerの設定
        /// </summary>
        ViewManagerSetting _viewManagerSetting = new ViewManagerSetting();

        /// <summary>
        /// ViewerManagerのUser設定
        /// </summary>
        ViewManagerSetting _viewManagerUserSetting = new ViewManagerSetting();

        /// <summary>
        /// 前回よまれた、コンバイナファイルパス
        /// </summary>
        string _lastLoadedCombinerFilePath = string.Empty;

        #endregion フィールド

        #region 公開イベント
        /// <summary>
        /// アプリケーション設定変更イベント
        /// </summary>
        public event EventHandler Modified;
        #endregion 公開イベント

        #region プロパティ
        /// <summary>
        /// 安全フレームを設定、取得します。
        /// </summary>
        public SafeFrame SafeFrame
        {
            get { return _safeFrame; }
            set { _safeFrame = value; }
        }

        /// <summary>
        /// 座標系種類を設定、取得します。
        /// </summary>
        public CoordinateType Coordinate
        {
            get { return _coordinateType; }
            set
            {
                // TODO:
                // 途中で、変更があった場合には、
                // データ変換などの実施を行う必要があるかもしれない。
                _coordinateType = value;
            }
        }

        /// <summary>
        /// 隠されたペインをnullとして描画する。
        /// </summary>
        public bool DrawHiddenPanesAsNull
        {
            get { return _drawHiddenPanesAsNull; }
            set { _drawHiddenPanesAsNull = value; }
        }

        /// <summary>
        /// テキストボックス内の制御コードを解釈する。
        /// </summary>
        public bool ConvertTextboxControlCode
        {
            get { return _convertTextboxControlCode; }
            set { _convertTextboxControlCode = value; }
        }

        /// <summary>
        /// 子階層に平行移動操作を影響させるか？
        /// </summary>
        public bool TransAffectHierarchy
        {
            get { return _transAffectHierarchy ^ _transAffectsChildrenFlip; }
            set { _transAffectHierarchy = value; }
        }

        /// <summary>
        /// ドラッグ操作でサイズを変更するか？
        /// (falseの場合は、スケールを変更します。)
        /// </summary>
        public bool PaneDragModifierChangesSize
        {
            get { return _paneDragModifierChangesSize; }
            set { _paneDragModifierChangesSize = value; }
        }


        /// <summary>
        /// 子階層に平行移動操作を影響させるか？
        /// フリップ
        /// </summary>
        public bool TransAffectsChildrenFlip
        {
            get { return _transAffectsChildrenFlip; }
            set { _transAffectsChildrenFlip = value;}
        }

        /// <summary>
        /// バイナリコンバート・タイムアウト時間
        /// </summary>
        public int BinaryConvertTimeOutMiliSec
        {
            get { return _binaryConvertTimeOutMiliSec; }
            set { _binaryConvertTimeOutMiliSec = value; }
        }

        /// <summary>
        /// アニメーション情報を貼り付ける
        /// </summary>
        public bool PasteAnimation
        {
            get { return _bPasteAnimation; }
            set { _bPasteAnimation = value; }
        }

        /// <summary>
        /// 前回よまれた、フォントパス
        /// </summary>
        public string LastLoadedFontFilePath
        {
            get { return _lastLoadedFontFilePath; }
            set { _lastLoadedFontFilePath = value; }
        }

        /// <summary>
        /// 前回設定された画面サイズ
        /// </summary>
        public PointF LastSetScreenSize
        {
            get { return _lastSetScreenSize.AsPointF; }
            set { _lastSetScreenSize.Set( value.X, value.Y ); }
        }

        /// <summary>
        /// 前回バイナリファイルが出力されたパス
        /// </summary>
        public string LastExportedBinaryFilePath
        {
            get { return _lastExportedBinaryFilePath; }
            set { _lastExportedBinaryFilePath = value; }
        }

        /// <summary>
        /// 前回レイアウトファイルが再出力されたフォルダパス
        /// </summary>
        public string LastOutputtedLayoutFolderPath
        {
            get { return _lastOutputtedLayoutFolderPath; }
            set { _lastOutputtedLayoutFolderPath = value; }
        }

        /// <summary>
        /// 前回一括同期されたフォルダパス
        /// </summary>
        public string LastUpdatedLayoutFolderPath
        {
            get { return _lastUpdatedLayoutFolderPath; }
            set { _lastUpdatedLayoutFolderPath = value; }
        }

        /// <summary>
        /// 前回サムネールが一括同期されたフォルダパス
        /// </summary>
        public string LastUpdatedThumbnailFolderPath { get; set; } = string.Empty;

        /// <summary>
        /// Viewer設定
        /// </summary>
        public ViewManagerSetting ViewManagerSetting
        {
            get { return _viewManagerSetting; }
            set { _viewManagerSetting = value; }
        }

        /// <summary>
        /// ViewerのUser設定
        /// </summary>
        public ViewManagerSetting ViewManagerUserSetting
        {
            get { return _viewManagerUserSetting; }
            set { _viewManagerUserSetting = value; }
        }

        /// <summary>
        /// レンダラの種類
        /// </summary>
        public RendererType RendererKind
        {
            get { return _rendererType; }
            set
            {
                if( IsAvailableRrendererType( value ) )
                {
                    _rendererType = value;
                }
            }
        }

        /// <summary>
        /// 接続先の種類
        /// </summary>
        /// <remarks>
        /// プラットフォームの変更に連動して変化します。
        /// 保存はしません。
        /// </remarks>
        [XmlIgnore]
        public TargetViewerType TargetViewerKind
        {
            get
            {
                // 以前の設定の取得
                var target = TargetForUiPlatformNames.FirstOrDefault(x => x.UiPlatformName == UiPlatformName)
                    ?.Target ?? TargetViewerType.AEViewer;

                // 対応している型を返す。
                var preference = LayoutEditorCore.PlatformDetail?.PlatformPreference;
                if (target == TargetViewerType.PCViewer)
                {
                    if (!string.IsNullOrEmpty(preference?.PcTransferProtocol))
                    {
                        return TargetViewerType.PCViewer;
                    }
                }

                if (!string.IsNullOrEmpty(preference?.TargetTransferProtocol))
                {
                    return TargetViewerType.AEViewer;
                }
                else
                {
                    // 少なくとも PC か実機のどちらかには対応しているはず。
                    return TargetViewerType.PCViewer;
                }
            }

            set {
                var item = TargetForUiPlatformNames.FirstOrDefault(x => x.UiPlatformName == UiPlatformName);

                if (item == null)
                {
                    item = new TargetForUiPlatformName { UiPlatformName = UiPlatformName };
                    TargetForUiPlatformNames.Add(item);
                }

                item.Target = value;
            }
        }

        /// <summary>
        /// キー固定処理の種類
        /// </summary>
        public BakeKeyMode BakeKeyKind
        {
            get { return _bakeKeyType; }
            set { _bakeKeyType = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public TargetScreen TargetScreenKind
        {
            get { return _TargetScreen; }
            set { _TargetScreen = value; }
        }

        /// <summary>
        /// 互換性にかんする警告アイコンを表示するか？
        /// </summary>
        public bool ShowCompatibilityAlertIcon
        {
            get { return _bShowCompatibilityAlertIcon; }
            set { _bShowCompatibilityAlertIcon = value; }
        }

        /// <summary>
        /// 警告ダイアログを無視するか？
        /// </summary>
        public List<Forms.MessageReportDlg.MessageTypes> IgnoreSimilarMessage
        {
            get { return _ignoreSimilarMessage; }
            set { _ignoreSimilarMessage = value; }
        }

        /// <summary>
        /// 親子階層ウインドウで矢印キーによるペイン操作を行うか？
        /// </summary>
        public bool HierarchyWindowPaneControl
        {
            get { return _bHierarchyWindowPaneControl; }
            set { _bHierarchyWindowPaneControl = value; }
        }

        /// <summary>
        /// "子階層に透明度の影響を伝える"の初期値(Nullペインのみ)
        /// </summary>
        public bool ConveyTransparencyInfluenceOfNull
        {
            get { return _ConveyTransparencyInfluenceOfNull; }
            set { _ConveyTransparencyInfluenceOfNull = value; }
        }

        /// <summary>
        /// 部品ルートフォルダ(プロジェクトフォルダ)
        /// </summary>
        public string PartsRootPath { get; set; }

        /// <summary>
        /// プラットフォーム名
        /// </summary>
        public string UiPlatformName { get; set; }

        /// <summary>
        /// プラットフォームに対応する接続先
        /// </summary>
        public List<TargetForUiPlatformName> TargetForUiPlatformNames { get; set; }
            = new List<TargetForUiPlatformName>();

        /// <summary>
        /// キャプチャ保存パス
        /// </summary>
        [XmlIgnore]
        public string CaptureDestinationPath { get; set; }

        /// <summary>
        /// キャプチャフレーム
        /// </summary>
        [XmlIgnore]
        public int CaptureFrame { get; set; }

        /// <summary>
        /// キャプチャサイズ
        /// </summary>
        [XmlIgnore]
        public Size CaptureSize { get; set; }

        /// <summary>
        /// プロジェクト設定
        /// </summary>
        [XmlIgnore]
        public ProjectSettings ProjectSettings { get; private set; }

        /// <summary>
        /// アニメーション区間ウインドウ表示中に選択タグ区間を再生範囲とするか
        /// </summary>
        public bool UseTargetFrameSectionRangeAsPlayRange { get; set; }

        /// <summary>
        /// アニメーション区間ウインドウ表示中に選択タグ区間を再生範囲とするか(分割モード用)
        /// </summary>
        public bool UseTargetFrameSectionRangeAsPlayRangeForSeparateMode { get; set; } = true;

        /// <summary>
        /// ビューア転送時に再生範囲を自動的に調整するか
        /// </summary>
        public bool AutoAdjustPlayRange { get; set; } = true;

        /// <summary>
        /// LayoutEditor での変更をビューアに自動的に反映させるか
        /// </summary>
        public bool AutoUpdateViewer { get; set; }

        /// <summary>
        /// アニメーションモードで転送を行うか？
        /// </summary>
        public bool SendViewerAnimation { get; set; }

        /// <summary>
        /// コンバータをプロセスとして実行する
        /// </summary>
        [XmlIgnore]
        public bool DebugConverter { get; set; }

        /// <summary>
        /// 最近使ったファイルの最大数
        /// </summary>
        public const int MaxRecentlyUsedFiles = 20;

        /// <summary>
        /// 最近使ったファイル
        /// </summary>
        public string[] RecentlyUsedFiles { get; set; } = new string[0];

        /// <summary>
        /// 前回よまれた、コンバイナファイルパス
        /// </summary>
        public string LastLoadedCombinerFilePath
        {
            get { return _lastLoadedCombinerFilePath; }
            set { _lastLoadedCombinerFilePath = value; }
        }

        #endregion プロパティ

        /// <summary>
        /// 更新イベントを通知します。
        /// </summary>
        public void RaiseModifyEvent()
        {
            if (Modified != null)
            {
                Modified(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AppSetting()
        {
            this.PartsRootPath = AppConstants.NwDefaultSampleRoot;
            this.CaptureDestinationPath = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);
            this.ProjectSettings = new ProjectSettings();
        }

        /// <summary>
        /// 有効なレンダラを指定します。
        /// 永続化しないデータです。
        /// </summary>
        public void SetAvailableRrendererType( RendererType availableRrendererType )
        {
            _AvailableRrendererType = availableRrendererType | RendererType.GDI;

            // 無効なレンダラが指定されていたら、GDIモードに設定する。
            if( !IsAvailableRrendererType( _rendererType ) )
            {
                _rendererType = RendererType.GDI;
            }
        }

        /// <summary>
        /// 有効なレンダラを取得します。
        /// 永続化しないデータです。
        /// </summary>
        public bool IsAvailableRrendererType( RendererType availableRrendererType )
        {
            return (availableRrendererType & _AvailableRrendererType) != 0x0;
        }

        /// <summary>
        /// プレビュー設定を取得します。
        /// </summary>
        public LECore.ViewerPreviewParam GetPreviewSettings()
        {
            ViewerPreviewParam settings = new ViewerPreviewParam();

            settings.InitTempPath(LECore.Util.TemporaryFileUtil.MakeViewerTempDirectory());

            settings.ConvertTimeOutMiliSec = ViewerPreviewParam.WaitTimeForExit;
            settings.Kind = this.TargetViewerKind == AppSetting.TargetViewerType.AEViewer ? ViewerPreviewParam.ViewerKind.Target : ViewerPreviewParam.ViewerKind.PC;
            settings.CustomPCViewerWindowTitleHint = this.ProjectSettings.CustomPCViewerWindowTitleHint;
            settings.SearchPCViewerByTitle = this.ProjectSettings.SearchPCViewerByTitle
                || (LayoutEditorCore.PlatformDetail?.PlatformPreference?.SearchPCViewerByTitle ?? false);

            settings.IsPerspectiveProjection = this.ProjectSettings.IsPerspectiveDefault;
            settings.PerspectiveFOV = this.ProjectSettings.PerspectiveFOV;
            settings.PerspectiveNear = this.ProjectSettings.PerspectiveNear;
            settings.PerspectiveFar = this.ProjectSettings.PerspectiveFar;
            settings.Is30Fps = this.ProjectSettings.Is30Fps;
            settings.PreArchiverCommandPath = this.ProjectSettings.PreArchiverCommandPath;

            settings.CaptureFrameCount = this.CaptureFrame;
            settings.CaptureSize = this.CaptureSize;
            settings.CaptureFilePath = this.CaptureDestinationPath;

            settings.ConvertTextboxControlCode = this.ConvertTextboxControlCode;

            settings.SRGBWriteEnabled = this.ProjectSettings.SRGBWriteEnabled;
            settings.DegammaParameter = this.ProjectSettings.DegammaParameterConversionEnabled;
            settings.DegammaTexture = this.ProjectSettings.DegammaTextureConversionEnabled;

            settings.PreviewDisplayWidth = this.ProjectSettings.PreviewDisplayWidth;
            settings.PreviewDisplayHeight = this.ProjectSettings.PreviewDisplayHeight;

            settings.ViewMode = this.ProjectSettings.ViewMode;

            settings.DrawHiddenPanesAsNull = this.DrawHiddenPanesAsNull;
            settings.PreviewConverterOptionOmitNoKeyAllEnabled = this.ProjectSettings.PreviewConverterOptionOmitNoKeyAllEnabled;
            settings.AllowNoGroupAnimationTag = this.ProjectSettings.AllowNoGroupAnimationTag;

            settings.AutoFilteredFontGenerationEnabled = this.ProjectSettings.AutoFilteredFontGenerationEnabled;
            settings.AutoFilteredFontGenerationFromSameNameFfntEnabled = this.ProjectSettings.AutoFilteredFontGenerationFromSameNameFfntEnabled;

            settings.PreviewTileOptimize = this.ProjectSettings.TileOptimize;
            settings.PreviewTileSizeThreshold = this.ProjectSettings.TileSizeThreshold;
            settings.IsPreviewGpuEncodingEnabled = this.ProjectSettings.IsGpuEncodingEnabled;

            if (ProjectSettings.IsCombinerUserShaderEnabled && ProjectSettings.UserCombinerShaderSettings.CombinerBlockDefinitionDirectories != null)
            {
                settings.CombinerBlockDefinitionDirectories =
                    ProjectSettings
                    .UserCombinerShaderSettings.CombinerBlockDefinitionDirectories
                    .Select(s => SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, s))
                    .ToList();
            }

            if (!string.IsNullOrWhiteSpace(ProjectSettings.UserCombinerShaderSettings.CombinerEditorPath))
            {
                // 絶対パスか相対パスか調べます。
                // 先頭の文字が '\'(または '/')か、二文字目が ':' の時に True になります。
                if (Path.IsPathRooted(ProjectSettings.UserCombinerShaderSettings.CombinerEditorPath))
                {
                    settings.CombinerEditorPath = ProjectSettings.UserCombinerShaderSettings.CombinerEditorPath;
                }
                else
                {
                    settings.CombinerEditorPath = SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, ProjectSettings.UserCombinerShaderSettings.CombinerEditorPath);
                }
            }

            settings.InitializeParametersDependingOnPlatform();

            settings.ExecuteConverterAsProcess = DebugConverter;

            return settings;
        }

        public static AppSetting DefaultSetting = new AppSetting();
    }
    #endregion AppSetting
}
