﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.Windows.Forms;
using System.Threading;
using System.ComponentModel;
using System.IO;

namespace LayoutEditor
{
    using LECore;
    using LECore.Structures;
    using LECore.Save_Load;
    using LECore.Manipulator;

    using ShellShortcut = LayoutEditor.Utility.ShellShortcut;

    class SaveLoadUIHelper
    {
       #region 保存関連

        /// <summary>
        /// 最後に入出力したフォルダ名
        /// </summary>
        public static string LastFileIODirectoryName { get; set; }

        /// <summary>
        /// 有効なディレクトリか
        /// </summary>
        public static bool IsValidDirectory(string dirName)
        {
            return !string.IsNullOrEmpty(dirName) && Directory.Exists(dirName);
        }

        /// <summary>
        /// 絶対パスを取得します。
        /// </summary>
        public static string GetAbsolutePathForExistDirectory(string path)
        {
            Ensure.Argument.True(Directory.Exists(path));
            var dirInfo = new DirectoryInfo(path);
            return dirInfo.FullName;
        }

        /// <summary>
        /// 出力ダイアログを尋ねる
        /// </summary>
        static public bool AskOutputFilePath(
            ImportExportFileKind kind, string initialFilePath, out string outFilePath)
        {
            outFilePath = string.Empty;

            var exporters = LayoutEditorCore.GetExporter(kind, new ExportOption());
            if (exporters != null && exporters.Count() > 0)
            {
                SaveFileDialog saveFileDlg = new SaveFileDialog();
                saveFileDlg.AddExtension = true;
                saveFileDlg.Filter = SaveLoadHelper.GetFliterString(LayoutEditorCore.GetExporter(ImportExportFileKind.Rlyt, new ExportOption()));

                if (!string.IsNullOrEmpty(initialFilePath))
                {
                    saveFileDlg.FileName = Path.GetFileNameWithoutExtension(initialFilePath);
                    var dirName = Path.GetDirectoryName(initialFilePath);
                    if (IsValidDirectory(dirName))
                    {
                        saveFileDlg.InitialDirectory = GetAbsolutePathForExistDirectory(dirName);
                    }
                }

                if(!IsValidDirectory(saveFileDlg.InitialDirectory))
                {
                    if (IsValidDirectory(LastFileIODirectoryName))
                    {
                        saveFileDlg.InitialDirectory = GetAbsolutePathForExistDirectory(LastFileIODirectoryName);
                    }
                }

                if (saveFileDlg.ShowDialog() == DialogResult.OK)
                {
                    outFilePath = saveFileDlg.FileName;
                    LastFileIODirectoryName = Path.GetDirectoryName(saveFileDlg.FileName);
                }
            }

            return outFilePath != string.Empty;
        }

       #endregion 保存関連

        static SceneManipulator _SceneManipulator;
        static SaveLoadUIHelper()
        {
            _SceneManipulator = new SceneManipulator();
            _SceneManipulator.BindTarget( LayoutEditorCore.Scene );
        }

        /// <summary>
        /// ファイル拡張子がドロップに適しているか判定します。
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        static public bool IsFileExtValidForDrop( string filePath )
        {
            string ext = Path.GetExtension( filePath );
            return string.Compare(ext, AppConstants.LayoutFileExt, true) == 0 ||
                    string.Compare( ext, AppConstants.AnimationFileExt, true ) == 0;
        }

        /// <summary>
        /// コモンダイアログを開き
        /// 入力ファイルパスを取得します。
        /// </summary>
        static public bool AskInputFilePath(
            ImportExportFileKind kind,
            out string[] inputPathSet )
        {
            var importers = LayoutEditorCore.GetImporter(kind);

            // コモンダイアログを開き、入力パスの取得をおこなう。
            OpenFileDialog openFileDlg = new OpenFileDialog();

            // ユーザが拡張子を指定しない場合、拡張子を追加する。
            openFileDlg.AddExtension = true;
            openFileDlg.Multiselect = true;
            openFileDlg.Filter = SaveLoadHelper.GetFliterString(importers);
            if (IsValidDirectory(LastFileIODirectoryName))
            {
                openFileDlg.InitialDirectory = GetAbsolutePathForExistDirectory(LastFileIODirectoryName);
            }

            if( openFileDlg.ShowDialog() == DialogResult.OK )
            {
                inputPathSet = openFileDlg.FileNames;
                LastFileIODirectoryName = Path.GetDirectoryName(openFileDlg.FileName);
            }
            else
            {
                inputPathSet = new string[0];
            }

            return inputPathSet.Length > 0;
        }

        /// <summary>
        /// 最近使ったファイルのファイル名リストを取得します。
        /// </summary>
        public static string[] GetRecentlyUsedFiles( int maxNum )
        {
            // 最近使ったファイル名を取得します。
            List<string> filePathSet = new List<string>();

            string recentPath = Environment.GetFolderPath( Environment.SpecialFolder.Recent );
            string[] recentFilePathSet = Directory.GetFiles( recentPath );
            foreach( string recentFilePath in recentFilePathSet )
            {
                // リンクか？
                if( Path.GetExtension( recentFilePath ) != ".lnk" )
                {
                    continue;
                }

                ShellShortcut shortcut = null;
                try
                {
                    shortcut = new ShellShortcut( recentFilePath );
                }
                catch
                {
                    // 不正なデータで、ShellShortcutインスタンスの生成に失敗
                    // するケースが報告されています。
                    // そのため、不正なデータを無視して、処理を続行するように
                    // します。
                    continue;
                }

                string ext = Path.GetExtension( shortcut.Path );

                // リンク先がレイアウト関連ファイルか？
                if (ext != AppConstants.LayoutFileExt &&
                    ext != AppConstants.AnimationFileExt)
                {
                    continue;
                }

                // ファイルは存在するか？
                if( !File.Exists( shortcut.Path ) )
                {
                    continue;
                }

                // 同一ファイル名(拡張子違い)が存在するか？
                string sameNameFilePath = filePathSet.Find( delegate( string filePath )
                {
                    // 同一ディレクトリで...
                    if( Path.GetDirectoryName( filePath ) ==
                        Path.GetDirectoryName( shortcut.Path ) )
                    {
                        // 拡張子以外が同一であるなら重複とみなし、登録しない。
                        if( Path.GetFileNameWithoutExtension( filePath ) ==
                            Path.GetFileNameWithoutExtension( shortcut.Path ) )
                        {
                            return true;
                        }
                    }
                    return false;
                } );

                if( sameNameFilePath != null )
                {
                    // 古い場合は、削除して登録する。
                    if( File.GetLastAccessTime( sameNameFilePath ) <
                        File.GetLastAccessTime( shortcut.Path ) )
                    {
                        filePathSet.Remove( sameNameFilePath );
                    }
                    else
                    {
                        // 新しい場合は、登録しない。
                        continue;
                    }
                }

                // 有効なファイルとして記録します。
                filePathSet.Add( shortcut.Path );
            }

            // 最終アクセス時間が、新しい順にならべます
            filePathSet.Sort( delegate( string lhs, string rhs )
            {
                if( File.GetLastAccessTime( lhs ) > File.GetLastAccessTime( rhs ) )
                {
                    return -1;
                }
                else if( File.GetLastAccessTime( lhs ) < File.GetLastAccessTime( rhs ) )
                {
                    return  1;
                }
                else
                {
                    return 0;
                }
            } );

            return filePathSet.GetRange( 0, Math.Min( filePathSet.Count, maxNum ) ).ToArray();
        }
    }
}
