﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;
using System.Diagnostics;
using System.Collections.Generic;

namespace LayoutEditor
{
    using LECore;
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using MathUtil = LECore.Util.MathUtil;

    using TexWrap = LECore.Structures.Nsrif.Attributes.AttrTexWrap;
    using TexFilterMin = LECore.Structures.Nsrif.Attributes.AttrTexFilterMin;
    using TexFilterMag = LECore.Structures.Nsrif.Attributes.AttrTexFilterMag;

    using ImageAttributes = System.Drawing.Imaging.ImageAttributes;
    using ColorMatrix = System.Drawing.Imaging.ColorMatrix;
    using LECore.Structures.Nsrif.Attributes;

    class RendererOption
    {
        uint _optionFlag = 0x00000000;

        /// <summary>
        /// 簡易描画モードが有効か調べます。
        /// </summary>
        public bool DrawOutlineEnable
        {
            get { return ( _optionFlag & IRenderOptionFlag.DrawOutline ) != 0x0; }
        }

        /// <summary>
        /// デバック描画が有効か調べます。
        /// </summary>
        public bool DrawDebugInfoEnable
        {
            get { return ( _optionFlag & IRenderOptionFlag.DrawDebugInfo ) != 0x0; }
        }

        /// <summary>
        /// オプションフラグを設定します。
        /// </summary>
        public uint OptionFlag
        {
            get { return _optionFlag; }
            set { _optionFlag = value; }
        }

        public void EnableOptionFlag( uint flag )
        {
            _optionFlag |= flag;
        }

        public void DisableOptionFlag( uint flag )
        {
            _optionFlag &= ~flag;
        }

        public void ToggleOptionFlag( uint flag )
        {
            uint mask = flag;
            uint invMask = ~mask;

            _optionFlag = ( ~( _optionFlag & mask ) & mask ) | ( _optionFlag & invMask );
        }
    }

    /// <summary>
    /// GDI＋モジュールを使用して、描画処理を行うクラスです。
    /// IRenderer インタフェースを実装します。
    /// </summary>
    public class GDIRenderer : IRenderer
    {
        #region -------------- フィールド --------------
        // GDI-グラフィックスインスタンス
        // すべての描画命令の前に、BeginRendering() で設定されます。
        Control _targetControl = null;
        Graphics     _graphics = null;

        // レンダラが使用するペンオブジェクト
        Pen          _pen              = new Pen( Color.Black );
        SolidBrush   _brush            = new SolidBrush( Color.Black );
        Color        _currentColor     = Color.Red;
        Color _materialWhiteColor = Color.White;
        Color _materialBlackColor = Color.Black;
        Stack        _mtxStack         = new Stack( 32 );
        int		  _transparency = PaneHelper.MaxTransparency;

        RendererOption _rendererOption = new RendererOption();

        ColorMatrix _colorMtx = new ColorMatrix();
        ImageAttributes _imageAttribute = new ImageAttributes();

        Bitmap _currentBmp = null;

        #endregion

        #region -------------- プロパティ --------------

        /// <summary>
        /// リニア色空間モードを指定する
        /// (GDIRendererではサポートされていません。テクスチャの変換処理だけが行われるため、テクスチャが白っぽくなります。)
        /// </summary>
        public bool LinearGammmaSRGBFetchEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// リニア色空間でSRGB書き込みを行うかどうかを指定する
        /// (GDIRendererではサポートされていません。)
        /// </summary>
        public bool LinearGammmaSRGBWriteEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// リニア色空間色パラメータモードを指定する(GDIRendererではサポートされていません。)
        /// </summary>
        public bool LinearGammmaColorParamatersEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// レンダリング結果を画面に表示するかどうか(GDIRenderer では意味はありません)
        /// </summary>
        public bool NeedPresent
        {
            get;
            set;
        }

        /// <summary>
        /// 視野角（ラジアン）
        /// </summary>
        public float FOV { get; set; }

        /// <summary>
        /// ニア平面
        /// </summary>
        public float PerseNear { get; set; }

        /// <summary>
        /// ニア平面
        /// </summary>
        public float PerseFar { get; set; }

        /// <summary>
        /// パースぺクティブ表示が有効
        /// </summary>
        public bool PersepectiveRendering { get; set; }

        /// <summary>
        /// 深度テストが有効
        /// </summary>
        public bool DepthWriteTestEnabled { get; set; }

        /// <summary>
        /// モジュールが描画処理を行える状態か判定します。
        /// </summary>
        bool _IsReadyToDraw
        {
            get{ return ( _graphics != null);}
        }

        #endregion



        void PreDrawCoorsinateAdjust_( float x, float y )
        {
//            this.PushMtx();
//            this.Trans( x, y );
//            this.Scale( 1.0f, -1.0f );
        }

        void AfterDrawCoorsinateAdjust_()
        {
//             this.PopMtx();
        }
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GDIRenderer()
        {
#if DEBUG
            // デバック描画有効
            EnableOptionFlag( IRenderOptionFlag.DrawDebugInfo );

#endif // DEBUG
        }



       #region -------------- IRenderer 実装 --------------
        /// <summary>
        ///
        /// </summary>
        public bool Initialize( object initParams )
        {
            _targetControl = initParams as Control;
            return true;
        }

        /// <summary>
        /// 描画処理の準備を行います。
        /// 描画開始時に呼び出してください。
        /// </summary>
        /// <param name="initParams">Graphicsインスタンス。</param>
        public void BeginRendering( object initParams )
        {
            _graphics = initParams as Graphics;
            Debug.Assert( _IsReadyToDraw );
        }

        /// <summary>
        /// 描画処理の後処理を行います。
        /// 描画終了時に呼び出してください。
        /// </summary>
        public void EndRendering()
        {
            Debug.Assert( _IsReadyToDraw );
            _graphics = null;
        }

        /// <summary>
        /// オプションフラグを設定します。
        /// </summary>
        public uint OptionFlag
        {
            get { return _rendererOption.OptionFlag; }
            set { _rendererOption.OptionFlag = value; }
        }

        public void EnableOptionFlag( uint flag )
        {
            _rendererOption.EnableOptionFlag( flag );
        }

        public void DisableOptionFlag( uint flag )
        {
            _rendererOption.DisableOptionFlag( flag );
        }

        public void ToggleOptionFlag( uint flag )
        {
            _rendererOption.ToggleOptionFlag( flag );
        }

        /// <summary>
        /// ポイントセットを現在の変換行列で変換します。
        /// </summary>
        public void TransformPointsToView( PointF[] pointSet )
        {
            for ( int i = 0 ; i < pointSet.Length ; i++ )
            {
                pointSet[i] = MathUtil.MtxTransformPoint( CurrentMtx, pointSet[i] );
            }
        }

        /// <summary>
        /// ポイントセットを現在の逆変換行列で変換します。
        /// </summary>
        public void TransformPointsToScene(PointF[] pointSet)
        {
            Matrix mtx = CurrentMtx;
            mtx.Invert();
            for (int i = 0; i < pointSet.Length; i++)
            {
                pointSet[i] = MathUtil.MtxTransformPoint(mtx, pointSet[i]);
            }
        }

        public void DrawRectangle(float x, float y, float z, float w, float h)
        {
            Debug.Assert( _IsReadyToDraw );

            PreDrawCoorsinateAdjust_( x, y );

            _pen.Color = Color;

            _graphics.DrawRectangle( _pen, x, y, w, h );
            AfterDrawCoorsinateAdjust_();
        }

        public void DrawRectangleOrtho(float x, float y, float z, float w, float h)
        {
            DrawRectangle(x, y, z, w, h);
        }

        public void DrawPoint( PointF p )
        {
            DrawLine(
                new PointF(  p.X, p.Y - LineWidth * 0.5f ),
                new PointF(  p.X, p.Y + LineWidth * 0.5f ) );

            DrawLine(
                new PointF( p.X - LineWidth * 0.5f, p.Y ),
                new PointF( p.X + LineWidth * 0.5f, p.Y ) );
        }

        public void DrawLine( Point p0, Point p1 )
        {
            //PreDrawCoorsinateAdjust_();

            _pen.Color = Color;

            _graphics.DrawLine( _pen, p0, p1 );
            //AfterDrawCoorsinateAdjust_();
        }

        public void DrawLine( PointF p0, PointF p1 )
        {
            //PreDrawCoorsinateAdjust_();

            _pen.Color = Color;

            _graphics.DrawLine( _pen, p0, p1 );
            //AfterDrawCoorsinateAdjust_();
        }

        public void FillRectangle( int x, int y, int z, int w, int h )
        {
            // 簡易描画モード時には無視されます。
            if( _rendererOption.DrawOutlineEnable )
            {
                DrawRectangle(x, y, z, w, h);
                return;
            }

            PreDrawCoorsinateAdjust_( x, y );

            _brush.Color = Color;

            _graphics.FillRectangle( _brush, x, y, w, h );
            AfterDrawCoorsinateAdjust_();
        }
        public void FillRectangle( float x, float y, float z, float w, float h )
        {
            // 簡易描画モード時には無視されます。
            if( _rendererOption.DrawOutlineEnable )
            {
                DrawRectangle( x, y, z, w, h );
                return;
            }


            PreDrawCoorsinateAdjust_( x, y );

            _brush.Color = Color;

            _graphics.FillRectangle( _brush, x, y, w, h );
            AfterDrawCoorsinateAdjust_();
        }

        public void DrawString( string str, System.Drawing.Font font, int x, int y )
        {
            PreDrawCoorsinateAdjust_( x, y );

            _brush.Color = Color;

            _graphics.DrawString( str, font, _brush, x, y );

            AfterDrawCoorsinateAdjust_();
        }

        public void DrawString( string str, System.Drawing.Font font, float x, float y )
        {
            PreDrawCoorsinateAdjust_( x, y );

            _brush.Color = Color;

            _graphics.DrawString( str, font, _brush, x, y );
            AfterDrawCoorsinateAdjust_();
        }

        public void DrawImage(FVec3 imgPos, FVec2 imgSize, TexCoord4[] coord)
        {

            // 簡易描画モード時には無視されます。
            if( _rendererOption.DrawOutlineEnable )
            {
                DrawRectangle(imgPos.X, imgPos.Y, imgPos.Z, imgSize.X, imgSize.Y);
                return;
            }


            Rectangle    destRect = new Rectangle( (int)imgPos.X, (int)imgPos.Y, (int)imgSize.X, (int)imgSize.Y );
            PreDrawCoorsinateAdjust_( destRect.X, destRect.Y );

            if (_currentBmp == null)
            {
                FillRectangle( imgPos.X, imgPos.Y, imgPos.Z, imgSize.X, imgSize.Y );
                return;
            }
            else
            {
                FVec2 uv00 = coord[0].LT;
                FVec2 uv11 = coord[0].RB;

                int srcX = (int)(_currentBmp.Width * uv00.X);
                int srcY = (int)(_currentBmp.Height * uv00.Y);
                int srcW = (int)(_currentBmp.Width * uv11.X) - srcX;
                int srcH = (int)(_currentBmp.Height * uv11.Y) - srcY;


                if( _materialWhiteColor.R != 255 || _materialWhiteColor.G != 255 || _materialWhiteColor.B != 255 ||
                    _materialBlackColor.R != 0 || _materialBlackColor.G != 0 || _materialBlackColor.B != 0 )
                {
                    _colorMtx.Matrix00 = (float)_materialWhiteColor.R / 255.0f;
                    _colorMtx.Matrix11 = (float)_materialWhiteColor.G / 255.0f;
                    _colorMtx.Matrix22 = (float)_materialWhiteColor.B / 255.0f;
                    _colorMtx.Matrix33 = (float)_materialWhiteColor.A / 255.0f;

                    _colorMtx.Matrix40 = (float)_materialBlackColor.R / 255.0f;
                    _colorMtx.Matrix41 = (float)_materialBlackColor.G / 255.0f;
                    _colorMtx.Matrix42 = (float)_materialBlackColor.B / 255.0f;
                    _colorMtx.Matrix43 = (float)_materialBlackColor.A / 255.0f;

                    _imageAttribute.SetColorMatrix( _colorMtx );

                    _graphics.DrawImage(_currentBmp, destRect, srcX, srcY, srcW, srcH, GraphicsUnit.Pixel, _imageAttribute);
                }
                else
                {
                    _graphics.DrawImage(_currentBmp, destRect, srcX, srcY, srcW, srcH, GraphicsUnit.Pixel);
                }
            }

            AfterDrawCoorsinateAdjust_();
        }

        /// <summary>
        /// 頂点カラーを指定しての描画。
        /// </summary>
        public void DrawImage(FVec3 imgPos, FVec2 imgSize, TexCoord4[] coord, Color[] vtxColors)
        {
            Debug.Assert( vtxColors != null );
            // GDI では、頂点カラーをサポートしません。
            // 平均カラーを求めて、設定します。

            int r = 0, g = 0, b = 0, a = 0;
            foreach( Color color in vtxColors )
            {
                r += color.R / vtxColors.Length;
                g += color.G / vtxColors.Length;
                b += color.B / vtxColors.Length;
                a += color.A / vtxColors.Length;
            }

            Color tempColor = _materialWhiteColor;
            _materialWhiteColor = Color.FromArgb( a, r, g, b );

            DrawImage(imgPos, imgSize, coord);

            _materialWhiteColor = tempColor;
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawMesh(float[] vertices, float[] texCoords, int[] indices)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void TryUpdateTextureIfOld(Bitmap bmp, DateTime lastModifyTime, RendererTextrureFormat format)
        {
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="bmp"></param>
        /// <param name="format"></param>
        public void TryRemoveTextureFromBitmap(Bitmap bmp, RendererTextrureFormat format)
        {
        }

        /// <summary>
        /// テクスチャステージを設定
        /// </summary>
        public void SetTextureState(Bitmap bmp, int slotIdx, AttrCombineMode colorBlend, AttrCombineMode alphaBlend, FVec2 indirectScale, float indirectRotate, int indirectImageChunnelNum, RendererTextrureFormat format)
        {
            _currentBmp = bmp;
        }

        /// <summary>
        /// 詳細マテリアルを設定
        /// </summary>
        public void SetDetailedCombinerState(Color[] colors, ITevStage[] tevStage, int stageCount)
        {
        }

        /// <summary>
        /// 角丸機能の基本設定
        /// </summary>
        /// <param name="paneWidth"></param>
        /// <param name="paneHeight"></param>
        /// <param name="exp"></param>
        /// <param name="radius"></param>
        public void SetProceduralShapeState(float paneWidth, float paneHeight, float exp, float radius)
        {
        }

        /// <summary>
        /// 角丸の各効果のブレンド設定。
        /// シェーダーの都合により使用するレジスタが同一のため一緒に設定します。
        /// </summary>
        /// <param name="innerStrokeBlendType"></param>
        /// <param name="innerShadowBlendType"></param>
        /// <param name="colorOverlayBlendType"></param>
        /// <param name="gradationOverlayBlendType"></param>
        public void SetProceduralShapeEffectBlendState(
            ProceduralShapeEffectBlendMode innerStrokeBlendType,
            ProceduralShapeEffectBlendMode innerShadowBlendType,
            ProceduralShapeEffectBlendMode colorOverlayBlendType,
            ProceduralShapeEffectBlendMode gradationOverlayBlendType)
        {
        }

        /// <summary>
        /// 角丸の境界線設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="innerStrokeSize"></param>
        /// <param name="innerStrokeColor"></param>
        public void SetProceduralShapeInnerStrokeState(bool enabled, float innerStrokeSize, Color innerStrokeColor)
        {
        }

        /// <summary>
        /// 角丸のシャドウ(内側)設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="paneWidth"></param>
        /// <param name="paneHeight"></param>
        /// <param name="radius"></param>
        /// <param name="innerShadowSize"></param>
        /// <param name="angle"></param>
        /// <param name="distance"></param>
        /// <param name="innerShadowColor"></param>
        /// <param name="shadowType"></param>
        public void SetProceduralShapeInnerShadowState(bool enabled, float paneWidth, float paneHeight, float radius, float innerShadowSize, float angle, float distance, Color innerShadowColor, ProceduralShapeShadowType shadowType)
        {
        }

        /// <summary>
        /// 角丸のカラーオーバーレイ設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="color"></param>
        public void SetProceduralShapeColorOverlayState(bool enabled, Color color)
        {
        }

        /// <summary>
        /// 角丸のグラデーションオーバーレイ設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="controlPoints"></param>
        /// <param name="colors"></param>
        /// <param name="angle"></param>
        public void SetProceduralShapeGradationOverlayState(bool enabled, float[] controlPoints, Color[] colors, float angle)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void SetBlendMode(AttrBlendType colorBlendtype, AttrBlendFactor srcColor, AttrBlendFactor dstColor, AttrBlendOp colorOp, AttrBlendFactor srcAlpha, AttrBlendFactor dstAlpha, AttrBlendOp alphaOp)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void SetAlphaCompare(bool enabled, AttrCompareFunc alphaCompare, float alpha)
        {
        }

        /// <summary>
        /// デバック描画用：ライン描画
        /// </summary>
        public void DbgDrawLine( PointF p0, PointF p1 )
        {
            if( _rendererOption.DrawOutlineEnable ) { DrawLine( p0, p1 ); }
        }

        /// <summary>
        /// デバック描画用：矩形描画
        /// </summary>
        public void DbgDrawRectangle(float x, float y, float z, float w, float h)
        {
            if (_rendererOption.DrawDebugInfoEnable) { DrawRectangle(x, y, z, w, h); }
        }

        /// <summary>
        /// デバック描画用：文字列描画
        /// </summary>
        public void DbgDrawString( string str, System.Drawing.Font font, float x, float y )
        {
            if( _rendererOption.DrawDebugInfoEnable ) { DrawString( str, font, x, y ); }
        }


        public void SetLineCap( LineCap start, LineCap end )
        {
            _pen.StartCap = start;
            _pen.EndCap = end;
        }

        public void SetTextureMtx( int index, FVec2 trans, FVec2 scale, float rotate )
        {
            // 何もしません。
        }

        public void SetTextureProjectionState(int index, FVec2 imageSize, Matrix34 mxt)
        {
            // 何もしません。
        }

        public void SetPackedFontMode(bool ignoreBorder)
        {
            // 何もしません。
        }

        public void SetTextureSamplingState(
            int idx,
            TexWrap wrapS,
            TexWrap wrapT,
            TexFilterMin minFilter,
            TexFilterMag magFilter,
            bool isSrgb)
        {
            // 何もしません。
        }

        public void SetTextureSamplingSrgbState(
            int idx,
            bool isSrgb)
        {
            // 何もしません。
        }

        public void GetMatrialColorBlend( out Color whiteBlendColor, out Color blackBlendColor )
        {
            whiteBlendColor = _materialWhiteColor;
            blackBlendColor = _materialBlackColor;
        }

        public void SetMatrialColorBlend(FloatColor whiteBlendColor, FloatColor blackBlendColor, bool isThresholdingAlphaInterpolationEnabled)
        {
            _materialWhiteColor = whiteBlendColor.ToSystemColor();
            _materialBlackColor = blackBlendColor.ToSystemColor();
        }

        public void PushMtx()
        {
            Debug.Assert( _IsReadyToDraw );
            _mtxStack.Push( _graphics.Transform );
        }
        public void PopMtx()
        {
            Debug.Assert( _IsReadyToDraw );
            _graphics.Transform = _mtxStack.Pop() as Matrix;
        }
        public void IdentityMtx()
        {
            _graphics.Transform = new Matrix();
        }

        public void Trans( int x, int y )
        {
            Debug.Assert( _IsReadyToDraw );

            Matrix mtx = _graphics.Transform;
            mtx.Translate( x, y );
            _graphics.Transform = mtx;

        }

        public void Trans( float x, float y )
        {
            Debug.Assert( _IsReadyToDraw );

            Matrix mtx = _graphics.Transform;
            mtx.Translate( x, y );
            _graphics.Transform = mtx;
        }

        public void Trans(int x, int y, int z)
        {
            Trans((float)x, (float)y, (float)z);
        }

        public void Trans(float x, float y, float z)
        {
            Debug.Assert(_IsReadyToDraw);

            Matrix mtx = _graphics.Transform;
            mtx.Translate(x, y);
            _graphics.Transform = mtx;
        }

        public void Scale( float sx, float sy )
        {
            Debug.Assert( _IsReadyToDraw );

            Matrix mtx = _graphics.Transform;
            mtx.Scale( sx, sy );
            _graphics.Transform = mtx;
        }

        public void Scale( float sx, float sy, float sz )
        {
            // 3D モデル表示に使用する機能だが、GDIRenderer ではモデル表示に対応していないため行列設定も対応しない。
            Scale(sx, sy);
        }

        /// <summary>
        /// ビュー行列を設定します。
        /// </summary>
        public void SetViewTransform(float sx, float sy, float tx, float ty, float tz, float pz)
        {
            this.Scale(sx, -sy);
            this.Trans(tx, ty);
        }

        /// <summary>
        /// 基準スクリーンサイズ（≒レイアウトサイズ）を設定します。
        /// </summary>
        public void SetDefaultScreenSize(float width, float height)
        {
        }

        public void ClearBitmapTextureMap()
        {
            // 何もしない
        }

        /// <summary>
        /// カレント行列を設定、取得します。
        /// </summary>
        /// <param name="mtx"></param>
        public Matrix CurrentMtx
        {
            get{ return _graphics.Transform;}
            set{ _graphics.Transform = value;}
        }

        /// <summary>
        /// カレント行列を設定、取得します。
        /// </summary>
        /// <param name="mtx"></param>
        public Matrix34 CurrentMtx34
        {
            get { return new Matrix34(); }
            set { }
        }

        /// <summary>
        /// 現在の色を設定、取得します。
        /// </summary>
        public Color Color
        {
            get
            {
                // return _currentColor;
                return Color.FromArgb( (int)(
                    PaneHelper.GetNormalizedTransparency( Transparency ) *
                    _currentColor.A), _currentColor );
            }
            set
            {
                _currentColor   = value;
            }
        }

        /// <summary>
        /// 線の太さを指定します。
        /// </summary>
        public float LineWidth
        {
            get{return _pen.Width;}
            set{_pen.Width = value;}
        }

        /// <summary>
        /// 線のダッシュスタイルを設定します。
        /// </summary>
        public DashStyle LineDashStyle
        {
            get{ return _pen.DashStyle;}
            set{ _pen.DashStyle = value;}
        }

        /// <summary>
        ///
        /// </summary>
        public int Transparency
        {
            get { return _transparency; }
            set
            {
                Debug.Assert( value >= 0 && value <= PaneHelper.MaxTransparency );
                _transparency = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float RectangleTopXOffset
        {
            get;
            set;
        }

        public float FontHeight
        {
            get;
            set;
        }

        public SizeF ViewportSize
        {
            get { return _targetControl != null ? new SizeF(_targetControl.Size) : SizeF.Empty; }
        }

        public bool IsDeviceLost
        {
            get { return false; }
        }
        #endregion

      #region IDisposable メンバ

      /// <summary>
      ///
      /// </summary>
      public void Dispose()
      {
          _pen.Dispose();
          _brush.Dispose();
      }

        #endregion

        /// <summary>
        /// キャプチャ
        /// </summary>
        public Bitmap Capture(DrawerForCapture drawer, Rectangle bounds, bool outputAlpha)
        {
            // TODO: 必要なら outputAlpha に対応

            Bitmap bitmap = new Bitmap(bounds.Width, bounds.Height, PixelFormat.Format32bppArgb);
            drawer(Graphics.FromImage(bitmap));
            return bitmap;
        }
    }
}
