﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;

namespace LayoutEditor.DynamicBinding
{
    using System.IO;
    using System.Runtime.CompilerServices;
    using System.Runtime.InteropServices;
    using LayoutEditor.Utility;
    using LECore;
    using LECore.Structures;
    using LECore.Structures.LECoreInterface;
    using LECore.Structures.Core;
    using SharpDX.Direct3D9;
    using DxCustomVertex = PositionColoredTextured;
    using DXFont = SharpDX.Direct3D9.Font;
    using DXMatrix = SharpDX.Matrix;
    using DXVector3 = SharpDX.Vector3;

    using Vector4 = SharpDX.Vector4;
    using Vector3 = SharpDX.Vector3;
    using Vector2 = SharpDX.Vector2;

    using MathUtil = LECore.Util.MathUtil;
    using SysFont = System.Drawing.Font;
    using SysMatrix = System.Drawing.Drawing2D.Matrix;
    using TexFilterMag = LECore.Structures.Nsrif.Attributes.AttrTexFilterMag;
    using TexFilterMin = LECore.Structures.Nsrif.Attributes.AttrTexFilterMin;
    using TexWrap = LECore.Structures.Nsrif.Attributes.AttrTexWrap;
    using LECore.Structures.Nsrif.Attributes;
    using System.Threading.Tasks;
    using System.Linq;

    public struct DetailedCombinerStage
    {
        public int SourceRgb0;
        public int SourceRgb1;
        public int SourceRgb2;

        public int OperandRgb0;
        public int OperandRgb1;
        public int OperandRgb2;

        public int SourceAlpha0;
        public int SourceAlpha1;
        public int SourceAlpha2;

        public int OperandAlpha0;
        public int OperandAlpha1;
        public int OperandAlpha2;

        public int CombineRgb;
        public int CombineAlpha;

        public int ScaleRgb;
        public int ScaleAlpha;

        public int ConstantRegRgb;
        public int ConstantRegAlpha;
        public int SavePrevRgb;
        public int SavePrevAlpha;

        public int SourceCountRgb;
        public int SourceCountAlpha;
        public int dummy0;
        public int dummy1;
    };

    class DXMatrixStack
    {
        public DXMatrix Top = DXMatrix.Identity;
        public object Tag = null;
        private List<DXMatrix> mStack = new List<DXMatrix>();

        public DXMatrixStack()
        {
            LoadIdentity();
        }

        public void LoadIdentity()
        {
            //mStack.Clear();
            Top = DXMatrix.Identity;
        }

        public void LoadMatrix(DXMatrix pMat)
        {
            //mStack.Clear();
            Top = pMat;
        }

        public void Push()
        {
            mStack.Add(Top);
        }

        public void Pop()
        {
            if (mStack.Count > 0)
            {
                Top = mStack[mStack.Count - 1];
                mStack.RemoveAt(mStack.Count - 1);
            }
        }

        public void MultiplyMatrix(DXMatrix pMat)
        {
            Top = DXMatrix.Multiply(Top, pMat);
        }

        public void MultiplyMatrixLocal(DXMatrix pMat)
        {
            Top = DXMatrix.Multiply(pMat, Top);
        }

        public void Scale(float pX, float pY, float pZ)
        {
            DXMatrix tmp = DXMatrix.Scaling(pX, pY, pZ);
            Top = DXMatrix.Multiply(Top, tmp);
        }

        public void ScaleLocal(float pX, float pY, float pZ)
        {
            DXMatrix tmp = DXMatrix.Scaling(pX, pY, pZ);
            Top = DXMatrix.Multiply(tmp, Top);
        }

        public void TranslateLocal(float pX, float pY, float pZ)
        {
            DXMatrix tmp = DXMatrix.Translation(pX, pY, pZ);
            Top = DXMatrix.Multiply(tmp, Top);
        }
    }

    // 概要:
    /// <summary>
    ///
    /// </summary>
    [NativeCppClassAttribute()]
    [StructLayout(LayoutKind.Sequential)]
    public struct PositionColoredTextured
    {
        //
        // 概要:
        //     Retrieves or sets the x component of the position.
        public float X;
        //
        // 概要:
        //     Retrieves or sets the y component of the position.
        public float Y;
        //
        // 概要:
        //     Retrieves or sets the z component of the position.
        public float Z;
        // 概要:
        //     Retrieves or sets the vertex color.
        public int Color;
        //
        // 概要:
        //     Retrieves or sets the u component of the texture coordinate.
        public float Tu;
        //
        // 概要:
        //     Retrieves or sets the v component of the texture coordinate.
        public float Tv;

        //
        // 概要:
        //     Retrieves or sets the u component of the texture coordinate.
        public float Tu2;
        //
        // 概要:
        //     Retrieves or sets the v component of the texture coordinate.
        public float Tv2;

        //
        // 概要:
        //     Retrieves or sets the u component of the texture coordinate.
        public float Tu3;
        //
        // 概要:
        //     Retrieves or sets the v component of the texture coordinate.
        public float Tv3;

        //
        // 概要:
        //     Retrieves or sets the u component of the texture coordinate.
        public float Tu4;
        //
        // 概要:
        //     Retrieves or sets the v component of the texture coordinate.
        public float Tv4;

        /// <summary>
        /// 位置
        /// </summary>
        public DXVector3 Position
        {
            get
            {
                return new DXVector3(this.X, this.Y, this.Z);
            }

            set
            {
                this.X = value.X;
                this.Y = value.Y;
                this.Z = value.Z;
            }
        }
    }

    public static class DXRendererHelper
    {
        /// <summary>
        /// シェーダー内に指定のコンスタントバッファが存在する場合のみ値を設定するメソッドです。
        /// </summary>
        /// <param name="device"></param>
        /// <param name="ct"></param>
        /// <param name="name"></param>
        /// <param name="data"></param>
        /// <returns></returns>
        public static bool SetPixelShaderConstant(Device device, ConstantTable ct, string name, Vector4[] data)
        {
            EffectHandle handle = ct.GetConstantByName(null, name);
            if ((IntPtr)handle != (IntPtr)0)
            {
                device.SetPixelShaderConstant(ct.GetConstantDescription(handle).RegisterIndex, data);

                return true;
            }

            return false;
        }

        /// <summary>
        /// シェーダー内に指定のコンスタントバッファが存在する場合のみ値を設定するメソッドです。
        /// </summary>
        /// <param name="device"></param>
        /// <param name="ct"></param>
        /// <param name="name"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static bool SetPixelShaderConstant(Device device, ConstantTable ct, string name, int value)
        {
            EffectHandle handle = ct.GetConstantByName(null, name);
            if ((IntPtr)handle != (IntPtr)0)
            {
                ct.SetValue(device, handle, value);

                return true;
            }

            return false;
        }
    }

    /// <summary>
    /// Direct3Dを利用した描画モジュール
    /// </summary>
    class D3DRenderer : IRenderer
    {
        class Mtx23
        {
            public float[] m = new float[6];

            public Mtx23(FVec2 scale, float rotDeg)
            {
                FVec2 center = new FVec2(0.5f, 0.5f);
                float sinR = (float)Math.Sin(MathUtil.DegToRad(rotDeg));
                float cosR = (float)Math.Cos(MathUtil.DegToRad(rotDeg));
                float a0 = cosR * scale.X * 2.0f;
                float a1 = -sinR * scale.Y * 2.0f;

                // SRT の順

                m[0] = a0;
                m[1] = a1;
                m[2] = a0 * (-center.X) + a1 * (-center.Y);

                a0 = sinR * scale.X * 2.0f;
                a1 = cosR * scale.Y * 2.0f;
                m[3] = a0;
                m[4] = a1;
                m[5] = a0 * (-center.X) + a1 * (-center.Y);
            }
        }

        /// <summary>
        /// 頂点バッファの長さ
        /// </summary>
        const int VertexBufferLength = 16384;

        /// <summary>
        /// インデックスバッファの長さ
        /// </summary>
        const int IndexBufferLength = VertexBufferLength * 3;

        /// <summary>
        /// レンダリングステートを変更するコマンドの既定クラス。
        /// </summary>
        class RendererStateChangerBase
        {
            public virtual void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {

            }

            public virtual void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {

            }

            public virtual void Apply(Device device, ConstantTable vsct)
            {

            }

            public virtual void Reset(Device device, ConstantTable vsct)
            {

            }
        }

        /// <summary>
        /// フチフォントシェーダの設定
        /// </summary>
        class PackedFontFormat : RendererStateChangerBase
        {
            readonly bool _ignoreBorder;

            /// <summary>
            ///
            /// </summary>
            public PackedFontFormat(bool ignoreBorder)
            {
                _ignoreBorder = ignoreBorder;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                psct.SetValue(device, psct.GetConstantByName(null, "PackedFontBorderDisabled"), _ignoreBorder ? 1 : 0);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                psct.SetValue(device, psct.GetConstantByName(null, "PackedFontBorderDisabled"), 0);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class ColorBlendData : RendererStateChangerBase
        {
            readonly Color _whiteColor;
            readonly Color _blackColor;
            readonly bool _isThresholdingAlphaInterpolationEnabled;

            public ColorBlendData(Color whiteColor, Color blackColor, bool isThresholdingAlphaInterpolationEnabled)
            {
                _whiteColor = whiteColor;
                _blackColor = blackColor;
                _isThresholdingAlphaInterpolationEnabled = isThresholdingAlphaInterpolationEnabled;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] width = new Vector4[] { new Vector4(_whiteColor.R / 255.0f, _whiteColor.G / 255.0f, _whiteColor.B / 255.0f, _whiteColor.A / 255.0f) };
                Vector4[] offset = new Vector4[] { new Vector4(_blackColor.R / 255.0f, _blackColor.G / 255.0f, _blackColor.B / 255.0f, _blackColor.A / 255.0f) };
                width[0] = width[0] - offset[0];

                if (_isThresholdingAlphaInterpolationEnabled)
                {
                    float a = 1.0f / ((_whiteColor.A / 255.0f) - (_blackColor.A / 255.0f) + 0.0001f); // 傾きaを求めます。発散を防ぐために微小な値を足しています。

                    width[0].W = a;
                    offset[0].W = -(_blackColor.A / 255.0f) * a;
                }

                DXRendererHelper.SetPixelShaderConstant(device, psct, "InterpolateWidth", width);
                DXRendererHelper.SetPixelShaderConstant(device, psct, "InterpolateOffset", offset);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] width = new Vector4[] { new Vector4(1.0f, 1.0f, 1.0f, 1.0f) };
                Vector4[] offset = new Vector4[] { new Vector4(0f, 0f, 0f, 0.0f) };

                DXRendererHelper.SetPixelShaderConstant(device, psct, "InterpolateWidth", width);
                DXRendererHelper.SetPixelShaderConstant(device, psct, "InterpolateOffset", offset);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class PEEnviromentData : RendererStateChangerBase
        {
            readonly AttrBlendFactor _srcColor;
            readonly AttrBlendFactor _dstColor;
            readonly AttrBlendOp _colorOp;

            readonly AttrBlendFactor _srcAlpha;
            readonly AttrBlendFactor _dstAlpha;
            readonly AttrBlendOp _alphaOp;
            readonly bool _alphaBlendEnabled;

            public PEEnviromentData(
                AttrBlendType colorBlendtype,
                AttrBlendFactor srcColor, AttrBlendFactor dstColor, AttrBlendOp colorOp,
                AttrBlendFactor srcAlpha, AttrBlendFactor dstAlpha, AttrBlendOp alphaOp)
            {
                _alphaBlendEnabled = colorBlendtype == AttrBlendType.Blend;

                _srcColor = srcColor;
                _dstColor = dstColor;
                _colorOp = colorOp;

                _srcAlpha = srcAlpha;
                _dstAlpha = dstAlpha;
                _alphaOp = alphaOp;
            }

            BlendOperation ConvertBlendOp_(AttrBlendOp op)
            {
                switch (op)
                {
                    case AttrBlendOp.Add: return BlendOperation.Add;
                    case AttrBlendOp.Subtract: return BlendOperation.Subtract;
                    case AttrBlendOp.ReverseSubtract: return BlendOperation.ReverseSubtract;
                    case AttrBlendOp.SelectMax: return BlendOperation.Maximum;
                    case AttrBlendOp.SelectMin: return BlendOperation.Minimum;
                    default: return BlendOperation.Add;
                }
            }

            Blend ConvertBlend_(AttrBlendFactor blend)
            {
                switch (blend)
                {
                    case AttrBlendFactor.DstAlpha: return Blend.DestinationAlpha;
                    case AttrBlendFactor.DstClr: return Blend.DestinationColor;
                    case AttrBlendFactor.InvDstAlpha: return Blend.InverseDestinationAlpha;
                    case AttrBlendFactor.InvDstClr: return Blend.InverseDestinationColor;
                    case AttrBlendFactor.InvSrcAlpha: return Blend.InverseSourceAlpha;
                    case AttrBlendFactor.InvSrcClr: return Blend.InverseSourceColor;
                    case AttrBlendFactor.SrcAlpha: return Blend.SourceAlpha;
                    case AttrBlendFactor.SrcClr: return Blend.SourceColor;
                    case AttrBlendFactor.V0: return Blend.Zero;
                    case AttrBlendFactor.V1_0: return Blend.One;
                    default: return Blend.Zero;
                }
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetRenderState(RenderState.AlphaBlendEnable, _alphaBlendEnabled);

                device.SetRenderState(RenderState.SourceBlend, ConvertBlend_(_srcColor));
                device.SetRenderState(RenderState.DestinationBlend, ConvertBlend_(_dstColor));
                device.SetRenderState(RenderState.BlendOperation, ConvertBlendOp_(_colorOp));

                device.SetRenderState(RenderState.SeparateAlphaBlendEnable, true);

                device.SetRenderState(RenderState.SourceBlendAlpha, ConvertBlend_(_srcAlpha));
                device.SetRenderState(RenderState.DestinationBlendAlpha, ConvertBlend_(_dstAlpha));
                device.SetRenderState(RenderState.BlendOperationAlpha, ConvertBlendOp_(_alphaOp));
            }

            ///
            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetRenderState(RenderState.AlphaBlendEnable, true);

                device.SetRenderState(RenderState.SourceBlend, Blend.SourceAlpha);
                device.SetRenderState(RenderState.DestinationBlend, Blend.InverseSourceAlpha);
                device.SetRenderState(RenderState.BlendOperation, BlendOperation.Add);

                device.SetRenderState(RenderState.SeparateAlphaBlendEnable, false);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class AlphaCompare : RendererStateChangerBase
        {
            readonly bool _enabled;
            readonly AttrCompareFunc _alphaCompare;
            readonly float _alpha;

            public AlphaCompare(bool enabled, AttrCompareFunc alphaCompare, float alpha)
            {
                _enabled = enabled;
                _alphaCompare = alphaCompare;
                _alpha = alpha;
            }

            Compare ConvertBlendOp_(AttrCompareFunc alphaCompare)
            {
                switch (alphaCompare)
                {
                    case AttrCompareFunc.Always: return Compare.Always;
                    case AttrCompareFunc.Equal: return Compare.Equal;
                    case AttrCompareFunc.Gequal: return Compare.GreaterEqual;
                    case AttrCompareFunc.Greater: return Compare.Greater;
                    case AttrCompareFunc.Lequal: return Compare.LessEqual;
                    case AttrCompareFunc.Less: return Compare.Less;
                    case AttrCompareFunc.Nequal: return Compare.NotEqual;
                    case AttrCompareFunc.Never: return Compare.Never;
                    default: return Compare.Always;
                }
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetRenderState(RenderState.AlphaTestEnable, _enabled);
                device.SetRenderState(RenderState.AlphaFunc, ConvertBlendOp_(_alphaCompare));
                device.SetRenderState(RenderState.AlphaRef, (int)(_alpha * 255));
            }

            ///
            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetRenderState(RenderState.AlphaTestEnable, false);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class TextureEnviromentData : RendererStateChangerBase
        {
            const int COMBINE_REPLACE = 2;
            const int COMBINE_MODULATE = 3;
            const int COMBINE_ADD = 4;
            const int COMBINE_SUBTRACT = 5;
            const int COMBINE_EXCLUSION = 6;
            const int COMBINE_COLOR_DODGE = 7;
            const int COMBINE_COLOR_BURN = 8;
            const int COMBINE_OVERLAY = 9;
            const int COMBINE_LIGHTEN = 10;
            const int COMBINE_DARKEN = 11;

            const int COMBINE_INDIRECT = 14;
            const int COMBINE_BLEND_INDIRECT = 15;
            const int COMBINE_EACH_INDIRECT = 16;

            readonly Texture _texture;
            readonly int     _textureSlotIdx;
            readonly AttrCombineMode _colorBlendMode;
            readonly AttrCombineMode _alphaBlendMode;
            readonly Mtx23 _indirectMtx;
            readonly int _indirectImgChannelNum;

            public TextureEnviromentData(Texture texture, int textureSlotIdx, AttrCombineMode colorBlendMode, AttrCombineMode alphaBlendMode, FVec2 indirectScale, float indirectRotate, int indirectImgChannelNum)
            {
                Ensure.Argument.True(textureSlotIdx < 8);

                _texture = texture;
                _textureSlotIdx = textureSlotIdx;
                _colorBlendMode = colorBlendMode;
                _alphaBlendMode = alphaBlendMode;
                _indirectMtx = new Mtx23(indirectScale, indirectRotate);
                _indirectImgChannelNum = indirectImgChannelNum;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetTexture(_textureSlotIdx, _texture);

                DXRendererHelper.SetPixelShaderConstant(device, psct, "EnabledTextureCount", _textureSlotIdx + 1);

                if (_textureSlotIdx == 1 || _textureSlotIdx == 2)
                {
                    DXRendererHelper.SetPixelShaderConstant(device, psct, "CombineModeStage" + _textureSlotIdx.ToString(), SetTexOptId_(_colorBlendMode));
                    DXRendererHelper.SetPixelShaderConstant(device, psct, "SelectAlphaMaxStage" + _textureSlotIdx.ToString(), SetTexAlphaBlendOptId_(_alphaBlendMode));

                    if (_colorBlendMode >= AttrCombineMode.Indirect)
                    {
                        Vector4[] mtx1 = new Vector4[] { new Vector4(_indirectMtx.m[0],_indirectMtx.m[1],0.0f, _indirectMtx.m[2]) };
                        Vector4[] mtx2 = new Vector4[] { new Vector4(_indirectMtx.m[3],_indirectMtx.m[4],0.0f, _indirectMtx.m[5]) };

                        DXRendererHelper.SetPixelShaderConstant(device, psct, "IndirectMtx1", mtx1);
                        DXRendererHelper.SetPixelShaderConstant(device, psct, "IndirectMtx2", mtx2);
                        DXRendererHelper.SetPixelShaderConstant(device, psct, "IndirectImgChannelNum", _indirectImgChannelNum);
                    }
                }
            }

            ///
            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                // D3DRS_ALPHABLENDENABLE
                // D3DRS_SRCBLEND
                // D3DRS_DESTBLEND
                // D3DRS_BLENDOP

                // D3DRS_SRCBLENDALPHA
                // D3DRS_DESTBLENDALPHA
                // D3DRS_BLENDOPALPHA

                // D3DRS_ALPHAREF
                // D3DRS_ALPHAFUNC
                // D3DRS_ALPHATESTENABLE

                DXRendererHelper.SetPixelShaderConstant(device, psct, "EnabledTextureCount", 0);
                if (_textureSlotIdx == 1 || _textureSlotIdx == 2)
                {
                    DXRendererHelper.SetPixelShaderConstant(device, psct, "CombineModeStage" + _textureSlotIdx.ToString(), 0);
                    DXRendererHelper.SetPixelShaderConstant(device, psct, "SelectAlphaMaxStage" + _textureSlotIdx.ToString(), 0);
                }
            }

            private int SetTexOptId_(AttrCombineMode blendMode)
            {
                switch (blendMode)
                {
                    case AttrCombineMode.Modulate: return COMBINE_MODULATE;
                    case AttrCombineMode.Add:return COMBINE_ADD;
                    case AttrCombineMode.Subtract: return COMBINE_SUBTRACT;
                    case AttrCombineMode.Replace: return COMBINE_REPLACE;

                    case AttrCombineMode.AddSigned: return COMBINE_EXCLUSION;
                    case AttrCombineMode.AddMultDMP: return COMBINE_COLOR_DODGE;
                    case AttrCombineMode.MultAddDMP: return COMBINE_COLOR_BURN;
                    case AttrCombineMode.Overlay: return COMBINE_OVERLAY;
                    case AttrCombineMode.Lighten: return COMBINE_LIGHTEN;
                    case AttrCombineMode.Darken: return COMBINE_DARKEN;

                    case AttrCombineMode.Indirect: return COMBINE_INDIRECT;
                    case AttrCombineMode.BlendIndirect: return COMBINE_BLEND_INDIRECT;
                    case AttrCombineMode.EachIndirect: return COMBINE_EACH_INDIRECT;

                    default:return COMBINE_REPLACE;
                }
            }

            private int SetTexAlphaBlendOptId_(AttrCombineMode blendMode)
            {
                switch (blendMode)
                {
                    case AttrCombineMode.Replace: return 1;
                    case AttrCombineMode.Modulate: return 0;
                    default: return COMBINE_REPLACE;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private class DetailedCombinerData : RendererStateChangerBase
        {
            private ITevStage[] _tevStage;
            private Color[] _colors;
            private int _stageCount;
            private int _texCount;
            private ConstantTable _constantTable;

            public DetailedCombinerData(Color[] colors, ITevStage[] tevStage, int stageCount, int texCount, ConstantTable psct)
            {
                _tevStage = tevStage;
                _colors = colors;
                _stageCount = stageCount;
                _texCount = texCount;
                _constantTable = psct;
            }

            ///
            public override void Apply(Device device, ConstantTable vsct)
            {
                if (_stageCount <= 0 || _tevStage == null)
                {
                    if (_colors != null)
                    {
                        Vector4 color = new Vector4(_colors[5].R / 255.0f, _colors[5].G / 255.0f, _colors[5].B / 255.0f, _colors[5].A / 255.0f);

                        EffectHandle handle = _constantTable.GetConstantByName(null, "DetailedCombiner_ConstantColor" + (5).ToString());
                        _constantTable.SetValue<Vector4>(device, handle, color);
                    }

                    return;
                }

                // 詳細マテリアル
                if (_stageCount > 0)
                {
                    _constantTable.SetValue(device, _constantTable.GetConstantByName(null, "DetailedCombinerTexCount"), _texCount);
                }

                for (int i = 0; i < _stageCount; i++)
                {
                    DetailedCombinerStage combinerStage = new DetailedCombinerStage();

                    // ＲＧＢソース段数
                    switch (_tevStage[i].ColorStage.CombineMode)
                    {
                    case AttrCombineMode.Replace:
                        combinerStage.SourceCountRgb = 1;
                        break;
                    case AttrCombineMode.Add:
                    case AttrCombineMode.Modulate:
                    case AttrCombineMode.AddSigned:
                    case AttrCombineMode.Subtract:
                        combinerStage.SourceCountRgb = 2;
                        break;
                    case AttrCombineMode.Interpolate:
                    case AttrCombineMode.AddMultDMP:
                    case AttrCombineMode.MultAddDMP:
                        combinerStage.SourceCountRgb = 3;
                        break;
                    default:
                        combinerStage.SourceCountRgb = 0;
                        break;
                    }

                    combinerStage.CombineRgb = (int)_tevStage[i].ColorStage.CombineMode;

                    // アルファソース段数
                    switch(_tevStage[i].AlphaStage.CombineMode)
                    {
                    case AttrCombineMode.Replace:
                        combinerStage.SourceCountAlpha = 1;
                        break;
                    case AttrCombineMode.Add:
                    case AttrCombineMode.Modulate:
                    case AttrCombineMode.AddSigned:
                    case AttrCombineMode.Subtract:
                        combinerStage.SourceCountAlpha = 2;
                        break;
                    case AttrCombineMode.Interpolate:
                    case AttrCombineMode.AddMultDMP:
                    case AttrCombineMode.MultAddDMP:
                        combinerStage.SourceCountAlpha = 3;
                        break;
                    default:
                        combinerStage.SourceCountAlpha = 0;
                        break;
                    }

                    combinerStage.CombineAlpha = (int)_tevStage[i].AlphaStage.CombineMode;

                    for (int j = 0; j < _tevStage[i].ColorStage.TevArgs.Length; j++)
                    {
                        combinerStage.ConstantRegRgb = (int)_tevStage[i].ColorStage.Konst;
                        combinerStage.ConstantRegAlpha = (int)_tevStage[i].AlphaStage.Konst;
                        combinerStage.SavePrevRgb = (int)(_tevStage[i].ColorStage.CopyReg ? 1 : 0);
                        combinerStage.SavePrevAlpha = (int)(_tevStage[i].AlphaStage.CopyReg ? 1 : 0);
                        combinerStage.ScaleRgb = (int)_tevStage[i].ColorStage.Scale;
                        combinerStage.ScaleAlpha = (int)_tevStage[i].AlphaStage.Scale;

                        switch(j)
                        {
                        case 0:
                            combinerStage.SourceRgb0 = (int)_tevStage[i].ColorStage.TevArgs[j].Source;
                            combinerStage.OperandRgb0 = (int)_tevStage[i].ColorStage.TevArgs[j].Op;

                            combinerStage.SourceAlpha0 = (int)_tevStage[i].AlphaStage.TevArgs[j].Source;
                            combinerStage.OperandAlpha0 = (int)_tevStage[i].AlphaStage.TevArgs[j].Op;
                            break;
                        case 1:
                            combinerStage.SourceRgb1 = (int)_tevStage[i].ColorStage.TevArgs[j].Source;
                            combinerStage.OperandRgb1 = (int)_tevStage[i].ColorStage.TevArgs[j].Op;

                            combinerStage.SourceAlpha1 = (int)_tevStage[i].AlphaStage.TevArgs[j].Source;
                            combinerStage.OperandAlpha1 = (int)_tevStage[i].AlphaStage.TevArgs[j].Op;
                            break;
                        case 2:
                            combinerStage.SourceRgb2 = (int)_tevStage[i].ColorStage.TevArgs[j].Source;
                            combinerStage.OperandRgb2 = (int)_tevStage[i].ColorStage.TevArgs[j].Op;

                            combinerStage.SourceAlpha2 = (int)_tevStage[i].AlphaStage.TevArgs[j].Source;
                            combinerStage.OperandAlpha2 = (int)_tevStage[i].AlphaStage.TevArgs[j].Op;
                            break;
                        default:
                            Debug.Assert(false);
                            break;
                        }
                    }

                    EffectHandle handle = _constantTable.GetConstantByName(null, "DetailedCombiner_Stages" + i.ToString());
                    _constantTable.SetValue<DetailedCombinerStage>(device, handle, combinerStage);
                }

                // コンスタントカラー
                for (int i = 0; i < _colors.Length; i++)
                {
                    Vector4 color = new Vector4(_colors[i].R / 255.0f, _colors[i].G / 255.0f, _colors[i].B / 255.0f, _colors[i].A / 255.0f);

                    EffectHandle handle = _constantTable.GetConstantByName(null, "DetailedCombiner_ConstantColor" + i.ToString());
                    _constantTable.SetValue<Vector4>(device, handle, color);
                }
            }

            ///
            public override void Reset(Device device, ConstantTable vsctt)
            {
            }

        }

        /// <summary>
        /// テクスチャ投影を設定します。
        /// </summary>
        class TextureProjectionState : RendererStateChangerBase
        {
            readonly int _index;
            FVec2 _imageSize;
            Matrix34 _mxt;

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public TextureProjectionState(int index, FVec2 imageSize, Matrix34 mxt)
            {
                _index = index;
                _imageSize = imageSize;
                _mxt = mxt;
            }

            /// <summary>
            /// 設定
            /// </summary>
            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                float hW = _imageSize.X;
                float hH = _imageSize.Y;

                DXMatrix mtxInvView = DXMatrix.Invert(device.GetTransform(TransformState.View)); // 現在のビュー行列を取得します。
                DXMatrix mtxTexProjection = _mxt.ToSharpDXMatrix34();
                DXMatrix mtxTexCenterAdjust = DXMatrix.Translation(0.5f, 0.5f, 0.0f);

                // 投影変換を行う、新しいテクスチャ行列を計算します。
                DXMatrix mtxNewTex =
                    mtxTexProjection *　// 投影スケール、オフセット変換
                    CalcTextureProjectionOrtho_(-hW, hW, -hH, hH, 1.0f, -1.0f, -mtxInvView.M14, -mtxInvView.M24) * // 投影行列
                    mtxTexCenterAdjust;  // テクスチャ中心調整

                vsct.SetValue<int>(device, vsct.GetConstantByName(null, "isTexProjection" + _index.ToString()), 1);
                device.SetVertexShaderConstant(vsct.GetConstantDescription(vsct.GetConstantByName(null, "texProjectionMtx" + _index.ToString())).RegisterIndex, mtxNewTex);
            }

            /// <summary>
            /// 設定の復元
            /// </summary>
            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                vsct.SetValue<int>(device, vsct.GetConstantByName(null, "isTexProjection" + _index.ToString()), 0);
            }

            /// <summary>
            /// テクスチャマトリクス用平行射影行列を作成します。
            /// </summary>
            private DXMatrix CalcTextureProjectionOrtho_(
                float l, float r, float b, float t, float scaleS, float scaleT, float translateS, float translateT)
            {
                DXMatrix res = DXMatrix.Identity;
                float reverseWidth = 1.0f / (r - l);

                res.M11 = 2.0f * reverseWidth * scaleS;
                res.M12 = 0.0f;
                res.M13 = 0.0f;
                res.M14 = ((-(r + l) * reverseWidth) * scaleS) + translateS;

                float reverseHeight = 1.0f / (t - b);
                res.M21 = 0.0f;
                res.M22 = (2.0f * reverseHeight) * scaleT;
                res.M23 = 0.0f;
                res.M24 = ((-(t + b) * reverseHeight) * scaleT) + translateT;

                res.M31 = 0.0f;
                res.M32 = 0.0f;
                res.M33 = 0.0f;
                res.M34 = 1.0f;
                return DXMatrix.Transpose(res);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class TextureMtxData : RendererStateChangerBase
        {
            readonly int _index;
            readonly DXMatrix _mtx;

            /// <summary>
            ///
            /// </summary>
            public TextureMtxData(int index, DXMatrix mtx)
            {
                _index = index;
                _mtx = DXMatrix.Transpose(mtx);
            }

            /// <summary>
            ///
            /// </summary>
            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetVertexShaderConstant(vsct.GetConstantDescription(vsct.GetConstantByName(null, "texture_matrix" + _index.ToString())).RegisterIndex, _mtx);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class SamplerStateData : RendererStateChangerBase
        {
            readonly int _index;
            readonly TextureAddress _textureAddressU;
            readonly TextureAddress _textureAddressV;
            readonly TextureFilter _minFilter;
            readonly TextureFilter _magFilter;
            readonly bool _isSrgb;

            public SamplerStateData(int index, TextureAddress addressU, TextureAddress addressV, TextureFilter minFilter, TextureFilter magFilter, bool isSrgb)
            {
                _index = index;

                _textureAddressU = addressU;
                _textureAddressV = addressV;

                _minFilter = minFilter;
                _magFilter = magFilter;

                _isSrgb = isSrgb;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetSamplerState(_index, SamplerState.AddressU, _textureAddressU);
                device.SetSamplerState(_index, SamplerState.AddressV, _textureAddressV);
                device.SetSamplerState(_index, SamplerState.MinFilter, _minFilter);
                device.SetSamplerState(_index, SamplerState.MagFilter, _magFilter);
                device.SetSamplerState(_index, SamplerState.SrgbTexture, _isSrgb ? 1 : 0);
            }
        }

        class SamplerStateSrgbData : RendererStateChangerBase
        {
            readonly int _index;
            readonly bool _isSrgb;

            public SamplerStateSrgbData(int index, bool isSrgb)
            {
                _index = index;
                _isSrgb = isSrgb;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                device.SetSamplerState(_index, SamplerState.SrgbTexture, _isSrgb ? 1 : 0);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class DepthTestStateData : RendererStateChangerBase
        {
            readonly bool _depthTestEnabled;

            public DepthTestStateData(bool depthTestEnabled)
            {
                _depthTestEnabled = depthTestEnabled;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                if (_depthTestEnabled)
                {
                    device.SetRenderState(RenderState.ZEnable, true);
                    device.SetRenderState(RenderState.ZWriteEnable, true);
                    device.SetRenderState(RenderState.ZFunc, Compare.LessEqual);
                }
                else
                {
                    device.SetRenderState(RenderState.ZEnable, false);
                    device.SetRenderState(RenderState.ZWriteEnable, false);
                }
            }
        }

        /// <summary>
        /// 角丸に関するシェーダー設定
        /// </summary>
        class ProceduralShapeState : RendererStateChangerBase
        {
            readonly float _paneWidth;
            readonly float _paneHeight;
            readonly float _exp;
            readonly float _radius;

            public ProceduralShapeState(float paneWidth, float paneHeight, float exp, float radius)
            {
                _paneWidth = paneWidth;
                _paneHeight = paneHeight;
                _exp = RendererHelper.Clamp(exp, 0.5f, 3.0f);
                _radius = radius;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                DXRendererHelper.SetPixelShaderConstant(device, psct, "ProceduralShapeEnabled", 1);

                // ui2d ランタイムでのコンスタントバッファ設定を移植
                float halfWidth = _paneWidth * 0.5f;
                float halfHeight = _paneHeight * 0.5f;
                float cornerWidth = _radius / halfWidth;
                if (cornerWidth == 0.0f)
                {
                    cornerWidth = 0.00001f;
                }
                float cornerHeight = _radius / halfHeight;
                if (cornerHeight == 0.0f)
                {
                    cornerHeight = 0.00001f;
                }
                float cornerOffsetX = 1.0f - cornerWidth;
                cornerOffsetX = RendererHelper.Clamp(cornerOffsetX, 0.0f, 1.0f);
                float cornerOffsetY = 1.0f - cornerHeight;
                cornerOffsetY = RendererHelper.Clamp(cornerOffsetY, 0.0f, 1.0f);

                float radius = _radius;
                radius = RendererHelper.Clamp(radius, 0.0f, halfWidth);
                radius = RendererHelper.Clamp(radius, 0.0f, halfHeight);

                Vector4[] proceduralShapeParams = new Vector4[] { new Vector4(_exp, cornerOffsetX, cornerOffsetY, 0.0f) };
                Vector4[] proceduralShapeSizeParams = new Vector4[] { new Vector4(_paneWidth, _paneHeight, radius, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uProceduralShapeParams", proceduralShapeParams);
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uProceduralShapeSizeParams", proceduralShapeSizeParams);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                DXRendererHelper.SetPixelShaderConstant(device, psct, "ProceduralShapeEnabled", 0);
            }
        }

        /// <summary>
        /// 角丸の各エフェクトのブレンド設定
        /// </summary>
        class ProceduralShapeEffectBlendState : RendererStateChangerBase
        {
            readonly ProceduralShapeEffectBlendMode _innerStrokeBlendMode;
            readonly ProceduralShapeEffectBlendMode _innerShadowBlendMode;
            readonly ProceduralShapeEffectBlendMode _colorOverlayBlendMode;
            readonly ProceduralShapeEffectBlendMode _gradationOverlayBlendMode;

            public ProceduralShapeEffectBlendState(
                ProceduralShapeEffectBlendMode innerStrokeBlendType,
                ProceduralShapeEffectBlendMode innerShadowBlendType,
                ProceduralShapeEffectBlendMode colorOverlayBlendType,
                ProceduralShapeEffectBlendMode gradationOverlayBlendType)
            {
                _innerStrokeBlendMode = innerStrokeBlendType;
                _innerShadowBlendMode = innerShadowBlendType;
                _colorOverlayBlendMode = colorOverlayBlendType;
                _gradationOverlayBlendMode = gradationOverlayBlendType;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] blendParams = new Vector4[] { new Vector4((float)_innerStrokeBlendMode, (float)_innerShadowBlendMode, (float)_colorOverlayBlendMode, (float)_gradationOverlayBlendMode) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uBlendType", blendParams);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
            }
        }

        /// <summary>
        /// 角丸の境界線効果設定
        /// </summary>
        class ProceduralShapeInnerStrokeState : RendererStateChangerBase
        {
            readonly float _innerStrokeSize;
            readonly Vector4[] _innerStrokeColor;
            public ProceduralShapeInnerStrokeState(bool enabled, float innerStrokeSize, Color innerStrokeColor)
            {
                _innerStrokeSize = innerStrokeSize;
                _innerStrokeColor = new Vector4[] {
                    new Vector4((float)innerStrokeColor.R / 255.0f,
                                (float)innerStrokeColor.G / 255.0f,
                                (float)innerStrokeColor.B / 255.0f,
                                enabled ? (float)innerStrokeColor.A / 255.0f : 0.0f) };
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] innerStrokeParams = new Vector4[] { new Vector4(_innerStrokeSize, 0.0f, 0.0f, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uInnerStrokeParams", innerStrokeParams);
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uInnerStrokeColor", _innerStrokeColor);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] innerStrokeColor = new Vector4[] { new Vector4(0.0f, 0.0f, 0.0f, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uInnerStrokeColor", innerStrokeColor);
            }
        }

        /// <summary>
        /// 角丸のシャドウ(内側)設定
        /// </summary>
        class ProceduralShapeInnerShadowState : RendererStateChangerBase
        {
            readonly float _innerShadowSize;
            readonly float _angle;
            readonly float _distance;
            readonly Vector4[] _innerShadowColor;
            readonly ProceduralShapeShadowType _shadowType;
            readonly float _paneWidth;
            readonly float _paneHeight;
            readonly float _radius;

            public ProceduralShapeInnerShadowState(bool enabled, float paneWidth, float paneHeight, float radius, float innerShadowSize, float angle, float distance, Color innerShadowColor, ProceduralShapeShadowType shadowType)
            {
                _innerShadowSize = innerShadowSize;
                _innerShadowColor = new Vector4[] {
                    new Vector4((float)innerShadowColor.R / 255.0f,
                                (float)innerShadowColor.G / 255.0f,
                                (float)innerShadowColor.B / 255.0f,
                                enabled ? (float)innerShadowColor.A / 255.0f : 0.0f) };
                _angle = RendererHelper.ToRadian(angle);
                _distance = distance;
                _shadowType = shadowType;
                _paneWidth = paneWidth;
                _paneHeight = paneHeight;
                _radius = radius;
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                // ui2d ランタイムのコンスタントバッファ設定を移植
                float halfWidth = _paneWidth * 0.5f;
                float halfHeight = _paneHeight * 0.5f;
                float radius = _radius;
                radius = RendererHelper.Clamp(radius, 0.0f, halfWidth);
                radius = RendererHelper.Clamp(radius, 0.0f, halfHeight);

                float innerShadowOffsetX = (float)Math.Cos(_angle) * _distance / _paneWidth;
                float innerShadowOffsetY = -(float)Math.Sin(_angle) * _distance / _paneHeight;

                float size = _innerShadowSize / radius + 0.00001f;

                Vector4[] innerShadowParams = new Vector4[] { new Vector4(size, innerShadowOffsetX, innerShadowOffsetY, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uInnerShadowParams", innerShadowParams);
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uInnerShadowColor", _innerShadowColor);
                Vector4[] innerShadowTypeCoefficient = new Vector4[] { new Vector4(0.0f, 0.0f, 0.0f, 0.0f) };
                switch (_shadowType)
                {
                    case ProceduralShapeShadowType.Weak:
                        innerShadowTypeCoefficient[0].X = 1.0f;
                        break;
                    case ProceduralShapeShadowType.Normal:
                        innerShadowTypeCoefficient[0].Y = 1.0f;
                        break;
                    case ProceduralShapeShadowType.Strong:
                        innerShadowTypeCoefficient[0].Z = 1.0f;
                        break;
                }
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uInnerShadowAttenuation", innerShadowTypeCoefficient);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] innerStrokeColor = new Vector4[] { new Vector4(0.0f, 0.0f, 0.0f, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uInnerShadowColor", innerStrokeColor);
            }
        }

        /// <summary>
        /// 角丸のカラーオーバーレイ効果
        /// </summary>
        class ProceduralShapeColorOverlayState : RendererStateChangerBase
        {
            readonly Vector4[] _color;
            public ProceduralShapeColorOverlayState(bool enabled, Color color)
            {
                _color = new Vector4[] {
                    new Vector4((float)color.R / 255.0f,
                                (float)color.G / 255.0f,
                                (float)color.B / 255.0f,
                                enabled ? (float)color.A / 255.0f : 0.0f) };
            }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uColorOverlayColor", _color);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] color = new Vector4[] { new Vector4(0.0f, 0.0f, 0.0f, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uColorOverlayColor", color);
            }
        }

        /// <summary>
        /// 角丸のグラデーションオーバーレイ設定効果
        /// </summary>
        class ProceduralShapeGradationOverlayState : RendererStateChangerBase
        {
            readonly Vector4[] _colors;
            readonly Vector4[] _controlPoints;
            readonly float _angle;

            public ProceduralShapeGradationOverlayState(bool enabled, float[] controlPoints, Color[] colors, float angle)
            {
                Debug.Assert(controlPoints.Length == colors.Length);

                _controlPoints = new Vector4[] { new Vector4(-1.0f, -1.0f, -1.0f, -1.0f) };
                _colors = new Vector4[ProceduralShapeConstants.GradationControlPointCount]
                {
                    new Vector4(1.0f, 1.0f, 1.0f, 1.0f),
                    new Vector4(1.0f, 1.0f, 1.0f, 1.0f),
                    new Vector4(1.0f, 1.0f, 1.0f, 1.0f),
                    new Vector4(1.0f, 1.0f, 1.0f, 1.0f)
                };

                if (enabled)
                {
                    // LayoutConverter でのデータの操作と ui2d ランタイムでの操作を再現する
                    // LayoutConverter では不足分のコントロールポイントが 1.0 で最後に出力したカラーを出力する。
                    for (int i = 0; i < ProceduralShapeConstants.GradationControlPointCount; ++i)
                    {
                        if (i < controlPoints.Length)
                        {
                            _controlPoints[0][i] = controlPoints[i];
                            _colors[i] = new Vector4((float)colors[i].R / 255.0f,
                                                     (float)colors[i].G / 255.0f,
                                                     (float)colors[i].B / 255.0f,
                                                     (float)colors[i].A / 255.0f);
                        }
                        else
                        {
                            _controlPoints[0][i] = 1.0f;
                            _colors[i] = new Vector4((float)colors[controlPoints.Length - 1].R / 255.0f,
                                                     (float)colors[controlPoints.Length - 1].G / 255.0f,
                                                     (float)colors[controlPoints.Length - 1].B / 255.0f,
                                                     (float)colors[controlPoints.Length - 1].A / 255.0f);
                        }
                    }

                    // ui2d ランタイムではアニメーションした値のクリップを行っています。
                    float prevControlPointValue = 0.0f;
                    int finishColorIndex = -1;
                    for (int i = 0; i < ProceduralShapeConstants.GradationControlPointCount; ++i)
                    {
                        float value = _controlPoints[0][i];
                        value = RendererHelper.Clamp(value, 0.0f, 1.0f);

                        // コントロールポイントの値が 1.0 を超えた場合は、最初に 1.0 を超えたコントロールポイントのカラーを使用する
                        if (finishColorIndex >= 0)
                        {
                            _controlPoints[0][i] = 1.0f;
                            _colors[i] = _colors[finishColorIndex];
                        }
                        else
                        {
                            // ひとつ前のコントロールポイントよりも小さい値が設定されないように値をクリップします。
                            if (value < prevControlPointValue)
                            {
                                _controlPoints[0][i] = prevControlPointValue;
                            }
                            else
                            {
                                _controlPoints[0][i] = value;
                            }

                            prevControlPointValue = value;

                            // コントロールポイントが 1.0 を超えていないか判定
                            // 超えている場合はこれ以降、このコントロールポイントの値が使用されるようにする。
                            if (value >= 1.0f)
                            {
                                finishColorIndex = i;
                            }
                        }
                    }
                    _angle = RendererHelper.ToRadian(angle);
                }
           }

            public override void Apply(Device device, ConstantTable vsct, ConstantTable psct)
            {
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uGradationOverlayControlPoints", _controlPoints);

                Vector4[] angleParams = new Vector4[] { new Vector4(_angle, 0.0f, 0.0f, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uGradationOverlayAngleParams", angleParams);
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uGradationOverlayColors", _colors);
            }

            public override void Reset(Device device, ConstantTable vsct, ConstantTable psct)
            {
                Vector4[] controlPoints = new Vector4[] { new Vector4(-1.0f, -1.0f, -1.0f, -1.0f) };
                DXRendererHelper.SetPixelShaderConstant(device, psct, "uGradationOverlayControlPoints", controlPoints);
            }
        }


        /// <summary>
        /// 描画タスク
        /// </summary>
        class DrawTask
        {
        }

        /// <summary>
        ///
        /// </summary>
        class PrimitiveDrawTask : DrawTask
        {
            public enum ProjectionMode
            {
                Normal,
                ForceOrtho
            }

            readonly int _index;
            readonly int _vertexCount;
            readonly int _numPrimitive;
            readonly PrimitiveType _primitiveType;

            readonly int _indexOffset;

            readonly DXMatrix _mtxProjection;
            readonly DXMatrix _mtxView;
            readonly DXMatrix _mtxCurrent;
            readonly PixelShaderResourceContainer _currentPixelShaderResource;

            readonly bool _depthWriteTestEnabled;

            List<RendererStateChangerBase> _stateChangerSet;

            //----------------------------------------------------------

            public int Index { get { return _index; } }
            public int VertexCount { get { return _vertexCount; } }
            public PrimitiveType PrimitiveType { get { return _primitiveType; } }
            public int NumPrimitive { get { return _numPrimitive; } }
            public int IndexOffset { get { return _indexOffset; } }

            //----------------------------------------------------------

            private PixelShaderResourceContainer SelectPixelShader_(D3DRenderer renderer, List<RendererStateChangerBase> stateChangerSet)
            {
                if (renderer._currentUsedTextureStageNum == 0)
                {
                    if (stateChangerSet.Exists((item) => item is ColorBlendData))
                    {
                        return renderer._builtInNoTexBlendPixelShader;
                    }else{
                        return renderer._builtInNoTexPixelShader;
                    }
                }

                if (renderer._lowLevelCombinerSettingsEnabled
                    && renderer._builtInDetailedCombinerShaders != null
                    && renderer._builtInDetailedCombinerShaders.Count > 0)
                {
                    return renderer._builtInDetailedCombinerShaders[(int)renderer._detailedCombinerValiationType];
                }
                else if (renderer._currentUsedTextureStageNum > 1)
                {
                    return renderer._builtInPixelShader;
                }
                else
                {
                    if (stateChangerSet.Exists((item) => item is PackedFontFormat))
                    {
                        return renderer._builtInPackedFontPixelShader;
                    }

                    if (stateChangerSet.Exists((item) => item is ColorBlendData))
                    {
                        return renderer._builtIn1TexBlendPixelShader;
                    }

                    return renderer._builtIn1TexPixelShader;
                }
            }

            /// <summary>
            ///
            /// </summary>
            public PrimitiveDrawTask(
                int index,
                int vertexCount,
                PrimitiveType primitiveType,
                int numPrimitive,
                int indexOffset,
                List<RendererStateChangerBase> stateChangerSet,
                D3DRenderer renderer,
                ProjectionMode projectionMode = ProjectionMode.Normal)
            {
                _index = index;
                _vertexCount = vertexCount;
                _primitiveType = primitiveType;
                _numPrimitive = numPrimitive;

                if(projectionMode == ProjectionMode.ForceOrtho)
                {
                    _mtxProjection = renderer.CalcProjectionMatrix_(false);
                }
                else
                {
                    _mtxProjection = renderer._d3dDevice.GetTransform(TransformState.Projection);
                }

                _mtxView = renderer._mtxViewStack.Top;
                _mtxCurrent = renderer._mtxStack.Top;

                _indexOffset = indexOffset;

                _currentPixelShaderResource = SelectPixelShader_(renderer, stateChangerSet);

                _depthWriteTestEnabled = renderer.DepthWriteTestEnabled;

                _stateChangerSet = null;

                if (stateChangerSet.Count > 0)
                {
                    renderer._lowLevelCombinerSettingsEnabled = false;
                    renderer._detailedCombinerValiationType = DetailedCombinerVariationType.DetailedCombinerVariationType_Stage0;

                    _stateChangerSet = new List<RendererStateChangerBase>();
                    _stateChangerSet.AddRange(stateChangerSet);

                    stateChangerSet.Clear();
                }
            }

            /// <summary>
            ///
            /// </summary>
            public void RestoreStateBeforeDraw(Device d3dDevice, ConstantTable vsct)
            {
                ConstantTable psct = _currentPixelShaderResource.constantTable;

                d3dDevice.SetTransform(TransformState.Projection, _mtxProjection);
                d3dDevice.SetTransform(TransformState.View, _mtxView);
                d3dDevice.SetTransform(TransformState.World, _mtxCurrent);

                d3dDevice.SetVertexShaderConstant(vsct.GetConstantDescription(vsct.GetConstantByName(null, "view_model_matrix")).RegisterIndex, _mtxCurrent * _mtxView * _mtxProjection);
                d3dDevice.SetVertexShaderConstant(vsct.GetConstantDescription(vsct.GetConstantByName(null, "world_matrix")).RegisterIndex, _mtxCurrent);

                d3dDevice.PixelShader = _currentPixelShaderResource.pixelShader;

                DXRendererHelper.SetPixelShaderConstant(d3dDevice, psct, "ProceduralShapeEnabled", 0);

                if (_stateChangerSet != null)
                {
                    foreach (RendererStateChangerBase stateChanger in _stateChangerSet)
                    {
                        if (stateChanger.GetType() == typeof(DetailedCombinerData))
                        {
                            stateChanger.Apply(d3dDevice, vsct);
                        }
                        else
                        {
                            stateChanger.Apply(d3dDevice, vsct, psct);
                        }
                    }
                }
            }

            /// <summary>
            ///
            /// </summary>
            public void RestoreStateAfterDraw(Device d3dDevice, ConstantTable vsct)
            {
                ConstantTable psct = _currentPixelShaderResource.constantTable;

                if (_stateChangerSet != null)
                {
                    foreach (RendererStateChangerBase stateChanger in _stateChangerSet)
                    {
                        if (stateChanger.GetType() == typeof(DetailedCombinerData))
                        {
                            stateChanger.Reset(d3dDevice, vsct);
                        }
                        else
                        {
                            stateChanger.Reset(d3dDevice, vsct, psct);
                        }
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        class FontDrawTask : DrawTask
        {
            public bool IsFontDraw { get { return _Font != null; } }
            DXFont _Font;
            string _Text;
            Point _TextPos;
            Color _TextColor;

            /// <summary>
            ///
            /// </summary>
            public void DrawFont()
            {
                _Font.DrawText(null, _Text, _TextPos.X, _TextPos.Y, new SharpDX.ColorBGRA(_TextColor.ToArgb()));
            }

            /// <summary>
            ///
            /// </summary>
            public FontDrawTask(DXFont font, string text, Point pos, Color color)
            {
                _Font = font;
                _Text = text;
                _TextPos = pos;
                _TextColor = color;
            }
        }

        //-------------------------------------------------------------------
        #region 定数
        const float _viewWidth = 640.0f;
        const float _viewHeight = 640.0f;

        #endregion


        //-------------------------------------------------------------------
        #region フィールド

        Direct3D _direct3D = null;

        /// <summary>
        /// デバイス
        /// </summary>
        Device _d3dDevice = null;

        ShaderBytecode _shaderBytecodeVS = null;
        ConstantTable _builtInVertexShaderConstantTable = null;

        PixelShaderResourceContainer _builtInPixelShader = null;
        PixelShaderResourceContainer _builtInNoTexPixelShader = null;
        PixelShaderResourceContainer _builtInNoTexBlendPixelShader = null;

        PixelShaderResourceContainer _builtIn1TexPixelShader = null;
        PixelShaderResourceContainer _builtIn1TexBlendPixelShader = null;
        PixelShaderResourceContainer _builtInPackedFontPixelShader = null;

        class PixelShaderResourceContainer : IDisposable
        {
            public ShaderBytecode shaderBytecode { get; set; }
            public ConstantTable constantTable { get; set; }
            public PixelShader pixelShader { get; set; }

            public PixelShaderResourceContainer()
            {
            }

            /// <summary>
            /// 初期化処理付きコンストラクタです。
            /// </summary>
            /// <param name="device"></param>
            /// <param name="filePath"></param>
            public PixelShaderResourceContainer(Device device, ShaderBytecode byteCode)
            {
                shaderBytecode = byteCode;
                constantTable = shaderBytecode.ConstantTable;
                pixelShader = new PixelShader(device, shaderBytecode);
            }

            /// <summary>
            /// 終了処理を行います。
            /// </summary>
            public void Dispose()
            {
                if (pixelShader != null)
                {
                    pixelShader.Dispose();
                    pixelShader = null;
                }
                if (constantTable != null)
                {
                    constantTable.Dispose();
                    constantTable = null;
                }
                if (shaderBytecode != null)
                {
                    shaderBytecode.Dispose();
                    shaderBytecode = null;
                }
            }

            private ShaderBytecode _LoadShaderBytecode(string filePath)
            {
                using (var fileStream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    Debug.Assert(fileStream != null, "シェーダバイナリ '{0}' が見つかりません。", filePath);
                    return ShaderBytecode.Load(fileStream);
                }
            }
        };
        List<PixelShaderResourceContainer> _builtInDetailedCombinerShaders;

        enum DetailedCombinerVariationType {
            DetailedCombinerVariationType_Stage0 = 0,

            DetailedCombinerVariationType_Stage1_Source1,
            DetailedCombinerVariationType_Stage1_Source2,
            DetailedCombinerVariationType_Stage1_Source3,

            DetailedCombinerVariationType_Stage2_Source1,
            DetailedCombinerVariationType_Stage2_Source2,
            DetailedCombinerVariationType_Stage2_Source3,

            DetailedCombinerVariationType_Stage3_Source1,
            DetailedCombinerVariationType_Stage3_Source2,
            DetailedCombinerVariationType_Stage3_Source3,

            DetailedCombinerVariationType_Stage4_Source1,
            DetailedCombinerVariationType_Stage4_Source2,
            DetailedCombinerVariationType_Stage4_Source3,

            DetailedCombinerVariationType_Stage5_Source1,
            DetailedCombinerVariationType_Stage5_Source2,
            DetailedCombinerVariationType_Stage5_Source3,

            DetailedCombinerVariationType_Stage6_Source1,
            DetailedCombinerVariationType_Stage6_Source2,
            DetailedCombinerVariationType_Stage6_Source3
        };

        DetailedCombinerVariationType _detailedCombinerValiationType = DetailedCombinerVariationType.DetailedCombinerVariationType_Stage0;

        VertexShader _builtInVertexShader = null;

        /// <summary>
        /// 頂点バッファ
        /// </summary>
        VertexBuffer _vertexBufferRect;

        /// <summary>
        /// インデックスバッファ
        /// </summary>
        IndexBuffer _indexBuffer;



        /// <summary>
        /// マテリアル
        /// </summary>
        Material _material = new Material();

        class TextureMapProxy
        {
            public Texture Texture { get; set; }
            public Texture TextureWith1bitAlpha { get; set; }
            public Texture TextureIntencity { get; set; }
            public Texture TextureIntencityAlpha { get; set; }
            public DateTime SrcBmpLastModifyTime { get; set; }

            public void SetTexture(Texture dxTexture, RendererTextrureFormat format)
            {
                switch (format)
                {
                    case RendererTextrureFormat.ARGB:
                        this.Texture = dxTexture;
                        break;
                    case RendererTextrureFormat.BC1FromARGB:
                        this.TextureWith1bitAlpha = dxTexture;
                        break;
                    case RendererTextrureFormat.A8FromARGB:
                    case RendererTextrureFormat.A8FromRGB:
                    case RendererTextrureFormat.L8:
                        this.TextureIntencity = dxTexture;
                        break;
                    case RendererTextrureFormat.LA8:
                        this.TextureIntencityAlpha = dxTexture;
                        break;
                    default:
                        break;
                }
            }

            public Texture GetTexture(RendererTextrureFormat format)
            {
                switch (format)
                {
                    case RendererTextrureFormat.ARGB:
                        return this.Texture;
                    case RendererTextrureFormat.BC1FromARGB:
                        return this.TextureWith1bitAlpha;
                    case RendererTextrureFormat.A8FromARGB:
                    case RendererTextrureFormat.A8FromRGB:
                    case RendererTextrureFormat.L8:
                        return this.TextureIntencity;
                    case RendererTextrureFormat.LA8:
                        return this.TextureIntencityAlpha;
                    default:
                        return null;
                }
            }

            public void Dispose()
            {
                if (this.Texture != null)
                {
                    this.Texture.Dispose();
                }

                if (this.TextureWith1bitAlpha != null)
                {
                    this.TextureWith1bitAlpha.Dispose();
                }

                if (this.TextureIntencity != null)
                {
                    this.TextureIntencity.Dispose();
                }

                if (this.TextureIntencityAlpha != null)
                {
                    this.TextureIntencityAlpha.Dispose();
                }
            }
        }

        /// <summary>
        /// Bitmapからテクスチャを検索する辞書
        /// </summary>
        Dictionary<Bitmap, TextureMapProxy> _bitmapTextureMap = new Dictionary<Bitmap, TextureMapProxy>();

        /// <summary>
        /// 表示パラメータ
        /// </summary>
        PresentParameters _presentParams;

        /// <summary>
        /// 描画内容を反映する対象コントロール
        /// </summary>
        Control _targetCtrl;

        /// <summary>
        /// 行列スタック
        /// </summary>
        DXMatrixStack _mtxStack = new DXMatrixStack();

        /// <summary>
        /// ビュー行列スタック
        /// </summary>
        DXMatrixStack _mtxViewStack = new DXMatrixStack();

        float _cameraDistZ = 0.0f;

        float _defaultScreenSizeW = 1920.0f;
        float _defaultScreenSizeH = 1080.0f;

        /// <summary>
        /// 既定の頂点色
        /// </summary>
        static Color[] _defaultVtxColors = new Color[] { Color.White, Color.White, Color.White, Color.White };

        /// <summary>
        /// 透明度
        /// </summary>
        int _transparency = PaneHelper.MaxTransparency; // ( 0 - 255 )

        /// <summary>
        /// 白カラー
        /// </summary>
        FloatColor _whiteBlendColor = FloatColor.White;

        /// <summary>
        /// 黒カラー
        /// </summary>
        FloatColor _blackBlendColor = FloatColor.Black;

        bool _isThresholdingAlphaInterpolationEnabled = false;

        /// <summary>
        /// ラインの太さ
        /// 現在はサポートしていません。
        /// </summary>
        float _lineWidth = 1.0f;

        /// <summary>
        /// 頂点バッファのバッファ
        /// </summary>
        List<DxCustomVertex> _vtxSet = new List<DxCustomVertex>();

        /// <summary>
        /// インデックスバッファ
        /// </summary>
        List<int> _indexSet = new List<int>();

        /// <summary>
        /// 描画タスク
        /// </summary>
        List<DrawTask> _drawTaskSet = new List<DrawTask>();
        /// <summary>
        /// ステート変更リスト
        /// </summary>
        List<RendererStateChangerBase> _stateChangerSet = new List<RendererStateChangerBase>();

        /// <summary>
        /// 現在のテクスチャステージの利用数
        /// </summary>
        int _currentUsedTextureStageNum = 0;

        DxCustomVertex[] _temporaryVtxSet = new DxCustomVertex[5];

        /// <summary>
        /// 描画オプション
        /// </summary>
        RendererOption _rendererOption = new RendererOption();

        DXFont _font;
        DXFont _fontBold;

        private bool _deviceLost = false;

        bool _bNeedPresent = true;

        /// <summary>
        /// レンダリング結果を画面に表示するかどうか
        /// </summary>
        public bool NeedPresent
        {
            get { return _bNeedPresent; }
            set { _bNeedPresent = value; }
        }

        /// <summary>
        /// 詳細コンバイナ利用
        /// </summary>
        bool _lowLevelCombinerSettingsEnabled = false;


        /// <summary>
        /// リニア色空間モードを指定する
        /// </summary>
        public bool LinearGammmaSRGBFetchEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// リニア色空間でSRGB書き込みを行うかどうかを指定する
        /// </summary>
        public bool LinearGammmaSRGBWriteEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// リニア色空間色パラメータモードを指定する
        /// </summary>
        public bool LinearGammmaColorParamatersEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// 視野角（ラジアン）
        /// </summary>
        public float FOV { get; set; }

        /// <summary>
        /// ニア平面
        /// </summary>
        public float PerseNear { get; set; }

        /// <summary>
        /// ニア平面
        /// </summary>
        public float PerseFar { get; set; }

        /// <summary>
        /// パースぺクティブ表示が有効
        /// </summary>
        public bool PersepectiveRendering { get; set; }

        private bool _depthWriteTestEnabled;
        /// <summary>
        /// 深度テストが有効
        /// </summary>
        public bool DepthWriteTestEnabled
        {
            get
            {
                return _depthWriteTestEnabled;
            }
            set
            {
                if (_depthWriteTestEnabled != value)
                {
                    _depthWriteTestEnabled = value;
                    if (_d3dDevice != null)
                    {
                        _stateChangerSet.Add(new DepthTestStateData(_depthWriteTestEnabled));
                    }
                }
            }
        }

        #endregion フィールド

        #region IRenderer メンバ

        /// <summary>
        ///
        /// </summary>
        public D3DRenderer()
        {
            this.FOV = (float)Math.PI / 4.0f;
            this.PerseNear = 1.0f;
            this.PerseFar = 10000.0f;
            this.PersepectiveRendering = false;
            this.DepthWriteTestEnabled = false;
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="initParams"></param>
        public bool Initialize(object initParams)
        {
            Control ctrl = initParams as Control;
            Debug.Assert(ctrl != null);

            try
            {
                // Deviceクラスの既定のハンドラは、
                // ウインドウサイズがゼロの場合の処理を正しく行えず、
                // 異常終了してしまうので、無効にします。
                // 適切な処理を独自に実装して登録します。
                // Device.IsUsingEventHandlers = false;


                // Direct3D デバイスの作成に使う構造体を設定する。
                _presentParams = new PresentParameters();
                _presentParams.BackBufferFormat = Format.A8R8G8B8;
                _presentParams.BackBufferCount = 1;
                _presentParams.BackBufferWidth = ctrl.ClientSize.Width;
                _presentParams.BackBufferHeight = ctrl.ClientSize.Height;

                _presentParams.MultiSampleType = MultisampleType.None;
                _presentParams.SwapEffect = SwapEffect.Discard;
                _presentParams.EnableAutoDepthStencil = true;
                _presentParams.AutoDepthStencilFormat = Format.D16;
                _presentParams.PresentationInterval = PresentInterval.Immediate;
                _presentParams.Windowed = true;
                _presentParams.PresentFlags = PresentFlags.LockableBackBuffer;

                _presentParams.DeviceWindowHandle = ctrl.Handle;

                // Direct3D デバイスの作成
                _direct3D = new Direct3D();
                try
                {
                    _d3dDevice = new Device(_direct3D, 0, DeviceType.Hardware, ctrl.Handle, CreateFlags.HardwareVertexProcessing, _presentParams);
                }
                catch (Exception e)
                {
                    MessageBox.Show(
                        StringResMgr.Get("SYSTEM_DIRECTX_FAILURE_IN_CREATE_DEVICE") + "\n\n" + e.ToString(),
                        StringResMgr.Get("SYSTEM_LAYOUTEDITOR"),
                        MessageBoxButtons.OK);

                    DbgConsole.WriteLine("Failure in DirectX-Device initialization.\n\n" + e.ToString());
                    return false;
                }

                // NOTE: winforms デザインモード時は 通常の出力フォルダではない C:\Users\<ﾛｰｶﾙﾕｰｻﾞｰ>\AppData\Local\Microsoft\VisualStudio\14.0\ProjectAssemblies 以下のフォルダに
                //       実行ファイルが生成され利用される為、必要とするシェーダバイナリの読み込みに失敗します。
                //       デザインモードの時には読み込みを行わないようにしますが、デザインモードを判定する方法がかなり不確定のようで、下記のサイトを参考にしました。
                //       http://www.geocities.jp/tinqwill/cs.html#deginer
                //
                bool designMode = false;
                if (System.ComponentModel.LicenseManager.UsageMode == System.ComponentModel.LicenseUsageMode.Designtime)
                {
                    designMode = true;
                }
                else if (Process.GetCurrentProcess().ProcessName.ToUpper().Equals("DEVENV") || Process.GetCurrentProcess().ProcessName.ToUpper().Equals("VCSEXPRESS"))
                {
                    designMode = true;
                }
                else if (AppDomain.CurrentDomain.FriendlyName == "DefaultDomain")
                {
                    designMode = true;
                }

                // 詳細コンバイナ用シェーダ
                _builtInDetailedCombinerShaders = new List<PixelShaderResourceContainer>();
                //--------------------------
                // シェーダ関連初期化
                {
                    if (designMode)
                    {
                        string psSource = new StreamReader(ImageResMgr.GetShader("layout_ps.fx")).ReadToEnd();
                        _builtInPixelShader = new PixelShaderResourceContainer(_d3dDevice, ShaderBytecode.Compile(psSource, "PShader", "ps_3_0", ShaderFlags.None));

                        _shaderBytecodeVS = ShaderBytecode.Compile(new StreamReader(ImageResMgr.GetShader("layout_vs.fx")).ReadToEnd(), "main", "vs_3_0", ShaderFlags.None);
                        _builtInVertexShaderConstantTable = _shaderBytecodeVS.ConstantTable;
                        _builtInVertexShader = new VertexShader(_d3dDevice, _shaderBytecodeVS);

                        _builtInNoTexPixelShader = new PixelShaderResourceContainer(_d3dDevice, ShaderBytecode.Compile(psSource, "PShader0Tex", "ps_3_0", ShaderFlags.None));
                        _builtInNoTexBlendPixelShader = new PixelShaderResourceContainer(_d3dDevice, ShaderBytecode.Compile(psSource, "PShader0TexBlend", "ps_3_0", ShaderFlags.None));
                        _builtIn1TexPixelShader = new PixelShaderResourceContainer(_d3dDevice, ShaderBytecode.Compile(psSource, "PShader1Tex", "ps_3_0", ShaderFlags.None));
                        _builtIn1TexBlendPixelShader = new PixelShaderResourceContainer(_d3dDevice, ShaderBytecode.Compile(psSource, "PShader1TexColorBlend", "ps_3_0", ShaderFlags.None));
                        _builtInPackedFontPixelShader = new PixelShaderResourceContainer(_d3dDevice, ShaderBytecode.Compile(psSource, "PShaderPackedFont", "ps_3_0", ShaderFlags.None));
                    }
                    else
                    {
                        string appDir = Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                        string shaderPath = appDir + "\\compiledShaders\\";

                        bool existShaderPath = true;
                        if (!System.IO.Directory.Exists(shaderPath))
                        {
                            shaderPath = AppConstants.NwToolsRootPath + "\\compiledShaders\\";
                            if (!System.IO.Directory.Exists(shaderPath))
                            {
                                existShaderPath = false;
                            }
                        }

                        if (existShaderPath)
                        {

                            // 標準シェーダ
                            _shaderBytecodeVS = _LoadShaderBytecode(shaderPath + "layout_vs.cfx");
                            _builtInVertexShaderConstantTable = _shaderBytecodeVS.ConstantTable;
                            _builtInVertexShader = new VertexShader(_d3dDevice, _shaderBytecodeVS);

                            _builtInPixelShader = new PixelShaderResourceContainer(_d3dDevice, _LoadShaderBytecode(shaderPath + "layout_ps_PShader.cfx"));
                            _builtInNoTexPixelShader = new PixelShaderResourceContainer(_d3dDevice, _LoadShaderBytecode(shaderPath + "layout_ps_PShader0Tex.cfx"));
                            _builtInNoTexBlendPixelShader = new PixelShaderResourceContainer(_d3dDevice, _LoadShaderBytecode(shaderPath + "layout_ps_PShader0TexBlend.cfx"));
                            _builtIn1TexPixelShader = new PixelShaderResourceContainer(_d3dDevice, _LoadShaderBytecode(shaderPath + "layout_ps_PShader1Tex.cfx"));
                            _builtIn1TexBlendPixelShader = new PixelShaderResourceContainer(_d3dDevice, _LoadShaderBytecode(shaderPath + "layout_ps_PShader1TexColorBlend.cfx"));
                            _builtInPackedFontPixelShader = new PixelShaderResourceContainer(_d3dDevice, _LoadShaderBytecode(shaderPath + "layout_ps_PShaderPackedFont.cfx"));

                            {
                                string fileName = "layout_ps_dc_stg0.cfx";
                                FileStream fs = new FileStream(shaderPath + fileName, FileMode.Open, FileAccess.Read);
                                Debug.Assert(fs != null, "詳細コンバイナ用シェーダバイナリ '{0}' が見つかりません。", fileName);

                                PixelShaderResourceContainer detailedCombinerShader = new PixelShaderResourceContainer();
                                detailedCombinerShader.shaderBytecode = ShaderBytecode.Load(fs);
                                detailedCombinerShader.constantTable = detailedCombinerShader.shaderBytecode.ConstantTable;
                                detailedCombinerShader.pixelShader = new PixelShader(_d3dDevice, detailedCombinerShader.shaderBytecode);
                                _builtInDetailedCombinerShaders.Add(detailedCombinerShader);

                                fs.Close();
                            }

                            for (int i = 0; i < 6; i++)
                            {
                                for (int j = 0; j < 3; j++)
                                {
                                    string fileName = "layout_ps_dc_stg" + (i + 1).ToString() + "_src" + (j + 1).ToString() + ".cfx";
                                    FileStream fs = new FileStream(shaderPath + fileName, FileMode.Open, FileAccess.Read);
                                    Debug.Assert(fs != null, "詳細コンバイナ用シェーダバイナリ '{0}' が見つかりません。", fileName);

                                    PixelShaderResourceContainer detailedCombinerShader = new PixelShaderResourceContainer();
                                    detailedCombinerShader.shaderBytecode = ShaderBytecode.Load(fs);
                                    detailedCombinerShader.constantTable = detailedCombinerShader.shaderBytecode.ConstantTable;
                                    detailedCombinerShader.pixelShader = new PixelShader(_d3dDevice, detailedCombinerShader.shaderBytecode);
                                    _builtInDetailedCombinerShaders.Add(detailedCombinerShader);

                                    fs.Close();
                                }
                            }
                        }
                    }
                }

                this.OnResetDevice_(null, null);

                //--------------------------
                _vertexBufferRect = new VertexBuffer(_d3dDevice,
                    Marshal.SizeOf(typeof(DxCustomVertex)) * VertexBufferLength,
                    Usage.WriteOnly,
                    VertexFormat.Position | VertexFormat.Diffuse | VertexFormat.Texture4,
                    Pool.Managed);

                _indexBuffer = new IndexBuffer(_d3dDevice,
                    Marshal.SizeOf(typeof(int)) * IndexBufferLength,
                    Usage.WriteOnly,
                    Pool.Managed,
                    false);

                _targetCtrl = ctrl;

                //----------------------------
                // 既定のハンドラは、問題があるので、
                // 自前で同等の処理を実装します。
                _targetCtrl.SizeChanged += new EventHandler(_targetCtrl_SizeChanged);
            }
            catch (Exception e)
            {
                MessageBox.Show(
                    StringResMgr.Get("SYSTEM_DIRECTX_FAILURE_IN_CREATE_DEVICE") + "\n\n" + e.ToString(),
                    StringResMgr.Get("SYSTEM_LAYOUTEDITOR"),
                    MessageBoxButtons.OK);

                DbgConsole.WriteLine("Failure in DirectX-Device initialization.\n\n" + e.ToString());
                return false;
            }

            return true;
        }

        private ShaderBytecode _LoadShaderBytecode(string filePath)
        {
            using (var fileStream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
            {
                Debug.Assert(fileStream != null, "シェーダバイナリ '{0}' が見つかりません。", filePath);
                return ShaderBytecode.Load(fileStream);
            }
        }

        /// <summary>
        /// 対象コントロールサイズ変更
        /// </summary>
        void _targetCtrl_SizeChanged(object sender, EventArgs e)
        {
            Control ctrl = sender as Control;
            Rectangle clientRectangle = ctrl.ClientRectangle;

            // 小さすぎなければ、バックバッファのサイズとして設定します。
            ResetSize_(ctrl.ClientRectangle.Width, ctrl.ClientRectangle.Height);
        }

        void ResetSize_(int width, int height)
        {
            // 小さすぎなければ、バックバッファのサイズとして設定します。
            if (width > 1 && height > 1)
            {
                _presentParams.BackBufferWidth = width;
                _presentParams.BackBufferHeight = height;

                _deviceLost = true;

                _vtxSet.Clear();
                _indexSet.Clear();
                _drawTaskSet.Clear();

                var res = _d3dDevice.TestCooperativeLevel();
                if (res.Success || res == ResultCode.DeviceNotReset)
                {
                    OnDeviceLostRelease_();
                    _d3dDevice.Reset(_presentParams);
                    OnResetDevice_(null, null);
                    _deviceLost = false;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        void OnDeviceLostRelease_()
        {
            if (_font != null)
            {
                _font.Dispose();
                _font = null;
            }

            if (_fontBold != null)
            {
                _fontBold.Dispose();
                _fontBold = null;
            }
        }

        /// <summary>
        /// デバイスリセットハンドラ
        /// </summary>
        void OnResetDevice_(object sender, EventArgs e)
        {
            _d3dDevice.Reset(_presentParams);

            DbgConsole.WriteLine("_d3dDevice_DeviceReset");

            _d3dDevice.SetRenderState(RenderState.CullMode, Cull.None);

            // Z バッファを有効にする
            if (this.DepthWriteTestEnabled)
            {
                _d3dDevice.SetRenderState(RenderState.ZEnable, true);
                _d3dDevice.SetRenderState(RenderState.ZWriteEnable, true);
                _d3dDevice.SetRenderState(RenderState.ZFunc, Compare.LessEqual);//
            }
            else
            {
                _d3dDevice.SetRenderState(RenderState.ZEnable, false);
                _d3dDevice.SetRenderState(RenderState.ZWriteEnable, false);
            }

            //--------------------------
            // フォント
            _font = new SharpDX.Direct3D9.Font(
                _d3dDevice, 16, 6, FontWeight.Regular, 0, false,
                FontCharacterSet.Default, FontPrecision.Default, FontQuality.ClearTypeNatural, FontPitchAndFamily.Default, "Tahoma");

            _fontBold = new SharpDX.Direct3D9.Font(
                _d3dDevice, 16, 6, FontWeight.ExtraBold, 0, false,
                FontCharacterSet.Default, FontPrecision.Default, FontQuality.ClearTypeNatural, FontPitchAndFamily.Default, "Tahoma");

            // マテリアルの設定

            // ライティングの設定(ライトは利用しません。)
            //--------------------------------------------
            // レンダーステートの設定。
            _d3dDevice.SetRenderState(RenderState.Lighting, false);
            _d3dDevice.SetRenderState(RenderState.ColorVertex, true);
            _d3dDevice.SetRenderState(RenderState.DiffuseMaterialSource, ColorSource.Color1);
            _d3dDevice.SetRenderState(RenderState.AlphaBlendEnable, true);
            _d3dDevice.SetRenderState(RenderState.SeparateAlphaBlendEnable, false);
            _d3dDevice.SetRenderState(RenderState.SourceBlend, Blend.SourceAlpha);
            _d3dDevice.SetRenderState(RenderState.DestinationBlend, Blend.InverseSourceAlpha);

            {
                _d3dDevice.PixelShader = _builtInPixelShader.pixelShader;
                _d3dDevice.VertexShader = _builtInVertexShader;

                DXRendererHelper.SetPixelShaderConstant(_d3dDevice, _builtInPixelShader.constantTable, "EnabledTextureCount", 0);
                Vector4[] width = new Vector4[] { new Vector4(1.0f, 1.0f, 1.0f, 1.0f) };
                Vector4[] offset = new Vector4[] { new Vector4(0f, 0f, 0f, 0.0f) };
                DXRendererHelper.SetPixelShaderConstant(_d3dDevice, _builtInPixelShader.constantTable, "InterpolateWidth", width);
                DXRendererHelper.SetPixelShaderConstant(_d3dDevice, _builtInPixelShader.constantTable, "InterpolateOffset", offset);
            }

            //--------------------------------------------
            // 頂点フォーマットの設定
            _d3dDevice.VertexFormat = VertexFormat.Position | VertexFormat.Diffuse | VertexFormat.Texture4;

            //--------------------------------------------
            // _bitmapTextureMapが管理する、Textureオブジェクトは、
            // Managed指定で初期化される、Deviceリソースなので、
            // 初期化の必要はありません。
            _d3dDevice.SetTransform(TransformState.View, DXMatrix.Identity);
        }

        /// <summary>
        /// レンダリング開始
        /// </summary>
        public void BeginRendering(object initParams)
        {
            // 投影行列を設定する
            _d3dDevice.SetTransform(TransformState.Projection, CalcProjectionMatrix_(this.PersepectiveRendering));
        }

        /// <summary>
        /// レンダリング終了
        /// </summary>
        public void EndRendering()
        {
            EndRendering(null);
        }

        private SharpDX.Matrix CalcProjectionMatrix_(bool isPerspective)
        {
            if (_targetCtrl.Width > 0 && _targetCtrl.Height > 0)
            {
                if (isPerspective)
                {
                    double targetW = (_targetCtrl.Width % 2) != 0 ? _targetCtrl.Width + 1 : _targetCtrl.Width;
                    double targetH = (_targetCtrl.Height % 2) != 0 ? _targetCtrl.Height + 1 : _targetCtrl.Height;

                    // 本来の視野角の内、一部分だけを表示するので補正係数を乗じます。
                    double fovYCompensation = (double)targetH / _defaultScreenSizeH;
                    double compensatedH = Math.Tan((double)this.FOV * 0.5) * fovYCompensation;
                    //double halfFovy = Math.Atan(compensatedH);

                    // SharpDX.Matrix.PerspectiveRH の評価結果と
                    // SharpDX.Matrix.PerspectiveFovRH の評価結果 から導出しました。
                    //
                    // double screenH = 1.0 / Math.Tan(halfFovy);
                    // screenH = (2.0 * (double)this.PerseNear) / screenH;
                    // ↓
                    // screenH = (2.0 * (double)this.PerseNear) * Math.Tan(halfFovy);
                    // ↓
                    double screenH = (2.0 * (double)this.PerseNear) * compensatedH;

                    double aspectRatio = targetW / targetH;
                    double screenW = screenH * aspectRatio;

                    SharpDX.Matrix mPers = SharpDX.Matrix.PerspectiveOffCenterRH(
                        -(float)screenW / 2, (float)screenW / 2,
                        -(float)screenH / 2, (float)screenH / 2,
                        this.PerseNear, this.PerseFar);

                    return mPers;
                }
                else
                {
                    return DXMatrix.OrthoRH(_targetCtrl.Width, _targetCtrl.Height, -1000.0f, 10000.0f);
                }
            }

            return DXMatrix.Identity;
        }

        /// <summary>
        /// レンダリング終了
        /// </summary>
        public void EndRendering(Action backBufferCapturingAction)
        {
            // デバイスロストのハンドリング
            if (_deviceLost)
            {
                _vtxSet.Clear();
                _indexSet.Clear();
                _drawTaskSet.Clear();

                var res = _d3dDevice.TestCooperativeLevel();
                if (res.Success || res == ResultCode.DeviceNotReset)
                {
                    OnDeviceLostRelease_();
                    _d3dDevice.Reset(_presentParams);
                    OnResetDevice_(null, null);
                    _deviceLost = false;
                }
                else
                {
                    System.Threading.Thread.Sleep(100);
                }

                return;
            }

            try
            {
                _d3dDevice.BeginScene();

                var clearColor = System.Drawing.Color.Gray;
                clearColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(clearColor) : clearColor;
                _d3dDevice.Clear(ClearFlags.Target | ClearFlags.ZBuffer, SharpDX.ColorBGRA.FromRgba(clearColor.ToArgb()), 10000.0f, 0);

                //
                InitializeTextureSamplingState();

                //--------------------------------------
                // 描画登録があれば描画を行います。
                if (_vtxSet.Count > 0)
                {
                    int taskIndex = 0;

                    int currentIdxIndex = 0;
                    int indexSizeToCopy = 0;
                    int indexSizeCopied = 0;

                    if (_indexSet.Count > 0)
                    {
                        var indexArray = _indexSet.ToArray();
                        var indexStream = _indexBuffer.Lock(0, 0, LockFlags.None);
                        indexSizeToCopy = indexArray.Length - currentIdxIndex;
                        indexSizeCopied = Math.Min(indexSizeToCopy, IndexBufferLength);
                        indexStream.WriteRange(indexArray, 0, indexSizeCopied);
                        _indexBuffer.Unlock();

                        _d3dDevice.Indices = _indexBuffer;
                    }

                    // _vtxSet の内容を頂点バッファにコピーしながら描画していく。
                    int currentVtxIndex = 0;
                    var vertexArray = _vtxSet.ToArray();
                    while (currentVtxIndex < _vtxSet.Count)
                    {
                        var stream = _vertexBufferRect.Lock(0, _vertexBufferRect.Description.SizeInBytes, LockFlags.None);
                        int sizeToCopy = vertexArray.Length - currentVtxIndex;
                        int sizeCopied = Math.Min(sizeToCopy,VertexBufferLength);
                        stream.WriteRange(vertexArray, currentVtxIndex, sizeCopied);

                        _vertexBufferRect.Unlock();

                        _d3dDevice.SetStreamSource(0, _vertexBufferRect, 0, Marshal.SizeOf(typeof(DxCustomVertex)));

                        while (taskIndex < _drawTaskSet.Count)
                        {
                            DrawTask drawTask = _drawTaskSet[taskIndex];
                            if (drawTask is FontDrawTask)
                            {
                                (drawTask as FontDrawTask).DrawFont();
                                taskIndex++;
                                continue;
                            }

                            var primDrawTask = drawTask as PrimitiveDrawTask;
                            if (primDrawTask == null)
                            {
                                taskIndex++;
                                continue;
                            }

                            if (primDrawTask.Index >= currentVtxIndex + sizeCopied)
                            {
                                break;
                            }

                            //---------------------------------------
                            // 描画コマンドの実行

                            primDrawTask.RestoreStateBeforeDraw(_d3dDevice, _builtInVertexShaderConstantTable);

                            if (primDrawTask.IndexOffset >= 0)
                            {
                                _d3dDevice.DrawIndexedPrimitive(
                                    primDrawTask.PrimitiveType,
                                    primDrawTask.Index,
                                    0,
                                    primDrawTask.VertexCount,
                                    primDrawTask.IndexOffset - currentIdxIndex,
                                    primDrawTask.NumPrimitive);
                            }
                            else
                            {
                                _d3dDevice.DrawPrimitives(
                                    primDrawTask.PrimitiveType,
                                    primDrawTask.Index - currentVtxIndex,
                                    primDrawTask.NumPrimitive);
                            }

                            primDrawTask.RestoreStateAfterDraw(_d3dDevice, _builtInVertexShaderConstantTable);

                            taskIndex++;
                        }

                        currentVtxIndex += sizeCopied;
                    }

                    _vtxSet.Clear();
                    _indexSet.Clear();
                    _drawTaskSet.Clear();
                }

                _d3dDevice.EndScene();

                // バックバッファのキャプチャ処理を行う。
                if (backBufferCapturingAction != null)
                {
                    backBufferCapturingAction();
                }

                //-------------------------------------
                // 対象コントロールへの表示を試みます。
                if (NeedPresent)
                {
                    Rectangle rect = _targetCtrl.ClientRectangle;

                    Debug.Assert(_d3dDevice.SwapChainCount == 1);
                    _d3dDevice.GetSwapChain(0).Present(this.LinearGammmaSRGBWriteEnabled ? Present.LinearContent : Present.None);
                }

                // 設定を元に戻し忘れていないか、チェックします。
                Debug.Assert(this.RectangleTopXOffset == 0.0f);
                Debug.Assert(this.FontHeight == 0.0f);
            }
            catch (SharpDX.SharpDXException e)
            {
                //デバイスロストかどうか判定
                if (e.ResultCode == ResultCode.DeviceLost)
                {
                    _deviceLost = true;
                }
                else
                {
                    throw;
                }
            }
        }

        //----------------------------------------------------
#region オプションフラグ関連
        /// <summary>
        ///
        /// </summary>
        public uint OptionFlag
        {
            get { return _rendererOption.OptionFlag; }
            set { _rendererOption.OptionFlag = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public void EnableOptionFlag(uint flag)
        {
            _rendererOption.EnableOptionFlag(flag);
        }

        /// <summary>
        ///
        /// </summary>
        public void DisableOptionFlag(uint flag)
        {
            _rendererOption.DisableOptionFlag(flag);
        }

        /// <summary>
        ///
        /// </summary>
        public void ToggleOptionFlag(uint flag)
        {
            _rendererOption.ToggleOptionFlag(flag);
        }
#endregion


        /// <summary>
        ///
        /// </summary>
        void SetVtxRectangle_(DxCustomVertex[] vtxs, float x, float y, float z, float w, float h)
        {
            Debug.Assert(vtxs.Length == 5);

            float italicOffset = FontHeight == 0.0f ? 0.0f : RectangleTopXOffset * h / FontHeight;
            vtxs[0].Position = new DXVector3(x + RectangleTopXOffset, y, z);
            vtxs[1].Position = new DXVector3(x + w + RectangleTopXOffset, y, z);
            vtxs[2].Position = new DXVector3(x + w - italicOffset + RectangleTopXOffset, y + h, z);
            vtxs[3].Position = new DXVector3(x - italicOffset + RectangleTopXOffset, y + h, z);
            vtxs[4].Position = new DXVector3(x + RectangleTopXOffset, y, z);

            // 角丸などで使用する、左上が (0.0, 0.0), 右下が (1.0, 1.0) となる頂点アトリビュートを設定する。
            // TexCoord4 に設定しているため、TexCoord4 をほかの用途で使用する場合はシェーダーと合わせて調整の必要がある。
            vtxs[0].Tu4 = 0.0f;
            vtxs[0].Tv4 = 0.0f;

            vtxs[1].Tu4 = 1.0f;
            vtxs[1].Tv4 = 0.0f;

            vtxs[2].Tu4 = 1.0f;
            vtxs[2].Tv4 = 1.0f;

            vtxs[3].Tu4 = 0.0f;
            vtxs[3].Tv4 = 1.0f;

            vtxs[4].Tu4 = 0.0f;
            vtxs[4].Tv4 = 0.0f;
        }

        /// <summary>
        ///
        /// </summary>
        void SetVtxRectangleColor_(DxCustomVertex[] vtxs, Color color)
        {
            Debug.Assert(vtxs.Length == 5);

            int c = color.ToArgb();

            vtxs[0].Color = c;
            vtxs[1].Color = c;
            vtxs[2].Color = c;
            vtxs[3].Color = c;
            vtxs[4].Color = c;
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawRectangle(float x, float y, float z, float w, float h)
        {
            int startIdx = _vtxSet.Count;

            SetVtxRectangle_(_temporaryVtxSet, x, y, z, w, h);

            var renderColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(Color) : Color;
            SetVtxRectangleColor_(_temporaryVtxSet, renderColor);

            _vtxSet.AddRange(_temporaryVtxSet);

            _drawTaskSet.Add(new PrimitiveDrawTask(startIdx, 5, PrimitiveType.LineStrip, 4, -1, _stateChangerSet, this));
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawRectangleOrtho(float x, float y, float z, float w, float h)
        {
            int startIdx = _vtxSet.Count;

            SetVtxRectangle_(_temporaryVtxSet, x, y, z, w, h);

            var renderColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(Color) : Color;
            SetVtxRectangleColor_(_temporaryVtxSet, renderColor);

            _vtxSet.AddRange(_temporaryVtxSet);

            _drawTaskSet.Add(new PrimitiveDrawTask(startIdx, 5, PrimitiveType.LineStrip, 4, -1, _stateChangerSet, this, PrimitiveDrawTask.ProjectionMode.ForceOrtho));
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawPoint(PointF p)
        {
            PointF[] temp = new PointF[] { p };
            this.TransformPointsToView(temp);
            p = temp[0];

            this.PushMtx();
            this.IdentityMtx();

            FillRectangle(p.X - LineWidth * 0.5f, p.Y - LineWidth * 0.5f, 0, LineWidth, LineWidth);

            this.PopMtx();
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawLine(Point p0, Point p1)
        {

            DrawLine(MathUtil.PointToPointF(p0), MathUtil.PointToPointF(p1));
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawLine(PointF p0, PointF p1)
        {
            int startIdx = _vtxSet.Count;

            var renderColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(Color) : Color;
            _temporaryVtxSet[0].Position = new DXVector3(p0.X, p0.Y, PerseNear);
            _temporaryVtxSet[0].Color = renderColor.ToArgb();
            _temporaryVtxSet[1].Position = new DXVector3(p1.X, p1.Y, PerseNear);
            _temporaryVtxSet[1].Color = renderColor.ToArgb();

            _vtxSet.Add(_temporaryVtxSet[0]);
            _vtxSet.Add(_temporaryVtxSet[1]);

            _drawTaskSet.Add(new PrimitiveDrawTask(startIdx, 2, PrimitiveType.LineList, 1, -1, _stateChangerSet, this));
        }

        /// <summary>
        ///
        /// </summary>
        public void FillRectangle(int x, int y, int z, int w, int h)
        {
            FillRectangle((float)x, (float)y, (float)z, (float)w, (float)h);
        }

        /// <summary>
        ///
        /// </summary>
        public void FillRectangle(float x, float y, float z, float w, float h)
        {
            if (_rendererOption.DrawOutlineEnable)
            {
                DrawRectangle(x, y, z, w, h);
                return;
            }

            int startIdx = _vtxSet.Count;

            SetVtxRectangle_(_temporaryVtxSet, x, y, z, w, h);

            var renderColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(Color) : Color;
            SetVtxRectangleColor_(_temporaryVtxSet, renderColor);

            _vtxSet.AddRange(_temporaryVtxSet);

            _drawTaskSet.Add(new PrimitiveDrawTask(startIdx, 5, PrimitiveType.TriangleFan, 2, -1, _stateChangerSet, this));
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawString(string str, SysFont font, int x, int y)
        {
            var renderColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(this.Color) : this.Color;
            _drawTaskSet.Add(new FontDrawTask(font.Bold ? _fontBold : _font, str, new Point(x, y), renderColor));
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawString(string str, SysFont font, float x, float y)
        {
            var renderColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(this.Color) : this.Color;
            _drawTaskSet.Add(new FontDrawTask(font.Bold ? _fontBold : _font, str, new Point((int)x, (int)y), renderColor));
        }

        /// <summary>
        ///
        /// </summary>
        int RoundupToPow2_(int n)
        {

            if (n <= 0)
            {
                return 0;
            }

            int orig = n;
            int pow2 = 0;
            while ((n >> 1) > 0)
            {
                pow2++;
                n = n >> 1;
            }

            int result = (int)Math.Pow(2, pow2);

            // 切り上げます。
            if (result < orig)
            {
                result *= 2;
            }

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        void DrawImage_(FVec3 imgPos, FVec2 imgSize, TexCoord4[] coords, Color[] vtxColors)
        {

            //------------------------------------------------------
            // 簡易モードなら ...
            if (_rendererOption.DrawOutlineEnable)
            {
                DrawRectangle(imgPos.X, imgPos.Y, imgPos.Z, imgSize.X, imgSize.Y);
                return;
            }

            int startIdx = _vtxSet.Count;

            //------------------------------------------------------
            // 頂点の登録
            DxCustomVertex[] vtxs = _temporaryVtxSet;

            SetVtxRectangle_(vtxs, imgPos.X, imgPos.Y, imgPos.Z, imgSize.X, imgSize.Y);

            for (int i = 0; i < vtxColors.Length; i++)
            {
                int alpha = (int)(vtxColors[i].A * PaneHelper.GetNormalizedTransparency(Transparency));
                var renderColor = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(vtxColors[i]) : vtxColors[i];
                vtxs[i].Color = Color.FromArgb(alpha, renderColor).ToArgb();
            }

            //------------------------------------------------------
            // UV の登録
            if (coords != null)
            {
                if (coords.Length > 0 && coords[0] != null)
                {
                    vtxs[0].Tu = coords[0].LT.X; vtxs[0].Tv = coords[0].LT.Y;
                    vtxs[1].Tu = coords[0].RT.X; vtxs[1].Tv = coords[0].RT.Y;
                    vtxs[2].Tu = coords[0].RB.X; vtxs[2].Tv = coords[0].RB.Y;
                    vtxs[3].Tu = coords[0].LB.X; vtxs[3].Tv = coords[0].LB.Y;
                }

                if (coords.Length > 1 && coords[1] != null)
                {
                    vtxs[0].Tu2 = coords[1].LT.X; vtxs[0].Tv2 = coords[1].LT.Y;
                    vtxs[1].Tu2 = coords[1].RT.X; vtxs[1].Tv2 = coords[1].RT.Y;
                    vtxs[2].Tu2 = coords[1].RB.X; vtxs[2].Tv2 = coords[1].RB.Y;
                    vtxs[3].Tu2 = coords[1].LB.X; vtxs[3].Tv2 = coords[1].LB.Y;
                }

                if (coords.Length > 2 && coords[2] != null)
                {
                    vtxs[0].Tu3 = coords[2].LT.X; vtxs[0].Tv3 = coords[2].LT.Y;
                    vtxs[1].Tu3 = coords[2].RT.X; vtxs[1].Tv3 = coords[2].RT.Y;
                    vtxs[2].Tu3 = coords[2].RB.X; vtxs[2].Tv3 = coords[2].RB.Y;
                    vtxs[3].Tu3 = coords[2].LB.X; vtxs[3].Tv3 = coords[2].LB.Y;
                }
            }

            _vtxSet.AddRange(vtxs);

            if (!_lowLevelCombinerSettingsEnabled)
            {
                // カラー補間ステージの設定
                Color defColorB = Color.FromArgb(0, Color.Black);
                Color defColorW = Color.FromArgb(255, Color.White);
                // 白黒カラーが HDR カラーの場合はデガンマを行わない。
                var wCol = (LinearGammmaColorParamatersEnabled && _whiteBlendColor.ByteConvert)
                                ? ColorHelper.ConvertToLinerGamma(_whiteBlendColor.ToSystemColor()) : _whiteBlendColor.ToSystemColor();
                var bCol = (LinearGammmaColorParamatersEnabled && _blackBlendColor.ByteConvert)
                                ? ColorHelper.ConvertToLinerGamma(_blackBlendColor.ToSystemColor()) : _blackBlendColor.ToSystemColor();

                // 0枚テクスチャでは黒補間は無効。
                if (_currentUsedTextureStageNum == 0)
                {
                    bCol = defColorB;
                }
                _stateChangerSet.Add(new ColorBlendData(wCol, bCol, _isThresholdingAlphaInterpolationEnabled));
            }

            _drawTaskSet.Add(new PrimitiveDrawTask(startIdx, 5, PrimitiveType.TriangleFan, 2, -1, _stateChangerSet, this));
            _currentUsedTextureStageNum = 0;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetPackedFontMode(bool ignoreBorder)
        {
            _stateChangerSet.Add(new PackedFontFormat(ignoreBorder));
        }

        /// <summary>
        ///
        /// </summary>
        public void TryUpdateTextureIfOld(Bitmap bmp, DateTime lastModifyTime, RendererTextrureFormat format)
        {
            TryGetUpdatedTextureMap_(bmp, lastModifyTime, format);
        }

        /// <summary>
        /// Bitmap をキーにテクスチャリソースを破棄する
        /// </summary>
        /// <param name="bmp"></param>
        /// <param name="format"></param>
        public void TryRemoveTextureFromBitmap(Bitmap bmp, RendererTextrureFormat format)
        {
            TextureMapProxy textureMapProxy = null;
            if (_bitmapTextureMap.TryGetValue(bmp, out textureMapProxy))
            {
                _bitmapTextureMap.Remove(bmp);

                var dxTexture = textureMapProxy.GetTexture(format);
                if (dxTexture != null)
                {
                    dxTexture.Dispose();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        TextureMapProxy TryGetUpdatedTextureMap_(Bitmap bmp, DateTime lastModifyTime, RendererTextrureFormat format)
        {
            if (bmp == null)
            {
                return null;
            }

            TextureMapProxy textureMapProxy = null;
            if (_bitmapTextureMap.TryGetValue(bmp, out textureMapProxy))
            {
                if (textureMapProxy.SrcBmpLastModifyTime < lastModifyTime)
                {
                    DbgConsole.WriteLine("*************** D3D -- TryUpdateTextureIfOld - UPDATED {0}->{1}", textureMapProxy.SrcBmpLastModifyTime, lastModifyTime);

                    TryRemoveTextureFromBitmap(bmp, format);

                    textureMapProxy.SetTexture(CreateDxTexture_(bmp, format), format);
                    textureMapProxy.SrcBmpLastModifyTime = lastModifyTime;

                    _bitmapTextureMap.Add(bmp, textureMapProxy);
                }
                else
                {
                    if (textureMapProxy.GetTexture(format) == null)
                    {
                        textureMapProxy.SetTexture(CreateDxTexture_(bmp, format), format);
                    }
                }
            }
            else
            {
                textureMapProxy = new TextureMapProxy();
                textureMapProxy.SetTexture(CreateDxTexture_(bmp, format), format);
                textureMapProxy.SrcBmpLastModifyTime = lastModifyTime;

                _bitmapTextureMap.Add(bmp, textureMapProxy);
            }

            return textureMapProxy;
        }

        /// <summary>
        /// テクスチャステージを設定
        /// </summary>
        public void SetTextureState(Bitmap bmp, int slotIdx, AttrCombineMode colorBlend, AttrCombineMode alphaBlend, FVec2 indirectScale, float indirectRotate, int indirectImageChunnelNum, RendererTextrureFormat format)
        {
            if (bmp == null)
            {
                return;
            }

            if (_rendererOption.DrawOutlineEnable)
            {
                return;
            }

            //------------------------------------------------------
            // テクスチャの登録(初回のみ)
            TextureMapProxy textureMapProxy = TryGetUpdatedTextureMap_(bmp, DateTime.MinValue, format);
            Texture dxTexture = textureMapProxy.GetTexture(format);

            // アルファ1チャンネルフォーマットの場合は、 _indirectImgChannelNum == -1 として、アルファチャンネルをフェッチするようにします。
            if (indirectImageChunnelNum == 1)
            {
                if (format == RendererTextrureFormat.A8FromRGB || format == RendererTextrureFormat.A8FromARGB)
                {
                    indirectImageChunnelNum = -1;
                }
            }

            _stateChangerSet.Add(new TextureEnviromentData(dxTexture, slotIdx, colorBlend, alphaBlend, indirectScale, indirectRotate, indirectImageChunnelNum));

            _currentUsedTextureStageNum++;
        }

        /// <summary>
        /// 詳細マテリアルを設定
        /// </summary>
        public void SetDetailedCombinerState(Color[] colors, ITevStage[] tevStage, int stageCount)
        {
            // 詳細コンバイナ用シェーダの選択
            int sourceMax = 0;
            for (int i = 0; i < stageCount; i++)
            {
                // ＲＧＢソース段数
                switch(tevStage[i].ColorStage.CombineMode)
                {
                case AttrCombineMode.Replace:
                    sourceMax = Math.Max(sourceMax, 1);
                    break;
                case AttrCombineMode.Add:
                case AttrCombineMode.Modulate:
                case AttrCombineMode.AddSigned:
                case AttrCombineMode.Subtract:
                    sourceMax = Math.Max(sourceMax, 2);
                    break;
                case AttrCombineMode.Interpolate:
                case AttrCombineMode.AddMultDMP:
                case AttrCombineMode.MultAddDMP:
                    sourceMax = Math.Max(sourceMax, 3);
                    break;
                default:
                    sourceMax = Math.Max(sourceMax, 1);
                    break;
                }

                switch(tevStage[i].AlphaStage.CombineMode)
                {
                case AttrCombineMode.Replace:
                    sourceMax = Math.Max(sourceMax, 1);
                    break;
                case AttrCombineMode.Add:
                case AttrCombineMode.Modulate:
                case AttrCombineMode.AddSigned:
                case AttrCombineMode.Subtract:
                    sourceMax = Math.Max(sourceMax, 2);
                    break;
                case AttrCombineMode.Interpolate:
                case AttrCombineMode.AddMultDMP:
                case AttrCombineMode.MultAddDMP:
                    sourceMax = Math.Max(sourceMax, 3);
                    break;
                default:
                    sourceMax = Math.Max(sourceMax, 1);
                    break;
                }
            }

            if (stageCount <= 0)
            {
                _detailedCombinerValiationType = DetailedCombinerVariationType.DetailedCombinerVariationType_Stage0;
            }
            else
            {
                _detailedCombinerValiationType = (DetailedCombinerVariationType)(1 + Math.Max(0, Math.Min(5, stageCount - 1)) * 3 + Math.Max(0, Math.Min(2, sourceMax - 1)));
            }

            // コンスタント設定
            if (_builtInDetailedCombinerShaders.Count > 0)
            {
                // 白黒カラーの HDR カラー設定を参照して、HDRカラーモードかどうかを判定します。
                // HDRカラーモードの場合はデガンマを行いません。
                Color[] renderColors = new Color[colors.Length];
                for(int i = 0;i < renderColors.Length;i++)
                {
                    renderColors[i] = (LinearGammmaColorParamatersEnabled && _whiteBlendColor.ByteConvert) ? ColorHelper.ConvertToLinerGamma(colors[i]) : colors[i];
                }

                _stateChangerSet.Add(new DetailedCombinerData(renderColors, tevStage, stageCount, _currentUsedTextureStageNum, _builtInDetailedCombinerShaders[(int)_detailedCombinerValiationType].constantTable));
            }

            _lowLevelCombinerSettingsEnabled = tevStage != null ? true : false;


        }

        /// <summary>
        /// 角丸の形状設定
        /// </summary>
        /// <param name="paneWidth"></param>
        /// <param name="paneHeight"></param>
        /// <param name="exp"></param>
        /// <param name="radius"></param>
        public void SetProceduralShapeState(float paneWidth, float paneHeight, float exp, float radius)
        {
            _stateChangerSet.Add(new ProceduralShapeState(paneWidth, paneHeight, exp, radius));
        }

        /// <summary>
        /// 角丸の各効果のブレンド設定
        /// </summary>
        /// <param name="innerStrokeBlendType"></param>
        /// <param name="innerShadowBlendType"></param>
        /// <param name="colorOverlayBlendType"></param>
        /// <param name="gradationOverlayBlendType"></param>
        public void SetProceduralShapeEffectBlendState(
            LECore.Structures.LECoreInterface.ProceduralShapeEffectBlendMode innerStrokeBlendType,
            LECore.Structures.LECoreInterface.ProceduralShapeEffectBlendMode innerShadowBlendType,
            LECore.Structures.LECoreInterface.ProceduralShapeEffectBlendMode colorOverlayBlendType,
            LECore.Structures.LECoreInterface.ProceduralShapeEffectBlendMode gradationOverlayBlendType)
        {
            _stateChangerSet.Add(new ProceduralShapeEffectBlendState(innerStrokeBlendType, innerShadowBlendType, colorOverlayBlendType, gradationOverlayBlendType));
        }

        /// <summary>
        /// 角丸の境界線効果の設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="innerStrokeSize"></param>
        /// <param name="innerStrokeColor"></param>
        public void SetProceduralShapeInnerStrokeState(bool enabled, float innerStrokeSize, Color innerStrokeColor)
        {
            _stateChangerSet.Add(new ProceduralShapeInnerStrokeState(
                enabled,
                innerStrokeSize,
                LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(innerStrokeColor) : innerStrokeColor));
        }

        /// <summary>
        /// 角丸のシャドウ(内側)設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="paneWidth"></param>
        /// <param name="paneHeight"></param>
        /// <param name="radius"></param>
        /// <param name="innerShadowSize"></param>
        /// <param name="angle"></param>
        /// <param name="distance"></param>
        /// <param name="innerShadowColor"></param>
        /// <param name="shadowType"></param>
        public void SetProceduralShapeInnerShadowState(bool enabled, float paneWidth, float paneHeight, float radius, float innerShadowSize, float angle, float distance, Color innerShadowColor, ProceduralShapeShadowType shadowType)
        {
            _stateChangerSet.Add(new ProceduralShapeInnerShadowState(
                enabled,
                paneWidth,
                paneHeight,
                radius,
                innerShadowSize,
                angle,
                distance,
                LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(innerShadowColor) : innerShadowColor,
                shadowType));
        }

        /// <summary>
        /// 角丸のカラーオーバーレイ設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="color"></param>
        public void SetProceduralShapeColorOverlayState(bool enabled, Color color)
        {
            _stateChangerSet.Add(new ProceduralShapeColorOverlayState(
                enabled,
                LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(color) : color));
        }

        /// <summary>
        /// 角丸のグラデーションオーバーレイ設定
        /// </summary>
        /// <param name="enabled"></param>
        /// <param name="controlPoints"></param>
        /// <param name="colors"></param>
        /// <param name="angle"></param>
        public void SetProceduralShapeGradationOverlayState(bool enabled, float[] controlPoints, Color[] colors, float angle)
        {
            for (int i = 0; i < colors.Length; ++i)
            {
                colors[i] = LinearGammmaColorParamatersEnabled ? ColorHelper.ConvertToLinerGamma(colors[i]) : colors[i];
            }
            _stateChangerSet.Add(new ProceduralShapeGradationOverlayState(enabled, controlPoints, colors, angle));
        }

        private Texture CreateDxTexture_(Bitmap bmp, RendererTextrureFormat format)
        {
            lock (bmp)
            {
                int newW = RoundupToPow2_(bmp.Width);
                int newH = RoundupToPow2_(bmp.Height);

                Bitmap textureSrcBmp;
                if (newW != bmp.Width || newH != bmp.Height || format != RendererTextrureFormat.ARGB)
                {
                    if (bmp.PixelFormat == PixelFormat.Format32bppArgb || bmp.PixelFormat == PixelFormat.Format24bppRgb)
                    {
                        // textureSrcBmp = new Bitmap(bmp, newW, newH) の コンストラクタの中で、Imageの描画が行われ、カラーチャンネルの情報がアルファと乗算された結果に変更されてしまうことから
                        // 結果として、カラーチャンネルだけを使うブレンド計算で、実機と表示結果が不一致となる不具合が起こっていた。
                        // そこで、自前でスケーリング処理を記述しつつ、ビット値をそのままコピーするようにした。
                        textureSrcBmp = new Bitmap(newW, newH);

                        BitmapData bdDst = textureSrcBmp.LockBits(new Rectangle(0, 0, textureSrcBmp.Width, textureSrcBmp.Height), ImageLockMode.WriteOnly, textureSrcBmp.PixelFormat);
                        BitmapData bdSrc = bmp.LockBits(new Rectangle(0, 0, bmp.Width, bmp.Height), ImageLockMode.ReadOnly, bmp.PixelFormat);

                        int dataSizeDst = bdDst.Stride * bdDst.Height;
                        int dataSizeSrc = bdSrc.Stride * bdSrc.Height;
                        byte[] bufDst = new byte[dataSizeDst];
                        byte[] bufSrc = new byte[dataSizeSrc];
                        Marshal.Copy(bdSrc.Scan0, bufSrc, 0, dataSizeSrc);

                        float magW = (float)bmp.Width / newW;
                        float magH = (float)bmp.Height / newH;

                        int numChanelSrc = bmp.PixelFormat == PixelFormat.Format32bppArgb ? 4 : 3;

                        for (int y = 0; y < textureSrcBmp.Height; y++)
                        {
                            float rawYSrc = y * magH;
                            int origYSrc = (int)rawYSrc;
                            int nearYSrc = Math.Min(origYSrc + 1, bmp.Height - 1);
                            float dYSrc = rawYSrc - (float)origYSrc;

                            int offsetYDst = bdDst.Stride * y;
                            int offsetY0Src = bdSrc.Stride * origYSrc;
                            int offsetY1Src = bdSrc.Stride * nearYSrc;

                            for (int x = 0; x < textureSrcBmp.Width; x++)
                            {
                                float rawXSrc = x * magW;
                                int origXSrc = ((int)rawXSrc + 0);
                                int nearXSrc = Math.Min(origXSrc + 1, bmp.Width - 1);
                                float dXSrc = rawXSrc - (float)origXSrc;

                                int ofsetDst = offsetYDst + x * 4;
                                int ofset00Src = offsetY0Src + origXSrc * numChanelSrc;
                                int ofset01Src = offsetY1Src + origXSrc * numChanelSrc;
                                int ofset10Src = offsetY0Src + nearXSrc * numChanelSrc;
                                int ofset11Src = offsetY1Src + nearXSrc * numChanelSrc;

                                for (int c = 0; c < numChanelSrc; c++)
                                {
                                    bufDst[ofsetDst + c] = (byte)(
                                        (1.0f - dXSrc) * (1.0f - dYSrc) * bufSrc[ofset00Src + c] +
                                        (1.0f - dXSrc) * (dYSrc) * bufSrc[ofset01Src + c] +
                                        (dXSrc) * (1.0f - dYSrc) * bufSrc[ofset10Src + c] +
                                        (dXSrc) * (dYSrc) * bufSrc[ofset11Src + c]);
                                }

                                // ソースがアルファチャンネルを持たない場合は、255を埋めます。
                                if (numChanelSrc == 3)
                                {
                                    bufDst[ofsetDst + 3] = 255;
                                }
                            }
                        }

                        Marshal.Copy(bufDst, 0, bdDst.Scan0, dataSizeDst);
                        textureSrcBmp.UnlockBits(bdDst);
                        bmp.UnlockBits(bdSrc);
                    }
                    else
                    {
                        textureSrcBmp = new Bitmap(bmp, newW, newH);
                    }
                }
                else
                {
                    textureSrcBmp = bmp;
                }

                // 全ピクセルのアルファがゼロだと、テクスチャ生成時に勝手にアルファチャンネルがドロップされてしまう。
                // 対策として、少しアルファ値を入れる。
                {
                    try
                    {
                        Color sampleColor = textureSrcBmp.GetPixel(0, 0);
                        textureSrcBmp.SetPixel(0, 0, Color.FromArgb(sampleColor.A == 0 ? sampleColor.A + 1 : sampleColor.A, sampleColor));
                    }
                    catch (Exception e)
                    {

                        throw e;
                    }
                }

                AdjustPixelsOfTextureSorceTexture_(textureSrcBmp, format);

                MemoryStream ms = new MemoryStream();
                textureSrcBmp.Save(ms, ImageFormat.Bmp);

                int srcTexW = textureSrcBmp.Width;
                int srcTexH = textureSrcBmp.Height;
                if (!object.ReferenceEquals(textureSrcBmp, bmp))
                {
                    textureSrcBmp.Dispose();
                }

                return Texture.FromMemory(_d3dDevice, ms.ToArray(), srcTexW, srcTexH, 0, Usage.None, GetDxTextureFormat_(format), Pool.Managed, Filter.None, Filter.Default, 0);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void AdjustPixelsOfTextureSorceTexture_(Bitmap textureSrcBmp, RendererTextrureFormat rendererFormat)
        {
            if (rendererFormat == RendererTextrureFormat.ARGB)
            {
                return;
            }

            if (textureSrcBmp.PixelFormat != PixelFormat.Format32bppArgb)
            {
                return;
            }

            BitmapData bd = textureSrcBmp.LockBits(new Rectangle(0, 0, textureSrcBmp.Width, textureSrcBmp.Height), ImageLockMode.ReadWrite, textureSrcBmp.PixelFormat);

            int dataSize = bd.Stride * bd.Height;
            byte[] buf = new byte[dataSize];
            Marshal.Copy(bd.Scan0, buf, 0, dataSize);


            switch (rendererFormat)
            {
                case RendererTextrureFormat.A8FromARGB:
                    for (int y = 0; y < textureSrcBmp.Height; y++)
                    {
                        int offsetY = bd.Stride * y;
                        for (int x = 0; x < textureSrcBmp.Width; x++)
                        {
                            int ofset = offsetY + x * 4;
                            buf[ofset + 0] = 254;
                            buf[ofset + 1] = 255;
                            buf[ofset + 2] = 255;
                            // buf[offsetY + x * 4 + 3] = buf[offsetY + x * 4 + 3]; // A はそのまま使う
                        }
                    }
                    break;
                case RendererTextrureFormat.A8FromRGB:
                    for (int y = 0; y < textureSrcBmp.Height; y++)
                    {
                        int offsetY = bd.Stride * y;
                        for (int x = 0; x < textureSrcBmp.Width; x++)
                        {
                            int ofset = offsetY + x * 4;
                            byte r = buf[ofset + 2];

                            buf[ofset + 0] = 254;
                            buf[ofset + 1] = 255;
                            buf[ofset + 2] = 255;
                            buf[ofset + 3] = r;   // R 成分をAとする
                        }
                    }
                    break;
                case RendererTextrureFormat.L8:
                    for (int y = 0; y < textureSrcBmp.Height; y++)
                    {
                        int offsetY = bd.Stride * y;
                        for (int x = 0; x < textureSrcBmp.Width; x++)
                        {
                            int ofset = offsetY + x * 4;
                            byte r = buf[ofset + 2];

                            buf[ofset + 0] = r;
                            buf[ofset + 1] = r;
                            buf[ofset + 2] = r;
                            buf[ofset + 3] = 255;
                        }
                    }
                    break;
                case RendererTextrureFormat.LA8:
                    for (int y = 0; y < textureSrcBmp.Height; y++)
                    {
                        int offsetY = bd.Stride * y;
                        for (int x = 0; x < textureSrcBmp.Width; x++)
                        {
                            int ofset = offsetY + x * 4;
                            byte r = buf[ofset + 2];

                            buf[ofset + 0] = r;
                            buf[ofset + 1] = r;
                            buf[ofset + 2] = r;
                            // buf[offsetY + x * 4 + 3] = buf[offsetY + x * 4 + 3]; // A はそのまま使う
                        }
                    }
                    break;
                case RendererTextrureFormat.BC1FromARGB:
                    for (int y = 0; y < textureSrcBmp.Height; y++)
                    {
                        int offsetY = bd.Stride * y;
                        for (int x = 0; x < textureSrcBmp.Width; x++)
                        {
                            int ofset = offsetY + x * 4;
                            // 2値化します。
                            buf[ofset + 3] = (byte)(buf[ofset + 3] > 127 ? 255 : 0);
                        }
                    }
                    break;
                default:
                    break;
            }

            Marshal.Copy(buf, 0, bd.Scan0, dataSize);
            textureSrcBmp.UnlockBits(bd);
        }

        Format GetDxTextureFormat_(RendererTextrureFormat rendererFormat)
        {
            switch (rendererFormat)
            {
                case RendererTextrureFormat.ARGB:
                case RendererTextrureFormat.BC1FromARGB:
                default:
                    return Format.A8R8G8B8;
                case RendererTextrureFormat.A8FromARGB:
                case RendererTextrureFormat.A8FromRGB:
                case RendererTextrureFormat.L8:
                case RendererTextrureFormat.LA8:
                    return Format.A8L8;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawImage(FVec3 imgPos, FVec2 imgSize, TexCoord4[] coords)
        {
            // 描画
            DrawImage_(imgPos, imgSize, coords, _defaultVtxColors);
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawImage(FVec3 imgPos, FVec2 imgSize, TexCoord4[] coords, Color[] vtxColors)
        {
            // 描画
            DrawImage_(imgPos, imgSize, coords, vtxColors);
        }

        /// <summary>
        ///
        /// </summary>
        public void DrawMesh(float[] vertices, float[] texCoords, int[] indices)
        {
            //------------------------------------------------------
            // 簡易モードなら ...
            if (_rendererOption.DrawOutlineEnable)
            {
                //DrawRectangle(imgPos.X, imgPos.Y, imgPos.Z, imgSize.X, imgSize.Y);
                return;
            }

            int vertexStartIdx = _vtxSet.Count;
            int indexStartIdx = _indexSet.Count;

            //------------------------------------------------------
            // 頂点の登録
            int vtxCount = vertices.Length/ 3;
            DxCustomVertex[] vtxs = new DxCustomVertex[vtxCount];

            int alpha = (int)(PaneHelper.GetNormalizedTransparency(Transparency) * 255.0f);

            for (int i = 0; i < vtxCount; ++i)
            {
                DXVector3 pos;
                pos.X = vertices[i * 3    ];
                pos.Y = vertices[i * 3 + 1];
                pos.Z = vertices[i * 3 + 2];

                vtxs[i].Position = pos;
                vtxs[i].Color = Color.FromArgb(alpha, 255, 255, 255).ToArgb();
                vtxs[i].Tu = 0.0f;
                vtxs[i].Tv = 0.0f;
            }

            if (texCoords != null &&
                texCoords.Length / 2 == vtxCount)
            {
                for (int i = 0; i < vtxCount; ++i)
                {
                    vtxs[i].Tu = texCoords[i * 2];
                    vtxs[i].Tv = texCoords[i * 2 + 1];
                    vtxs[i].Tu2 = texCoords[i * 2];
                    vtxs[i].Tv2 = texCoords[i * 2 + 1];
                    vtxs[i].Tu3 = texCoords[i * 2];
                    vtxs[i].Tv3 = texCoords[i * 2 + 1];
                }
            }

            if (!_lowLevelCombinerSettingsEnabled)
            {
                // カラー補間ステージの設定
                Color defColorB = Color.FromArgb(0, Color.Black);
                Color defColorW = Color.FromArgb(255, Color.White);
                if (_whiteBlendColor.ToSystemColor() != defColorW || _blackBlendColor.ToSystemColor() != defColorB)
                {
                    // 白黒カラーが HDR カラーの場合はデガンマを行わない。
                    var wCol = (LinearGammmaColorParamatersEnabled && _whiteBlendColor.ByteConvert)
                                    ? ColorHelper.ConvertToLinerGamma(_whiteBlendColor.ToSystemColor()) : _whiteBlendColor.ToSystemColor();
                    var bCol = (LinearGammmaColorParamatersEnabled && _blackBlendColor.ByteConvert)
                                    ? ColorHelper.ConvertToLinerGamma(_blackBlendColor.ToSystemColor()) : _blackBlendColor.ToSystemColor();

                    // 0枚テクスチャでは黒補間は無効。
                    if (_currentUsedTextureStageNum == 0)
                    {
                        bCol = defColorB;
                    }
                    _stateChangerSet.Add(new ColorBlendData(wCol, bCol, _isThresholdingAlphaInterpolationEnabled));
                }
            }

            _vtxSet.AddRange(vtxs);

            _indexSet.AddRange(indices);

            _drawTaskSet.Add(new PrimitiveDrawTask(vertexStartIdx, vtxCount, PrimitiveType.TriangleList, indices.Length / 3, indexStartIdx, _stateChangerSet, this));
            _currentUsedTextureStageNum = 0;

        }

        /// <summary>
        ///
        /// </summary>
        public void SetLineCap(LineCap start, LineCap end)
        {
            // throw new Exception( "The method or operation is not implemented." );
        }

        /// <summary>
        ///
        /// </summary>
        public void TransformPointsToView(PointF[] pointSet)
        {
            DXMatrix proj = this._d3dDevice.GetTransform(TransformState.Projection);

            DXMatrix modelView = _mtxStack.Top * _mtxViewStack.Top;
            for (int i = 0; i < pointSet.Length; i++)
            {
                Vector4 result = Vector3.Transform(new Vector3(pointSet[i].X, pointSet[i].Y, this.PerseNear), modelView);

                pointSet[i] = new PointF(result.X, result.Y);
            }
        }

        /// <summary>
        /// ポイントセットを現在の逆変換行列で変換します。
        /// </summary>
        public void TransformPointsToScene(PointF[] pointSet)
        {
            if (PersepectiveRendering)
            {
                DXMatrix proj = this._d3dDevice.GetTransform(TransformState.Projection);
                Vector3 nearScrn = this._d3dDevice.Viewport.Project(new Vector3(0.0f, 0.0f, this.PerseNear), proj, _mtxViewStack.Top, _mtxStack.Top);

                for (int i = 0; i < pointSet.Length; i++)
                {
                    Vector3 dst = this._d3dDevice.Viewport.Unproject(new Vector3(pointSet[i].X, pointSet[i].Y, nearScrn.Z), proj, _mtxViewStack.Top, _mtxStack.Top);
                    pointSet[i] = new PointF(dst.X, dst.Y);
                }
            }
            else
            {
                // 正規化されたビューポートを使わない分、上の計算より誤差が小さくなることが多い
                var mtx = DXMatrix.Multiply(_mtxStack.Top, _mtxViewStack.Top);
                Debug.Assert(mtx.Determinant() != 0);
                mtx.Invert();
                for (int i=0;i < pointSet.Length; i++)
                {
                    var pos = new Vector3(pointSet[i].X - _d3dDevice.Viewport.Width / 2, -(pointSet[i].Y - _d3dDevice.Viewport.Height / 2), 0);
                    var result = DXVector3.Transform(pos, mtx);
                    pointSet[i] = new PointF(result.X, result.Y);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void DbgDrawLine(PointF p0, PointF p1)
        {
            // throw new Exception( "The method or operation is not implemented." );
        }

        /// <summary>
        ///
        /// </summary>
        public void DbgDrawRectangle(float x, float y, float z, float w, float h)
        {
            // throw new Exception( "The method or operation is not implemented." );
        }

        /// <summary>
        ///
        /// </summary>
        public void DbgDrawString(string str, SysFont font, float x, float y)
        {
            // throw new Exception( "The method or operation is not implemented." );
        }

        /// <summary>
        ///
        /// </summary>
        public void SetTextureMtx(int index, FVec2 trans, FVec2 scale, float rotate)
        {

            // 行列を作成する。
            // 平行移動成分を指定する要素に注意が必要です。


            // アフィン変換中心を調整する行列を作成します。
            DXMatrix mtxAdjustCenter = DXMatrix.Identity;
            mtxAdjustCenter.M41 = 0.5f;
            mtxAdjustCenter.M42 = 0.5f;

            DXMatrix mtxAdjustCenterInv = mtxAdjustCenter;
            mtxAdjustCenter.M41 = -0.5f;
            mtxAdjustCenter.M42 = -0.5f;


            DXMatrix mtx = DXMatrix.Transformation2D(
                Vector2.Zero,
                0.0f,
                new Vector2(scale.X, scale.Y),
                Vector2.Zero,
                MathUtil.DegToRad(rotate),
                Vector2.Zero);
            mtx.M41 = trans.X;
            mtx.M42 = trans.Y;


            mtx = mtxAdjustCenter * mtx * mtxAdjustCenterInv;

            _stateChangerSet.Add(new TextureMtxData(index, mtx));
        }

        public void SetTextureProjectionState(int index, FVec2 imageSize, Matrix34 mxt)
        {
            _stateChangerSet.Add(new TextureProjectionState(index, imageSize, mxt));
        }

        /// <summary>
        ///
        /// </summary>
        public void SetBlendMode(AttrBlendType colorBlendtype, AttrBlendFactor srcColor, AttrBlendFactor dstColor, AttrBlendOp colorOp, AttrBlendFactor srcAlpha, AttrBlendFactor dstAlpha, AttrBlendOp alphaOp)
        {
            _stateChangerSet.Add(new PEEnviromentData(colorBlendtype, srcColor, dstColor, colorOp, srcAlpha, dstAlpha, alphaOp));
        }

        /// <summary>
        ///
        /// </summary>
        public void SetAlphaCompare(bool enabled, AttrCompareFunc alphaCompare, float alpha)
        {
            _stateChangerSet.Add(new AlphaCompare(enabled, alphaCompare, alpha));
        }

        /// <summary>
        ///
        /// </summary>
        static TextureAddress TexWrapToTextureAddress_(TexWrap texWrap)
        {
            switch (texWrap)
            {
                case TexWrap.Clamp: return TextureAddress.Clamp;
                case TexWrap.Mirror: return TextureAddress.Mirror;
                case TexWrap.Repeat: return TextureAddress.Wrap;
                default: Debug.Assert(false); return TextureAddress.Border;
            }
        }

        /// <summary>
        ///
        /// </summary>
        static TextureFilter TexFilterMinTextureFilter_(TexFilterMin minFilter)
        {
            switch (minFilter)
            {
                case TexFilterMin.Near: return TextureFilter.Point;
                case TexFilterMin.Linear: return TextureFilter.Linear;
                default: Debug.Assert(false); return TextureFilter.Point;
            }
        }

        /// <summary>
        ///
        /// </summary>
        static TextureFilter TexFilterMagTextureFilter_(TexFilterMag texWrap)
        {
            switch (texWrap)
            {
                case TexFilterMag.Near: return TextureFilter.Point;
                case TexFilterMag.Linear: return TextureFilter.Linear;
                default: Debug.Assert(false); return TextureFilter.Point;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void InitializeTextureSamplingState()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void SetTextureSamplingState(
            int idx,
            TexWrap wrapS,
            TexWrap wrapT,
            TexFilterMin minFilter,
            TexFilterMag magFilter,
            bool isSrgb)
        {
            Debug.Assert(idx >= 0 && idx <= 8);

            TextureAddress textureAddressU = TexWrapToTextureAddress_(wrapS);
            TextureAddress textureAddressV = TexWrapToTextureAddress_(wrapT);
            TextureFilter minTexFilter = TexFilterMinTextureFilter_(minFilter);
            TextureFilter magTexFilter = TexFilterMagTextureFilter_(magFilter);

            _stateChangerSet.Add(new SamplerStateData(idx, textureAddressU, textureAddressV, minTexFilter, magTexFilter, this.LinearGammmaSRGBFetchEnabled && isSrgb));
        }

        public void SetTextureSamplingSrgbState(int idx, bool isSrgb)
        {
            _stateChangerSet.Add(new SamplerStateSrgbData(idx, this.LinearGammmaSRGBFetchEnabled && isSrgb));
        }

        /// <summary>
        /// 白黒カラー補間色を取得します。
        /// </summary>
        public void GetMatrialColorBlend(out System.Drawing.Color whiteBlendColor, out System.Drawing.Color blackBlendColor)
        {
            whiteBlendColor = _whiteBlendColor.ToSystemColor();
            blackBlendColor = _blackBlendColor.ToSystemColor();
        }

        /// <summary>
        /// 白黒カラー補間色を設定します。
        /// </summary>
        public void SetMatrialColorBlend(FloatColor whiteBlendColor, FloatColor blackBlendColor, bool isThresholdingAlphaInterpolationEnabled)
        {
            _whiteBlendColor = whiteBlendColor;
            _blackBlendColor = blackBlendColor;
            _isThresholdingAlphaInterpolationEnabled = isThresholdingAlphaInterpolationEnabled;
        }

        //---------------------------------------------------
#region 行列操作関連
        /// <summary>
        ///
        /// </summary>
        public void PushMtx()
        {
            _mtxStack.Push();
            _mtxViewStack.Push();
        }

        /// <summary>
        ///
        /// </summary>
        public void PopMtx()
        {
            _mtxStack.Pop();
            _mtxViewStack.Pop();
        }

        /// <summary>
        ///
        /// </summary>
        public void IdentityMtx()
        {
            _mtxStack.LoadIdentity();
            _mtxViewStack.LoadIdentity();

            _mtxViewStack.LoadMatrix(DXMatrix.LookAtRH(new Vector3(0.0f, 0.0f, _cameraDistZ), new Vector3(0.0f, 0.0f, _cameraDistZ - 5.0f), new Vector3(0.0f, 1.0f, 0.0f)));
        }

        /// <summary>
        ///
        /// </summary>
        public void Trans(int x, int y)
        {
            _mtxStack.TranslateLocal((float)x, (float)y, 0.0f);
        }

        /// <summary>
        ///
        /// </summary>
        public void Trans(float x, float y)
        {
            _mtxStack.TranslateLocal(x, y, 0.0f);
        }

        /// <summary>
        ///
        /// </summary>
        public void Trans(int x, int y, int z)
        {
            _mtxStack.TranslateLocal((float)x, (float)y, (float)z);
        }

        /// <summary>
        ///
        /// </summary>
        public void Trans(float x, float y, float z)
        {
            _mtxStack.TranslateLocal(x, y, z);
        }

        /// <summary>
        ///
        /// </summary>
        public void Scale(float sx, float sy)
        {
            _mtxStack.ScaleLocal(sx, sy, 1.0f);
        }

        /// <summary>
        ///
        /// </summary>
        public void Scale(float sx, float sy, float sz)
        {
            _mtxStack.ScaleLocal(sx, sy, sz);
        }

        /// <summary>
        /// ビュー行列を設定します。
        /// </summary>
        public void SetViewTransform(float sx, float sy, float tx, float ty, float tz, float pz)
        {
            _cameraDistZ = pz;

            _mtxViewStack.LoadIdentity();
            _mtxViewStack.LoadMatrix(DXMatrix.LookAtRH(new Vector3(tx, ty, pz), new Vector3(tx, ty, tz), new Vector3(0.0f, 1.0f, 0.0f)));
            _mtxViewStack.Scale(sx, sy, 1.0f);
        }

        /// <summary>
        /// 基準スクリーンサイズ（≒レイアウトサイズ）を設定します。
        /// </summary>
        public void SetDefaultScreenSize(float width, float height)
        {
            _defaultScreenSizeW = width;
            _defaultScreenSizeH = height;
        }

        /// <summary>
        ///
        /// </summary>
        static SysMatrix DxMtx2SysMtx(DXMatrix dxmtx)
        {
            return new SysMatrix(dxmtx.M11, dxmtx.M12, dxmtx.M21, dxmtx.M22, dxmtx.M41, dxmtx.M42);
        }

        /// <summary>
        ///
        /// </summary>
        static DXMatrix SysMtx2DxMtx(SysMatrix sysmtx)
        {
            DXMatrix mxt = DXMatrix.Identity;

            mxt.M11 = sysmtx.Elements[0];
            mxt.M12 = sysmtx.Elements[1];
            mxt.M21 = sysmtx.Elements[2];
            mxt.M22 = sysmtx.Elements[3];
            mxt.M41 = sysmtx.Elements[4];
            mxt.M42 = sysmtx.Elements[5];

            return mxt;
        }

        /// <summary>
        /// カレント行列を設定、取得します。
        /// </summary>
        /// <param name="mtx"></param>
        public Matrix34 CurrentMtx34
        {
            get
            {

                var m = new Matrix34();
                m[0, 0] = _mtxStack.Top.M11;
                m[0, 1] = _mtxStack.Top.M21;
                m[0, 2] = _mtxStack.Top.M31;
                m[0, 3] = _mtxStack.Top.M41;

                m[1, 0] = _mtxStack.Top.M12;
                m[1, 1] = _mtxStack.Top.M22;
                m[1, 2] = _mtxStack.Top.M32;
                m[1, 3] = _mtxStack.Top.M42;

                m[2, 0] = _mtxStack.Top.M13;
                m[2, 1] = _mtxStack.Top.M23;
                m[2, 2] = _mtxStack.Top.M33;
                m[2, 3] = _mtxStack.Top.M43;

                return m;
            }
            set
            {
                var m = new DXMatrix();
                m.M11 = value[0, 0];
                m.M21 = value[0, 1];
                m.M31 = value[0, 2];
                m.M41 = value[0, 3];

                m.M12 = value[1, 0];
                m.M22 = value[1, 1];
                m.M32 = value[1, 2];
                m.M42 = value[1, 3];

                m.M13 = value[2, 0];
                m.M23 = value[2, 1];
                m.M33 = value[2, 2];
                m.M43 = value[2, 3];

                m.M14 = 0;
                m.M24 = 0;
                m.M34 = 0;
                m.M44 = 1;

                _mtxStack.LoadMatrix(m);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public SysMatrix CurrentMtx
        {
            get
            {
                return DxMtx2SysMtx(_mtxStack.Top);
            }
            set
            {
                _mtxStack.LoadMatrix(SysMtx2DxMtx(value));
            }
        }
#endregion

        /// <summary>
        ///
        /// </summary>
        public Color Color
        {
            get
            {
                return Color.FromArgb(
                    (int)(_material.Diffuse.Alpha * 255.0f * PaneHelper.GetNormalizedTransparency(Transparency)),
                    (int)(_material.Diffuse.Red * 255.0f), (int)(_material.Diffuse.Green * 255.0f), (int)(_material.Diffuse.Blue * 255.0f));
            }
            set
            {
                _material.Diffuse = new SharpDX.Color4(new SharpDX.Color4(value.ToArgb()).ToBgra());
                _material.Ambient = new SharpDX.Color4(new SharpDX.Color4(value.ToArgb()).ToBgra());
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float LineWidth
        {
            get
            {
                return _lineWidth;
            }
            set
            {
                if (_lineWidth != value)
                {
                    _lineWidth = value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public DashStyle LineDashStyle
        {
            get
            {
                // throw new Exception( "The method or operation is not implemented." );
                return DashStyle.Dash;
            }
            set
            {
                // throw new Exception( "The method or operation is not implemented." );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int Transparency
        {
            get { return _transparency; }
            set
            {
                Debug.Assert(value >= 0 && value <= PaneHelper.MaxTransparency);
                _transparency = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float RectangleTopXOffset
        {
            get;
            set;
        }

        public float FontHeight
        {
            get;
            set;
        }

        public SizeF ViewportSize
        {
            get { return _d3dDevice != null ? new SizeF(_d3dDevice.Viewport.Width, _d3dDevice.Viewport.Height) : SizeF.Empty; }
        }

        public bool IsDeviceLost
        {
            get { return _deviceLost; }
        }
#endregion

#region IDisposable メンバ

        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            if (_targetCtrl != null)
            {
                _targetCtrl.SizeChanged -= new EventHandler(_targetCtrl_SizeChanged);
                _targetCtrl = null;
            }

            // デバイスを破棄します。
            if (_d3dDevice != null)
            {
                OnDeviceLostRelease_();

                foreach (var detailedCombiner in _builtInDetailedCombinerShaders)
                {
                    if (detailedCombiner.shaderBytecode != null)
                    {
                        detailedCombiner.shaderBytecode.Dispose();
                    }
                    if (detailedCombiner.pixelShader != null)
                    {
                        detailedCombiner.pixelShader.Dispose();
                    }
                    if (detailedCombiner.constantTable != null)
                    {
                        detailedCombiner.constantTable.Dispose();
                    }
                }
                _builtInDetailedCombinerShaders = null;

                if (_builtInVertexShader != null) { _builtInVertexShader.Dispose(); _builtInVertexShader = null; }
                if (_builtInVertexShaderConstantTable != null) { _builtInVertexShaderConstantTable.Dispose(); _builtInVertexShaderConstantTable = null; }
                if (_shaderBytecodeVS != null) { _shaderBytecodeVS.Dispose(); _shaderBytecodeVS = null; }

                if (_builtInPixelShader != null) { _builtInPixelShader.Dispose(); _builtInPixelShader = null; }
                if (_builtInNoTexPixelShader != null) { _builtInNoTexPixelShader.Dispose(); _builtInNoTexPixelShader = null; }
                if (_builtInNoTexBlendPixelShader != null) { _builtInNoTexBlendPixelShader.Dispose(); _builtInNoTexBlendPixelShader = null; }
                if (_builtIn1TexPixelShader != null) { _builtIn1TexPixelShader.Dispose(); _builtIn1TexPixelShader = null; }
                if (_builtIn1TexBlendPixelShader != null) { _builtIn1TexBlendPixelShader.Dispose(); _builtIn1TexBlendPixelShader = null; }
                if (_builtInPackedFontPixelShader != null) { _builtInPackedFontPixelShader.Dispose(); _builtInPackedFontPixelShader = null; }

                _indexBuffer.Dispose();
                _vertexBufferRect.Dispose();

                foreach (var item in _bitmapTextureMap)
                {
                    item.Value.Dispose();
                }

                _bitmapTextureMap.Clear();

                _d3dDevice.Dispose();
                _d3dDevice = null;
            }

            if (_direct3D != null)
            {
                _direct3D.Dispose();
                _direct3D = null;
            }
        }

#endregion

        /// <summary>
        /// ビットマップに対応するテクスチャを削除する
        /// </summary>
        public void ClearBitmapTextureMap()
        {
            foreach (var item in _bitmapTextureMap.ToArray())
            {
                lock (item.Key)
                {
                    item.Value.Dispose();
                    _bitmapTextureMap.Remove(item.Key);
                }
            }
        }


        /// <summary>
        /// キャプチャ
        /// </summary>
        public Bitmap Capture(DrawerForCapture drawer, System.Drawing.Rectangle bounds, bool outputAlpha)
        {
            Bitmap bitmap = null;
            Action getCaptureImage = () =>
            {
                var res = _d3dDevice.TestCooperativeLevel();
                if (res.Success)
                {
                    try
                    {
                        using (Surface backBuffer = _d3dDevice.GetBackBuffer(0, 0))
                        {
                            using (SharpDX.DataStream stream = Surface.ToStream(
                                backBuffer, outputAlpha ? ImageFileFormat.Png: ImageFileFormat.Bmp, new SharpDX.Rectangle(bounds.X, bounds.Y, bounds.Width, bounds.Height)))
                            {
                                bitmap = new Bitmap(stream);

                                if (LinearGammmaSRGBWriteEnabled)
                                {
                                    // 近似計算で sRGB に変換する
                                    using (var g = Graphics.FromImage(bitmap))
                                    {
                                        g.CompositingMode = CompositingMode.SourceCopy;
                                        using (var ia = new ImageAttributes())
                                        {
                                            ia.SetGamma(1.0f / 2.2f);
                                            g.DrawImage(bitmap,
                                                new Rectangle(0, 0, bitmap.Width, bitmap.Height),
                                                0, 0, bitmap.Width, bitmap.Height, GraphicsUnit.Pixel, ia);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    catch
                    {
                        bitmap = new Bitmap(16, 16);
                    }
                }
            };

            int currentW = _presentParams.BackBufferWidth;
            int currentH = _presentParams.BackBufferHeight;

            try
            {
                this.BeginRendering(null);
                this.IdentityMtx();
                this.SetViewTransform(1.0f, 1.0f, 0, 0, -5.0f, 0.0f);
                this.Scale(1.0f, 1.0f);

                this.PushMtx();

                // キャプチャする画像を描画します。
                drawer(null);

                this.PopMtx();
            }
            finally
            {
                this.EndRendering(getCaptureImage);
            }

            return bitmap;
        }
    }
}
