﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Text;
using System.Net;
using System.Runtime.InteropServices;

/*

    ==========================================================================
    ※このプロトコルヘッダは、
    $(NW4R_ROOT)/ToolUtil/ProtocolUtil/Protocol/ProtocolSample/ProtocolSample.xml
    から自動生成されて作られたものです。
    プロトコルを編集する際には、.h及び.csファイルを直接編集するのではなく
    ProtocolSample.xmlファイルを編集し再コンバートする必要があります。
    ==========================================================================

    Viewer - ProtocolSampleApplication通信プロトコル

    通信メッセージの種類
        ビューアメッセージ  ビューアからエディタに送るメッセージ
        エディタメッセージ  エディタからビューアに送るメッセージ
        エラーメッセージ    エラーが起こった際に送るメッセージ

    通信の際には必ずMessageHeaderを送信し、その次にMessageDataを送信します。
    MessageHeaderは2byteのMessageTypeと2byteのMessageSizeからなります。

    +---------------------------+--------------------+
    |      MessageHeader        |    MessageData     |
    +-------------+-------------+                    |
    | MessageType | MessageSize | (MessageSize byte) |
    | (2byte)     | (2byte)     |                    |
    +-------------+-------------+--------------------+

    MessageTypeは1bitのErrorFlag、1bitのReplyFlag、6bitのCategory
    8bitのCommandからなります。

    15             8 7             0
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    |E|R| Category  |    Command    |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    Errorがtrueの時にはReplyは必ずfalseになります。
 */
namespace NW4R.ProtocolLayout{

    //-----------------------------------------------------------------------
    /** <summary>Constants</summary> */
    public class Constants{

        //-------------------------------------------------------------------
        /*
            バージョン情報の定義
            バージョン番号の運用は各モジュールとツールでの運用ルールにあわせてください
         */
        /** <summary>プロトコルバージョン</summary> */
        public const uint	PROTOCOL_VERSION =	0x00010000;

        //-------------------------------------------------------------------
        /* メッセージのフラグとマスクの定義 */
        /** <summary>エラー用フラグ  </summary> */
        public const ushort	HEADER_ERROR_FLAG =	0x8000;
        /** <summary>返信用フラグ    </summary> */
        public const ushort	HEADER_REPLY_FLAG =	0x4000;
        /** <summary>カテゴリ用マスク</summary> */
        public const ushort	HEADER_CATEGORY_MASK =	0x3F00;
        /** <summary>コマンド用マスク</summary> */
        public const ushort	HEADER_COMMAND_MASK =	0x00FF;

        //-------------------------------------------------------------------
        /* カテゴリの定義 */
        /** <summary>システム関連カテゴリ       </summary> */
        public const ushort	CATEGORY_SYSTEM =	0x0000;
        /** <summary>レイアウト関連カテゴリ     </summary> */
        public const ushort	CATEGORY_LAYOUT =	0x0100;
        /** <summary>MCS通信テストカテゴリ      </summary> */
        public const ushort	CATEGORY_TEST =	0x3F00;

        //-------------------------------------------------------------------
        /* システム関連カテゴリのメッセージ */
        /** <summary>バージョン要求コマンド       </summary> */
        public const ushort	TOOL_SYSTEM_VERSION =	0x00 + CATEGORY_SYSTEM + HEADER_REPLY_FLAG;
        /** <summary>同期用コマンド               </summary> */
        public const ushort	TOOL_SYSTEM_SYNC =	0x01 + CATEGORY_SYSTEM + HEADER_REPLY_FLAG;
        /** <summary>モジュールリセットコマンド   </summary> */
        public const ushort	TOOL_SYSTEM_RESET =	0x02 + CATEGORY_SYSTEM;

        //-------------------------------------------------------------------
        /* レイアウト関連カテゴリのメッセージ */
        /** <summary>レイアウトデータの読み込みコマンド           </summary> */
        public const ushort	TOOL_LAYOUT_LOAD_LAYOUT =	0x00 + CATEGORY_LAYOUT + HEADER_REPLY_FLAG;

        //-------------------------------------------------------------------
        /* テストカテゴリのメッセージ */
        /** <summary>EFBカラー変更コマンド           </summary> */
        public const ushort	TOOL_TEST_EFB_COLOR =	0x00 + CATEGORY_TEST;

        //-------------------------------------------------------------------
        /* ビューア主導 システム関連カテゴリのメッセージ */
        /** <summary>ビューア主導のテストコマンド</summary> */
        public const ushort	VIEWER_TEST_NOP =	0x00 + CATEGORY_TEST;

        //-------------------------------------------------------------------
        /* システムエラーカテゴリのメッセージ */
        /** <summary>未定義コマンドエラー</summary> */
        public const ushort	ERR_SYSTEM_UNKNOWN_COMMAND =	0x0 + CATEGORY_SYSTEM + HEADER_ERROR_FLAG;

        //-------------------------------------------------------------------
        /* データエラーカテゴリのメッセージ */
        /** <summary>データロードエラー        </summary> */
        public const ushort	ERR_LAYOUT_LOAD =	0x0 + CATEGORY_LAYOUT + HEADER_ERROR_FLAG;

    }

    //-----------------------------------------------------------------------
    /** <summary>Sampleプロトコルのメッセージ共通ヘッダ</summary> */
    public class Header {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+2+2;
        /** <summary>メッセージタイプ   </summary> */
        public ushort	type;
        /** <summary>メッセージサイズ   </summary> */
        public ushort	size;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            type = reader.ReadUInt16();
            size = reader.ReadUInt16();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(type);
            writer.Write(size);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
         バージョン要求応答メッセージ
            ツールのプロトコルバージョンを受け取り、ビューアのプロトコルバージョンを返します。
        </summary> */
    public class ToolSystemVersion {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: バージョン番号</summary> */
        public uint	version;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            version = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(version);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
        同期メッセージ

              Viewerは、この同期コマンドを受け取ると、指定されたtagName
              名をエコーした応答コマンドをアプリケーションに返信します。

              tagNameは､ツール側が自由に設定できます。
        </summary> */
    public class ToolSystemSync {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: タグ名</summary> */
        public uint	tagName;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            tagName = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(tagName);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            レイアウトデータの読み込みメッセージ
        </summary> */
    public class ToolLayoutLoadLayout {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+1*260+2+2+1*260;
        /** <summary>260B: PC上のデータのパス名 </summary> */
        public byte[]	pathName = new byte[260];
        /** <summary>  2B: ファイル形態</summary> */
        public ushort	fileType;
        /** <summary>  2B: </summary> */
        public ushort	padding;
        /** <summary>260B: ロードするレイアウトファイル名</summary> */
        public byte[]	layoutName = new byte[260];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            for(int i = 0; i < 260; i++){
                pathName[i] = reader.ReadByte();
            }
            fileType = reader.ReadUInt16();
            padding = reader.ReadUInt16();
            for(int i = 0; i < 260; i++){
                layoutName[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            for(int i = 0; i < 260; i++){
                writer.Write(pathName[i]);
            }
            writer.Write(fileType);
            writer.Write(padding);
            for(int i = 0; i < 260; i++){
                writer.Write(layoutName[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルデータの読み込み応答メッセージ
        </summary> */
    public class ToolLayoutLoadLayoutReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>  4B: ロードされたモデルの管理ID  </summary> */
        public uint	id;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            id = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(id);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            ロードされているモデルリスト応答メッセージ
            この後にsize個のモデルIDが続く可変長の応答メッセージです。
        </summary> */
    public class Tool3DGetModelListReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>                4B: Viewerにロードされているモデルの数           </summary> */
        public uint	modelCount;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            modelCount = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(modelCount);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルリストの要素構造体
        </summary> */
    public class Tool3DGetModelListReplyElement {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4;
        /** <summary>          4B: モデルの管理ID              </summary> */
        public uint	id;
        /** <summary>          4B: モデル名の文字列長(NUL込み) </summary> */
        public uint	nameSize;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            id = reader.ReadUInt32();
            nameSize = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(id);
            writer.Write(nameSize);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルのノードリスト取得メッセージ
        </summary> */
    public class Tool3DGetNodeList {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>  4B: モデルの管理ID           </summary> */
        public uint	id;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            id = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(id);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルのノードリスト応答メッセージ
            モデルが存在しない場合にはnameLengthが0で返る
        </summary> */
    public class Tool3DGetNodeListReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>               4B: モデル内のノード数    </summary> */
        public int	nodeCount;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            nodeCount = reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(nodeCount);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            ノードリストの要素構造体
        </summary> */
    public class Tool3DGetNodeListReplyElement {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4;
        /** <summary>          4B: ノードのインデックス                                 </summary> */
        public int	idx;
        /** <summary>          4B: 親ノードのインデックス(ルートノードの場合には負の値) </summary> */
        public int	parentIdx;
        /** <summary>          4B: ノード名の文字列長(NUL込み)                          </summary> */
        public uint	nameSize;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            idx = reader.ReadInt32();
            parentIdx = reader.ReadInt32();
            nameSize = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(idx);
            writer.Write(parentIdx);
            writer.Write(nameSize);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
         EFBカラー変更メッセージ
        </summary> */
    public class ToolTestEfbColor {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+1+1+1+1;
        /** <summary>1B: R成分</summary> */
        public byte	r;
        /** <summary>1B: G成分</summary> */
        public byte	g;
        /** <summary>1B: B成分</summary> */
        public byte	b;
        /** <summary>1B: A成分</summary> */
        public byte	a;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            r = reader.ReadByte();
            g = reader.ReadByte();
            b = reader.ReadByte();
            a = reader.ReadByte();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(r);
            writer.Write(g);
            writer.Write(b);
            writer.Write(a);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            未定義コマンドエラーメッセージ

            受け取った未定義コマンドをcommandパラメータで返します。
        </summary> */
    public class ErrSystemUnknownCommand {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+2;
        /** <summary>  2B: 受け取った未定義コマンド  </summary> */
        public ushort	command;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            command = reader.ReadUInt16();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write(command);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>データの読み込みエラー種別</summary> */
    public enum ErrDataLoadCode {
        /** <summary>ファイルのロードに失敗しました。        </summary> */
        ERR_DATA_LOAD_FILE_NOT_FOUND =	0,
        /** <summary>メモリのアロケーションに失敗しました。  </summary> */
        ERR_DATA_LOAD_MEMORY_ALLOCATE =	1,
        /** <summary>有効なデータではありませんでした。      </summary> */
        ERR_DATA_LOAD_INVALID_DATA =	2,
        /** <summary>MCS通信で異常が発生しました。           </summary> */
        ERR_DATA_LOAD_MCS_STREAM_FAIL =	3,
    }

    //-----------------------------------------------------------------------
    /** <summary>
            データの読み込みエラー
        </summary> */
    public class ErrDataLoad {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>  4B: エラーコード          </summary> */
        public ErrDataLoadCode	code;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolLayoutReader reader){
            code = (ErrDataLoadCode)reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolLayoutWriter writer){
            writer.Write((int)code);
        }

    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolLayoutReader
    /// short,ushort,int,uint,floatのエンディアンを変換し、読み込みます
    /// </summary>
    public class ProtocolLayoutReader : BinaryReader{

        /// <summary>コンストラクタ</summary>
        public ProtocolLayoutReader(Stream input) : base(input){}
        /// <summary>コンストラクタ</summary>
        public ProtocolLayoutReader(Stream input, Encoding encoding) :
            base(input, encoding){}

        ///	<summary>short読み込み</summary>
        public override short ReadInt16(){
            return IPAddress.NetworkToHostOrder(base.ReadInt16());
        }

        ///	<summary>ushort読み込み</summary>
        public override ushort ReadUInt16(){
            short result = (short)base.ReadUInt16();
            return (ushort)IPAddress.NetworkToHostOrder(result);
        }

        ///	<summary>int読み込み</summary>
        public override int ReadInt32(){
            return IPAddress.NetworkToHostOrder(base.ReadInt32());
        }

        ///	<summary>uint読み込み</summary>
        public override uint ReadUInt32(){
            int result = (int)base.ReadUInt32();
            return (uint)IPAddress.NetworkToHostOrder(result);
        }

        ///	<summary>float読み込み</summary>
        public override float ReadSingle(){
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(base.ReadSingle());
            Array.Reverse(floatBytes);
            return BitConverter.ToSingle(floatBytes, 0);
        }

    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolLayoutWriter
    /// short,ushort,int,uint,floatのエンディアンを変換し、書き込みます
    /// </summary>
    public class ProtocolLayoutWriter : BinaryWriter{

        /// <summary>コンストラクタ</summary>
        public ProtocolLayoutWriter() : base(){}
        /// <summary>コンストラクタ</summary>
        public ProtocolLayoutWriter(Stream input) : base(input){}
        /// <summary>コンストラクタ</summary>
        public ProtocolLayoutWriter(Stream input, Encoding encoding) :
            base(input, encoding){}

        ///	<summary>short書き出し</summary>
        public override void Write(short value){
            base.Write(IPAddress.HostToNetworkOrder(value));
        }

        ///	<summary>ushort書き出し</summary>
        public override void Write(ushort value){
            base.Write((ushort)IPAddress.HostToNetworkOrder((short)value));
        }

        ///	<summary>int書き出し</summary>
        public override void Write(int value){
            base.Write(IPAddress.HostToNetworkOrder(value));
        }

        ///	<summary>uint書き出し</summary>
        public override void Write(uint value){
            base.Write((uint)IPAddress.HostToNetworkOrder((int)value));
        }

        ///	<summary>float書き出し</summary>
        public override void Write(float value){
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(value);
            Array.Reverse(floatBytes);
            base.Write(BitConverter.ToSingle(floatBytes, 0));
        }

    }

}
