﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Drawing;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.common
{
    /// <summary>
    /// 最小化、最大化状態のウインドウの通常の位置、サイズを
    /// NativeAPIコールを利用して取得する。
    ///
    /// http://www.thescripts.com/forum/thread435683.html を参考に実装
    ///
    /// </summary>
    static public class WindowPlacementHelper
    {
        #region 構造体定義

        /// <summary>
        /// The WINDOWPLACEMENT structure for the Win32 interface.
        /// See Microsoft documentation for details.
        /// </summary>
        struct Win32WindowPlacement
        {
            public UInt32 length;
            public UInt32 flags;
            public UInt32 showCmd;
            public Win32Point minPosition;
            public Win32Point maxPosition;
            public Win32Rect normalPosition;

            public Win32WindowPlacement( UInt32 length, UInt32 flags,
            UInt32 showCmd, Win32Point minPosition,
            Win32Point maxPosition, Win32Rect normalPosition )
            {
                this.length = length;
                this.flags = flags;
                this.showCmd = showCmd;
                this.minPosition = minPosition;
                this.maxPosition = maxPosition;
                this.normalPosition = normalPosition;
            }

            /// <summary>
            /// ウインドウサイズ。
            /// </summary>
            public Size WindowSize
            {
                get { return new Size( this.normalPosition.Width, this.normalPosition.Height ); }
            }

            /// <summary>
            /// ウインドウ位置。
            /// </summary>
            public Point WindowPosition
            {
                get { return new Point( this.normalPosition.left, this.normalPosition.top ); }
            }
        }

        /// <summary>
        /// The POINT structure for the Win32 interface.
        /// See Microsoft documentation for details.
        /// </summary>
        struct Win32Point
        {
            public Int32 x;
            public Int32 y;
            public Win32Point( Int32 x, Int32 y )
            {
                this.x = x;
                this.y = y;
            }
        }

        /// <summary>
        /// The RECT structure for the Win32 interface.
        /// See Microsoft documentation for details.
        /// </summary>
        struct Win32Rect
        {
            public Int32 left;
            public Int32 top;
            public Int32 right;
            public Int32 bottom;


            public Win32Rect( Int32 top, Int32 left, Int32 bottom, Int32 right )
            {
                this.top = top;
                this.left = left;
                this.bottom = bottom;
                this.right = right;
            }

            public Int32 Width
            {
                get { return this.right - this.left; }
            }

            public Int32 Height
            {
                get { return this.bottom - this.top; }
            }
        }

        #endregion // 構造体定義

        /// <summary>
        /// Native DLL コール
        /// </summary>
        [DllImport( "user32.dll", SetLastError = true )]
        static extern bool GetWindowPlacement(
            IntPtr hDc,
            out Win32WindowPlacement placementInfo );

        /// <summary>
        /// フォームの通常の位置、サイズを取得します。
        /// フォームが最小化、最大化されているときに利用します。
        /// </summary>
        static public void GetWindowNormalPosition( Form form, out Point position, out Size size )
        {
            Win32WindowPlacement placement;
            if( GetWindowPlacement( form.Handle, out placement ) )
            {
                position = placement.WindowPosition;
                size = placement.WindowSize;
            }
            else
            {
                position = form.Location;
                size = form.Size;
            }
        }
    }
}
