﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.common
{
    using LECore.Structures;
    using ContentNodeDict = Dictionary<object, TreeNodeState>;


    /// <summary>
    /// TreeNode の 状態変更処理を定義するクラスです。
    /// </summary>
    internal interface ITreeNodeStateCheckPolicy
    {
        /// <summary>
        /// 更新されているか確認します。
        /// </summary>
        bool CheckChanged( object current, object old );
        /// <summary>
        /// 状態を管理する複製を生成します。
        /// </summary>
        void MakeClone( object current, out object clone );
        /// <summary>
        /// 複製の状態を最新に更新します。
        /// </summary>
        void UpdateClone( object current, object clone );
    }


    /// <summary>
    /// TreeNode の状態を管理するクラス。
    /// </summary>
    class TreeNodeState
    {
       #region フィールド
        // ツリーノードが最後に更新されたときの状態
        object _targetClone;
        object _target;
        readonly List<TreeNode>           _treeNodeSet = new List<TreeNode>();
        readonly ITreeNodeStateCheckPolicy _nodeStateCheckPolicy;
       #endregion



       #region private メソッド
        /// <summary>
        /// ノード状態のチェック機能が有効か ?
        /// _nodeStateCheckPolicy が target を正しく処理できる場合には true になるはずです。
        /// </summary>
        bool NodeStateCheckEnabled_
        {
            get { return _nodeStateCheckPolicy != null && _targetClone != null; }
        }
       #endregion

        /// <summary>
        /// ツリーノード
        /// </summary>
        public List<TreeNode> TreeNodeSet
        {
            get { return _treeNodeSet; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TreeNodeState( ITreeNodeStateCheckPolicy nodeStateCheckPolicy, TreeNode node )
        {
            _nodeStateCheckPolicy = nodeStateCheckPolicy;

            object target = node.Tag;
            object clone;

            _nodeStateCheckPolicy.MakeClone( target, out clone );

            _targetClone = clone;
            _target = target;

            AddTreeNode( node );
        }

        /// <summary>
        /// 変更されているか確認します。
        /// </summary>
        public bool CheckChanged( object currentTarget )
        {
            // 判定ポリシークラスに委譲します。
            if( NodeStateCheckEnabled_ )
            {
                return _nodeStateCheckPolicy.CheckChanged( currentTarget, _targetClone );
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// 状態を更新します。
        /// </summary>
        public void Update()
        {
            if( NodeStateCheckEnabled_ )
            {
                _nodeStateCheckPolicy.UpdateClone( _target, _targetClone );
            }
        }

        /// <summary>
        /// TreeNodeを登録します。
        /// </summary>
        public void AddTreeNode( TreeNode node )
        {
            Debug.Assert( node != null );

            // 以下の制約は設けない。
            // 同一キーを持つ nodeが異なる、内容を持っていてもそれは、正しい。
            //
            // 例：IAnmCurve を キーとして、node.Tag に
            //      (IAnmCurve)共有参照アトリビュートを持つ場合。
            // Debug.Assert( node.Tag == _target );
            _treeNodeSet.Add( node );
        }
    }

    /// <summary>
    /// ツリーノードを生成するクラス。
    ///
    /// ツリーノードの状態を管理し、
    /// TreeNode.Tagの参照から、TreeNodeを検索する機能を提供します。
    /// </summary>
    internal class TreeNodeFactory
    {
        /// <summary>
        /// nodeContent から 対応するノードの状態を検索するためのマップ
        /// </summary>
        readonly ContentNodeDict _nodeStateLookup = new ContentNodeDict();
        static readonly List<TreeNode>   _EmptyList = new List<TreeNode>();
        readonly ITreeNodeStateCheckPolicy _targetStateCheckPolicy;


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TreeNodeFactory( ITreeNodeStateCheckPolicy targetStateCheckPolicy )
        {
            _targetStateCheckPolicy = targetStateCheckPolicy;
        }


        /// <summary>
        /// 新しいノードを取得します。
        /// </summary>
        public TreeNode MakeNewNode( object nodeContent )
        {
            return MakeNewNode( nodeContent, nodeContent );
        }


        /// <summary>
        /// 新しいノードを取得します。
        /// キーオブジェクトを別に指定します。
        /// </summary>
        public TreeNode MakeNewNode( object nodeContent, object keyObject )
        {
            // ノードを生成します。
            TreeNode node = new TreeNode();
            node.Tag = nodeContent;

            // コンテンツに対応するエントリが存在すれば...
            TreeNodeState nodeState = null;
            if( _nodeStateLookup.TryGetValue( keyObject, out nodeState ) )
            {
                // ノードを追加
                nodeState.AddTreeNode( node );
            }
            else
            {
                // テーブルに新しいエントリを追加します。
                nodeState = new TreeNodeState( _targetStateCheckPolicy, node );
                _nodeStateLookup.Add( keyObject, nodeState );
            }
            return node;
        }

        /// <summary>
        /// ノードコンテンツから、対応する更新されたノードを取得します。
        /// 取得時に、更新状態がリセットされます。
        /// </summary>
        public List<TreeNode> FindUpdatedNodeByContents(object nodeContent)
        {
            return FindUpdatedNodeByContents(nodeContent, true);
        }

        /// <summary>
        /// ノードコンテンツから、対応する更新されたノードを取得します。
        /// 取得時に、更新状態がリセットされます。
        /// </summary>
        public List<TreeNode> FindUpdatedNodeByContents( object nodeContent, bool collectChangedOnly )
        {
            TreeNodeState nodeState = null;
            if( _nodeStateLookup.TryGetValue( nodeContent, out nodeState ) )
            {
                if (!collectChangedOnly || nodeState.CheckChanged(nodeContent))
                {
                    nodeState.Update();
                    return nodeState.TreeNodeSet;
                }
            }
            return _EmptyList;
        }

        /// <summary>
        ///
        /// </summary>
        public void Clear()
        {
            _nodeStateLookup.Clear();
        }
    }
}
