﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace LayoutEditor.Forms.ToolWindows.common
{
    using LECore.Structures;

    /// <summary>
    /// サブシーンごとのUI情報を記憶するクラス。
    /// 利用者は、派生クラスで abstract メンバを実装する必要があります。
    /// </summary>
    abstract class SubsceneUIStateCache
    {
        //---------------------------------------------
        // フィールド　
        //---------------------------------------------
        #region

        /// <summary>
        /// 対象サブシーン
        /// </summary>
        ISubScene _currentSubScene = null;
        /// <summary>
        /// サブシーンごとのUIデータマップ
        /// </summary>
        Dictionary<ISubScene, object> _uiStateMap = new Dictionary<ISubScene, object>();

        #endregion

        //---------------------------------------------
        // abstract
        //---------------------------------------------
        /// <summary>
        /// サブシーンごとにUIが保存すべきデータを取得します。
        /// SubsceneUIStateCache の派生クラスで、利用者が任意の処理を実装してください。
        /// 関数の返り値が、_uiStateMapに保存されます。
        /// </summary>
        protected abstract object GetSubSceneUIState_( ISubScene oldCurrentSubScene );
        /// <summary>
        /// SubsceneUIStateCacheが保存しているデータを新しいUIに設定します。
        /// </summary>
        protected abstract void SetSubSceneUIState_( ISubScene newCurrentSubScene, object uiState );


        //---------------------------------------------
        // private
        //---------------------------------------------
        #region

        /// <summary>
        /// UIの状態を保存します。
        /// </summary>
        void StoreOldUIState_( ISubScene oldCurrentSubScene )
        {
            object uiState = GetSubSceneUIState_( oldCurrentSubScene );
            if( uiState != null )
            {
                _uiStateMap[oldCurrentSubScene] = uiState;
            }
        }

        /// <summary>
        /// 新しいUIの状態を復元します。
        /// </summary>
        void RestoreNewUIState_( ISubScene newCurrentSubScene )
        {
            object uiState;
            if( _uiStateMap.TryGetValue( newCurrentSubScene, out uiState ) )
            {
                SetSubSceneUIState_( newCurrentSubScene, uiState );
            }
        }

        #endregion
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SubsceneUIStateCache()
        {
            // 何もしません。
        }

        //---------------------------------------------
        // public
        //---------------------------------------------
        /// <summary>
        /// 対象サブシーンを変更します。
        /// </summary>
        public void UpdateSubsceneUIState( ISubScene newCurrentSubScene )
        {
            // 状態を保存します。
            if( _currentSubScene != null )
            {
                StoreOldUIState_( _currentSubScene );
            }

            // 対象サブシーンを変更します。
            _currentSubScene = newCurrentSubScene;

            // 状態が保存されていれば復元します。
            if( newCurrentSubScene != null )
            {
                RestoreNewUIState_( newCurrentSubScene );
            }
        }
    }
}
