﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;

using LECore.Structures.Core;

namespace LayoutEditor.Forms.ToolWindows
{
    /// <summary>
    /// マウスドラック操作時に表示される、範囲選択矩形です。
    /// </summary>
    public class SelectRect : IDrawable
    {
        #region --------------- フィールド ---------------
        // スクリーン座標系での矩形情報
        RectangleF               _rect = RectangleF.Empty;
        #endregion


        // コンストラクタ
        public SelectRect(){}

        /// <summary>
        /// 選択を開始します。
        /// </summary>
        /// <param name="pntInScreen"></param>
        public void BeginSelecting( PointF pntInScreen )
        {
            _rect.Location = pntInScreen;
        }

        /// <summary>
        /// 矩形情報を更新します。
        /// </summary>
        /// <param name="pntInScreen"></param>
        public void UpDataSize( PointF pntInScreen )
        {
            SizeF sz = new SizeF( pntInScreen.X - _rect.Location.X,
                                  pntInScreen.Y - _rect.Location.Y );
            _rect.Size = sz;
        }

        /// <summary>
        /// 選択処理を終了します。
        /// 矩形情報をリセットし、終了点の矩形情報を返します。
        /// </summary>
        /// <returns></returns>
        public RectangleF EndSelecting()
        {
            RectangleF               result = MakePositiveSizeRectangle_();

            _rect = RectangleF.Empty;

            return result;
        }

        /// <summary>
        /// 描画する必要があるか調べます。
        /// </summary>
        public bool ShouldDraw()
        {
            return _rect != RectangleF.Empty;
        }


        #region --------------- 非公開 メソッド ---------------
        /// <summary>
        /// サイズが正の値を持つ、矩形を生成して返します。
        /// </summary>
        /// <returns></returns>
        RectangleF MakePositiveSizeRectangle_()
        {
            RectangleF       newRect = _rect;

            if (_rect.Width < 0.0f)
            {
                newRect.X = _rect.X + _rect.Width;
                newRect.Width = -_rect.Width;
            }

            if (_rect.Height < 0.0f)
            {
                newRect.Y = _rect.Y + _rect.Height;
                newRect.Height = -_rect.Height;
            }

            return newRect;
        }

        #endregion // --------------- 非公開 メソッド ---------------

        #region --------------- IDrawable メンバ ---------------
        /// <summary>
        /// サイズが正当な場合に、灰色の矩形を描画します
        /// </summary>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            if( ShouldDraw() )
            {
                RectangleF rc = MakePositiveSizeRectangle_();

                renderer.Color = option.SystemRed;
                renderer.DrawRectangleOrtho( rc.X, rc.Y, 0.0f, rc.Width, rc.Height );
            }
        }

        #endregion // --------------- IDrawable メンバ ---------------
    }
}
