﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.common
{
    /// <summary>
    /// コントロールに対して、
    /// マウス位置からスクロールをするか判定する。
    /// 登録されたハンドラを呼び出して、スクロールをする。
    /// </summary>
    public class ControlScroller
    {
        /// <summary>
        /// スクロール処理ハンドラ
        /// </summary>
        public delegate void OnScrollHandler( Control target, PointF vDiff );

        /// <summary>
        /// スクロール処理対象コントロール
        /// </summary>
        readonly Control _targetControl = null;
        /// <summary>
        /// 内部タイマ
        /// </summary>
        readonly Timer _scrollingTimer = new Timer();

        /// <summary>
        /// スクロール処理ハンドラ
        /// </summary>
        readonly OnScrollHandler _OnScrollHandler;
        /// <summary>
        /// 判定閾値
        /// </summary>
        readonly float _edgeThreshold = 0.0f; // _targetTreeView.ItemHeight * 2

        /// <summary>
        /// スクロール量。
        /// </summary>
        PointF _differnt = PointF.Empty;

        /// <summary>
        /// スクロールしているか
        /// </summary>
        public bool Scrolling
        {
            get { return _scrollingTimer.Enabled == true; }
        }

        float Left { get { return _edgeThreshold; } }
        float Right { get { return _targetControl.Width - _edgeThreshold; } }
        float Top { get { return _edgeThreshold; } }
        float Bottom { get { return _targetControl.Height - _edgeThreshold; } }



        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ControlScroller( Control ctrl, OnScrollHandler onScrollHandler, float edgeThreshold )
        {
            _targetControl = ctrl;
            _OnScrollHandler = onScrollHandler;
            _edgeThreshold = edgeThreshold;

            _scrollingTimer.Interval = 100;
            _scrollingTimer.Tick += new EventHandler( Event_TreeScrollingTimer_Tick );
        }

        /// <summary>
        /// マウスがエッジ上か判定します。
        /// </summary>
        protected virtual bool CheckNeedToStartScroll_( Point posInCliant )
        {
            return
                posInCliant.X < Left || posInCliant.X > Right ||
                posInCliant.Y < Top || posInCliant.Y > Bottom;
        }

        /// <summary>
        /// スクロール更新
        /// </summary>
        /// <param name="mousePos"></param>
        public void UpdateScrolling( Point mousePos )
        {
            Point posInCliant = _targetControl.PointToClient( mousePos );

            if( CheckNeedToStartScroll_( posInCliant ) )
            {
                _scrollingTimer.Enabled = true;
            }
            else
            {
                StopScrolling();
            }
        }

        /// <summary>
        /// スクロール停止
        /// </summary>
        public void StopScrolling()
        {
            _scrollingTimer.Interval = 100;
            _scrollingTimer.Enabled = false;
        }

        /// <summary>
        /// スクロールタイマハンドラ
        /// </summary>
        private void Event_TreeScrollingTimer_Tick( object sender, EventArgs e )
        {
            Point mp = _targetControl.PointToClient( Control.MousePosition );

            // スクロール変化量を計算します。
            // X
            if( mp.X < Left )
            {
                _differnt.X = mp.X - Left;
            }
            else if( mp.X > Right )
            {
                _differnt.X = mp.X - Right;
            }
            else
            {
                _differnt.X = 0.0f;
            }

            // Y
            if( mp.Y < Top )
            {
                _differnt.Y = mp.Y - Top;
            }
            else if( mp.Y > Bottom )
            {
                _differnt.Y = mp.Y - Bottom;
            }
            else
            {
                _differnt.Y = 0.0f;
            }

            int diff = (int)Math.Floor( Math.Max( Math.Abs( _differnt.X ), Math.Abs( _differnt.Y ) ) );

            // 更新イベントの周期を変更します。
            _scrollingTimer.Interval = 100 - Math.Min( 99, diff * 5 );

            // スクロールメッセージの送信
            int numScrollMsgSend = 1 + ( diff / 50 );
            while( numScrollMsgSend > 0 )
            {
                if( _OnScrollHandler != null )
                {
                    _OnScrollHandler( _targetControl, _differnt );
                }
                numScrollMsgSend--;
            }

            // マウスが離れていたら停止
            if( ( Control.MouseButtons & MouseButtons.Left ) == 0 )
            {
                StopScrolling();
            }
        }
    }
}
