﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LECore.Structures.Core;
using System.Diagnostics;
using System.Timers;

using LayoutEditor.Structures.SerializableObject;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    public partial class PlayContorolPanel : UserControl
    {
        //----------------------------------------------------------

        // 前後のキーの時間を取得する関数。
        public delegate bool GetNeighborKeyTimeFunction(
            float currentTime,
            bool bPrev,
            ref float resultTime);

        GetNeighborKeyTimeFunction _GetNeighborKeyTimeFunc = null;

        /// <summary>
        /// 前後のキーの時間を取得する関数。
        /// 設定されている場合にのみ、利用可能となる。
        /// </summary>
        public GetNeighborKeyTimeFunction GetNeighborKeyTimeFunc
        {
            set
            {
                // 一度だけ設定される
                Debug.Assert(_GetNeighborKeyTimeFunc == null);
                _GetNeighborKeyTimeFunc = value;
            }
        }

        /// <summary>
        /// グローバル時間クラス（クラス中でよく使用するので）
        /// </summary>
        GlobalTime _GrbTime
        {
            get { return GlobalTime.Inst; }
        }

        bool _IsTargetKeyValid
        {
            get { return _GetNeighborKeyTimeFunc != null; }
        }

        //----------------------------------------------------------

        public PlayContorolPanel()
        {
            InitializeComponent();
            timer.Interval = 5;
            timer.Elapsed += (s, e) => this.Invoke(new Action(() => { Event_Timer_Tick(s, e); }));
        }

        //----------------------------------------------------------

        public void UpdateProperty()
        {
            GlobalTime gt = GlobalTime.Inst;

            _tbxCurrTime.Text = gt.Time.ToString();

            // 次のキー、前のキー
            bool bTargetKeyValid = _IsTargetKeyValid;
            _btnNextKey.Enabled = bTargetKeyValid;
            _btnPrevKey.Enabled = bTargetKeyValid;
            _pctFps.BackgroundImage = Is30FpsSetting()
                                ? Properties.Resources.CurveEditor_AnimationInfo_30fps
                                : Properties.Resources.CurveEditor_AnimationInfo_60fps;
        }

        public bool IsPlaying => timer.Enabled;

        public void Suspend()
        {
            timer.Stop();
        }

        public void Resume()
        {
            timer.Start();
        }

        /// <summary>
        /// グローバル時間を設定します。
        /// </summary>
        void SetGlobalTime_(int newTime)
        {
            if (_GrbTime.Time != newTime)
            {
                // 一旦、再生状態に設定し、値変更後、再生停止します。
                // 本システムでは、アニメーション再生中は、
                // 更新メッセージの伝播が効率的に行われるためです。
                _GrbTime.Play();
                _GrbTime.Time = newTime;
                _GrbTime.Stop();
            }
        }

        /// <summary>
        /// 順再生中か判定します。
        /// </summary>>
        bool IsPlayingForword_()
        {
            return _btnPlayForword.Enabled == false;
        }

        /// <summary>
        /// 逆再生中か判定します。
        /// </summary>>
        bool IsPlayingBackword_()
        {
            return _btnPlayBackword.Enabled == false;
        }

        /// <summary>
        /// 進めた時間を取得します。
        /// </summary>
        int _IncrementTime
        {
            get
            {
                // 順再生。
                int nextTime = _GrbTime.Time + 1;
                if (nextTime >= _GrbTime.AnimPlayEndTime + 1)
                {
                    nextTime = _GrbTime.AnimPlayStartTime;
                }
                return nextTime;
            }
        }

        /// <summary>
        /// 戻った時間を取得します。
        /// </summary>
        int _DecrementTime
        {
            get
            {
                // 逆再生。
                int nextTime = _GrbTime.Time - 1;
                if (nextTime <= _GrbTime.AnimPlayStartTime - 1)
                {
                    nextTime = _GrbTime.AnimPlayEndTime;
                }
                return nextTime;
            }
        }

        //----------------------------------------------------------
        System.Timers.Timer timer = new System.Timers.Timer();
        DateTime lastUpdateTime = DateTime.Now;

        static readonly TimeSpan FrameSpanFps30X2 = new TimeSpan(10000000 / 15);
        static readonly TimeSpan FrameSpanFps30 = new TimeSpan(10000000 / 30);
        static readonly TimeSpan FrameSpanFps60 = new TimeSpan(10000000 / 60);

        /// <summary>
        /// タイマイベントハンドラ
        /// </summary>
        private void Event_Timer_Tick(object sender, System.EventArgs e)
        {
            if (!timer.Enabled)
            {
                // すでに Timer が止まっている場合は処理しない
                return;
            }

            var now = DateTime.Now;
            var elapsed = now - lastUpdateTime;

            if (Is30FpsSetting())
            {
                if (elapsed < FrameSpanFps30)
                {
                    return;
                }
                // 1フレームのレンダリングに2フレーム分以上時間がかかっている場合は、次のイベントでもレンダリングされるように適当に補正
                if (elapsed > FrameSpanFps30X2)
                {
                    lastUpdateTime = now - FrameSpanFps30;
                }
                else
                {
                    lastUpdateTime = lastUpdateTime + FrameSpanFps30;
                }
            }
            else
            {
                if (elapsed < FrameSpanFps60)
                {
                    return;
                }
                // 1フレームのレンダリングに2フレーム分以上時間がかかっている場合は、次のイベントでもレンダリングされるように適当に補正
                if (elapsed > FrameSpanFps30)
                {
                    lastUpdateTime = now - FrameSpanFps60;
                }
                else
                {
                    lastUpdateTime = lastUpdateTime + FrameSpanFps60;
                }
            }

            _GrbTime.IsPlaying = true;
            if (IsPlayingForword_())
            {
                _GrbTime.Time = this._IncrementTime;
            }
            else if (IsPlayingBackword_())
            {
                _GrbTime.Time = this._DecrementTime;
            }
        }

        private bool Is30FpsSetting()
        {
            var parent = ParentForm as LEToolWindow;
            if (parent == null || parent.ViewManager == null || parent.ViewManager.AppSetting == null)
            {
                return false;
            }
            return parent.ViewManager.AppSetting.ProjectSettings.Is30Fps;
        }

        /// <summary>
        /// 再生ボタンクリック
        /// </summary>
        private void Event_BtnPlayForword_Click(object sender, System.EventArgs e)
        {
            Event_BtnStop_Click(sender, e);
            _btnPlayForword.Enabled = false;
            _btnStop.Enabled = true;
            timer.Start();

            _GrbTime.Play();
        }

        /// <summary>
        /// 停止ボタンクリック
        /// </summary>
        private void Event_BtnStop_Click(object sender, System.EventArgs e)
        {
            Stop();
        }

        /// <summary>
        /// 停止
        /// </summary>
        public void Stop()
        {
            _btnPlayForword.Enabled = true;
            _btnPlayBackword.Enabled = true;
            _btnStop.Enabled = false;
            timer.Stop();

            _GrbTime.Stop();
        }

        /// <summary>
        /// 現在の時間
        /// </summary>
        private void Event_TbxCurrTime_ValueChanged(object sender, System.EventArgs e)
        {
            Controls.IntTextBox itb = sender as Controls.IntTextBox;

            if (_GrbTime.CheckValidtyForCurrentTime(itb.Value))
            {
                _GrbTime.Play();
                _GrbTime.Time = itb.Value;
                _GrbTime.Stop();
            }
            else
            {
                itb.SetValue(_GrbTime.Time);
            }
        }

        /// <summary>
        /// 逆再生ボタンクリック
        /// </summary>
        private void Event_Btn_PlayBackword_Click(object sender, System.EventArgs e)
        {
            Event_BtnStop_Click(sender, e);
            _btnPlayBackword.Enabled = false;
            _btnStop.Enabled = true;
            timer.Start();

            _GrbTime.Play();
        }

        /// <summary>
        /// 前のキー
        /// </summary>
        private void Event_BtnPrevKey_Click(object sender, System.EventArgs e)
        {
            Event_BtnStop_Click(sender, e);
            Debug.Assert(_GetNeighborKeyTimeFunc != null);

            float prevTime = _GrbTime.Time;
            if (_GetNeighborKeyTimeFunc(_GrbTime.Time, true, ref prevTime))
            {
                SetGlobalTime_((int)prevTime);
            }
        }

        /// <summary>
        /// 次のキー
        /// </summary>
        private void Event_BtnNextKey_Click(object sender, System.EventArgs e)
        {
            Event_BtnStop_Click(sender, e);
            Debug.Assert(_GetNeighborKeyTimeFunc != null);

            float nextTime = _GrbTime.Time;
            if (_GetNeighborKeyTimeFunc(_GrbTime.Time, false, ref nextTime))
            {
                SetGlobalTime_((int)nextTime);
            }
        }

        /// <summary>
        /// 前の1フレーム
        /// </summary>
        private void Event_BtnPrevFrame_Click(object sender, System.EventArgs e)
        {
            SetGlobalTime_(_DecrementTime);
        }

        /// <summary>
        /// 次の1フレーム
        /// </summary>
        private void Event_BtnNextFrame_Click(object sender, System.EventArgs e)
        {
            SetGlobalTime_(_IncrementTime);
        }

        /// <summary>
        /// 開始フレーム
        /// </summary>
        private void Event_BtnStart_Click(object sender, System.EventArgs e)
        {
            SetGlobalTime_(_GrbTime.AnimPlayStartTime);
        }

        /// <summary>
        /// 終了フレーム
        /// </summary>
        private void Event_BtnEnd_Click(object sender, System.EventArgs e)
        {
            SetGlobalTime_(_GrbTime.AnimPlayEndTime);
        }
    }
}
