﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using Adapters;

    using LECore.Structures;
    using src.Controls;
    using System.Collections.Generic;
    using Structures.SerializableObject;

    /// <summary>
    /// オブジェクトプロパティページクラス。
    /// </summary>
    public class PropertyPage : UserControl, ISceneModifyListener
    {
        #region デザイナ変数
        private System.ComponentModel.Container components = null;
        #endregion

        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            //
            // PropertyPage
            //
            this.Name = "PropertyPage";
            this.Size = new System.Drawing.Size(328, 150);

        }
        #endregion

        #region ---------------- フィールド ----------------

        // オーナー
        private PropertyPanel     _owner        = null;

        // サイズ変更
        private bool                 _sizable		= false;
        private Size                 _defaultSize	= Size.Empty;

        // イベントハンドラデリゲート
        private readonly DocPropertyChangedEventHandler _eventHandler;

        #endregion ---------------- フィールド ----------------

        /// <summary>
        /// GUIの更新中である
        /// GUIハンドラは参照対象データを
        /// 更新する必要がありません。
        /// UpdateProperty 呼び出し前後で更新されます。
        /// </summary>
        protected bool IsGUIUpdating => UpdatingCount > 0;

        private int UpdatingCount { get; set; }

        protected class GUIUpdating : IDisposable
        {
            private PropertyPage page { get; set; }
            public GUIUpdating(PropertyPage page)
            {
                this.page = page;
                page.UpdatingCount++;
            }
            public void Dispose()
            {
                page.UpdatingCount--;
            }
        }

        /// <summary>
        /// 強制的に表示を無効化します。
        /// </summary>
        public bool ForceDisabled
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャだけ上書きをしているか
        /// </summary>
        public bool IsTextureOverwritingEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// 上書き設定か
        /// </summary>
        [Browsable(false)]
        public bool IsOverwrite
        {
            get;
            set;
        }

        /// <summary>
        /// ページＩＤ。
        /// </summary>
        public virtual PropertyPageID PageID
        {
            get { return PropertyPageID.Null; }
        }

        /// <summary>
        /// オーナー。
        /// </summary>
        [Browsable(false)]
        public PropertyPanel Owner
        {
            get { return _owner;  }
            set { _owner = value; }
        }

        /// <summary>
        /// ターゲット。
        /// </summary>
        [Browsable(false)]
        public PaneGuiAdapterGroup Target
        {
            get { return _owner != null ? _owner.Target : null; }
        }

        public AppSetting AppSetting
        {
            get{ return Owner?.Owner?.AppSetting; }
        }

        /// <summary>
        /// サイズ変更可能なページかどうか。
        /// </summary>
        [DefaultValue(false)]
        [Description("サイズ変更可能なページかどうか。")]
        public bool Sizable
        {
            get { return _sizable;  }
            set { _sizable = value; }
        }

        /// <summary>
        /// デフォルトサイズ。
        /// </summary>
        [Browsable(false)]
        public Size DefaultPageSize
        {
            get { return _defaultSize;   }
            set { _defaultSize = value;  }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected PropertyPage()
        {
            InitializeComponent();

            // プロパティ編集イベント登録
            _eventHandler = new DocPropertyChangedEventHandler(Event_DocumentObject_DocPropertyChanged);
            this.ForceDisabled = false;
        }

        /// <summary>
        /// フォーム初期化。
        /// </summary>
        public void InitializeForm()
        {
            // 拡張用に１クッション置く
            InitializeProperty();

            //	最初に設定したサイズがデフォルト
            _defaultSize	= this.Size;
        }

        /// <summary>
        /// フォーム更新。
        /// </summary>
        public void UpdateForm()
        {
            this.BeforeUpdateTargetUpdate();

            using (new GUIUpdating(this))
            {
                UpdateProperty();
            }

            // 読み込み専用ロック状態なら強制的に無効にします。
            if (this.ForceDisabled || Target == null || Target.Active == null ||
                (Target.Active.Target.IsReadOnlyLocked && !AllowEditReadOnlyTarget()))
            {
                this.Enabled = false;
            }
            else
            {
                this.Enabled = true;
            }
        }

        /// <summary>
        /// 同時編集の識別に使われるカラーを取得します。
        /// </summary>
        protected Color GetMultiEditColor()
        {
            return Target.IsMulti != false ? Color.MintCream : SystemColors.Window;
        }

        /// <summary>
        /// 同時編集のボタン識別に使われるカラーを取得します。
        /// </summary>
        protected Color GetMultiEditButtonColor()
        {
            return Target.IsMulti != false ? Color.MintCream : SystemColors.ButtonFace;
        }

        /// <summary>
        /// 同時編集のパネル識別に使われるカラーを取得します。
        /// </summary>
        protected Color GetMultiEditPanelColor()
        {
            return GetMultiEditPanelColor(false);
        }

        /// <summary>
        /// 同時編集のパネル識別に使われるカラーを取得します。
        /// </summary>
        protected Color GetMultiEditPanelColor(bool disabledConditionFlag)
        {
            if (disabledConditionFlag)
            {
                return SystemColors.Control;
            }

            return Target.IsMulti != false ? Color.MintCream : SystemColors.Control;
        }

        /// <summary>
        /// アニメーションマークが利用可能を判定する。
        /// </summary>
        protected bool CanUseAnimationMark(ISubScene subScene)
        {
            // 上書き部分はアニメーションが不可です。
            if (this.IsOverwrite)
            {
                return false;
            }

            // シーンがアニメーション編集可能な状態でなければ不可です。
            if (!subScene.IsAnimationEditable())
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 複数同時編集
        /// </summary>
        protected virtual void MultiEdit_<T>(Action<object> modifyAction, T active, IEnumerable<T> targets)
        {
            if (IsGUIUpdating)
            {
                // 各コントロール等で弾いているはず
                Debug.Assert(false);
                return;
            }

            if (Target.IsMulti)
            {
                ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
                subScene.BeginMassiveModify();
                {
                    foreach (var target in targets)
                    {
                        modifyAction(target);
                    }
                }
                subScene.EndMassiveModify();
            }
            else
            {
                // 変更実行途中での UI の更新を抑える
                ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
                subScene.BeginMassiveModify();
                modifyAction(active);
                subScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// アニメーションマークが利用可能なコントロールの一覧を取得する。
        /// </summary>
        protected virtual IEnumerable<IAnimationMarkUsable> GetAnimationMarkUsableControl()
        {
            return new List<IAnimationMarkUsable>();
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public virtual bool CanCopy(){ return false; }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public virtual string CanPaste(Object pasteObject)
        {
            return string.Empty;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public virtual Object Copy(){ return null; }

        /// <summary>
        /// ペースト。
        /// </summary>
        public virtual void Paste(Object pasteObject){ Debug.Assert(false); }

        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        protected virtual void InitializeProperty()
        {
            // 派生クラスで実装する
            // 抽象にしないのはフォームデザイナでインスタンス化させる為
            Debug.Assert(false);
        }

        /// <summary>
        /// プロパティ更新。
        /// </summary>
        protected virtual void UpdateProperty()
        {
            // 派生クラスで実装する
            // 抽象にしないのはフォームデザイナでインスタンス化させる為
            Debug.Assert(false);
        }

        /// <summary>
        /// 更新処理です。
        /// </summary>
        public virtual void UpdateTarget(IPane newTargetPane)
        {
            // 特に何もしません。
        }

        /// <summary>
        /// ターゲット変更時にすべてのページについて呼び出される関数
        /// </summary>
        public virtual void OnTargetChangeHandler()
        {
            // 何もしません。
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// 再描画前の対象更新です。
        /// </summary>
        protected virtual void BeforeUpdateTargetUpdate()
        {
            // 何もしません。// マテリアル関連ページだけで処理されます。
        }

        /// <summary>
        /// 読み込み専用でも編集を許可するかどうか？
        /// <remarks>
        /// 例外的に、編集を許可したいページでオーバーライドします。
        /// </remarks>
        /// </summary>
        protected virtual bool AllowEditReadOnlyTarget()
        {
            return false;
        }

        /// <summary>
        /// 変更を通知します。
        /// </summary>
        protected virtual void NotifyPropertyChanged()
        {
            // TODO:
            // 自らの操作対象に変更があったことを通知します。
            // 変更通知を受け取る性質をIGuiAdapterに追加して、
            // そのメソッドを利用することにします。

            // 何もしません。// マテリアル関連ページだけで処理されます。
        }

        /// <summary>
        /// 変更を通知します。
        /// </summary>
        protected virtual void NotifyPropertyChanged(IBaseGuiAdapter paneAdapter)
        {
            // TODO:
            // 自らの操作対象に変更があったことを通知します。
            // 変更通知を受け取る性質をIGuiAdapterに追加して、
            // そのメソッドを利用することにします。

            // 何もしません。// マテリアル関連ページだけで処理されます。
        }

        /// <summary>
        /// 操作対象が自分以外で変更された通知を受け取るメソッド。
        /// </summary>
        private void Event_DocumentObject_DocPropertyChanged(object sender, DocPropertyChangedEventArgs e)
        {
            // 表示中なら
            if (this.Visible)
            {
                // 何もしない
                Debug.Assert( false );
            }
        }

        /// <summary>
        /// 非表示になったときに呼び出されるメソッド。
        /// OnTargetChangeHandler() は非表示になった際には呼び出されません。
        /// </summary>
        public virtual void OnHide()
        {
            // 何もしません。
        }

        #region ISceneModifyListener メンバ
        /// <summary>
        /// シーン変更イベントハンドラ
        /// </summary>
        public virtual void OnSceneModifyHandler( object sender, SceneModifyEventArgs e )
        {
            // アニメーションの追加削除変更があったら、ＵＩを更新します。
            if (e.Target == SceneModifyEventArgs.Kind.PaneAnimAddRemove ||
                e.Target == SceneModifyEventArgs.Kind.PaneAnimModify)
            {
                using (new GUIUpdating(this))
                {
                    UpdateProperty();
                }
            }
        }

        #endregion
    }

    #region PropertyPageID
    /// <summary>
    /// プロパティページＩＤ。
    /// </summary>
    public enum PropertyPageID
    {
        /// <summary>ＮＵＬＬ。</summary>
        Null,
        /// <summary>ユーザーデータ。</summary>
        ///UserData,
        /// <summary>モデル全般。</summary>
        //ModelGeneral,
        /// <summary>ノード全般。</summary>
        //NodeGeneral,
        /// <summary>ノード表示。</summary>
        //NodeVisibility,
        /// <summary>ノードビルボード。</summary>
        //NodeBillboard,
        /// <summary>ノード変換オプション。</summary>
        //NodeConvertOption,
        /// <summary>ポリゴン全般ページ。</summary>
        //PolygonGeneral,
        /// <summary>ポリゴン表示面。</summary>
        //PolygonDisplayFace,
        /// <summary>マテリアル全般。</summary>
        //MaterialGeneral,
        /// <summary>マテリアルカラー。</summary>
        MaterialColor,
        /// <summary>マテリアルテクスチャ。</summary>
        MaterialTexture,
        /// <summary>マテリアルＴＥＶ。</summary>
        MaterialTev,
        /// <summary>マテリアルＴＥＶスワップテーブル。</summary>
        //MaterialTevSwapTable,
        /// <summary>マテリアルＴＥＶカラーレジスタ。</summary>
        MaterialTevColor,
        /// <summary>マテリアルＴＥＶコンスタントレジスタ。</summary>
        //MaterialTevKColor,
        /// <summary>マテリアルＴＥＶステージ。</summary>
        MaterialTevStage,
        /// <summary>マテリアルＰＥ。</summary>
        //MaterialPE,
        /// <summary>マテリアル変換オプション。</summary>
        //MaterialConvertOption,
        /// <summary>テクスチャ全般。</summary>
        //TextureGeneral,
        /// <summary>テクスチャ使用状況。</summary>
        //TextureUsingState,
        /// <summary>パレット全般。</summary>
        //PaletteGeneral,
        /// <summary>パレット使用状況。</summary>
        //PaletteUsingState,
        /// <summary>カラー補間。</summary>
        MaterialColorBelnd,
        /// <summary>フラグメントオペレーション。</summary>
        MaterialFragmentOperation,
        /// <summary>マテリアルＴＥＶステージ(LowLevel)。</summary>
        MaterialTevStageLowLevel,
    }
    #endregion
}
