﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Linq;
using System.Windows.Forms;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
using LECore.Structures.SerializableObject.Lyt;
using LayoutEditor.Controls;
using LECore.Structures;

namespace LayoutEditor.src.Forms.ToolWindows.PropertyEditWindow.Pane.Window
{
    /// <summary>
    /// ウインドウ操作パネルです。
    /// </summary>
    public partial class WindowKindPanel : UserControl
    {

        private IWindowPaneAdapterEntry _targetWindowPane;
        private LEWindowPartsID _lastSelectedWindowPartsID;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public WindowKindPanel()
        {
            InitializeComponent();

            _cmbKind.AddItem(StringResMgr.Get("WINDOWKINDPANEL_KIND_ROUND"), WindowKind.Around, false);
            _cmbKind.AddItem(StringResMgr.Get("WINDOWKINDPANEL_KIND_HORIZONTAL"), WindowKind.Horizontal, false);
            _cmbKind.AddItem(StringResMgr.Get("WINDOWKINDPANEL_KIND_HORIZONTALNOCONT"), WindowKind.HorizontalNoContent, false);

            InitNumTextureCombo_(WindowKind.Around);

            _ctlTargetPartsPanel.OnWindowPartsClicked += new Action<LEWindowPartsID>(TargetWindowpartsChanged_);

            UpdateWarningMessage_();
        }

        public void TargetWindowpartsChanged(LEWindowPartsID id)
        {
            _lastSelectedWindowPartsID = id;

            this.UpdateProperty();

            if (OnTargetPartsChanged != null) { OnTargetPartsChanged(); }
        }

        private void TargetWindowpartsChanged_(LEWindowPartsID id)
        {
            if (this.TargetWindowPane == null)
            {
                return;
            }

            if (id != this.TargetWindowPane.CurrentTargetPartsID)
            {
                this.TargetWindowPane.CurrentTargetPartsID = id;
                TargetWindowpartsChanged(this.TargetWindowPane.CurrentTargetPartsID);
            }
        }

        /// <summary>
        /// 捜査対象ペインアダプタ
        /// </summary>
        public IWindowPaneAdapterEntry TargetWindowPane
        {
            get { return _targetWindowPane; }
            set
            {
                // 以前に編集対象だった部位を対象に設定する。
                if (!object.ReferenceEquals(_targetWindowPane, value))
                {
                    if(value != null)
                    {
                        value.CurrentTargetPartsID = _lastSelectedWindowPartsID;
                    }
                }

                _targetWindowPane = value;
            }
        }

        /// <summary>
        /// 対象サブシーン
        /// </summary>
        private ISubScene _TargetSubScene
        {
            get { return TargetWindowPane.Manipulator.ILEWindow.OwnerPane.OwnerSubScene; }
        }

        /// <summary>
        /// 対象部位変更イベント
        /// </summary>
        public event Action OnTargetPartsChanged;

        /// <summary>
        /// 枚数コンボ初期化
        /// </summary>
        private void InitNumTextureCombo_(WindowKind newKind)
        {
            _cmbNumTexture.Items.Clear();
            if(newKind == WindowKind.Around)
            {
                _cmbNumTexture.AddItem(StringResMgr.Get("WINDOWKINDPANEL_1"), 2, false);
                _cmbNumTexture.AddItem(StringResMgr.Get("WINDOWKINDPANEL_4"), 5, false);
                _cmbNumTexture.AddItem(StringResMgr.Get("WINDOWKINDPANEL_8"), 9, false);
            }else{
                _cmbNumTexture.AddItem(StringResMgr.Get("WINDOWKINDPANEL_H1"), 2, false);
                _cmbNumTexture.AddItem(StringResMgr.Get("WINDOWKINDPANEL_H2"), 3, false);
            }

            _cmbNumTexture.Tag = newKind;
        }

        /// <summary>
        /// UI更新
        /// </summary>
        public void UpdateProperty()
        {
            if (TargetWindowPane == null)
            {
                _lblParts.Text = string.Empty;
                _ctlTargetPartsPanel.Adapter = null;
                _cmbKind.Enabled = false;
                _cmbNumTexture.Enabled = false;
                return;
            }

            var wnd = TargetWindowPane.Manipulator.ILEWindow;

            // 部位選択
            _ctlTargetPartsPanel.Adapter = TargetWindowPane;
            _ctlTargetPartsPanel.Invalidate();

            // 種類コンボ
            _cmbKind.SetSelectedByData(wnd.WindowKind);
            _cmbKind.Enabled = true;

            // 枚数コンボ
            if ((WindowKind)_cmbNumTexture.Tag != wnd.WindowKind)
            {
                InitNumTextureCombo_(wnd.WindowKind);
            }
            _cmbNumTexture.SetSelectedByData(wnd.NumTexture);
            _cmbNumTexture.Enabled = true;

            switch (TargetWindowPane.CurrentTargetPartsID)
            {
                case LEWindowPartsID.Content: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_CONTENTs"); break;
                case LEWindowPartsID.CornerLT: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_LT"); break;
                case LEWindowPartsID.CornerLB: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_LB"); break;
                case LEWindowPartsID.CornerRT: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_RT"); break;
                case LEWindowPartsID.CornerRB: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_RB"); break;
                case LEWindowPartsID.FrameL: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_LF"); break;
                case LEWindowPartsID.FrameR: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_RF"); break;
                case LEWindowPartsID.FrameT: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_TF"); break;
                case LEWindowPartsID.FrameB: _lblParts.Text = StringResMgr.Get("WINDOWKINDPANEL_BF"); break;
            }

            // エラー表示
            UpdateWarningMessage_();
        }

        /// <summary>
        /// 警告を表示します。
        /// </summary>
        private void UpdateWarningMessage_()
        {
            if (TargetWindowPane == null)
            {
                return;
            }

            var wnd = TargetWindowPane.Manipulator.ILEWindow;

            _lblMessage.Visible = false;
            if (wnd.WindowKind == WindowKind.Horizontal)
            {
                if (wnd.GetMaterial(LEWindowPartsID.Content).IMaterialTexMapSet.Length == 0)
                {
                    _lblMessage.Visible = true;
                    _lblMessage.Text = StringResMgr.Get("WINDOWKINDPANEL_HORIZ_NOTEX_CONTENTS");
                    return;
                }
            }

            if (!wnd.IsMultiTextureCountValid())
            {
                _lblMessage.Visible = true;
                _lblMessage.Text = StringResMgr.Get("WINDOWKINDPANEL_FRAME_DIFF_TEXNUM");
                return;
            }

            if (!wnd.IsFrameTextureSizeValid())
            {
                _lblMessage.Visible = true;
                _lblMessage.Text = StringResMgr.Get("WINDOWKINDPANEL_TEXSIZE");
                return;
            }
        }

        //----------------------------------------------------------

        private void Event_CmbKind_SelectedIndexChanged(object sender, EventArgs e)
        {
            var newKind = (WindowKind)_cmbKind.SelectedItemData;
            if(TargetWindowPane.Manipulator.ILEWindow.WindowKind != newKind )
            {
                _TargetSubScene.BeginMassiveModify();
                {
                    TargetWindowPane.Manipulator.WindowKind = newKind;
                }

                this.TargetWindowPane.CurrentTargetPartsID = LEWindowPartsID.Content;
                TargetWindowpartsChanged_(_lastSelectedWindowPartsID);

                _TargetSubScene.EndMassiveModify();
            }
        }

        private void Event_CmbNumTexture_SelectedIndexChanged(object sender, EventArgs e)
        {
            var newNumTex = (int)_cmbNumTexture.SelectedItemData;
            if (TargetWindowPane.Manipulator.ILEWindow.NumTexture != newNumTex)
            {
                _TargetSubScene.BeginMassiveModify();
                {
                    TargetWindowPane.Manipulator.NumTexture = newNumTex;
                }

                this.TargetWindowPane.CurrentTargetPartsID = LEWindowPartsID.Content;
                TargetWindowpartsChanged_(_lastSelectedWindowPartsID);

                _TargetSubScene.EndMassiveModify();
            }
        }
    }
}
