﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LECore;
using LECore.Structures;
using LECore.Manipulator;
using LayoutEditor.Structures.SerializableObject;

namespace LayoutEditor.Forms.ToolWindows
{
    public partial class UserDataPanel : UserControl
    {
        #region プロパティ

        static UserDataDlg _userDataDlg = null;
        UserDataHolderManipulator _targetMnp = new UserDataHolderManipulator();
        bool _extUserDataAnimEnabled = false;
        bool _kindHintVisible = false;
        bool _isOverwriteUserData = false;
        Func<ExUserDataPreset, bool> _filterFunc = null;
        IEnumerable<ExUserDataPreset> _userDataPresets = null;

        /// <summary>
        /// 操作対象ペインを操作するクラスを取得します。
        /// </summary>
        UserDataHolderManipulator _TargetMpn
        {
            get{return _targetMnp;}
        }

        /// <summary>
        /// 16進表記をするか？
        /// </summary>
        bool ShowAsHex_
        {
            get { return _chkShowAsHex.Checked; }
        }


        /// <summary>
        /// 選択ユーザ情報を取得します。
        /// </summary>
        IUserDataElement SelectedUserDataElement_
        {
            get
            {
                if( _lvwUserData.SelectedIndices.Count > 0 )
                {
                    return _lvwUserData.Items[_lvwUserData.SelectedIndices[0]].Tag as IUserDataElement;
                }
                return null;
            }
        }

        public Font _boldFont;
        private Color _defaultItemColor = Color.Black;
        private Color _inheritedItemColor = Color.DarkGray;
        private Color _deletedItemColor = Color.Red;

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UserDataPanel()
        {
            InitializeComponent();

            // GUIに関するユーザ独自の初期化処理を記述します。
            _boldFont = new Font(_lvwUserData.Font, FontStyle.Bold);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public void Initialize(IUserDataHolder target, bool extUserDataAnimEnabled, bool kindHintVisible, bool isOverwriteUserData, IEnumerable<ExUserDataPreset> userDataPresets, Func<ExUserDataPreset, bool> filterFunc)
        {
            _TargetMpn.BindTarget(target);
            _extUserDataAnimEnabled = extUserDataAnimEnabled;
            _kindHintVisible = kindHintVisible;
            _userDataPresets = userDataPresets;
            _isOverwriteUserData = isOverwriteUserData;
            _filterFunc = filterFunc;

            _pbxListInfo.Visible = isOverwriteUserData;
        }

        /// <summary>
        /// ボタンの状態を更新します。
        /// </summary>
        void UpdateButtonState_()
        {
            bool bItemSelected = _lvwUserData.SelectedIndices.Count > 0;
            IUserDataElement userData = bItemSelected ? _lvwUserData.Items[_lvwUserData.SelectedIndices[0]].Tag as IUserDataElement : null;

            _btnNext.Enabled = bItemSelected && _lvwUserData.SelectedIndices[0] < _lvwUserData.Items.Count - 1;
            _btnPrev.Enabled = bItemSelected && _lvwUserData.SelectedIndices[0] > 0;
            _btnDelete.Enabled = bItemSelected && userData.UserDataKind != UserDataKind.None;
            _btnReset.Enabled = bItemSelected && userData.Overwrite;
            _btnEdit.Enabled = bItemSelected;
        }

        /// <summary>
        /// ListViewItem に 値を設定します。
        /// </summary>
        void SetUserDataToListViewItem_( ListViewItem lvi, IUserDataElement userData )
        {
            string valueString = UserDataElementHelper.ToString(userData.UserDataKind, userData.Value, ShowAsHex_, " ", true);
            lvi.Text = userData.Name;
            lvi.SubItems.Add( UserDataElementHelper.GetUserDataKindUIString( userData.UserDataKind ) );
            lvi.SubItems.Add( userData.Size.ToString() );
            lvi.SubItems.Add(valueString);
            lvi.SubItems.Add("");// 値の説明
            lvi.SubItems.Add("");// Enumの説明
            lvi.Tag = userData;
            lvi.Font = _lvwUserData.Font;
            lvi.ForeColor = _defaultItemColor;

            if (_isOverwriteUserData && userData.Overwrite == false)
            {
                // 部品レイアウトから引き継いだデータの文字を薄くする
                lvi.ForeColor = _inheritedItemColor;
            }
            else if (userData.UserDataKind == UserDataKind.None)
            {
                // 削除したデータの文字を赤くする
                lvi.ForeColor = _deletedItemColor;
            }

            // UIListViewはListViewItem.ForeColorが反映されないのでListViewSubItem.ForeColorを設定する
            foreach (ListViewItem.ListViewSubItem subItem in lvi.SubItems)
            {
                subItem.ForeColor = lvi.ForeColor;
            }

            // プリセットを探して、説明文章を追加する
            var preset = this._userDataPresets.FirstOrDefault((udp) => udp.Name == userData.Name);
            if (!string.IsNullOrEmpty(preset.Name))
            {
                lvi.ToolTipText = preset.Descriptions;
                lvi.Font = _boldFont;


                if (preset.MultiSelect)
                {
                    string valueStringNoHex = UserDataElementHelper.ToString(userData.UserDataKind, userData.Value, false, Environment.NewLine, true);
                    if (valueStringNoHex != null)
                    {
                        var entries = valueStringNoHex.Split(new[] { Environment.NewLine }, StringSplitOptions.RemoveEmptyEntries);
                        lvi.SubItems[4].Text =
                            string.Join(" ", entries.Select(x => preset.EnumValues.FirstOrDefault(y => y.Text == x)?.Descriptions ?? ""));
                    }
                }
                else
                {
                    string valueStringNoHex = UserDataElementHelper.ToString(userData.UserDataKind, userData.Value, false, " ", true);
                    var enumVal = preset.EnumValues.FirstOrDefault((e) => e.Text == valueStringNoHex);
                    if (enumVal != null)
                    {
                        lvi.SubItems[4].Text = enumVal.Descriptions;
                    }
                }

                lvi.SubItems[5].Text = preset.Descriptions;
            }
        }

        /// <summary>
        /// GUIを更新します。
        /// </summary>
        public void UpdateProperty()
        {
            // ターゲットを参照してUIを更新する
            var target = _TargetMpn.IUserDataHolder;
            if (target != null)
            {
                IUserDataElement[] userDataSet = target.UserDataElementSet;

                IUserDataElement selectedData = SelectedUserDataElement_;
                string selectedName = ( selectedData != null ) ? selectedData.Name : null;


                // 存在しないノードを探します。新規追加します。
                _lvwUserData.BeginUpdate();
                _lvwUserData.Items.Clear();

                foreach( IUserDataElement userData in userDataSet )
                {
                    ListViewItem lvi = new ListViewItem();

                    SetUserDataToListViewItem_( lvi, userData );

                    _lvwUserData.Items.Add( lvi );

                    if( userData.Name == selectedName )
                    {
                        lvi.Selected = true;
                    }
                }
                _lvwUserData.EndUpdate();
            }

            UpdateButtonState_();
        }

        #region イベントハンドラ

        /// <summary>
        /// 上ボタン
        /// </summary>
        private void Event_BtnPrev_Click( object sender, EventArgs e )
        {
            if( _lvwUserData.SelectedIndices.Count > 0 &&
                _lvwUserData.SelectedIndices[0] > 0 )
            {
                int index = _lvwUserData.SelectedIndices[0];
                IUserDataElement userData = _lvwUserData.Items[index].Tag as IUserDataElement;

                _TargetMpn.ChangeUserDataIndex(userData.Name, index - 1);

                // 移動したアイテムを選択状態にする
                int selectIndex = Array.IndexOf(_TargetMpn.IUserDataHolder.UserDataElementSet, userData);
                if (selectIndex >= 0 && selectIndex < _lvwUserData.Items.Count)
                {
                    _lvwUserData.Items[selectIndex].Selected = true;
                }
            }
        }

        /// <summary>
        /// 下ボタン
        /// </summary>
        private void Event_BtnNext_Click( object sender, EventArgs e )
        {
            if( _lvwUserData.SelectedIndices.Count > 0 &&
                _lvwUserData.SelectedIndices[0] < _lvwUserData.Items.Count - 1 )
            {
                int index = _lvwUserData.SelectedIndices[0];
                IUserDataElement userData = _lvwUserData.Items[index].Tag as IUserDataElement;

                _TargetMpn.ChangeUserDataIndex(userData.Name, index + 1);

                // 移動したアイテムを選択状態にする
                int selectIndex = Array.IndexOf(_TargetMpn.IUserDataHolder.UserDataElementSet, userData);
                if (selectIndex >= 0 && selectIndex < _lvwUserData.Items.Count)
                {
                    _lvwUserData.Items[selectIndex].Selected = true;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        UserDataDlg GetuserDataDlg_()
        {
            if (_userDataDlg == null)
            {
                _userDataDlg = new UserDataDlg();
                _userDataDlg.Location = this.PointToScreen(this.Location) + new Size(this.Width / 2, this.Height / 2) - new Size(_userDataDlg.Width / 2, _userDataDlg.Height / 2);
                _userDataDlg.Bounds = LEToolFormSetting.ClampWindowRectangle(_userDataDlg.Bounds);
            }

            _userDataDlg.SetUserDataPresets(this._userDataPresets, (preset) => this._filterFunc(preset));
            _userDataDlg.KindHintVisible = _kindHintVisible;
            return _userDataDlg;
        }

        /// <summary>
        /// 追加ボタン
        /// </summary>
        private void Event_BtnAdd_Click( object sender, EventArgs e )
        {
            IUserDataHolder userDataHolder = _TargetMpn.IUserDataHolder;

            // ダイアログを表示する
            // ダイアログからデータを受け取って新規にデータを追加する
            UserDataDlg userDataDlg = GetuserDataDlg_();
            userDataDlg.SetShowParameter(true, userDataHolder, null);
            userDataDlg.ShowDialog( this );

            if( userDataDlg.DialogResult == System.Windows.Forms.DialogResult.OK )
            {
                _TargetMpn.AddUserDataElement(
                    userDataDlg.UserDataName,
                    userDataDlg.Kind,
                    userDataDlg.Value );

                // 追加したアイテムを選択状態にする
                int selectIndex = _lvwUserData.Items.Count - 1;
                if (selectIndex >= 0 && selectIndex < _lvwUserData.Items.Count)
                {
                    _lvwUserData.Items[selectIndex].Selected = true;
                }
            }
        }

        /// <summary>
        /// 削除ボタンハンドラ
        /// </summary>
        private void Event_BtnDelete_Click( object sender, EventArgs e )
        {
            if( _lvwUserData.SelectedIndices.Count > 0 )
            {
                int selectedIndex = _lvwUserData.SelectedIndices[0];

                IUserDataElement userData = _lvwUserData.Items[_lvwUserData.SelectedIndices[0]].Tag as IUserDataElement;
                _TargetMpn.RemoveUserDataElement(userData.Name);

                // 削除したインデックスを選択状態にする
                int selectIndex = selectedIndex < _lvwUserData.Items.Count ? selectedIndex : selectedIndex - 1;
                if (selectIndex >= 0 && selectIndex < _lvwUserData.Items.Count)
                {
                    _lvwUserData.Items[selectIndex].Selected = true;
                }
            }
        }

        /// <summary>
        /// リセットボタン
        /// </summary>
        private void Event_BtnReset_Click( object sender, EventArgs e )
        {
            if( _lvwUserData.SelectedIndices.Count > 0 )
            {
                int selectedIndex = _lvwUserData.SelectedIndices[0];

                IUserDataElement userData = _lvwUserData.Items[_lvwUserData.SelectedIndices[0]].Tag as IUserDataElement;
                _TargetMpn.ResetUserDataElement(userData.Name);

                // リセットしたインデックスを選択状態にする
                int selectIndex = selectedIndex < _lvwUserData.Items.Count ? selectedIndex : selectedIndex - 1;
                if (selectIndex >= 0 && selectIndex < _lvwUserData.Items.Count)
                {
                    _lvwUserData.Items[selectIndex].Selected = true;
                }
            }
        }

        /// <summary>
        /// 選択インデックス変更
        /// </summary>
        private void Event_LvwUserData_SelectedIndexChanged( object sender, EventArgs e )
        {
            UpdateButtonState_();
        }

        /// <summary>
        /// ノード選択されたユーザデータを編集する。
        /// </summary>
        private void EditSelectedUserDataByDlg_()
        {
            if( _lvwUserData.SelectedIndices.Count > 0 )
            {
                IUserDataElement userData = _lvwUserData.Items[_lvwUserData.SelectedIndices[0]].Tag as IUserDataElement;
                IUserDataHolder userDataHolder = _TargetMpn.IUserDataHolder;

                UserDataDlg userDataDlg = GetuserDataDlg_();
                userDataDlg.SetShowParameter(false, userDataHolder, userData);
                userDataDlg.ShowDialog(this);

                if( userDataDlg.DialogResult == System.Windows.Forms.DialogResult.OK )
                {
                    _TargetMpn.SetUserDataPamaters(
                        userData.Name,
                        userDataDlg.UserDataName,
                        userDataDlg.Kind,
                        userDataDlg.Value );
                }
            }
        }

        /// <summary>
        /// 編集ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void Event_BtnEdit_Click(object sender, EventArgs e)
        {
            // ユーザーデータ編集ダイアログを表示する
            EditSelectedUserDataByDlg_();
        }

        /// <summary>
        /// リストアイテムをダブルクリックしたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void Event_LvwUserData_DoubleClick_(object sender, EventArgs e)
        {
            // ユーザーデータ編集ダイアログを表示する
            EditSelectedUserDataByDlg_();
        }

        /// <summary>
        /// 16進数表示のチェックを切り替えたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void Event_ChkShowAsHex_CheckedChanged_(object sender, EventArgs e)
        {
            UpdateProperty();
        }

        #endregion イベントハンドラ
    }
}
