﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Linq;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows
{
    using LayoutEditor.Forms.Dialogs;
    using LECore.Structures;
    using LayoutEditor.Structures.SerializableObject;
    using System.Windows.Forms.VisualStyles;

    public partial class UserDataDlg : LEBaseDlg
    {
        /// <summary>
        /// 型選択のヒントの文字列
        /// </summary>
        private readonly string _lblKindText;

        /// <summary>
        /// 拡張ユーザーデータの値変更モードかどうか
        /// </summary>
        private bool _editMode;

        #region プロパティ

        public string UserDataName { get { return _tbxName.Text; } }

        /// <summary>
        /// コンボボックスで選択されたユーザーデータの型を取得します。
        /// </summary>
        public UserDataKind Kind
        {
            get
            {
                if (_cmbKind.SelectedItemData != null)
                {
                    return (UserDataKind)_cmbKind.SelectedItemData;
                }
                else
                {
                    return UserDataKind.None;
                }
            }
        }

        /// <summary>
        /// テキストボックスに入力されたユーザーデータの値を取得します。
        /// </summary>
        public object Value
        {
            get { return UserDataElementHelper.ParseString( Kind, _tbxValue.Text, Environment.NewLine ); }
        }

        /// <summary>
        /// 拡張ユーザーデータの値変更モードかどうか取得します。
        /// </summary>
        public bool EditMode
        {
            get
            {
                return _editMode;
            }

            private set
            {
                bool isKindEditable;
                if (value == true)
                {
                    // 操作対象がアニメーション型なら種類は変更禁止。
                    if (_data != null && UserDataElementHelper.IsAnimationKind(_data.UserDataKind))
                    {
                        isKindEditable = false;
                    }else
                    {
                        isKindEditable = true;
                    }
                }
                else
                {
                    isKindEditable = true;
                }

                _btnPreset.Enabled = isKindEditable;
                _grpKind.Enabled = isKindEditable;

                _editMode = value;
            }
        }

        /// <summary>
        /// 型選択のヒントを表示するかどうか取得または設定します。
        /// </summary>
        public bool KindHintVisible
        {
            get
            {
                return !string.IsNullOrEmpty(_lblKind.Text);
            }

            set
            {
                // VisibleプロパティはShowDialogでtrueになってしまうので
                // Textが空かどうかで判断する
                bool visible = !string.IsNullOrEmpty(_lblKind.Text);

                if (value != visible)
                {
                    _lblKind.Text = (value ? _lblKindText : null);

                    // 型選択グループのサイズを更新
                    int height = _lblKind.Height + 3;
                    _grpKind.Height += (value ? height : -height);
                }
            }
        }

        #endregion プロパティ

        private IEnumerable<ExUserDataPreset> _userDataPresets;
        private SelectUserDataPresetDlg _presetForm = new SelectUserDataPresetDlg();
        private IUserDataHolder _holder;
        private IUserDataElement _data;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UserDataDlg()
        {
            InitializeComponent();
            InitializeProperty_();

            _presetForm.Leave += Event_lvwPreset_Leave;
            _presetForm.Deactivate += Event_lvwPreset_Leave;
            _presetForm.ListClicked += Event_lvwPreset_Click;

            _lblKindText = _lblKind.Text;
        }

        /// <summary>
        /// 表示パラメータを設定します。
        /// </summary>
        /// <param name="doAdd">追加用の表示を行うか</param>
        /// <param name="holder">拡張ユーザデータホルダ</param>
        /// <param name="data">拡張ユーザデータ</param>
        public void SetShowParameter (bool doAdd, IUserDataHolder holder, IUserDataElement data)
        {
            _holder = holder;
            _data = data;
            bool animationEnabled = (holder != null ? holder.AnimationEnabled : false);

            if (doAdd)
            {
                // コンボボックスのアイテムを更新
                this.UpdateComboboxItems(animationEnabled);

                this.EditMode = false;
                _cmbKind.SelectedItem = _cmbKind.Items[0];
                _tbxValue.Text = string.Empty;
                _tbxName.Value = string.Empty;
                _tbxName.ReadOnly = false;

                this.UpdateButtonState_();
            }
            else
            {
                Debug.Assert(data != null);

                // コンボボックスのアイテムを更新
                this.UpdateComboboxItems(animationEnabled && UserDataElementHelper.IsAnimationKind(data.UserDataKind));

                this.EditMode = true;
                _cmbKind.SelectedItemData = data.UserDataKind;

                _tbxValue.Text =
                    UserDataElementHelper.ToString(
                        data.UserDataKind,
                        data.Value,
                        false,
                        Environment.NewLine,
                        true);

                _tbxName.Value = data.Name;
                _tbxName.ReadOnly = (holder.FindOriginalData(data.Name) != null);

                this.UpdatePropertyPresetByNameChanged_();
                this.UpdateButtonState_();
            }
        }

        public void SetUserDataPresets(IEnumerable<ExUserDataPreset> userDataPresets, Func<ExUserDataPreset, bool> filterFunc)
        {
            _userDataPresets = userDataPresets;
            _presetForm.SetUserDataPresets(userDataPresets, filterFunc);
        }

        /// <summary>
        /// コンボボックスのアイテムを更新します。
        /// </summary>
        /// <param name="extUserDataAnimEnabled"></param>
        void UpdateComboboxItems(bool extUserDataAnimEnabled)
        {
            if (extUserDataAnimEnabled)
            {
                _cmbKind.Items.Clear();

                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_STRING"), UserDataKind.String);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_INT"), UserDataKind.Int);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_FLOAT"), UserDataKind.Float);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT"), UserDataKind.AnmInt);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT_VEC2"), UserDataKind.AnmIntVec2);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_ANM_COLOR_RGBA"), UserDataKind.AnmByteRGBA4);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT"), UserDataKind.AnmFloat);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC2"), UserDataKind.AnmFloatVec2);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC3"), UserDataKind.AnmFloatVec3);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_ANM_COLOR_RGBA_DEGAMMA"), UserDataKind.AnmByteRGBA4Degamma);
            }
            else
            {
                _cmbKind.Items.Clear();

                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_STRING"), UserDataKind.String);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_INT"), UserDataKind.Int);
                _cmbKind.AddItem(StringResMgr.Get("PROP_USERDATA_KIND_FLOAT"), UserDataKind.Float);
            }
        }

        /// <summary>
        /// GUIの状態を初期化します。
        /// </summary>
        void InitializeProperty_()
        {
            _tbxName.InitializeTextValidator((newText) => !LECore.AppConstants.RegexInvalidCharForUserDataExName.IsMatch(newText));

            // 初期化しておく。
            this.UpdatePropertyPreset_(new ExUserDataPreset());

            this.SetShowParameter(true, null, null);
        }

        /// <summary>
        /// GUIの状態を更新します。
        /// </summary>
        void UpdateProperty_( IUserDataElement data, bool extUserDataAnimEnabled )
        {
            // コンボボックスのアイテムを更新
            this.UpdateComboboxItems(extUserDataAnimEnabled);

            this.EditMode = true;
            _cmbKind.SelectedItemData = data.UserDataKind;

            _tbxValue.Text = UserDataElementHelper.ToString(
                    data.UserDataKind,
                    data.Value,
                    false,
                    Environment.NewLine,
                    true);

            _tbxName.Value = data.Name;

            UpdatePropertyPresetByNameChanged_();
            UpdateButtonState_();
        }

        /// <summary>
        /// OKボタンの状態を更新します。
        /// </summary>
        void UpdateButtonState_()
        {
            string valifiedStr = UserDataElementHelper.GetValifiedStringForValue( Kind, _tbxValue.Text);
            _tbxValue.Text = valifiedStr;

            bool bValueValid = UserDataElementHelper.CheckParamatersValid(
                UserDataName,
                Kind,
                Value );

            _btnOK.Enabled = bValueValid;
        }

        #region イベントハンドラ
        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOK_Click( object sender, EventArgs e )
        {
            // 同名チェックを行う
            if (EditMode)
            {
                var item = _holder.UserDataElementSet.FirstOrDefault(x => x.Name == UserDataName);
                if (item != null && item != _data)
                {
                    MessageBox.Show(string.Format(StringResMgr.Get("USERDATADLG_DATA_EXISTS_EDIT"), UserDataName), this.Text);
                    return;
                }
            }
            else if (_holder.UserDataElementSet.Any(x => x.Name == UserDataName))
            {
                MessageBox.Show(string.Format(StringResMgr.Get("USERDATADLG_DATA_EXISTS"), UserDataName), this.Text);
                return;
            }

            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click( object sender, EventArgs e )
        {
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_TbxName_TextChanged( object sender, EventArgs e )
        {
            UpdatePropertyPresetByNameChanged_();
            UpdateButtonState_();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_TbxValue_TextChanged( object sender, EventArgs e )
        {
            UpdateButtonState_();
        }

        /// <summary>
        /// データ型を変更したときのイベント処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void Event_cmbType_SelectedIndexChanged(object sender, EventArgs e)
        {
            // 設定項目の状況に応じてOKボタンの有効/無効を切り替える
            UpdateButtonState_();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_btnPreset_Click(object sender, EventArgs e)
        {
            _presetForm.Show();
            _presetForm.Location = _btnPreset.PointToScreen(new Point(0, 0));
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_lvwPreset_Click(object sender, EventArgs e)
        {
            ListView lv = sender as ListView;
            if (lv.SelectedItems.Count > 0)
            {
                ExUserDataPreset preset = (ExUserDataPreset)lv.SelectedItems[0].Tag;

                UserDataKind userDataKind = GetUserDataKind_(preset.Type);
                _cmbKind.SelectedItemData = userDataKind;

                _tbxName.Value = preset.Name;
            }

            UpdateButtonState_();

            _presetForm.Hide();
        }

        /// <summary>
        /// stringをUserDataKindに変換します。
        /// </summary>
        /// <param name="typeStr">string</param>
        private UserDataKind GetUserDataKind_(string typeStr)
        {
            UserDataKind kind = UserDataKind.None;
            bool tryResult = Enum.TryParse(typeStr, out kind);

            Debug.Assert(tryResult == true);

            return kind;
        }

        /// <summary>
        /// プリセット用UIを初期化します。
        /// </summary>
        private void UpdatePropertyPresetByNameChanged_()
        {
            if (_userDataPresets != null)
            {
                var preset = _userDataPresets.FirstOrDefault((udp) => udp.Name == _tbxName.Text);
                UpdatePropertyPreset_(preset);
            }
        }

        private string CalcEnamValueRenderLableString_(ExUserDataPreset.EnumValue enumValue)
        {
            if(enumValue == null)
            {
                return string.Empty;
            }

            string lable = enumValue.Text;
            if (!string.IsNullOrEmpty(enumValue.Descriptions))
            {
                lable = ExUserDataPreset.EnumValue.CalcFormattedString(enumValue);
            }

            return lable;
        }

        /// <summary>
        /// プリセット用UIを初期化します。
        /// </summary>
        private void UpdatePropertyPreset_(ExUserDataPreset preset)
        {
            // データがプリセットのときか、アニメーション型の場合種類を変更できないようにする
            {
                bool isPreset = string.IsNullOrEmpty(preset.Name) == false;
                bool isAnimKindRelated = _data != null && UserDataElementHelper.IsAnimationKind(_data.UserDataKind);
                if (isPreset || isAnimKindRelated)
                {
                    _grpKind.Enabled = false;
                }
                else
                {
                    _grpKind.Enabled = true;
                }
            }


            // プリセット説明文章
            if (!string.IsNullOrEmpty(preset.Descriptions))
            {
                _lblPresetDescription.Text = preset.Descriptions;
                _ttpMain.SetToolTip(_lblPresetDescription, preset.Descriptions);
                _pnlPreset.Visible = true;
            }
            else
            {
                _lblPresetDescription.Text = string.Empty;
                _ttpMain.SetToolTip(_lblPresetDescription, string.Empty);
                _pnlPreset.Visible = false;
            }

            // EnumValuesが存在する場合は、入力欄をコンボボックスに変更します。
            if (preset.EnumValues != null && preset.EnumValues.Count() > 0)
            {
                _tbxValue.Visible = false;
                _lblValue.Visible = false;
                if (preset.MultiSelect)
                {
                    _cmbEnum.Visible = false;
                    var texts = _tbxValue.Text.Split(new[] { Environment.NewLine }, StringSplitOptions.RemoveEmptyEntries);
                    _lvwEnum.Items.Clear();
                    {
                        foreach (var enumValue in preset.EnumValues)
                        {
                            var item = new ListViewItem(new[] { enumValue.Text, enumValue.Descriptions })
                            {
                                Tag = enumValue
                            };
                            UserDataKind kind = GetUserDataKind_(preset.Type);
                            item.Checked = texts.Any(x =>
                                UserDataElementHelper.IsStringValueSame(kind, enumValue.Text, x, Environment.NewLine));
                            _lvwEnum.Items.Add(item);
                        }
                    }
                    _lvwEnum.Visible = true;
                }
                else
                {
                    _lvwEnum.Visible = false;
                    _cmbEnum.Items.Clear();
                    // ドロップダウン用コンボを構築します。
                    {
                        float maxDropDownWidth = 0.0f;
                        using (var graphics = this.CreateGraphics())
                        {
                            foreach (ExUserDataPreset.EnumValue enumValue in preset.EnumValues)
                            {
                                // 改行はスペース区切りにする。
                                string lable = enumValue.Text.Replace("\n", " ");
                                _cmbEnum.AddItem(lable, enumValue);

                                maxDropDownWidth = Math.Max(maxDropDownWidth, graphics.MeasureString(CalcEnamValueRenderLableString_(enumValue), _cmbEnum.Font).Width);
                            }
                        }

                        const float margin = 4.0f;
                        _cmbEnum.DropDownWidth = (int)(maxDropDownWidth + margin);
                    }

                    UserDataKind kind = GetUserDataKind_(preset.Type);
                    _cmbEnum.SetSelectedByCondition(_tbxValue.Text, (dataL, dataR) =>
                    UserDataElementHelper.IsStringValueSame(kind, dataL.ToString(), dataR.ToString(), Environment.NewLine));

                    // 該当アイテムが無い場合は先頭のアイテムを設定する。
                    if (_cmbEnum.SelectedIndex == -1)
                    {
                        _cmbEnum.SelectedIndex = 0;
                    }
                    _tbxValue.Text = _cmbEnum.SelectedItem != null ? _cmbEnum.SelectedItem.ToString() : string.Empty;

                    _cmbEnum.Visible = true;
                }
            }
            else
            {
                _tbxValue.Visible = true;
                _lblValue.Visible = true;
                _cmbEnum.Visible = false;
                _lvwEnum.Visible = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_lvwPreset_Leave(object sender, EventArgs e)
        {
            _presetForm.Hide();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_cmbEnum_SelectionChangeCommitted(object sender, EventArgs e)
        {
            _tbxValue.Text = _cmbEnum.SelectedItem != null ? _cmbEnum.SelectedItem.ToString() : string.Empty;
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_cmbEnum_DrawItem(object sender, DrawItemEventArgs e)
        {
            e.DrawBackground();
            if (_cmbEnum.Focused)
            {
                e.DrawFocusRectangle();
            }

            if (e.Index < 0 || e.Index >= _cmbEnum.Items.Count)
            {
                return;
            }

            var item = _cmbEnum.Items[e.Index] as LayoutEditor.Controls.UIListControlItem;
            var enumValue = item.Data as ExUserDataPreset.EnumValue;

            string text = CalcEnamValueRenderLableString_(enumValue);

            e.Graphics.DrawString(text, e.Font, SystemBrushes.ControlText, e.Bounds.Left, e.Bounds.Top + e.Bounds.Height / 2 - e.Font.Height / 2);
        }
        #endregion イベントハンドラ

        private void _lvwEnum_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            _tbxValue.Text = string.Join(
                Environment.NewLine,
                _lvwEnum.CheckedItems.OfType<ListViewItem>().Select(x => ((ExUserDataPreset.EnumValue)x.Tag).Text));
        }
    }
}
