﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.TextureControls
{
    using LECore.Structures;
    using LayoutEditor.Controls;
    using src.Controls;

    public partial class TextureSRTMtxEditPanel : UserControl
    {
        /// <summary>
        /// 行列変更イベント
        /// </summary>
        public event EventHandler OnMatrixChanged;

        // イベントを通知するか？
        bool _NotifyEvent   = true;

        ITexMtx _texMtx = null;
        Color _controlColor;

        #region プロパティ
        /// <summary>
        /// スケール
        /// </summary>
        public FVec2 MtxScale
        {
            get { return _nudScaleUV.FVec2Value; }
        }

        /// <summary>
        /// 回転
        /// </summary>
        public float MtxRotate
        {
            get { return Convert.ToSingle( _nudRotate.Value ); }
        }

        /// <summary>
        /// 平行移動
        /// </summary>
        public FVec2 MtxTrans
        {
            get { return _nudTransUV.FVec2Value; }
        }

        /// <summary>
        /// 操作対象テクスチャマトリクス
        /// </summary>
        public ITexMtx TargetTexMtx
        {
            get
            {
                return _texMtx;
            }
            set
            {
                if (_texMtx != value)
                {
                    _texMtx = value;
                    UpdateLinkButton();
                }
            }
        }

        /// <summary>
        /// アニメーションマークを利用するか
        /// </summary>
        public bool UseAnimationMark
        {
            get;
            set;
        }

        /// <summary>
        /// 同時編集時の色選択方法
        /// </summary>
        public Color ControlColor
        {
            get
            {
                return _controlColor;
            }
            set
            {
                if (_controlColor != value)
                {
                    _controlColor = value;
                    UpdateControlColor();
                }
            }
        }

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureSRTMtxEditPanel()
        {
            InitializeComponent();

            // コンテキストメニューを追加する
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblScale, _nudScaleUV.AnimationMarkArray);
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblTransU, _nudTransUV.AnimationMarkArray);

            // アニメーションマークを関連付ける
            _lblRotate.BindAnimationMark(_nudRotate.AnimationMarkInst);
        }

        /// <summary>
        /// ViewManagerへのメッセージハンドラを設定します
        /// </summary>
        public void SetViewManagerMessageHandler(Action<ViewManagerMessage> handler)
        {
            _nudScaleUV.SetViewManagerMessageHandler(handler);
            _nudRotate.SetViewManagerMessageHandler(handler);
            _nudTransUV.SetViewManagerMessageHandler(handler);
        }

       #region イベントハンドラ
        /// <summary>
        /// スケール値変更
        /// </summary>
        private void _nudScaleUV_ValueChanged(object sender, FVec2EditEventArgs e)
        {
            MaterialTextureEditEventKind kind = MaterialTextureEditEventKind.None;
            if (e.Name == "X") kind = MaterialTextureEditEventKind.Scale_U;
            else if (e.Name == "Y") kind = MaterialTextureEditEventKind.Scale_V;
            else if (e.Name == "XY") kind = MaterialTextureEditEventKind.Scale_UV;

            NotifyChangeEvent_(new MaterialTextureEditEventArgs(kind));
        }

        /// <summary>
        /// 回転値変更
        /// </summary>
        private void _nudRotate_ValueChanged( object sender, EventArgs e )
        {
            NotifyChangeEvent_(new MaterialTextureEditEventArgs(MaterialTextureEditEventKind.Rotate));
        }

        /// <summary>
        /// 移動値変更
        /// </summary>
        private void _nudTransV_ValueChanged(object sender, FVec2EditEventArgs e)
        {
            MaterialTextureEditEventKind kind = MaterialTextureEditEventKind.None;
            if (e.Name == "X") kind = MaterialTextureEditEventKind.Trans_U;
            else if (e.Name == "Y") kind = MaterialTextureEditEventKind.Trans_V;
            else if (e.Name == "XY") kind = MaterialTextureEditEventKind.Trans_UV;

            NotifyChangeEvent_(new MaterialTextureEditEventArgs(kind));
        }

        /// <summary>
        /// リセットボタン
        /// </summary>
        private void Event_BtnReset_Click( object sender, EventArgs e )
        {
            Set( 1.0f, 1.0f, 0.0f, 0.0f, 0.0f );
            NotifyChangeEvent_(new MaterialTextureEditEventArgs(MaterialTextureEditEventKind.Reset));
        }
       #endregion イベントハンドラ



        /// <summary>
        /// GUIに値を設定します。
        /// 変更通知イベントは発生しません。
        /// </summary>
        public void Set( float su, float sv, float rot, float tu, float tv )
        {
            lock( this )
            {
                _NotifyEvent = false;

                _nudScaleUV.SetTargetAttribute(TargetTexMtx.ScaleIAnmAttr);
                _nudScaleUV.FVec2Value = new FVec2(su, sv);
                _nudRotate.SetTargetAttribute(TargetTexMtx.RotateIAnmAttr);
                _nudRotate.Value = (decimal)rot;
                _nudTransUV.SetTargetAttribute(TargetTexMtx.TransIAnmAttr);
                _nudTransUV.FVec2Value = new FVec2(tu, tv);
                UpdateAnimationMark();

                // 同時編集
                UpdateControlColor();

                _NotifyEvent = true;
            }
        }

        /// <summary>
        /// コントロールの色を更新します。
        /// </summary>
        private void UpdateControlColor()
        {
            _nudScaleUV.TextBoxBackColor = ControlColor;
            _nudRotate.NumericUpDownBackColor = ControlColor;
            _nudTransUV.TextBoxBackColor = ControlColor;
        }

        /// <summary>
        /// アニメーションマークを更新します。
        /// </summary>
        public void UpdateAnimationMark()
        {
            // アニメーションマークとコンテキストメニューの有効無効
            List<IAnimationMarkUsable> animCtrlList = GetAnimationMarkUsableControl();
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UseAnimationMark = this.UseAnimationMark;
            }

            // アニメーションマークの更新
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UpdateAnimationMark();
            }
        }

        /// <summary>
        /// アニメーションマークが利用可能なコントロールの一覧を取得する。
        /// </summary>
        protected List<IAnimationMarkUsable> GetAnimationMarkUsableControl()
        {
            List<IAnimationMarkUsable> list = new List<IAnimationMarkUsable>();

            list.Add(_lblScale);
            list.Add(_lblTransU);
            list.Add(_lblRotate);
            list.Add(_nudScaleUV);
            list.Add(_nudRotate);
            list.Add(_nudTransUV);

            return list;
        }

        /// <summary>
        /// 連動ボタンを更新します
        /// </summary>
        public void UpdateLinkButton()
        {
            if (TargetTexMtx == null) return;

            _nudScaleUV.UpdateLinkButton(TargetTexMtx.Scale.X == TargetTexMtx.Scale.Y);
        }

        /// <summary>
        /// 変更イベントを通知します。
        /// </summary>
        void NotifyChangeEvent_(EventArgs args)
        {
            // 抑止されていなければ...
            if( _NotifyEvent )
            {
                // 通知します。
                OnMatrixChanged( this, args );
            }
        }
    }
}
