﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages
{
    using LayoutEditor.Controls;
    using LayoutEditor.Utility;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters ;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Structures;
    using App = LayoutEditor;

    public partial class TexCombStagePanelDetailCtr : UserControl
    {
        // 編集用データ
        private readonly TevDirectStage _data = new TevDirectStage(null, 0, false);

        ///
        private static Dictionary<AttrTevOp, string>   _formatDictionary = new Dictionary<AttrTevOp, string>();

        /// <summary>データ変更イベント。</summary>
        public event EventHandler DataChanged = null;

        ///
        protected const int MaxSourceCount = 3;
        private int _stageNo = -1;

        /// <summary>
        /// スタティックコンストラクタ
        /// </summary>
        static TexCombStagePanelDetailCtr()
        {
            _formatDictionary.Add( AttrTevOp.InvAlpha,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_ALPHA_LBL"));
            _formatDictionary.Add( AttrTevOp.InvB,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_B_LBL"));
            _formatDictionary.Add( AttrTevOp.InvBbb,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_BBB_LBL"));
            _formatDictionary.Add( AttrTevOp.InvG,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_G_LBL"));
            _formatDictionary.Add( AttrTevOp.InvGgg,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_GGG_LBL"));
            _formatDictionary.Add( AttrTevOp.InvR,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_R_LBL"));
            _formatDictionary.Add( AttrTevOp.InvRrr,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_RRR_LBL"));
            _formatDictionary.Add( AttrTevOp.InvRgb,
                                   StringResMgr.Get( "PROP_MAT_TEV_STAGE_OP_INV_RGB_LBL"));
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexCombStagePanelDetailCtr()
        {
            InitializeComponent();
        }

        /// <summary>
        /// データ。
        /// </summary>
        public TevDirectStage Data
        {
            get { return _data; }
        }

        public void SetData( int stageNo, TevDirectStage data)
        {
            _stageNo = stageNo;

            // 設定をコピー
            _data.Set(data);

            // プロパティ更新
            UpdateProperty();
        }

        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        public virtual void InitializeProperty()
        {
            //コンバインモード
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_REPLACE",      AttrCombineMode.Replace, true);
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_MODULATE",     AttrCombineMode.Modulate, true);
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_ADD",          AttrCombineMode.Add, true);
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_ADD_SIGNED",   AttrCombineMode.AddSigned, true);
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_INTERPOLATE",  AttrCombineMode.Interpolate, true);
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_SUBTRACT",     AttrCombineMode.Subtract, true);
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_MULT_ADD_DMP", AttrCombineMode.MultAddDMP, true);
            cmbCombineMode.AddItem( "PROP_MAT_TEXCOMB_STAGE_ADD_MULT_DMP", AttrCombineMode.AddMultDMP, true);
            cmbCombineMode.SelectedIndexChanged += Event_CmbCombineMode_SelectedIndexChanged;

            //スケール
            cmbScale.AddItem( "PROP_MAT_TEXCOMB_STAGE_SCALE1", AttrTevScale.Scale1, true);
            cmbScale.AddItem( "PROP_MAT_TEXCOMB_STAGE_SCALE2", AttrTevScale.Scale2, true);
            cmbScale.AddItem( "PROP_MAT_TEXCOMB_STAGE_SCALE4", AttrTevScale.Scale4, true);
            cmbScale.SelectedIndexChanged += Event_CmbScale_SelectedIndexChanged;

            //
            for( int i = 0; i < MaxSourceCount; i++ )
            {
                UIComboBox cmb = GetSrcComboBox( i);
                cmb.SelectedIndexChanged += Event_CmbSrc_SelectedIndexChanged;

                cmb = GetOperandComboBox( i);
                cmb.SelectedIndexChanged += Event_CmbOperand_SelectedIndexChanged;
            }

            // バッファの入力元
            chkCopyReg.CheckedChanged += Event_chkCopyReg_CheckedChanged;
        }

        /// <summary>
        /// "ソース"コンボボックスの更新
        /// </summary>
        private void UpdateSourceComboBox()
        {
            for( int i = 0; i < MaxSourceCount; i++ )
            {
                UIComboBox cmb = GetSrcComboBox( i);

                cmb.BeginUpdate();
                cmb.Items.Clear();
                cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_TEXTURE0",  AttrTevSource.Texture0, true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_TEXTURE1",  AttrTevSource.Texture1, true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_TEXTURE2",  AttrTevSource.Texture2, true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_TEXTURE3",  AttrTevSource.Texture3, true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_CONSTANT",  AttrTevSource.Constant, true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_PRIMARY",   AttrTevSource.Primary,  true);

                if( _stageNo >= 1 )
                {
                    cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_PREVIOUS", AttrTevSource.Previous, true);
                    cmb.AddItem( "PROP_MAT_TEV_STAGE_SOURCE_REGISTER", AttrTevSource.Register, true);
                }

                cmb.EndUpdate();
            }
        }

        /// <summary>
        /// プロパティ更新。
        /// </summary>
        public virtual void UpdateProperty()
        {
            UpdateSourceComboBox();

            // バッファの入力元
            if( _stageNo >= 1 && _stageNo <= 4 ) {
                lblCopyReg.Enabled = true;
                chkCopyReg.Enabled = true;
                chkCopyReg.Checked = _data.CopyReg;
            } else {

                lblCopyReg.Enabled = false;
                chkCopyReg.Enabled = false;
                chkCopyReg.Checked = false;
            }

            //
            switch( _data.CombineMode ) {
            case AttrCombineMode.Replace:
                lblS1.Visible       = false;
                cmbSrc1.Visible     = false;
                cmbOperand1.Visible = false;
                lblS2.Visible       = false;
                cmbSrc2.Visible     = false;
                cmbOperand2.Visible = false;
                break;

            case AttrCombineMode.Modulate:
            case AttrCombineMode.Add:
            case AttrCombineMode.AddSigned:
            case AttrCombineMode.Subtract:
                lblS1.Visible       = true;
                cmbSrc1.Visible     = true;
                cmbOperand1.Visible = true;
                lblS2.Visible       = false;
                cmbSrc2.Visible     = false;
                cmbOperand2.Visible = false;
                break;

            case AttrCombineMode.Interpolate:
            case AttrCombineMode.AddMultDMP:
            case AttrCombineMode.MultAddDMP:
                lblS1.Visible       = true;
                cmbSrc1.Visible     = true;
                cmbOperand1.Visible = true;
                lblS2.Visible       = true;
                cmbSrc2.Visible     = true;
                cmbOperand2.Visible = true;
                break;
            }

            //
            cmbCombineMode.SetSelectedItemData( _data.CombineMode);
            cmbScale.SetSelectedItemData( _data.Scale);

            //
            if( _data.CombineMode == AttrCombineMode.Interpolate ) {
                //pnlExpressionInterpolate.Visible = true;
                //pnlExpression.Visible            = false;
            } else {
                //pnlExpressionInterpolate.Visible = false;
                //pnlExpression.Visible            = true;
                //pnlExpression.SetCombineMode( _data.CombineMode);
            }

            //
            UIComboBox  cmb = null;
            string      sourceText = null;
            string      opText = null;
            string      text = null;

            for( int i = 0; i < MaxSourceCount; i++ )
            {
                cmb = GetSrcComboBox( i);
                cmb.SetSelectedItemData( _data.TevArgs[ i ].Source);
                sourceText = cmb.Text;

                cmb = GetOperandComboBox( i);
                cmb.SetSelectedItemData( _data.TevArgs[ i ].Op);

                //
                if( _formatDictionary.ContainsKey( _data.TevArgs[ i ].Op) != false ) {
                    opText = _formatDictionary[ _data.TevArgs[ i ].Op ];
                    text = string.Format( opText, sourceText);
                } else {

                    opText = GetOperandComboBox( i).Text;
                    text = string.Format( "{0} {1}", sourceText, opText);
                }

                //
                //pnlExpression.SetSourceText( i, text);
                //pnlExpressionInterpolate.SetSourceText( i, text);
            }

            //
            //pnlExpression.SetScaleText( cmbScale.Text);
            //pnlExpressionInterpolate.SetScaleText( cmbScale.Text);
        }

        /// <summary>
        /// データ変更ハンドラ。
        /// </summary>
        protected virtual void OnDataChanged(EventArgs e)
        {
            if (this.DataChanged != null)
            {
                this.DataChanged(this, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected UIComboBox GetSrcComboBox( int index)
        {
            Debug.Assert( index >= 0 && index <= 2);
            UIComboBox[]  comboBoxs = new [] { cmbSrc0, cmbSrc1, cmbSrc2 };
            return comboBoxs[ index ];
        }

        /// <summary>
        ///
        /// </summary>
        protected UIComboBox GetOperandComboBox( int index)
        {
            Debug.Assert( index >= 0 && index <= 2);
            UIComboBox[]  comboBoxs = new [] { cmbOperand0, cmbOperand1, cmbOperand2 };
            return comboBoxs[ index ];
        }

        /// <summary>
        ///
        /// </summary>
        private int GetTevArgsIndexByComboBox( UIComboBox cmb)
        {
            if( cmb == cmbSrc0 || cmb == cmbOperand0 ) { return 0; }
            if( cmb == cmbSrc1 || cmb == cmbOperand1 ) { return 1; }
            if( cmb == cmbSrc2 || cmb == cmbOperand2 ) { return 2; }
            Debug.Assert( false);
            return -1;
        }

        /// <summary>
        ///
        /// </summary>
        private int GetEnabledSourceCount()
        {
            return GetEnabledSourceCount(_data.CombineMode);
        }

        public static int GetEnabledSourceCount(AttrCombineMode mode)
        {
            //
            switch( mode ) {
            case AttrCombineMode.Replace:
                return 1;

            case AttrCombineMode.Modulate:
            case AttrCombineMode.Add:
            case AttrCombineMode.AddSigned:
            case AttrCombineMode.Subtract:
                return 2;

            case AttrCombineMode.Interpolate:
            case AttrCombineMode.AddMultDMP:
            case AttrCombineMode.MultAddDMP:
                return 3;

            default:
                Debug.Assert( false);
                return 0;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowInvalidSourcesErrorDialog()
        {
            MessageBox.Show(
                            StringResMgr.Get( "PROP_MAT_TEV_MSG_INVALID_SOURCES" ),
                            StringResMgr.Get( "PROP_ERROR_DLG_TITLE" ),
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Error );
        }

        #region イベントハンドラ
        private void Event_CmbCombineMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            AttrCombineMode prevCombineMode = _data.CombineMode;
            _data.CombineMode = (AttrCombineMode)cmbCombineMode.SelectedItemData;

            // 使われないソースにソースを設定
            // ステージ0 = コンスタント、ステージ1〜6 = 前段の出力
            AttrTevSource source = _stageNo == 0 ? AttrTevSource.Constant : AttrTevSource.Previous;
            for( int index = GetEnabledSourceCount(); index < MaxSourceCount; index++ )
            {
                _data.TevArgs[ index ].Source = source;
            }

            OnDataChanged(EventArgs.Empty);
        }

        private void Event_CmbScale_SelectedIndexChanged(object sender, EventArgs e)
        {
            _data.Scale = (AttrTevScale)cmbScale.SelectedItemData;
            OnDataChanged(EventArgs.Empty);
        }

        private void Event_CmbSrc_SelectedIndexChanged(object sender, EventArgs e)
        {
            UIComboBox  cmb = (UIComboBox)sender;
            int         index = GetTevArgsIndexByComboBox( cmb);

            AttrTevSource prevSource = _data.TevArgs[ index ].Source;
            _data.TevArgs[ index ].Source = (AttrTevSource)cmb.SelectedItemData;

            OnDataChanged(EventArgs.Empty);
        }

        private void Event_CmbOperand_SelectedIndexChanged(object sender, EventArgs e)
        {
            UIComboBox  cmb = (UIComboBox)sender;
            int         index = GetTevArgsIndexByComboBox( cmb);

            _data.TevArgs[ index ].Op = (AttrTevOp)cmb.SelectedItemData;
            OnDataChanged(EventArgs.Empty);
        }

        /// バッファの入力元
        private void Event_chkCopyReg_CheckedChanged(object sender, EventArgs e)
        {
            _data.CopyReg = chkCopyReg.Checked;
            OnDataChanged(EventArgs.Empty);
        }
        #endregion
    }

    #region TevColorStagePanel
    /// <summary>
    /// カラーステージパネルクラス。
    /// </summary>
    public sealed class TexCombColorStagePanelDetailCtr : TexCombStagePanelDetailCtr
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void InitializeProperty()
        {
            base.InitializeProperty();

            for( int i = 0; i < MaxSourceCount; i++ )
            {
                UIComboBox cmb = GetOperandComboBox(i);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_RGB",       AttrTevOp.Rgb,      true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_RGB",   AttrTevOp.InvRgb,   true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_ALPHA",     AttrTevOp.Alpha,    true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_ALPHA", AttrTevOp.InvAlpha, true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_RRR",       AttrTevOp.Rrr,      true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_RRR",   AttrTevOp.InvRrr,   true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_GGG",       AttrTevOp.Ggg,      true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_GGG",   AttrTevOp.InvGgg,   true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_BBB",       AttrTevOp.Bbb,      true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_BBB",   AttrTevOp.InvBbb,   true);
            }
        }
    }
    #endregion

    #region TevColorStagePanel
    /// <summary>
    /// アルファステージパネルクラス。
    /// </summary>
    public sealed class TexCombAlphaStagePanelDetailCtr : TexCombStagePanelDetailCtr
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void InitializeProperty()
        {
            base.InitializeProperty();

            for( int i = 0; i < MaxSourceCount; i++ )
            {
                UIComboBox cmb = GetOperandComboBox( i);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_ALPHA",     AttrTevOp.Alpha,    true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_ALPHA", AttrTevOp.InvAlpha, true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_R",         AttrTevOp.R,        true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_R",     AttrTevOp.InvR,     true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_G",         AttrTevOp.G,        true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_G",     AttrTevOp.InvG,     true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_B",         AttrTevOp.B,        true);
                cmb.AddItem( "PROP_MAT_TEV_STAGE_OP_INV_B",     AttrTevOp.InvB,     true);
            }
        }
    }
    #endregion
}
