﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.TevControls
{
    using LayoutEditor.Controls;
    using LayoutEditor.Utility;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters ;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Structures;
    using App = LayoutEditor;

    /// <summary>
    /// ＴＥＶダイレクトステージパネルクラス。
    /// </summary>
    public class TevDirectStagePanel : UserControl
    {
        #region デザイナ変数

        private System.Windows.Forms.Panel pnlMode;
        private App.Controls.UIRadioButton radModeCalc;
        private App.Controls.UIRadioButton radModeComp;
        private System.Windows.Forms.Label labKSel;
        private App.Controls.UIComboBox cmbKSel;
        private System.Windows.Forms.Label labOutReg;
        private App.Controls.UIComboBox cmbOutReg;
        private App.Controls.BorderPanel pnlStage;
        private App.Controls.UICheckBox chkClamp;
        private System.ComponentModel.Container components = null;
        #endregion

        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager( typeof( TevDirectStagePanel ) );
            this.pnlStage = new LayoutEditor.Controls.BorderPanel();
            this.pnlMode = new System.Windows.Forms.Panel();
            this.radModeCalc = new LayoutEditor.Controls.UIRadioButton();
            this.radModeComp = new LayoutEditor.Controls.UIRadioButton();
            this.labKSel = new System.Windows.Forms.Label();
            this.cmbKSel = new LayoutEditor.Controls.UIComboBox();
            this.labOutReg = new System.Windows.Forms.Label();
            this.cmbOutReg = new LayoutEditor.Controls.UIComboBox();
            this.chkClamp = new LayoutEditor.Controls.UICheckBox();
            this.pnlMode.SuspendLayout();
            this.SuspendLayout();
            //
            // pnlStage
            //
            resources.ApplyResources( this.pnlStage, "pnlStage" );
            this.pnlStage.Name = "pnlStage";
            //
            // pnlMode
            //
            this.pnlMode.Controls.Add( this.radModeCalc );
            this.pnlMode.Controls.Add( this.radModeComp );
            resources.ApplyResources( this.pnlMode, "pnlMode" );
            this.pnlMode.Name = "pnlMode";
            //
            // radModeCalc
            //
            resources.ApplyResources( this.radModeCalc, "radModeCalc" );
            this.radModeCalc.Name = "radModeCalc";
            //
            // radModeComp
            //
            resources.ApplyResources( this.radModeComp, "radModeComp" );
            this.radModeComp.Name = "radModeComp";
            //
            // labKSel
            //
            resources.ApplyResources( this.labKSel, "labKSel" );
            this.labKSel.Name = "labKSel";
            //
            // cmbKSel
            //
            resources.ApplyResources( this.cmbKSel, "cmbKSel" );
            this.cmbKSel.Name = "cmbKSel";
            this.cmbKSel.SelectedItemData = null;
            //
            // labOutReg
            //
            resources.ApplyResources( this.labOutReg, "labOutReg" );
            this.labOutReg.Name = "labOutReg";
            //
            // cmbOutReg
            //
            resources.ApplyResources( this.cmbOutReg, "cmbOutReg" );
            this.cmbOutReg.Name = "cmbOutReg";
            this.cmbOutReg.SelectedItemData = null;
            //
            // chkClamp
            //
            resources.ApplyResources( this.chkClamp, "chkClamp" );
            this.chkClamp.Name = "chkClamp";
            //
            // TevDirectStagePanel
            //
            this.Controls.Add( this.chkClamp );
            this.Controls.Add( this.cmbOutReg );
            this.Controls.Add( this.labOutReg );
            this.Controls.Add( this.cmbKSel );
            this.Controls.Add( this.labKSel );
            this.Controls.Add( this.pnlMode );
            this.Controls.Add( this.pnlStage );
            resources.ApplyResources( this, "$this" );
            this.Name = "TevDirectStagePanel";
            this.pnlMode.ResumeLayout( false );
            this.ResumeLayout( false );

        }
        #endregion

        // ステージパネル
        private MaterialPropertyTevDirectStageCalcPanel _calcPanel = null;
        private TevCompStagePanel _compPanel = null;

        // 編集用データ
        private readonly TevDirectStage _data = new TevDirectStage(false);

        // 切り替え用保存値
        private AttrTevOp    _tempCalcOp    = AttrTevOp.Add;
        private AttrTevBias  _tempCalcBias  = AttrTevBias.Zero;
        private AttrTevScale _tempCalcScale = AttrTevScale.Scale1;
        private AttrTevOp    _tempCompOp    = AttrTevOp.CompR8Gt;

        /// <summary>データ変更イベント。</summary>
        public event EventHandler DataChanged = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevDirectStagePanel()
        {
            InitializeComponent();
        }

        /// <summary>
        /// データ。
        /// </summary>
        public TevDirectStage Data
        {
            get { return _data; }
            set
            {
                // 設定をコピー
                _data.Set(value);

                // 設定保存値初期化
                if (_data.IsCompareMode)
                {
                    _tempCalcOp    = AttrTevOp.Add;
                    _tempCalcBias  = AttrTevBias.Zero;
                    _tempCalcScale = AttrTevScale.Scale1;
                    _tempCompOp    = _data.Op;
                }
                else
                {
                    _tempCalcOp    = _data.Op;
                    _tempCalcBias  = _data.Bias;
                    _tempCalcScale = _data.Scale;
                    _tempCompOp    = AttrTevOp.CompR8Gt;
                }

                // プロパティ更新
                UpdateProperty();
            }
        }

        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        public virtual void InitializeProperty()
        {
            // 設定モード
            // タグは TevDirectStage.IsCompareMode に対応させる
            radModeCalc.Tag = false;
            radModeComp.Tag = true;
            radModeCalc.RadioChecked += new EventHandler(Event_RadMode_RadioChecked);
            radModeComp.RadioChecked += new EventHandler(Event_RadMode_RadioChecked);

            // ステージパネル
            _calcPanel.InitializeProperty();
            _compPanel.InitializeProperty();
            _calcPanel.DataChanged += new EventHandler(Event_CalcPanel_DataChanged);
            _compPanel.DataChanged += new EventHandler(Event_CompPanel_DataChanged);

            // 定数選択
            // 項目作成は派生クラスで
            cmbKSel.SelectedIndexChanged += new EventHandler(Event_CmbKSel_SelectedIndexChanged);

            // クランプフラグ
            chkClamp.CheckedChanged += new EventHandler(Event_ChkClamp_CheckedChanged);

            // 出力レジスタ
            cmbOutReg.AddItem("PROP_MAT_TEV_STAGE_OUTREG_C0", AttrTevOutReg.Reg0, true);
            cmbOutReg.AddItem("PROP_MAT_TEV_STAGE_OUTREG_C1", AttrTevOutReg.Reg1, true);
            cmbOutReg.AddItem("PROP_MAT_TEV_STAGE_OUTREG_C2", AttrTevOutReg.Reg2, true);
            cmbOutReg.AddItem("PROP_MAT_TEV_STAGE_OUTREG_C3", AttrTevOutReg.Prev, true);
            cmbOutReg.SelectedIndexChanged += new EventHandler(Event_CmbOutReg_SelectedIndexChanged);
        }

        /// <summary>
        /// プロパティ更新。
        /// </summary>
        public void UpdateProperty()
        {
            // 設定モード
            radModeCalc.SetCheckedByTag(_data.IsCompareMode);
            radModeComp.SetCheckedByTag(_data.IsCompareMode);

            // ステージパネル
            if (_data.IsCompareMode)
            {
                _calcPanel.Visible = false;
                _compPanel.Visible = true;
                _compPanel.UpdateProperty();
            }
            else
            {
                _compPanel.Visible = false;
                _calcPanel.Visible = true;
                _calcPanel.UpdateProperty();
            }

            // 定数選択
            cmbKSel.SetSelectedItemData(_data.KSel);

            // クランプフラグ
            chkClamp.SetChecked(_data.Clamp);

            // 出力レジスタ
            cmbOutReg.SetSelectedItemData(_data.OutReg);
        }

        /// <summary>
        /// ステージパネル登録。
        /// </summary>
        protected void RegisterStagePanel(MaterialPropertyTevDirectStageCalcPanel calcPanel, TevCompStagePanel compPanel)
        {
            _calcPanel         = calcPanel;
            _calcPanel.Parent  = pnlStage;
            _calcPanel.Dock    = DockStyle.Fill;
            _calcPanel.Visible = false;
            _calcPanel.RefData = _data;

            _compPanel         = compPanel;
            _compPanel.Parent  = pnlStage;
            _compPanel.Dock    = DockStyle.Fill;
            _compPanel.Visible = false;
            _compPanel.RefData = _data;
        }

        /// <summary>
        /// 定数選択項目追加。
        /// </summary>
        protected void AddKSelItem(string stringID, AttrTevKSel val)
        {
            cmbKSel.AddItem(stringID, val, true);
        }

        /// <summary>
        /// データ変更ハンドラ。
        /// </summary>
        protected virtual void OnDataChanged(EventArgs e)
        {
            if (this.DataChanged != null)
            {
                this.DataChanged(this, e);
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }
        #endregion

        #region イベントハンドラ
        private void Event_RadMode_RadioChecked(object sender, EventArgs e)
        {
            bool compareMode = (bool)(((UIRadioButton)sender).Tag);

            // コンペアモード
            if (compareMode)
            {
                // 計算式の設定を保存
                _tempCalcOp    = _data.Op;
                _tempCalcBias  = _data.Bias;
                _tempCalcScale = _data.Scale;

                // コンペアモード用の設定に変更
                _data.Op    = _tempCompOp;
                _data.Bias  = AttrTevBias.Zero;
                _data.Scale = AttrTevScale.Scale1;

                // 表示切り替え
                _calcPanel.Visible = false;
                _compPanel.Visible = true;
                _compPanel.UpdateProperty();
            }
            // 計算式
            else
            {
                // コンペアモードの設定を保存
                _tempCompOp = _data.Op;

                // 計算式用の設定に変更
                _data.Op    = _tempCalcOp;
                _data.Bias  = _tempCalcBias;
                _data.Scale = _tempCalcScale;

                // 表示切り替え
                _compPanel.Visible = false;
                _calcPanel.Visible = true;
                _calcPanel.UpdateProperty();
            }

            OnDataChanged(EventArgs.Empty);
        }

        private void Event_CalcPanel_DataChanged(object sender, EventArgs e)
        {
            // 外に伝えるだけ
            OnDataChanged(EventArgs.Empty);
        }

        private void Event_CompPanel_DataChanged(object sender, EventArgs e)
        {
            // 外に伝えるだけ
            OnDataChanged(EventArgs.Empty);
        }

        private void Event_CmbKSel_SelectedIndexChanged(object sender, EventArgs e)
        {
            _data.KSel = (AttrTevKSel)cmbKSel.SelectedItemData;
            OnDataChanged(EventArgs.Empty);
        }

        private void Event_ChkClamp_CheckedChanged(object sender, EventArgs e)
        {
            _data.Clamp = chkClamp.Checked;
            OnDataChanged(EventArgs.Empty);
        }

        private void Event_CmbOutReg_SelectedIndexChanged(object sender, EventArgs e)
        {
            _data.OutReg = (AttrTevOutReg)cmbOutReg.SelectedItemData;
            OnDataChanged(EventArgs.Empty);
        }

        // 引数項目
        private static void Event_StageArgs_DrawItem(object sender, DrawItemEventArgs e)
        {
            e.DrawBackground();

            if (e.Index >= 0)
            {
                string text  = ((ComboBox)sender).Items[e.Index].ToString();
                string title = null;
                string desc  = null;

                // 分割用制御文字がある場合
                int splitIndex = text.LastIndexOf(';');
                if (splitIndex != -1)
                {
                    // 制御文字の左側（タイトル）と右側（説明）に分割
                    title = text.Substring(0, splitIndex);
                    desc  = ": " + text.Substring(splitIndex + 1, text.Length - (splitIndex + 1));
                }
                else
                {
                    title = text;
                }

                // 状態
                bool comboEdit = (e.State & DrawItemState.ComboBoxEdit) != 0;
                bool selected  = (e.State & DrawItemState.Selected)     != 0;

                // 書式を２種類用意
                StringFormat sf1 = new StringFormat();
                StringFormat sf2 = new StringFormat();
                sf1.Alignment     = StringAlignment.Center;
                sf1.LineAlignment = StringAlignment.Center;
                sf2.Alignment     = StringAlignment.Near;
                sf2.LineAlignment = StringAlignment.Center;
                sf2.FormatFlags  |= StringFormatFlags.NoWrap;

                // ブラシ
                Brush brush;
                if (selected) { brush = SystemBrushes.HighlightText; }
                else          { brush = SystemBrushes.WindowText;    }

                // 描画
                using (sf1)
                using (sf2)
                {
                    // コンボボックス項目
                    if (comboEdit)
                    {
                        e.Graphics.DrawString(title, e.Font, brush, e.Bounds, sf1);
                    }
                    // ドロップダウン項目
                    else
                    {
                        // 解説あり
                        if (desc != null)
                        {
                            // タイトル
                            Rectangle rcTitle = e.Bounds;
                            rcTitle.Width = ((ComboBox)sender).Width;
                            e.Graphics.DrawString(title, e.Font, brush, rcTitle, sf1);

                            // 解説
                            Rectangle rcDesc = e.Bounds;
                            rcDesc.X     += rcTitle.Width;
                            rcDesc.Width -= rcTitle.Width;
                            e.Graphics.DrawString(desc, e.Font, brush, rcDesc, sf2);
                        }
                        // 解説なし
                        else
                        {
                            e.Graphics.DrawString(title, e.Font, brush, e.Bounds, sf1);
                        }
                    }
                }
            }

            e.DrawFocusRectangle();
        }
        #endregion
    }

    #region TevColorStagePanel
    /// <summary>
    /// ＴＥＶカラーステージパネルクラス。
    /// </summary>
    public sealed class TevColorStagePanel : TevDirectStagePanel
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevColorStagePanel()
        {
            // ステージパネル登録
            RegisterStagePanel(
                new TevColorCalcStagePanel(),
                new TevColorCompStagePanel()
            );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void InitializeProperty()
        {
            base.InitializeProperty();

            // 定数選択項目
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0",  AttrTevKSel.K0);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0R", AttrTevKSel.K0R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0G", AttrTevKSel.K0G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0B", AttrTevKSel.K0B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0A", AttrTevKSel.K0A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1",  AttrTevKSel.K1);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1R", AttrTevKSel.K1R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1G", AttrTevKSel.K1G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1B", AttrTevKSel.K1B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1A", AttrTevKSel.K1A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2",  AttrTevKSel.K2);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2R", AttrTevKSel.K2R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2G", AttrTevKSel.K2G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2B", AttrTevKSel.K2B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2A", AttrTevKSel.K2A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3",  AttrTevKSel.K3);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3R", AttrTevKSel.K3R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3G", AttrTevKSel.K3G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3B", AttrTevKSel.K3B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3A", AttrTevKSel.K3A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_8_8", AttrTevKSel.C8_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_7_8", AttrTevKSel.C7_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_6_8", AttrTevKSel.C6_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_5_8", AttrTevKSel.C5_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_4_8", AttrTevKSel.C4_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_3_8", AttrTevKSel.C3_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_2_8", AttrTevKSel.C2_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_1_8", AttrTevKSel.C1_8);
        }
    }
    #endregion

    #region TevAlphaStagePanel
    /// <summary>
    /// ＴＥＶアルファステージパネルクラス。
    /// </summary>
    public sealed class TevAlphaStagePanel : TevDirectStagePanel
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevAlphaStagePanel()
        {
            // ステージパネル登録
            RegisterStagePanel(
                new TevAlphaCalcStagePanel(),
                new TevAlphaCompStagePanel()
            );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void InitializeProperty()
        {
            base.InitializeProperty();

            // 定数選択項目
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0R", AttrTevKSel.K0R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0G", AttrTevKSel.K0G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0B", AttrTevKSel.K0B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K0A", AttrTevKSel.K0A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1R", AttrTevKSel.K1R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1G", AttrTevKSel.K1G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1B", AttrTevKSel.K1B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K1A", AttrTevKSel.K1A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2R", AttrTevKSel.K2R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2G", AttrTevKSel.K2G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2B", AttrTevKSel.K2B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K2A", AttrTevKSel.K2A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3R", AttrTevKSel.K3R);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3G", AttrTevKSel.K3G);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3B", AttrTevKSel.K3B);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_K3A", AttrTevKSel.K3A);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_8_8", AttrTevKSel.C8_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_7_8", AttrTevKSel.C7_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_6_8", AttrTevKSel.C6_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_5_8", AttrTevKSel.C5_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_4_8", AttrTevKSel.C4_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_3_8", AttrTevKSel.C3_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_2_8", AttrTevKSel.C2_8);
            AddKSelItem("PROP_MAT_TEV_STAGE_KSEL_1_8", AttrTevKSel.C1_8);
        }
    }
    #endregion
}
