﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using LayoutEditor.Controls;
    using LayoutEditor.Utility;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
    using LECore.Structures;
    using LECore.Structures.Nsrif.Attributes;

    using App = LayoutEditor;

    public partial class MaterialTexCombStagePage : MaterialPage
    {
        // ステージ選択ボタン管理
        private readonly UIRadioButton[] _stageButtons = new UIRadioButton[ 2 ];

        // ステージ番号
        private int _stageNo = 0;

        // 前回選択されたGUIオブジェクトグループ
        private PaneGuiAdapterGroup _oldTarget = null;

        // 前回選択されたTevData
        private TevData _oldTevData = null;

        /// <summary>データ変更イベント。</summary>
        public event EventHandler DataChanged = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialTexCombStagePage()
        {
            InitializeComponent();

            // ステージ選択ボタンを配列管理する
            _stageButtons[0]  = radStage0;
            _stageButtons[1]  = radStage1;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override PropertyPageID PageID
        {
            get { return PropertyPageID.MaterialTevStage; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void InitializeProperty()
        {
            // ステージ選択
            for (int i = 0; i < _stageButtons.Length; i++)
            {
                // On/Offが取りたいのでCheckedChangedOnは使用しない
                _stageButtons[i].Tag = i;
                _stageButtons[i].CheckedChanged += Event_BtnStage_CheckedChanged;
            }

            // ステージ設定
            colorStagePanel.InitializeProperty();
            alphaStagePanel.InitializeProperty();
            colorStagePanel.DataChanged += Event_ColorPanel_DataChanged;
            alphaStagePanel.DataChanged += Event_AlphaPanel_DataChanged;
            colorStagePanel.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
            alphaStagePanel.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);

            DoUpdateProperty_();

            _btnChangeMaterialMode.Visible = LECore.LayoutEditorCore.PlatformDetail.LowLevelCombinerSettingsEnabled;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void DoUpdateProperty_()
        {
            colorStagePanel.UseAnimationMark = this.CanUseAnimationMark(this._OwnerPane?.OwnerSubScene);
            alphaStagePanel.UseAnimationMark = this.CanUseAnimationMark(this._OwnerPane?.OwnerSubScene);

            TevStages stages = this.ActiveTarget.TevData.Stages;

            // ステージ番号調整
            int stageNo = stages.NumStages < 1 ? -1 : Math.Max(0, Math.Min(_stageNo, stages.NumStages - 1));
            _stageNo = stageNo;

            // ステージ番号
            for (int i = 0; i < _stageButtons.Length; i++)
            {
                UIRadioButton button = _stageButtons[i];

                button.SetCheckedByTag(stageNo);
                button.Enabled = i < stages.NumStages;

                if (i < stages.NumStages)
                {
                    button.Font = LayoutEditor.AppMain.GuiFontBold;
                    button.ForeColor = button.Checked ? Color.Red : SystemColors.ControlText;
                }
                else
                {
                    button.Font      = this.Font;
                    button.ForeColor = SystemColors.ControlText;
                }
            }

            // ステージ設定
            _lblIndirectWarning.Visible = false;
            if (stageNo >= 0)
            {
                TevStage stage = stages[stageNo];
                colorStagePanel.SetData(stageNo, stage.ColorStage, this.ActiveTarget.CoreMaterial);
                alphaStagePanel.SetData(stageNo, stage.AlphaStage, this.ActiveTarget.CoreMaterial);

                if (stage.ColorStage.CombineMode == AttrCombineMode.BlendIndirect ||
                    stage.ColorStage.CombineMode == AttrCombineMode.Indirect ||
                    stage.ColorStage.CombineMode == AttrCombineMode.EachIndirect)
                {
                    _lblIndirectWarning.Visible = true;
                }
            }

            colorStagePanel.Enabled = stageNo >= 0;
            alphaStagePanel.Enabled = stageNo >= 0;
            _lblMessage.Visible = stages.NumStages <= 0;

            this.Enabled = this.ActiveTarget.IsTextureStageEditable;
            this.grpStage.Enabled = this.Enabled;
            this.grpAlpha.Enabled = this.Enabled;
            this._btnChangeMaterialMode.Enabled = this.Enabled;

            this._lblNotice.Visible = !this.ActiveTarget.IsTextureStageEditable;

            // 連動ボタンの更新
            if (_oldTarget != Target ||
                _oldTevData != this.ActiveTarget.TevData)
            {
                _oldTarget = Target;
                _oldTevData = this.ActiveTarget.TevData;
                colorStagePanel.UpdateLinkButton();
                alphaStagePanel.UpdateLinkButton();
            }
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy(){ return true; }

        /// <summary>
        /// コピー。
        /// </summary>
        public override Object Copy()
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static Object Copy(MaterialGUIAdapter target)
        {
            return new TevStages(target.TevData.Stages);
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(Object pasteObject)
        {
            Paste(ActiveTarget, pasteObject);
            using (new GUIUpdating(this))
            {
                UpdateProperty();
            }
            NotifyPropertyChanged();
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static void Paste(MaterialGUIAdapter destination, Object pasteObject)
        {
            Debug.Assert(pasteObject is TevStages);
            TevStages source = (TevStages)pasteObject;
            destination.TevData.Stages.Set(source);
        }
        #endregion

        #region ターゲット取得ヘルパ
        /// <summary>
        /// ターゲットステージ。
        /// </summary>
        private TevStage TargetStage
        {
            get
            {
                Debug.Assert(_stageNo != -1);
                return this.ActiveTarget.TevData.Stages[_stageNo];
            }
        }
        #endregion

        /// <summary>
        /// データ変更ハンドラ。
        /// </summary>
        protected virtual void OnDataChanged(EventArgs e)
        {
            if (this.DataChanged != null)
            {
                this.DataChanged(this, e);
            }
        }

        /// <summary>
        /// ステージ変更
        /// </summary>
        private void SetTargetTextureNo_(int index)
        {
            _stageNo = index;
        }

        #region イベントハンドラ

        private void Event_ColorPanel_DataChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.TargetStage.ColorStage.Set( colorStagePanel.Data);
            this.ActiveTarget.TevData.Stages.SetIndirectMatrixAll(this.TargetStage.ColorStage);
            NotifyPropertyChanged();
        }

        private void Event_AlphaPanel_DataChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.TargetStage.AlphaStage.Set(alphaStagePanel.Data);
            NotifyPropertyChanged();
        }

        // ステージ選択
        private void Event_BtnStage_CheckedChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            UIRadioButton button = (UIRadioButton)sender;

            if (button.Checked)
            {
                button.Font      = LayoutEditor.AppMain.GuiFontBold;
                button.ForeColor = Color.Red;

                // ステージ変更
                SetTargetTextureNo_((int)button.Tag);

                using (new GUIUpdating(this))
                {
                    UpdateProperty();
                }
            }
            else
            {
                button.Font      = this.Font;
                button.ForeColor = SystemColors.ControlText;
            }
        }

        private void Event_btnChangeMaterialMode_Click(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.ActiveTarget.LowLevelCombinerSettingsEnabled = true;

            if (this.ActiveTarget.TevData.Stages.NumStages <= 0)
            {
                int maxTevStage = LECore.LayoutEditorCore.PlatformDetail.MaxTevStageCount;
                this.ActiveTarget.TevData.Stages.NumStages = Math.Min(maxTevStage, Math.Max(1, this.ActiveTarget.TevData.Stages.NumStages));
            }

            NotifyPropertyChanged();
        }
        #endregion
    }
}
