﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.FragmentOperationControls
{
    using LayoutEditor.Controls;
    using LECore.Structures;
    using LECore.Structures.Nsrif.Attributes;

    public partial class FragmentOperationBlendPanel : UserControl
    {
        // ブレンドパネル
        private readonly FragmentOperationCalcBlendPanel  _blendPanel;
        private readonly FragmentOperationLogicBlendPanel _logicPanel;

        private bool _isAlpha;

        // 編集用データ
        private readonly PEBlend _data = new PEBlend();

        /// <summary>データ変更イベント。</summary>
        public event EventHandler DataChanged = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FragmentOperationBlendPanel()
        {
            InitializeComponent();

            // パネル作成
            _blendPanel         = new FragmentOperationCalcBlendPanel();
            _blendPanel.Parent  = pnlBlend;
            _blendPanel.Dock    = DockStyle.Fill;
            _blendPanel.Visible = false;
            _blendPanel.RefData = _data;

            _logicPanel         = new FragmentOperationLogicBlendPanel();
            _logicPanel.Parent  = pnlBlend;
            _logicPanel.Dock    = DockStyle.Fill;
            _logicPanel.Visible = false;
            _logicPanel.RefData = _data;
        }

        /// <summary>
        /// データ。
        /// </summary>
        public PEBlend Data
        {
            get { return _data; }
            set
            {
                // 設定をコピー
                _data.Set(value);

                // プロパティ更新
                UpdateProperty();
            }
        }

        /// <summary>
        /// アルファかどうか
        /// </summary>
        public bool IsAlpha
        {
            get { return this._isAlpha; }
            set
            {
                this._isAlpha = value;
            }
        }

        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        public void InitializeProperty()
        {
            // モード
            radModeNone.Tag  = AttrBlendType.None;
            radModeBlend.Tag = AttrBlendType.Blend;
            radModeLogic.Tag = AttrBlendType.Logic;

            radModeNone.RadioChecked  += Event_RadMode_RadioChecked;
            radModeBlend.RadioChecked += Event_RadMode_RadioChecked;
            radModeLogic.RadioChecked += Event_RadMode_RadioChecked;

            // ブレンドパネル
            _blendPanel.InitializeProperty();
            _logicPanel.InitializeProperty();
            _blendPanel.DataChanged += Event_BlendPanel_DataChanged;
            _logicPanel.DataChanged += Event_LogicPanel_DataChanged;
        }

        /// <summary>
        /// プロパティ更新。
        /// </summary>
        public void UpdateProperty()
        {
            // ラベル
            this.radModeNone.Text = this.IsAlpha ?
                StringResMgr.Get("PROP_MAT_PE_BF_NONELABEL_ALPHA") : StringResMgr.Get("PROP_MAT_PE_BF_NONELABEL");

            // モード
            radModeNone.SetCheckedByTag(_data.Type);
            radModeBlend.SetCheckedByTag(_data.Type);
            radModeLogic.SetCheckedByTag(_data.Type);

            // 論理演算の表示
            radModeLogic.Enabled = !this.IsAlpha;
            radModeLogic.Visible = !this.IsAlpha;

            // ブレンドパネル
            switch (_data.Type)
            {
            case AttrBlendType.Blend:
                _logicPanel.Visible = false;
                _blendPanel.Visible = true;
                _blendPanel.UpdateProperty();
                break;
            case AttrBlendType.Logic:
                _blendPanel.Visible = false;
                _logicPanel.Visible = true;
                _logicPanel.UpdateProperty();
                break;
            default:
                _blendPanel.Visible = false;
                _logicPanel.Visible = false;
                break;
            }
        }

        /// <summary>
        /// データ変更ハンドラ。
        /// </summary>
        private void OnDataChanged(EventArgs e)
        {
            if (this.DataChanged != null)
            {
                this.DataChanged(this, e);
            }
        }

        #region イベントハンドラ
        // モード
        private void Event_RadMode_RadioChecked(object sender, EventArgs e)
        {
            _data.Type = (AttrBlendType)((UIRadioButton)sender).Tag;
            UpdateProperty();
            OnDataChanged(EventArgs.Empty);
        }

        // ブレンド計算
        private void Event_BlendPanel_DataChanged(object sender, EventArgs e)
        {
            // 外に伝えるだけ
            OnDataChanged(EventArgs.Empty);
        }

        // 論理演算
        private void Event_LogicPanel_DataChanged(object sender, EventArgs e)
        {
            // 外に伝えるだけ
            OnDataChanged(EventArgs.Empty);
        }
        #endregion
    }
}
