﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using LayoutEditor.Controls;
using LayoutEditor.Structures.SerializableObject;
using LECore.Manipulator;
using LECore.Structures;
using LECore.Structures.Core;
using LayoutEditor.Forms.Dialogs;
using LECore;
using LayoutEditor.Utility;
using LECore.Save_Load;
using LayoutEditor.Controls.UI;
using LayoutEditor.src.Forms.Dialogs;
using System.Text.RegularExpressions;
using System.Xml;
using System.Text;
using System.Runtime.InteropServices;
using LayoutEditor.Forms.ToolWindows.common;
using System.Threading.Tasks;
using System.Drawing.Imaging;

namespace LayoutEditor.Forms.ToolWindows.PartsPalletWindow
{
    /// <summary>
    /// 部品パレットです。
    /// </summary>
    public partial class PartsPalletWindow : LEToolWindow
    {
        private Image _checkImage;
        private bool _toggleVisiblePartsOnly = false;

        /// <summary>
        /// 検索文字列
        /// </summary>
        private string _searchedString = String.Empty;
        /// <summary>
        /// 検索カテゴリ名(「全て」)
        /// </summary>
        private string _searchCategoryAll = String.Empty;

        /// <summary>
        /// サムネール
        /// </summary>
        /// <remarks>
        /// 値が null はサムネール取得予定を示す
        /// </remarks>
        private Dictionary<IPartsSubScene, Image[]> _thumbnailImages = new Dictionary<IPartsSubScene, Image[]>();

        /// <summary>
        /// ImageList を更新する必要があるか？
        /// </summary>
        private bool _thumbnailImagesDirtyFlag = true;

        /// <summary>
        /// サムネールのサイズの種類
        /// </summary>
        private enum ThumbnailSizeType
        {
            // 15 は Detail 表示で SmallImageList を指定しなかったときに相当
            Smalleset,
            Small,
            Middle,
            Large,
            Largest,
        }

        /// <summary>
        /// 種類ごとのそれぞれの高さ
        /// </summary>
        private int ToInt(ThumbnailSizeType type)
        {
            switch (type)
            {
                case ThumbnailSizeType.Smalleset:
                    // 15 はサムネール対応以前の高さに相当
                    return 15;
                case ThumbnailSizeType.Small:
                    return 32;
                case ThumbnailSizeType.Middle:
                    return 48;
                case ThumbnailSizeType.Large:
                    return 64;
                case ThumbnailSizeType.Largest:
                    return 128;
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// 現在のサムネールサイズの種類
        /// </summary>
        private ThumbnailSizeType _thumbnailSizeType = ThumbnailSizeType.Smalleset;

        private bool _showPreview = false;

        private static Image _bmpPreviewOn = LayoutEditor.Properties.Resources.ThumbnailPreview_On;
        private static Image _bmpPreviewOff = LayoutEditor.Properties.Resources.ThumbnailPreview_Off;

        /// <summary>
        /// 現在のサムネールのサイズ
        /// </summary>
        private Size _thumbnailSize => new Size(ToInt(_thumbnailSizeType), ToInt(_thumbnailSizeType));

        /// <summary>
        /// ToolTipText の初期のテキスト
        /// </summary>
        private string _tsbThumbnailSizeInitialToolTipText;

        /// <summary>
        /// SmallImageList 用のダミーの画像
        /// </summary>
        private Bitmap _dummySmallImage = new Bitmap(1, 1);

        private Bitmap[] _transparentImages;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsPalletWindow()
        {
            InitializeComponent();

            _tsbThumbnailSizeInitialToolTipText = _tsbThumbnailSize.ToolTipText;

            // メニューにタグを設定
            foreach (var item in _tsbThumbnailSize.DropDownItems.OfType<ToolStripMenuItem>().Reverse()
                .Zip(Enum.GetValues(typeof(ThumbnailSizeType)).OfType<ThumbnailSizeType>(),
                     (x, y) => new KeyValuePair<ToolStripMenuItem, ThumbnailSizeType>(x,y)))
            {
                item.Key.Tag = item.Value;
            }

            // リストビューの項目の高さを設定
            _lvwPartsFiles.LargeImageList = new ImageList()
            {
                ImageSize = new Size(ToInt(ThumbnailSizeType.Largest), ToInt(ThumbnailSizeType.Largest)),
                ColorDepth = ColorDepth.Depth32Bit,
            };

            // 画像の表示には使わない
            _lvwPartsFiles.SmallImageList = new ImageList()
            {
                ImageSize = _thumbnailSize,
                ColorDepth = ColorDepth.Depth4Bit,
            };

            _transparentImages = Enum.GetValues(typeof(ThumbnailSizeType))
                .OfType<ThumbnailSizeType>()
                .Where(x => x != ThumbnailSizeType.Largest)
                .Select(x => new Bitmap(ToInt(x), ToInt(x))).ToArray();

            // サムネールは UI スレッド以外で作成されるので、Handle が作成されたときに情報を更新するようにする。
            HandleCreated += (s, e) => BeginInvoke((Action)OnThumbnailChanged);
            VisibleChanged += (s, e) => _UpdatePreviewVisible();

            InitializeDlgState_();

            // メッセージフィルタの設定
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_tspMain);
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_tspSearch);
        }

        /// <summary>
        /// 現在のサブシーンです。
        /// </summary>
        private ISubScene CurrentSubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        #region LEToolWindow メンバ

        //----------------------------------------------------------
        // LEToolWindow メンバ
        //----------------------------------------------------------

        /// <summary>
        /// シーン変更イベントハンドラ
        /// </summary>
        public override void OnSceneModifyHandler(object sender, SceneModifyEventArgs e)
        {
            switch (e.Target)
            {
                // カレントサブシーンの更新
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                    {
                        break;
                    }
            }

            base.OnSceneModifyHandler(sender, e);
        }

        /// <summary>
        /// 可視状態を変更できるかどうかです。
        /// </summary>
        public override bool AllowToChangeVisible { get { return true; } }

        /// <summary>
        /// カスタムショートカットキーです。
        /// </summary>
        public override Keys CustomShortcut { get { return Keys.Control | Keys.F11; } }

        //----------------------------------------------------------
        // データ保存
        //----------------------------------------------------------

        /// <summary>
        /// 設定書き出し
        /// </summary>
        public sealed override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            // リスト行の幅の書き出し
            setting.SaveColumnPotisionSettings(_lvwPartsFiles);
            _cmbPartsRootHistory.RemoveByCondition((item) => !Directory.Exists(item.ToString()));

            if (option.AlsoSaveOtherThanWorkspace)
            {
                setting.SaveComboboxStrings(_cmbPartsRootHistory);

                // 部品レイアウトフィルタボタンの状態保存
                setting.AddUserData("VisiblePartsOnly", _toggleVisiblePartsOnly.ToString());

                // プレビュー
                setting.AddUserData("ShowThumbnailPictureBox", _showPreview.ToString());

                // サムネールサイズ
                setting.AddUserData("ThumbnailSize", _thumbnailSizeType.ToString());
            }

            base.SaveSetting(setting, option);
        }

        /// <summary>
        /// 設定読み込み
        /// </summary>
        public sealed override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            // リスト行の幅の読み込み
            setting.LoadColumnPotisionSettings(_lvwPartsFiles);

            if (option.AlsoLoadOtherThanWorkspace)
            {
                setting.LoadComboboxStrings(_cmbPartsRootHistory);

                // 部品レイアウトフィルタボタンの状態読み込み
                bool result;
                if (setting.TryToFindUserBoolDataByName("VisiblePartsOnly", out result))
                {
                    _toggleVisiblePartsOnly = result;
                    _tsbToggleVisiblePartsOnly.Image = _toggleVisiblePartsOnly
                        ? Properties.Resources.PartsPalletOn
                        : Properties.Resources.PartsPallet;
                }

                // プレビュー
                if (setting.TryToFindUserBoolDataByName("ShowThumbnailPictureBox", out result))
                {
                    _showPreview = result;
                }
                else
                {
                    _showPreview = false;
                }

                // 表示状態を更新
                _UpdatePreviewVisible();

                // サムネールサイズ
                if (!setting.TryToFindUserEnumDataByName("ThumbnailSize", out _thumbnailSizeType))
                {
                    _thumbnailSizeType = ThumbnailSizeType.Smalleset;
                }

                // リストビューの表示形式を更新
                UpdateImageSize();
            }

            base.LoadSetting(setting, option);

            this.InitializePartsSubScenes();
        }

        #endregion LEToolWindow メンバ

        /// <summary>
        /// ダイアログを初期化します。
        /// </summary>
        private void InitializeDlgState_()
        {
            // LargeIcon 表示時にファイル名を見せるため、デフォルトの DisplayIndex と Index が異なる点に注意
            _lvwPartsFiles.Columns[0].Text = StringResMgr.Get("PARTS_LISTVIEW_FILENAME");
            _lvwPartsFiles.Columns[1].Text = StringResMgr.Get("PARTS_LISTVIEW_PARTNAME");
            _lvwPartsFiles.Columns[2].Text = StringResMgr.Get("PARTS_LISTVIEW_KIND");
            _lvwPartsFiles.Columns[3].Text = StringResMgr.Get("PARTS_LISTVIEW_BASE");
            _lvwPartsFiles.Columns[4].Text = StringResMgr.Get("PARTS_LISTVIEW_DESC");
            _lvwPartsFiles.Columns[5].Text = StringResMgr.Get("PARTS_LISTVIEW_DATE");
            _lvwPartsFiles.Columns[6].Text = StringResMgr.Get("PARTS_LISTVIEW_PANENUM");
            _lvwPartsFiles.Columns[7].Text = StringResMgr.Get("PARTS_LISTVIEW_PIXELNUM");

            _lvwPartsFiles.CustomDrawItem += Event_LvwPartsFiles_CustomDrawItem;

            IScene scene = LECore.LayoutEditorCore.Scene;
            scene.OnPartsSubSceneModified += OnPartsSubSceneModified_;

            _checkImage = ImageResMgr.GetManifestResourcePng("Checkmark.png");

            _tbxSearchWord.Tag = 0;
            _lblSerchResultCnt.Text = String.Empty;

            _searchCategoryAll = StringResMgr.Get("SEARCH_CATEGORY_ALL");
            _cmbSearchCategoryList.Items.Add(_searchCategoryAll);
            foreach (ColumnHeader column in _lvwPartsFiles.Columns)
            {
                _cmbSearchCategoryList.Items.Add(column.Text);
            }
            _cmbSearchCategoryList.SelectedIndex = 0;
        }

        /// <summary>
        /// 古いパスを履歴にしまう。
        /// </summary>
        void StorePartsRootPathHistory_(string rootPath)
        {
            if (string.IsNullOrEmpty(rootPath))
            {
                return;
            }

            ComboBox combo = _cmbPartsRootHistory.ComboBox as ComboBox;
            if (!combo.Items.Cast<string>().Any((item) => item == rootPath))
            {
                combo.Items.Insert(0, rootPath);

                // 最大数を超えたら捨てる
                if (combo.Items.Count > 5)
                {
                    combo.Items.RemoveAt(combo.Items.Count - 1);
                }

                UIComboBox.ExpandDropDownWidth(combo);
            }

            UpdateClearHistoryButton();
        }

        /// <summary>
        /// 部品ルートフォルダを設定します。
        /// </summary>
        void InitializePartsRootPath()
        {
            // 古いパスを履歴にしまう。
            StorePartsRootPathHistory_(_cmbPartsRootHistory.Text);
            // 新規登録パスを履歴にしまう。
            StorePartsRootPathHistory_(LECore.LayoutEditorCore.Scene.PartsRootPath);

            _cmbPartsRootHistory.ForeColor = SystemColors.ControlText;
            _cmbPartsRootHistory.SelectByCondition((item) => string.Equals(item, LECore.LayoutEditorCore.Scene.PartsRootPath));
            _cmbPartsRootHistory.ToolTipText = LECore.LayoutEditorCore.Scene.PartsRootPath;

            if (!Directory.Exists(LECore.LayoutEditorCore.Scene.PartsRootPath))
            {
                _erpMain.SetError(_cmbPartsRootHistory.ComboBox, StringResMgr.Get("PARTS_PALLET_INVALID_PARTSROOT"));
            }
            else
            {
                _erpMain.SetError(_cmbPartsRootHistory.ComboBox, null);
            }

            UpdateClearHistoryButton();
        }

        /// <summary>
        /// 部品レイアウトリストを初期化します。
        /// </summary>
        void InitializePartsSubScenes(bool updateTimestamp = true)
        {
            InitializePartsRootPath();

            _lvwPartsFiles.BeginUpdate();
            {
                _lvwPartsFiles.Items.Clear();
            }

            lock (_thumbnailImages)
            {
                // 既存のサムネールが更新されるようにする
                _thumbnailImagesDirtyFlag = true;

                // 不要になったサムネールを削除
                var oldItems = _thumbnailImages.Keys.Except(LECore.LayoutEditorCore.Scene.PartsSubScenes).ToArray();

                foreach (var oldItem in oldItems)
                {
                    var images = _thumbnailImages[oldItem];
                    if (images != null)
                    {
                        foreach (var image in images)
                        {
                            if (image != null)
                            {
                                image.Dispose();
                            }
                        }
                    }

                    _thumbnailImages.Remove(oldItem);
                }

                if (oldItems.Any())
                {
                    // いまのところ部分的に入れ替わることはないはず。
                    Debug.Assert(_thumbnailImages.Count == 0);

                    // 念のため削除
                    _thumbnailImages.Clear();

                    // すべて削除
                    // (部分的な削除はかえって遅いかもしれない)
                    _lvwPartsFiles.LargeImageList.Images.Clear();
                }
            }

            var thumbnailCreationTasks = new List<Task>();
            foreach (var partsSubScene in LECore.LayoutEditorCore.Scene.PartsSubScenes)
            {
                ListViewItem newItem = new ListViewItem(
                    new string[] { string.Empty, string.Empty, string.Empty, string.Empty, string.Empty, string.Empty, string.Empty, string.Empty });

                partsSubScene.UpdateBasePartsSubSceneModified();

                UpdateListNode_(newItem, partsSubScene);

                // 部品レイアウトのみフィルタの絞り込み対応
                if (_toggleVisiblePartsOnly && string.IsNullOrEmpty(newItem.SubItems[1].Text))
                {
                    continue;
                }

                // 検索機能対応
                if (_searchedString != String.Empty)
                {
                    string category = _cmbSearchCategoryList.Text;
                    string word = _searchedString;

                    bool find = false;
                    for (int index = 0; index < _lvwPartsFiles.Columns.Count; index++)
                    {
                        // カテゴリが一致するか
                        if (category != _searchCategoryAll &&
                            category != _lvwPartsFiles.Columns[index].Text)
                        {
                            continue;
                        }

                        // キーワードが含まれるか
                        string targetString;
                        if (index == _cmhPartsKind.Index)
                        {
                            // 「種類」の場合はPartsKindと比較する
                            var subScene = (IPartsSubScene)newItem.Tag;
                            if (subScene.PartsKind != null)
                            {
                                targetString = subScene.PartsKind;
                            }
                            else
                            {
                                targetString = String.Empty;
                            }
                        }
                        else
                        {
                            // それ以外の場合はTextと比較する
                            targetString = newItem.SubItems[index].Text;
                        }

                        if (targetString.ToLower().Contains(word.ToLower()))
                        {
                            find = true; // 条件一致する項目が見つかった
                            break;
                        }
                    }

                    if (!find)
                    {
                        continue;
                    }
                }

                _lvwPartsFiles.Items.Add(newItem);

                lock (_thumbnailImages)
                {
                    if (!_thumbnailImages.ContainsKey(partsSubScene))
                    {
                        _thumbnailImages.Add(partsSubScene, null);
                        bool isParts = !string.IsNullOrEmpty(partsSubScene.DescriptionName);
                        thumbnailCreationTasks.Add(Task.Run(() => LoadThumbnail(partsSubScene, isParts)));
                    }
                }
            }

            _lvwPartsFiles.EndUpdate();

            // ツールチップを更新する
            UpdateItemToolTips();

            // 検索UIの描画設定
            SetUISearchResult(_lvwPartsFiles.Items.Count);

            if (updateTimestamp)
            {
                _tsbDoUpdate.Text = DateTime.Now.ToString("HH:mm:ss");
            }

            this.UpdateDlgState();

            // 既存サムネールの再設定
            OnThumbnailChanged();
        }

        /// <summary>
        /// サムネールの読み込み
        /// </summary>
        private void LoadThumbnail(IPartsSubScene partsSubScene, bool isParts)
        {
            try
            {
                lock (_thumbnailImages)
                {
                    if (!_thumbnailImages.ContainsKey(partsSubScene))
                    {
                        // 既に不要になったので処理しない
                        return;
                    }

                    var partsPath = partsSubScene.FilePath;
                    var directory = $"{Path.GetDirectoryName(partsPath)}\\{AppConstants.ThumbnailDirectory}";
                    var path = $"{directory}\\{Path.GetFileName(partsPath)}{AppConstants.ThumbnailExt}";

                    if (File.Exists(path))
                    {
                        var sizeCount = Enum.GetValues(typeof(ThumbnailSizeType)).Length;
                        var images = new Image[sizeCount+1];

                        // サイズを調整する
                        using (var original = new Bitmap(path))
                        {
                            for (int i = 0; i < sizeCount; i++)
                            {
                                int height = ToInt((ThumbnailSizeType)i);
                                int width = (int)Math.Round((double)height / original.Height * original.Width);
                                images[i] = new Bitmap(width, height);
                                using (var g = Graphics.FromImage(images[i]))
                                {
                                    g.PixelOffsetMode = System.Drawing.Drawing2D.PixelOffsetMode.Half;
                                    if (original.Height < height)
                                    {
                                        g.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.NearestNeighbor;
                                    }

                                    g.DrawImage(original, 0, 0, images[i].Width, images[i].Height);
                                }
                            }
                        }

                        // 部品用のサムネール画像を作成
                        if (isParts)
                        {
                            var image = new Bitmap(images[sizeCount - 1]);
                            using (var g = Graphics.FromImage(image))
                            {
                                var partsIcon = PaneItemImageHelper.GetPaneImage(PaneKind.Parts, true);
                                g.DrawImage(partsIcon, 0, image.Height - partsIcon.Height);
                            }
                            images[sizeCount] = image;
                        }


                        _thumbnailImages[partsSubScene] = images;
                        if (!_thumbnailImagesDirtyFlag)
                        {
                            _thumbnailImagesDirtyFlag = true;
                            if (IsHandleCreated)
                            {
                                BeginInvoke((Action)(() =>
                                {
                                    OnThumbnailChanged();
                                }
                                ));
                            }
                        }
                    }
                }
            }
            catch (Exception e)
            {
                Debug.Assert(false, e.ToString());
            }
        }

        // サムネールが更新されたときの処理
        private void OnThumbnailChanged()
        {
            lock (_thumbnailImages)
            {
                if (!_thumbnailImagesDirtyFlag)
                {
                    return;
                }

                _lvwPartsFiles.BeginUpdate();

                List<Image> LargeImages = new List<Image>();
                List<ListViewItem> ListViewItems = new List<ListViewItem>();

                // 大きいサムネール表示用の更新
                foreach (ListViewItem item in _lvwPartsFiles.Items)
                {
                    if (item.ImageIndex != -1)
                    {
                        continue;
                    }

                    var partsSubScene = (IPartsSubScene)item.Tag;
                    Image[] images;
                    if (_thumbnailImages.TryGetValue(partsSubScene, out images) && images != null)
                    {
                        // 部品かどうかで使い分ける
                        LargeImages.Add(images[(int)ThumbnailSizeType.Largest + 1] ?? images[(int)ThumbnailSizeType.Largest]);


                        ListViewItems.Add(item);
                    }
                }

                // 更新が遅いのでまとめて末尾に追加
                var offset = _lvwPartsFiles.LargeImageList.Images.Count;
                _lvwPartsFiles.LargeImageList.Images.AddRange(LargeImages.ToArray());

                // LargeIcon 表示時以外に ImageIndex の情報が消えないようにサイズをそろえる
                if (_lvwPartsFiles.SmallImageList.Images.Count < _lvwPartsFiles.LargeImageList.Images.Count)
                {
                    _lvwPartsFiles.SmallImageList.Images.AddRange(
                        Enumerable.Repeat(
                            _dummySmallImage,
                            _lvwPartsFiles.LargeImageList.Images.Count - _lvwPartsFiles.SmallImageList.Images.Count).ToArray());
                }

                // インデックスの更新
                foreach (var item in ListViewItems)
                {
                    item.ImageIndex = offset;
                    Debug.Assert(item.ImageIndex != -1);
                    offset++;
                }

                _lvwPartsFiles.EndUpdate();

                // ツールチップの更新
                if (LECore.LayoutEditorCore.Scene.PartsSubScenes.Any() && _thumbnailImages.Values.All(x => x == null))
                {
                    _tsbThumbnailSize.ToolTipText = StringResMgr.Get("PARTSPALLET_NOTHUMBNAIL");
                }
                else
                {
                    _tsbThumbnailSize.ToolTipText = _tsbThumbnailSizeInitialToolTipText;
                }

                _thumbnailImagesDirtyFlag = false;

                UpdatePicturePanel();
            }
        }

        /// <summary>
        /// 部品レイアウトノードの更新。
        /// </summary>
        void UpdateListNode_(ListViewItem item, IPartsSubScene partsSubScene)
        {
            Ensure.Argument.True(item.SubItems.Count == 8);
            IPartsControlSetting srcControlSetting = LayoutEditorCore.Scene.FindPartsControlSetting(partsSubScene.PartsKind);

            // デフォルトの DisplayIndex と異なるので注意
            item.SubItems[0].Text = Path.GetFileName(partsSubScene.FilePath);
            item.SubItems[1].Text = partsSubScene.DescriptionName;
            item.SubItems[2].Text = srcControlSetting != null ? srcControlSetting.UIName : partsSubScene.PartsKind;
            item.SubItems[3].Text = partsSubScene.BasePartsName;
            item.SubItems[4].Text = UIHelper.ChangeLinesMultiToSingle(partsSubScene.Description);
            item.SubItems[5].Text = File.GetLastWriteTime(partsSubScene.FilePath).ToString("yy/MM/dd HH:mm:ss");
            item.SubItems[6].Text = partsSubScene.TotalPaneCount > 0 ? partsSubScene.TotalPaneCount.ToString() : "-";
            item.SubItems[7].Text = partsSubScene.TotalPixelCount > 0 ? partsSubScene.TotalPixelCount.ToString() : "-";

            item.Tag = partsSubScene;
        }

        /// <summary>
        /// 部品レイアウトの更新。
        /// </summary>
        /// <param name="kind"></param>
        void OnPartsSubSceneModified_(string subSceneFilePath)
        {
            bool isSubSceneNotFound = true;
            string subSceneName = Path.GetFileName(subSceneFilePath);
            foreach (ListViewItem item in _lvwPartsFiles.Items)
            {
                if (GetPartsLayoutFileNameFromListItem_(item) == subSceneName)
                {
                    item.ListView.BeginUpdate();
                    var partsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(subSceneName);
                    Ensure.Operation.ObjectNotNull(partsSubScene);
                    UpdateListNode_(item, partsSubScene);
                    item.ListView.EndUpdate();

                    item.Selected = true;
                    isSubSceneNotFound = false;
                    break;
                }
            }

            // ツールチップを更新する
            UpdateItemToolTips();

            if (isSubSceneNotFound)
            {
                InitializePartsSubScenes();
            }
            else
            {
                UpdateDlgState();
            }
        }

        /// <summary>
        /// 選択部品レイアウトファイル名
        /// </summary>
        private string SelectedPartsFileName
        {
            get
            {
                return _lvwPartsFiles.SelectedItems.Count > 0 ?
                    GetPartsLayoutFileNameFromListItem_(_lvwPartsFiles.SelectedItems[0]) : string.Empty;
            }
        }

        /// <summary>
        /// 選択部品レイアウトファイルパス
        /// </summary>
        private string SelectedPartsFilePath
        {
            get
            {
                if(_lvwPartsFiles.SelectedItems.Count <= 0)
                {
                    return string.Empty;
                }

                var parts = _lvwPartsFiles.SelectedItems[0].Tag as IPartsSubScene;
                return parts != null ? parts.FilePath : string.Empty;
            }
        }

        /// <summary>
        /// 選択部品のサブシーン
        /// </summary>
        private IPartsSubScene SelectedPartsSubScene
        {
            get
            {
                return !string.IsNullOrEmpty(SelectedPartsFileName) ?
                    LECore.LayoutEditorCore.Scene.FindPartsSubSceneByFileName(SelectedPartsFileName) : null;
            }
        }

        /// <summary>
        /// ノードから部品レイアウトのファイル名を取得します。
        /// </summary>
        private string GetPartsLayoutFileNameFromListItem_(ListViewItem item)
        {
            var parts = item.Tag as IPartsSubScene;
            return parts != null ? Path.GetFileName(parts.FilePath) : string.Empty;
        }

        /// <summary>
        /// ノードから部品レイアウトのファイル名を取得します。
        /// </summary>
        private string GetPartsLayoutPartsKindFromListItem_(ListViewItem item)
        {
            var parts = item.Tag as IPartsSubScene;
            return parts != null ? parts.PartsKind : string.Empty;
        }

        /// <summary>
        /// ノードから派生元部品レイアウトのファイル名を取得します。
        /// </summary>
        private string GetPartsDerivativeSourceFileNameFromListItem_(ListViewItem item)
        {
            var parts = item.Tag as IPartsSubScene;
            return parts != null ? parts.BasePartsName : string.Empty;
        }

        /// <summary>
        /// ノードから部品レイアウトのファイルパスを取得します。
        /// </summary>
        private string GetFilePathFromListItem_(ListViewItem item)
        {
            var parts = item.Tag as IPartsSubScene;
            return parts != null ? parts.FilePath : string.Empty;
        }

        /// <summary>
        /// ノードから部品レイアウトのファイル名を取得します。
        /// </summary>
        private string GetPartsLayoutFilePath_(string fileName)
        {
            var partsEntry = LECore.LayoutEditorCore.Scene.PartsSubScenes.FirstOrDefault(
                (PartsSubScene) => Path.GetFileName(PartsSubScene.FilePath) == fileName);
            return partsEntry.FilePath;
        }

        /// <summary>
        /// ダイアログを更新します。
        /// </summary>
        private void UpdateDlgState()
        {
            // ロード状態を更新します。
            {
                foreach (ListViewItem item in _lvwPartsFiles.Items)
                {
                    var partsSubScene = item.Tag as IPartsSubScene;

                    if (partsSubScene.IsLoaded)
                    {
                        if (partsSubScene.SubScene.IControlSettings.Any((ctrlSettings) => ctrlSettings.ContainsInvalidAnimationTag()) ||
                            partsSubScene.SubScene.IControlSettings.Any((ctrlSettings) => ctrlSettings.ContainsInvalidPane()))
                        {
                            item.ForeColor = Color.Red;

                            string ngPrefix = "[NG] ";
                            if (!item.SubItems[0].Text.StartsWith(ngPrefix))
                            {
                                item.SubItems[0].Text = ngPrefix + item.SubItems[0].Text;
                            }
                        }
                        else
                        {
                            item.ForeColor = SystemColors.ControlText;
                        }
                        item.BackColor = Color.White;
                    }
                    else
                    {
                        item.ForeColor = SystemColors.ControlText;
                        item.BackColor = LECore.AppConstants.InactiveBackColor;
                    }
                }
            }
        }

        private void UpdatePicturePanel()
        {
            var item = _lvwPartsFiles.SelectedItems.OfType<ListViewItem>().FirstOrDefault();
            if (item == null)
            {
                _pbxPreview.Image = null;
            }
            else
            {
                lock (_thumbnailImages)
                {
                    Image[] images;
                    if (_thumbnailImages.TryGetValue(item.Tag as IPartsSubScene, out images))
                    {
                        if (images != null)
                        {
                            _pbxPreview.Image = images[(int)ThumbnailSizeType.Largest];
                        }
                        else
                        {
                            _pbxPreview.Image = null;
                        }
                    }
                    else
                    {
                        _pbxPreview.Image = null;
                    }
                }

            }
        }

        //----------------------------------------------------------
        // イベントハンドラ
        //----------------------------------------------------------

        /// <summary>
        /// パーツファイルリスト選択変更。
        /// </summary>
        private void Event_LvwPartsFiles_SelectedIndexChanged(
            object sender, System.EventArgs e)
        {
            UpdateDlgState();
            UpdatePicturePanel();

        }

        /// <summary>
        /// リストダブルクリック（ペイン生成の仮仕様で利用）
        /// </summary>
        private void Event_LvwPartsFiles_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            var partsSubScene = SelectedPartsSubScene;
            if (partsSubScene == null || !partsSubScene.IsValidPartsSubScene)
            {
                return;
            }

            if (CurrentSubScene == null)
            {
                return;
            }

            LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(Path.GetFileName(SelectedPartsSubScene.FilePath));
            if (!partsSubScene.IsLoaded)
            {
                // エラーダイアログを表示
                {
                    MessageReportDlg reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                    reportDlg.Title = StringResMgr.Get("TAG_ERROR");
                    reportDlg.Message = StringResMgr.Get("PARTS_PANE_ERROR_FILE_READ_ERROR");
                    reportDlg.ShowDialog(this);
                }
                return;
            }

            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.BindTarget(CurrentSubScene);

            // 自分自身に部品ペインを配置しようとしていないかチェックします。
            {
                var arg = new ViewManagerMessage(ViewManagerMessageKind.FindLayoutPathBySubScene, CurrentSubScene);
                this.DoSendMessageToViewManager(arg);
                string layoutFilePath = arg.Result as string;

                if (!string.IsNullOrEmpty(layoutFilePath))
                {
                    if (!PartsSubSceneHelper.IsPartsSubSceneValidForBodyLayout(partsSubScene, layoutFilePath))
                    {
                        // 無限ループを生む不正な部品ペインの追加
                        // エラーダイアログを表示
                        {
                            MessageReportDlg reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                            reportDlg.Title = StringResMgr.Get("TAG_ERROR");
                            reportDlg.Message = StringResMgr.Get("PARTS_PANE_WANING_ADD_INVALID");
                            reportDlg.ShowDialog(this);
                        }
                        return;
                    }
                }
            }

            IPane newPartsPane = subSceneMnp.AddPartsPane(ViewManager.AppSetting.ProjectSettings.GetNewPartsPaneName(), SelectedPartsFileName, partsSubScene.SubScene);

            // 名前と階層構造が確定したのちにリロードしたパーツペインのキャプチャテクスチャ参照情報を更新します。
            SubSceneHelper.ResolvePartsLayoutCaptureTextureReference(newPartsPane);
        }

        /// <summary>
        /// 有効なパスを取得する。
        /// </summary>
        string GetValidPath_(string path)
        {
            string validPath = path;
            if (string.IsNullOrEmpty(path))
            {
                return string.Empty;
            }

            // さかのぼりながら存在するフォルダを探す。
            while (!Directory.Exists(validPath))
            {
                validPath = Path.GetDirectoryName(validPath);

                if (String.IsNullOrEmpty(validPath))
                {
                    return string.Empty;
                }
            }

            // フルパス形式で返す（フォルダダイアログの誤動作の原因となっているようなので...）。
            return new DirectoryInfo(validPath).FullName;
        }

        /// <summary>
        /// フォルダ指定ボタンクリック
        /// </summary>
        private void Event_TbtSetPartsFolder_Click(object sender, EventArgs e)
        {
            var scene = LECore.LayoutEditorCore.Scene;

            FolderBrowserUility folderBrowser = new FolderBrowserUility();
            string newDir = folderBrowser.SelectFolder(StringResMgr.Get("SELECT_PARTS_ROOT"), GetValidPath_(scene.PartsRootPath), this.Handle);
            if (newDir != null)
            {
                TrySetPartsRoot_(newDir);
            }
        }

        void DoSetPartsRoot_(string newPartsRoot)
        {
            SceneManipulator mnp = new SceneManipulator();
            mnp.BindTarget(LECore.LayoutEditorCore.Scene);
            mnp.RefreshPartsSubScenes(newPartsRoot);

            //--------------------------------------------
            // プロジェクト設定をロードします。
            bool projectFileLoaded = ProjectSettings.TryReadProjectSettingsFromPartsRoot(newPartsRoot, this.ViewManager.AppSetting.ProjectSettings);

            if (projectFileLoaded)
            {
                DoSendMessageToViewManager(new ViewManagerMessage(ViewManagerMessageKind.ProjectSettingChanged));
            }

            InitializePartsSubScenes();
        }

        /// <summary>
        /// 部品ルートの設定を試みる
        /// </summary>
        bool TrySetPartsRoot_(string newPartsRoot)
        {
            if (newPartsRoot == LECore.LayoutEditorCore.Scene.PartsRootPath)
            {
                return false;
            }

            DoSetPartsRoot_(newPartsRoot);

            return true;
        }

        /// <summary>
        /// 更新ボタンクリック
        /// </summary>
        private void Event_TsbLastModify_Click(object sender, EventArgs e)
        {
            try
            {
                // 連続実行できないように無効化して処理をスタート
                _tsbLastModify.Enabled = false;
                DoSetPartsRoot_(LECore.LayoutEditorCore.Scene.PartsRootPath);

                // 開いているレイアウトについてすべてリロードを実行します。
                using (WaitCursor waitCursor = new WaitCursor())
                {
                    DoSendMessageToViewManager(new ViewManagerMessage(ViewManagerMessageKind.ReloadPartsAll, null));
                }
            }
            finally
            {
                _tsbLastModify.Enabled = true;
            }
        }

        /// <summary>
        /// エクスプローラで開くメニュー
        /// </summary>
        private void Event_TmiOpenByExplorer_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(SelectedPartsFileName))
            {
                return;
            }

            var partsSubScene = _lvwPartsFiles.SelectedItems[0].Tag as IPartsSubScene;
            string partsFileFolderPath = Path.GetDirectoryName(partsSubScene.FilePath);

            Process p = new Process();
            ProcessStartInfo ps = new ProcessStartInfo();
            ps.FileName = "explorer.exe";
            ps.Arguments = string.Format("/select,{0}", partsSubScene.FilePath);
            p.StartInfo = ps;
            p.Start();
        }

        /// <summary>
        /// 派生部品を作る
        /// </summary>
        private void Event_TmiCreateDerivative_Click(object sender, EventArgs e)
        {
            if (!string.IsNullOrEmpty(SelectedPartsFileName))
            {
                try
                {
                    LayoutEditorCore.MsgReporter.BeginPacking(LECore.LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSLAYOUTLOADING"));

                    // 内部データを読み込みます。
                    LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(SelectedPartsFileName);

                    // 編集用に別途シーンを読み込みます。
                    var args = new ViewManagerMessage(
                        ViewManagerMessageKind.CreateDerivativeLayout,GetPartsLayoutFilePath_(SelectedPartsFileName));
                    this.DoSendMessageToViewManager(args);
                }
                finally
                {
                    LayoutEditorCore.MsgReporter.EndPacking();
                }
            }
        }

        /// <summary>
        /// 部品作成メニュー
        /// </summary>
        private void Event_TmiMakePartsPane_Click(object sender, EventArgs e)
        {
            Event_LvwPartsFiles_MouseDoubleClick(null, null);
        }

        /// <summary>
        /// すべての派生部品を更新//
        /// </summary>
        private void Event_TmiUpdateDerivativeAll_Click(object sender, EventArgs e)
        {
            // すべてのレイアウトウインドウを閉じてから実行してください。
            if (ViewManager.IsLayoutWindowExsit)
            {
                MessageBox.Show(StringResMgr.Get("PARTS_PALLET_CLOSELYTWND"), StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"));
                return;
            }

            // 古い派生部品を列挙
            List<IPartsSubScene> deribPartsToUpdate = new List<IPartsSubScene>();
            foreach (ListViewItem item in _lvwPartsFiles.Items)
            {
                IPartsSubScene partsSubScene = item.Tag as IPartsSubScene;

                partsSubScene.UpdateBasePartsSubSceneModified();
                if (partsSubScene.IsBasePartsSubSceneModified)
                {
                    deribPartsToUpdate.Add(partsSubScene);
                }
            }

            // 処理対象がなければ中断
            if (deribPartsToUpdate.Count <= 0)
            {
                // 処理対象がありませんでした。
                MessageBox.Show(StringResMgr.Get("NO_OPERATION_TARGET"), StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"));
                return;
            }
            else
            {
                // ユーザに処理対象を通知。
                string msg = StringResMgr.Get("PARTS_PALLET_UPDATE_BELOW") + Environment.NewLine + Environment.NewLine;
                foreach (var deribParts in deribPartsToUpdate)
                {
                    msg += string.Format("{0}({1}-{2}){3}", StringResMgr.Get("LAST_UPDATE"), Path.GetFileName(deribParts.FilePath), deribParts.BasePartsLastModify.ToString(), Environment.NewLine);
                }

                if (MessageBox.Show(msg, StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"), MessageBoxButtons.OKCancel) != DialogResult.OK)
                {
                    return;
                }
            }

            // 全リロードする
            Event_TsbLastModify_Click(null, null);

            // 開く。そのまま保存し。閉じる。
            foreach (var deribParts in deribPartsToUpdate)
            {
                // 内部データを読み込みます。
                string partsName = Path.GetFileName(deribParts.FilePath);

                // 読む
                var partsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(partsName);
                if (partsSubScene != null && !partsSubScene.IsLoaded)
                {
                    LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(partsName);
                }
                partsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(partsName);

                if (partsSubScene != null)
                {
                    ExportOption exportOption = new ExportOption()
                    {
                        UseBaseValue = true,
                        Frame = GlobalTime.Inst.Time
                    };
                    LayoutEditorCore.ExportToFileAll(partsSubScene.SubScene, partsSubScene.FilePath, exportOption);
                }
                else
                {
                    MessageBox.Show(
                        StringResMgr.Get("ERROR_PARTS_NOT_LOADED", partsName),
                        StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"),
                        MessageBoxButtons.OK);
                }
            }

            // 最後にもう一度全リロードする。
            Event_TsbLastModify_Click(null, null);

            MessageBox.Show(StringResMgr.Get("OPERATION_FINISH"), StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"));
        }

        /// <summary>
        /// すべてのボディレイアウトを更新
        /// </summary>
        private void Event_tmiUpdateAllBodyLayout_Click(object sender, EventArgs e)
        {
            // すべてのレイアウトウインドウを閉じてから実行してください。
            if (ViewManager.IsLayoutWindowExsit)
            {
                MessageBox.Show(StringResMgr.Get("PARTS_PALLET_CLOSELYTWND"), StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"));
                return;
            }

            string partsFilePath = this.SelectedPartsFilePath;
            if (!File.Exists(partsFilePath))
            {
                return;
            }

            // 部品ルートから、近くのBodyフォルダを探す。見つからなければ
            // ファイルダイアログで、探索対象を選ばせる。
            var bodyRoots = new List<string>();
            string partsRoot = LECore.LayoutEditorCore.Scene.PartsRootPath;
            string bodyCandDir1 = Path.Combine(partsRoot, "../Body");
            if (Directory.Exists(bodyCandDir1))
            {
                bodyRoots.Add(Path.GetFullPath(bodyCandDir1));
            }
            bodyRoots.Add(partsRoot);

            // 処理フォルダを示して実行を確認する。
            {
                string text = string.Format(
                    "{0}\n\r{1}\n\r\n\r{2}",
                    StringResMgr.Get("PARTS_PALLET_UPDATEPARAM_DESC"),
                    StringResMgr.Get("PARTS_PALLET_UPDATEPARAM_CONFIRM"), bodyRoots[0]);

                if (bodyRoots.Count == 2)
                {
                    text += "\n\r" + bodyRoots[1];
                }

                if (DialogResult.Cancel == MessageBox.Show(this, text, StringResMgr.Get("PARTS_PALLET_UPDATEPARAM"), MessageBoxButtons.OKCancel))
                {
                    return;
                }
            }

            // 変換の実行
            StringBuilder removeReport = UpdateAllBodyLayout_(partsFilePath, bodyRoots);

            // 結果をユーザに報告する。
            {
                MessageReportDlg reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                reportDlg.Title = string.Format("{0}：{1}", StringResMgr.Get("PARTS_PALLET_UPDATEPARAM"), StringResMgr.Get("TAG_RESULT"));
                reportDlg.Message = removeReport.ToString();
                reportDlg.ShowDialog(this);
            }
        }

        /// <summary>
        /// 更新します。
        /// </summary>
        private StringBuilder UpdateAllBodyLayout_(string partsFilePath, IEnumerable<string> bodyRoots)
        {
            // まず、上書きペイン情報を整理しておく
            List<string> overwritePaneNames = new List<string>();
            {
                XmlDocument doc = new XmlDocument();
                doc.Load(partsFilePath);

                XmlNamespaceManager xmlNsManager = new XmlNamespaceManager(doc.NameTable);
                xmlNsManager.AddNamespace("ns", "http://www.nintendo.co.jp/NW4F/LayoutEditor");

                // <propertyForm kind="Null" target="N_all">
                XmlNodeList nodes = doc.SelectNodes("//ns:propertyForm", xmlNsManager);
                foreach (XmlElement node in nodes)
                {
                    string paneName = node.GetAttribute("target");
                    overwritePaneNames.Add(paneName);
                }
            }

            // ファイルを列挙して...
            StringBuilder removeReport = new StringBuilder();
            string partsFileName = Path.GetFileName(this.SelectedPartsFilePath);
            string partsXPath = string.Format("//ns:parts[@path=\"{0}\"]", partsFileName);
            foreach (string lytPath in bodyRoots.SelectMany(x => Directory.GetFiles(x, AppConstants.LayoutFileExtPattern, SearchOption.AllDirectories)))
            {
                XmlDocument doc = new XmlDocument();
                doc.Load(lytPath);

                XmlNamespaceManager xmlNsManager = new XmlNamespaceManager(doc.NameTable);
                xmlNsManager.AddNamespace("ns", "http://www.nintendo.co.jp/NW4F/LayoutEditor");

                // 自分を使っている部品ペインを列挙
                // <parts path="XXXX.flyt">
                List<string> modifiedPaneMsgs = new List<string>();
                foreach (XmlElement partsNode in doc.SelectNodes(partsXPath, xmlNsManager))
                {
                    if (partsNode.ParentNode == null) { continue; }

                    // 上書きパラメータを列挙
                    XmlNodeList partsNodeSelectNodes = partsNode.SelectNodes("./ns:property", xmlNsManager);
                    if (partsNodeSelectNodes.Count > 0)
                    {
                        string partsPaneName = (partsNode.ParentNode as XmlElement).GetAttribute("name");

                        // "部品ペイン名:[{0}] - 破棄した上書き---"
                        string message = StringResMgr.Get("PARTS_PALLET_UPDATEPARAM_PANEREPORT", partsPaneName);
                        bool bRemoved = false;
                        foreach (XmlElement overwriteParamNode in partsNodeSelectNodes)
                        {
                            // 上書き対象に存在しなければ破棄する。
                            string targetPaneName = overwriteParamNode.GetAttribute("target");
                            if (!overwritePaneNames.Contains(targetPaneName))
                            {
                                // 破棄する。破棄したものは、あとでリポートするので覚えておく
                                message += string.Format("{0}, ", targetPaneName);
                                overwriteParamNode.ParentNode.RemoveChild(overwriteParamNode);
                                bRemoved = true;
                            }
                        }

                        if (bRemoved)
                        {
                            modifiedPaneMsgs.Add(message);
                        }
                    }
                }

                // 報告メッセージを作る。
                if (modifiedPaneMsgs.Count > 0)
                {
                    removeReport.AppendLine(string.Format("** {0} **", Path.GetFileName(lytPath)));
                    removeReport.AppendLine(string.Format("  ({0})", lytPath));
                    foreach (var msg in modifiedPaneMsgs)
                    {
                        removeReport.AppendLine("    " + msg);
                    }
                    removeReport.AppendLine();

                    // 結果を保存する。
                    doc.Save(lytPath);
                }
            }

            // 一覧で処理内容を報告する
            if (removeReport.Length > 0)
            {
                // "*** 以下のボディレイアウトで上書き情報が破棄されました ***\n\r\n\r"
                removeReport.Insert(0, string.Format("{0}\n\r\n\r", StringResMgr.Get("PARTS_PALLET_UPDATEPARAM_RESULTMSG")));
            }
            else
            {
                // "該当するデータがありませんでした。"
                removeReport.AppendLine(StringResMgr.Get("TAG_NORESULTMSG"));
            }

            return removeReport;
        }

        /// <summary>
        /// 派生元を再設定
        /// </summary>
        private void Event_tmiRestoreDerivativeSource_Click(object sender, EventArgs e)
        {
            // すべてのレイアウトウインドウを閉じてから実行してください。
            if (ViewManager.IsLayoutWindowExsit)
            {
                MessageBox.Show(
                    StringResMgr.Get("PARTS_PALLET_CLOSELYTWND"),
                    StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"));
                return;
            }

            // ダイアログを出す(古い名前と、新しい名前を指定する。他の候補も列挙している)
            string filePath = GetFilePathFromListItem_(_lvwPartsFiles.SelectedItems[0]);
            string basePartsName = GetPartsDerivativeSourceFileNameFromListItem_(_lvwPartsFiles.SelectedItems[0]);
            List<string> basePathSet = new List<string>();
            List<string> targetPartsFilePathSet = new List<string>();

            basePathSet.Add(basePartsName);
            targetPartsFilePathSet.Add(filePath);

            // ダイアログに表示する情報の収集
            foreach(ListViewItem item in _lvwPartsFiles.Items)
            {
                // 派生部品ではない通常部品か？
                string itemBasePartsName = GetPartsDerivativeSourceFileNameFromListItem_(item);
                if (string.IsNullOrEmpty(itemBasePartsName))
                {
                    // 新しい派生元候補として収集する。
                    string partsKind = GetPartsLayoutPartsKindFromListItem_(item);
                    string itemPartsName = GetPartsLayoutFileNameFromListItem_(item);

                    if (!string.IsNullOrEmpty(partsKind) && !basePathSet.Contains(itemPartsName))
                    {
                        basePathSet.Add(itemPartsName);
                    }
                }
                else
                {
                    // 同じ派生元を持つ派生部品ファイルを収集する。
                    string itemFilePath = GetFilePathFromListItem_(item);
                    if (itemBasePartsName == basePartsName && !targetPartsFilePathSet.Contains(itemFilePath))
                    {
                        targetPartsFilePathSet.Add(itemFilePath);
                    }
                }
            }

            // ダイアログ表示
            RestoreDerivativeSourceDlg dlg = new RestoreDerivativeSourceDlg();
            dlg.Initialize(basePartsName, basePathSet, targetPartsFilePathSet);
            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            // 変更にしたがって、対象ファイルをすべて書き換える
            foreach (var targetFilePath in dlg.TargetFilePathSet)
            {
                try
                {
                    RewriteBasePartsInfo_(targetFilePath, dlg.NewBasePartsName);
                }
                catch
                {
                    MessageBox.Show(
                        LECoreStringResMgr.Get("SYSTEM_ERROR_IN_EXEC_FILE", targetFilePath),
                        StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"));
                }
            }

            // 全リロードする
            Event_TsbLastModify_Click(null, null);
            MessageBox.Show(StringResMgr.Get("OPERATION_FINISH"), StringResMgr.Get("PARTS_PALLET_UPDATE_DERIV"));
        }

        /// <summary>
        /// 部品レイアウトの派生元情報を書き換えます。
        /// </summary>
        private void RewriteBasePartsInfo_(string filePath, string newBasePartsName)
        {
            // partsDerivativeBaseName タグの記述が存在することを前提にしています。
            {
                string allContent = File.ReadAllText(filePath);
                string newTagStr = string.Format("<partsDerivativeBaseName>{0}</partsDerivativeBaseName>", newBasePartsName);
                if (allContent.IndexOf("<partsDerivativeBaseName>") != -1)
                {
                    allContent = Regex.Replace(allContent, "<partsDerivativeBaseName>(.*?)</partsDerivativeBaseName>", newTagStr);
                }

                // 派生元に付加されている読み込み専用属性をすべてリセットする。
                // (派生先で読み取り専用にマークされている情報は利用せず破棄されるので。)
                allContent = Regex.Replace(allContent, "readonlyLocked=\"true\"", "readonlyLocked=\"false\"");

                File.WriteAllText(filePath, allContent);
            }

            // アニメーションファイルがあれば同様に属性をリセットする。
            string animFilePath = Path.ChangeExtension(filePath, AppConstants.AnimationFileExt);
            if (File.Exists(animFilePath))
            {
                string allAnimContent = File.ReadAllText(animFilePath);
                allAnimContent = Regex.Replace(allAnimContent, "readonlyLocked=\"true\"", "readonlyLocked=\"false\"");
                File.WriteAllText(animFilePath, allAnimContent);
            }
        }

        /// <summary>
        /// コンテキストメニュー状態更新
        /// </summary>
        private void Event_CmsMain_Opening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            IPartsSubScene selectedParts = SelectedPartsSubScene;
            bool isValidParts = selectedParts != null && selectedParts.IsValidPartsSubScene;

            _tmiMakePartsPane.Enabled = CurrentSubScene != null && !string.IsNullOrEmpty(SelectedPartsFileName) && isValidParts;
            _tmiOpenByExplorer.Enabled = !string.IsNullOrEmpty(SelectedPartsFileName);
            _tmiCreateDerivative.Enabled = !string.IsNullOrEmpty(SelectedPartsFileName) && isValidParts && !selectedParts.IsDerivativeParts();
            _tmiUpdateDerivativeAll.Enabled = !string.IsNullOrEmpty(SelectedPartsFileName) && isValidParts && !selectedParts.IsDerivativeParts();
            _tmiUpdateBodyLayoutOverWriteAll.Enabled = !string.IsNullOrEmpty(SelectedPartsFileName) && isValidParts;
            _tmiRestoreDerivativeSource.Enabled = !string.IsNullOrEmpty(SelectedPartsFileName) && isValidParts && selectedParts.IsDerivativeParts();
        }

        /// <summary>
        /// 部品アイテムドラッグ
        /// </summary>
        private void Event_LvwPartsFiles_ItemDrag(object sender, ItemDragEventArgs e)
        {
            ListViewItem lvi = e.Item as ListViewItem;
            Debug.Assert(lvi != null);

            string partsLayoutFileName = GetPartsLayoutFileNameFromListItem_(lvi);
            _lvwPartsFiles.DoDragDrop(
                GetPartsLayoutFilePath_(partsLayoutFileName),
                DragDropEffects.Move);
        }

        /// <summary>
        /// 部品レイアウトを開くメニュー
        /// </summary>
        private void Event_TmiOpenPartsLayout_Click(object sender, EventArgs e)
        {
            if (!string.IsNullOrEmpty(SelectedPartsFileName))
            {
                try
                {
                    LayoutEditorCore.MsgReporter.BeginPacking(LECore.LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSLAYOUTLOADING"));

                    // 編集用に別途シーンを読み込みます。// ツール内部への読み込みは行わないようにしました。
                    var args = new ViewManagerMessage(
                        ViewManagerMessageKind.OpenLayout, GetPartsLayoutFilePath_(SelectedPartsFileName));
                    this.DoSendMessageToViewManager(args);
                }
                finally
                {
                    LayoutEditorCore.MsgReporter.EndPacking();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        string GetControlNameText_(IPartsSubScene partsSubScene, int numCtrl)
        {
            if (numCtrl == 0)
            {
                return "( NO CONTROL )";
            }
            else if (numCtrl == 1)
            {
                return partsSubScene.ControlNames.First();
            }
            else
            {
                return string.Format("{0}...[{1}]", partsSubScene.ControlNames.First(), numCtrl);
            }
        }

        /// <summary>
        /// リストオーナー描画
        /// </summary>
        void Event_LvwPartsFiles_CustomDrawItem(object sender, CustomDrawListViewItemEventArgs e)
        {
            e.SpecificForeColor = e.Item.ForeColor;
            e.SpecificBackColor = e.Item.BackColor;

            switch (e.ColumnIndex)
            {
                case 0:
                    {
                        e.IgnoreDefaultImage = true;
                    }
                    break;
                case 2:
                    {
                        // 部品種類
                        var partsSubScene = e.Item.Tag as IPartsSubScene;
                        if (partsSubScene != null)
                        {
                            // 部品かどうか？
                            int numCtrl = partsSubScene.ControlNames.Count();
                            if (string.IsNullOrEmpty(partsSubScene.DescriptionName))
                            {
                                // 非部品レイアウト
                                e.SpecificText = StringResMgr.Get("PARTS_NORMAL_LAYOUT");
                                e.SpecificForeColor = LECore.AppConstants.SealedForeColor;

                                // 複数コントロール設定レイアウト
                                if (numCtrl > 0)
                                {
                                    e.SpecificText = e.SpecificText + "(" + GetControlNameText_(partsSubScene, numCtrl) + ")";
                                }
                            }
                            else
                            {
                                // 部品レイアウト
                                e.SpecificText = GetControlNameText_(partsSubScene, numCtrl);
                                e.SpecificImage = PaneItemImageHelper.GetPaneImage(PaneKind.Parts, true);
                            }
                        }
                        else
                        {
                            e.SpecificText = StringResMgr.Get("PARTS_NORMAL_LAYOUT");
                            e.SpecificForeColor = LECore.AppConstants.SealedForeColor;
                        }
                    }
                    break;
                case 3:
                    {
                        // 派生元
                        if (string.IsNullOrEmpty(e.SubItem.Text))
                        {
                            e.SpecificText = "---";
                            e.SpecificForeColor = LECore.AppConstants.SealedForeColor;
                        }
                        else
                        {
                            var partsSubScene = e.Item.Tag as IPartsSubScene;
                            if (partsSubScene != null && partsSubScene.IsBasePartsSubSceneModified)
                            {
                                e.SpecificImage = _checkImage;
                            }

                            e.SpecificText = e.SubItem.Text;
                        }
                    }
                    break;

            }

            if (_lvwPartsFiles.Columns[e.ColumnIndex].DisplayIndex == 0)
            {
                var partsSubScene = e.Item.Tag as IPartsSubScene;
                if (partsSubScene != null)
                {
                    lock (_thumbnailImages)
                    {
                        Image[] images;
                        if (_thumbnailImages.TryGetValue(partsSubScene, out images))
                        {
                            if (images != null)
                            {

                                e.SpecificImage = images[(int)_thumbnailSizeType];
                            }
                        }
                    }

                    if (e.SpecificImage == null && _lvwPartsFiles.LargeImageList.Images.Count > 0)
                    {
                        e.SpecificImage = _transparentImages[(int)_thumbnailSizeType];
                    }
                }
            }
        }

        /// <summary>
        /// 部品ルートコンボ
        /// </summary>
        private void Event_cmbPartsRootHistory_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (_cmbPartsRootHistory.SelectedIndex == -1)
            {
                return;
            }

            string newRoot = _cmbPartsRootHistory.Items[_cmbPartsRootHistory.SelectedIndex].ToString();
            var scene = LECore.LayoutEditorCore.Scene;
            if (newRoot != scene.PartsRootPath)
            {
                StorePartsRootPathHistory_(scene.PartsRootPath);
                TrySetPartsRoot_(newRoot);
            }

            _lvwPartsFiles.Focus();
        }

        /// <summary>
        /// ドラッグ引数から、フォルダを得る
        /// </summary>
        private string GetOneValidFoloder_(DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                string[] s = e.Data.GetData(DataFormats.FileDrop, false) as string[];
                if (s != null && s.Length == 1 && Directory.Exists(s[0]))
                {
                    return s[0];
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// ドラッグ開始
        /// </summary>
        private void Event_tspMain_DragEnter(object sender, DragEventArgs e)
        {
            string dir = GetOneValidFoloder_(e);
            e.Effect = string.IsNullOrEmpty(dir) ? DragDropEffects.None : DragDropEffects.All;
        }

        /// <summary>
        /// ドラッグ開始
        /// </summary>
        private void Event_tspMain_DragDrop(object sender, DragEventArgs e)
        {
            string newRoot = GetOneValidFoloder_(e);
            if (string.IsNullOrEmpty(newRoot))
            {
                return;
            }

            var scene = LECore.LayoutEditorCore.Scene;
            if (newRoot != scene.PartsRootPath)
            {
                StorePartsRootPathHistory_(scene.PartsRootPath);
                TrySetPartsRoot_(newRoot);
            }
        }

        /// <summary>
        /// 初回表示時
        /// </summary>
        private void Event_PartsPalletWindow_Shown(object sender, EventArgs e)
        {
            // 初回表示時に、まだ部品ルートUIが未設定状態なら
            // 関係する UI の初期化を行います。
            if (string.IsNullOrEmpty(_cmbPartsRootHistory.Text))
            {
                InitializePartsSubScenes();
            }
        }

        private void Event_tsbOpenFlpj_Click(object sender, EventArgs e)
        {
            string flpjPath = ViewManager?.AppSetting?.ProjectSettings?.FilePath;
            Debug.Assert(!string.IsNullOrEmpty(flpjPath) && File.Exists(flpjPath));

            System.Diagnostics.Process.Start(flpjPath);
        }

        /// <summary>
        /// 履歴のクリアボタンクリック
        /// </summary>
        private void Event_TsbClearHistory_Click(object sender, EventArgs e)
        {
            if (MessageBox.Show(
                StringResMgr.Get("PARTS_ROOT_PATH_HISTORY_ALL_CLEAR_MESSAGE"),
                StringResMgr.Get("PARTS_ROOT_PATH_HISTORY_ALL_CLEAR_CAPTION"),
                MessageBoxButtons.OKCancel) == DialogResult.OK)
            {
                var comboBox = _cmbPartsRootHistory.ComboBox;
                if (comboBox == null)
                {
                    return;
                }

                int index = comboBox.SelectedIndex;
                if (0 <= index && index < comboBox.Items.Count)
                {
                    var selectedPath = comboBox.Items[index];
                    comboBox.Items.Clear();
                    comboBox.Items.Add(selectedPath);
                    comboBox.SelectedIndex = 0;
                }
                else
                {
                    comboBox.Items.Clear();
                }

                UIComboBox.ExpandDropDownWidth(comboBox);
                UpdateClearHistoryButton();
            }
        }

        /// <summary>
        /// 履歴のクリアボタンの状態を更新します。
        /// </summary>
        private void UpdateClearHistoryButton()
        {
            if (_cmbPartsRootHistory.ComboBox == null)
            {
                _tsbClearHistory.Enabled = false;
                return;
            }

            var items = _cmbPartsRootHistory.ComboBox.Items;
            if (items.Count == 0)
            {
                _tsbClearHistory.Enabled = false;
                return;
            }

            if (items.Count == 1 &&
                string.Equals((string)items[0], LECore.LayoutEditorCore.Scene.PartsRootPath))
            {
                _tsbClearHistory.Enabled = false;
                return;
            }

            _tsbClearHistory.Enabled = true;
        }

        /// <summary>
        /// 部品レイアウトフィルタのトグルボタンクリック
        /// </summary>
        private void Event_TsbToggleVisiblePartsOnly_Click(object sender, EventArgs e)
        {
            _toggleVisiblePartsOnly = !_toggleVisiblePartsOnly;
            _tsbToggleVisiblePartsOnly.Image = _toggleVisiblePartsOnly
                ? Properties.Resources.PartsPalletOn
                : Properties.Resources.PartsPallet;

            InitializePartsSubScenes(false);
        }

        /// <summary>
        /// フィルタカテゴリ変更
        /// </summary>
        private void Event_CmbSearchCategory_SelectedIndexChanged(object sender, EventArgs e)
        {
            DoSearch();
        }

        /// <summary>
        /// 検索文字列変更
        /// </summary>
        private void Event_TbxSearchWord_TextChanged(object sender, EventArgs e)
        {
            var tbx = sender as ToolStripTextBox;
            _searchedString = tbx.Text;

            DoSearch();
        }

        /// <summary>
        /// 検索実行
        /// </summary>
        private void DoSearch()
        {
            // リストを更新
            InitializePartsSubScenes(false);
        }

        /// <summary>
        /// 検索結果数に応じてUIを変化させる
        /// </summary>
        private void SetUISearchResult(int resultCnt)
        {
            // 検索機能有効時に結果が空だった場合にラベルを表示
            if (_searchedString != String.Empty && resultCnt <= 0)
            {
                // 検索結果：無し
                _tbxSearchWord.BackColor = Color.Pink;
                _lblSerchResultCnt.Text = String.Format(StringResMgr.Get("SEARCH_RESULT_COUNT"), resultCnt);
            }
            else if (_searchedString != String.Empty && resultCnt > 0)
            {
                // 検索結果：有り
                _tbxSearchWord.BackColor = Color.LightCyan;
                _lblSerchResultCnt.Text = String.Format(StringResMgr.Get("SEARCH_RESULT_COUNT"), resultCnt);
            }
            else
            {
                // 検索していない
                _tbxSearchWord.BackColor = Color.White;
                _lblSerchResultCnt.Text = String.Empty;
            }
        }

        /// <summary>
        /// サムネールの高さの更新
        /// </summary>
        private void UpdateImageSize()
        {
            if (_thumbnailSizeType == ThumbnailSizeType.Largest)
            {
                _lvwPartsFiles.BeginUpdate();
                if (_lvwPartsFiles.View != View.LargeIcon)
                {
                    // 順番が変わっているかも知れないので、ここで更新
                    UpdateItemToolTips();

                    _lvwPartsFiles.ShowItemToolTips = true;
                    _lvwPartsFiles.View = View.LargeIcon;
                }
                _lvwPartsFiles.LargeImageList.ImageSize = _thumbnailSize;
                _lvwPartsFiles.EndUpdate();
            }
            else
            {
                _lvwPartsFiles.BeginUpdate();
                if (_lvwPartsFiles.View != View.Details)
                {
                    _lvwPartsFiles.ShowItemToolTips = false;
                    _lvwPartsFiles.View = View.Details;
                }

                _lvwPartsFiles.SmallImageList.ImageSize = _thumbnailSize;
                _lvwPartsFiles.EndUpdate();
            }

            foreach (ToolStripMenuItem item in _tsbThumbnailSize.DropDownItems)
            {
                if ((ThumbnailSizeType)item.Tag == _thumbnailSizeType)
                {
                    _tsbThumbnailSize.Image = item.Image;
                }
            }
        }

        private void UpdateItemToolTips()
        {
            var columns = _lvwPartsFiles.Columns.OfType<ColumnHeader>().Except(new[] { _cmhPartsFilesName }).OrderBy(x => x.DisplayIndex).ToArray();
            foreach (ListViewItem listViewItem in _lvwPartsFiles.Items)
            {
                listViewItem.ToolTipText = string.Join("\n",
                    columns.Where(x => !string.IsNullOrEmpty(listViewItem.SubItems[x.Index].Text))
                    .Select(x => $"{x.Text}: {listViewItem.SubItems[x.Index].Text}"));
            }
        }

        /// <summary>
        /// サイズ変更
        /// </summary>
        private void SizeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            var menuItem = (ToolStripMenuItem)sender;
            _thumbnailSizeType = (ThumbnailSizeType)menuItem.Tag;
            UpdateImageSize();
        }

        private void _tsbThumbnailSize_ButtonClick(object sender, EventArgs e)
        {
            // 一つずらす
            _thumbnailSizeType =
                (ThumbnailSizeType)(((int)_thumbnailSizeType + 1) % Enum.GetValues(typeof(ThumbnailSizeType)).Length);

            UpdateImageSize();
        }

        private void _tsbPreview_Click(object sender, EventArgs e)
        {
            _showPreview = !_showPreview;
            _UpdatePreviewVisible();
        }

        // プレビューの表示を更新する
        private void _UpdatePreviewVisible()
        {
            // ドッキング状態の非表示時に変更すると配置が崩れる
            if (Visible)
            {
                _pbxPreview.Visible = _showPreview;

                // エクスプローラに合わせて現在の状態の逆のアイコンを表示する
                _tsbPreview.Image = _showPreview ? _bmpPreviewOff : _bmpPreviewOn;
            }
        }

        private void _lvwPartsFiles_DrawItem(object sender, DrawListViewItemEventArgs e)
        {
            // Details の場合は、UIListView の OnDrawItem 内で処理される
            if (_lvwPartsFiles.View == View.LargeIcon)
            {
                // デフォルトの表示を行う
                e.DrawDefault = true;
            }
        }
    }
}
