﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Windows.Forms;
using LayoutEditor.Controls;
using LayoutEditor.Structures.SerializableObject;
using LECore;
using LECore.Structures;

namespace LayoutEditor.Forms.ToolWindows.LayoutWindow
{
    public partial class SelectSubPartsDlg : Form
    {
        /// <summary>
        /// ノードの状態
        /// </summary>
        private class NodeScene
        {
            public readonly ISubScene _subScene;
            public readonly string _descriptions;
            public readonly string _filePath;
            public readonly bool _isCurrent;
            public readonly bool _isChanged;

            public NodeScene(ISubScene subScene, string filePath, string descriptions, bool isCurrent, bool isChanged)
            {
                _subScene = subScene;
                _filePath = filePath;
                _descriptions = descriptions;
                _isCurrent = isCurrent;
                _isChanged = isChanged;
            }
        }

        /// <summary>
        /// サムネイル製造クラス
        /// </summary>
        private class PartsSubsceneThumbnailCreator : SubsceneThumbnailCreator
        {
            public PartsSubsceneThumbnailCreator(AppSetting.RendererType rendererType, AppSetting appSetting)
                : base(rendererType, appSetting)
            {
            }

            /// <summary>
            ///
            /// </summary>
            override protected ISubScene GetSubSceneFromNode_(ListViewItem item)
            {
                return (item.Tag as NodeScene)._subScene;
            }
        }

        //----------------------------------------------------------

        const int _ImgMargin = 4;
        const int _ItemHight = 96;
        const int _MaxItemNum = 8;
        const int _ItemColorGradientStep = 8;

        //----------------------------------------------------------

    　 /// <summary>
    　 /// コンストラクタです。
    　 /// </summary>
        public SelectSubPartsDlg()
        {
            InitializeComponent();

            _lvwMain.View = View.Tile;
            _lvwMain.LargeImageList = new ImageList();
            _lvwMain.LargeImageList.ColorDepth = ColorDepth.Depth32Bit;
            _lvwMain.LargeImageList.ImageSize = new Size(_ItemHight, _ItemHight);

            _newIcon = ImageResMgr.GetManifestResourcePng("Annotation_New.png");

            FormBorderStyle = FormBorderStyle.None;
        }

        //----------------------------------------------------------

        /// <summary>
        ///ユーザによって選択されたファイルパス
        /// </summary>
        public string SelectedPartsFilePath
        {
            get;
            private set;
        }

        /// <summary>
        /// シーン選択イベント
        /// </summary>
        public event EventHandler OnSceneSelected;

        PartsSubsceneThumbnailCreator _subsceneThumbnailCreator = null;
        Image _newIcon = null;

        //----------------------------------------------------------

        /// <summary>
        /// 設定。
        /// </summary>
        public void Setup(List<LayoutWindow.TargetSubSceneSaveState> subScenes, string currentTargetPath, AppSetting appSetting)
        {
            // 非表示状態で実行される想定です。
            System.Diagnostics.Debug.Assert(!this.Visible);

            _lvwMain.Items.Clear();

            this.SelectedPartsFilePath = currentTargetPath;

            // 最初に開いたレイアウト
            ISubScene initialSubScene = subScenes[0].ISubScene;
            {
                string initialFilePath = subScenes[0].LastSavedFilePath;

                ListViewItem itemInitialScene = new ListViewItem();
                itemInitialScene.Text = string.Format("{0}", Path.GetFileNameWithoutExtension(initialFilePath));
                itemInitialScene.Tag = new NodeScene(
                    initialSubScene,
                    initialFilePath,
                    string.Format("[{0}]", StringResMgr.Get("PARTS_INITIAL_LAYOUT")),
                    initialFilePath == currentTargetPath,
                    subScenes[0].IsUpdated);

                _lvwMain.Items.Add(itemInitialScene);
            }

            // 最初に開いたレイアウトの部品ペインを処理する。
            List<string> relatedFilesAll = new List<string>();
            foreach (var partsPane in initialSubScene.FindPanesByKind(PaneKind.Parts))
            {
                // 参照レイアウト
                // 依存階層が深い順に列挙されるので、ひっくり返す。
                List<string> relatedFiles = new List<string>(partsPane.IPartsLayout.EnumlateReferencingLayoutFiles());
                relatedFiles.Reverse();

                // 基底部品
                IPartsSubScene partsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(
                    partsPane.IPartsLayout.PartsLayoutName);
                if (partsSubScene != null && partsSubScene.IsDerivativeParts())
                {
                    relatedFiles.Add(partsSubScene.BasePartsName);
                }

                Color itemColor = Color.White;
                foreach (var relatedFile in relatedFiles)
                {
                    if (relatedFilesAll.Contains(relatedFile))
                    {
                        continue;
                    }

                    ListViewItem item = new ListViewItem();
                    IPartsSubScene ps = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(relatedFile);
                    if (ps != null && ps.IsLoaded)
                    {
                        var saveState = subScenes.Find((ss) => ss.LastSavedFileName == relatedFile);
                        bool IsUpdated = saveState != null ? saveState.IsUpdated : false;

                        item.Text = string.Format("{0}", Path.GetFileNameWithoutExtension(ps.FilePath));
                        item.Tag = new NodeScene(
                            ps.SubScene,
                            ps.FilePath,
                            ps.Description,
                            ps.FilePath == currentTargetPath,
                            IsUpdated);

                        itemColor = GetNextItemColor_(itemColor);
                        item.BackColor = itemColor;

                        _lvwMain.Items.Add(item);

                        relatedFilesAll.Add(relatedFile);
                    }
                }
            }

            // 非同期でサムネイルの作成をスタート
            CreateThumbnail_(_lvwMain, appSetting);

            // サイズ調整
            {
                bool needsScrollBar = _lvwMain.Items.Count >= _MaxItemNum;
                int ScrollBarWidth = needsScrollBar ? 16 : 0;

                const int ItemMargin = 10;

                int MaxHeight = _MaxItemNum * _lvwMain.Items[0].Bounds.Height;
                this.Width = _lvwMain.Items[0].Bounds.Width + ScrollBarWidth + ItemMargin;
                this.Height = Math.Min(_lvwMain.Items.Count * _lvwMain.Items[0].Bounds.Height, MaxHeight) + ItemMargin + 1;

                _lvwMain.Scrollable = needsScrollBar;
            }
        }

        /// <summary>
        /// サムネイル画像を取得します。
        /// </summary>
        public Image GetThumnailImage(string filePath)
        {
            foreach (ListViewItem item in _lvwMain.Items)
            {
                var state = item.Tag as NodeScene;
                if (state._filePath == filePath)
                {
                    if (item.ImageIndex != -1)
                    {
                        return _lvwMain.LargeImageList.Images[item.ImageIndex];
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// 次のノードの色を求める
        /// </summary>
        private Color GetNextItemColor_(Color currentColor)
        {
            return Color.FromArgb(
                Math.Max(currentColor.R - _ItemColorGradientStep, 0),
                Math.Max(currentColor.G - _ItemColorGradientStep, 0),
                currentColor.B);
        }

        /// <summary>
        /// サムネイルを作成します。
        /// </summary>
        private void CreateThumbnail_(ListView targetListView, AppSetting appSetting)
        {
            if (_subsceneThumbnailCreator == null)
            {
                _subsceneThumbnailCreator = new PartsSubsceneThumbnailCreator(AppSetting.RendererType.D3D, appSetting);
                _subsceneThumbnailCreator.Size = new Size(_ItemHight, _ItemHight);
                Disposed += (s, e) =>
                {
                    if (_subsceneThumbnailCreator != null)
                    {
                        _subsceneThumbnailCreator.Dispose();
                        _subsceneThumbnailCreator = null;
                    }
                };
            }
            else
            {
                _subsceneThumbnailCreator.SetAppSetting(appSetting);
            }

            var temp = new List<ListViewItem>();
            foreach (ListViewItem item in targetListView.Items)
            {
                temp.Add(item);
            }

            Action onStartAction = new Action(() =>
            {
                if (targetListView.LargeImageList.Images.Count > 30)
                {
                    for (int i = 0; i < targetListView.LargeImageList.Images.Count; i++)
                    {
                        Image img = targetListView.LargeImageList.Images[i];
                        if (img != null)
                        {
                            img.Dispose();
                        }
                    }

                    targetListView.LargeImageList.Images.Clear();
                    foreach (ListViewItem item in temp)
                    {
                        if (targetListView.LargeImageList.Images.Count == 0)
                        {
                            item.ImageIndex = -1;
                        }
                    }
                }
            });

            _subsceneThumbnailCreator.CaptureAsync(temp, onStartAction);
        }

        //----------------------------------------------------------
        // イベントハンドラ
        //----------------------------------------------------------

        /// <summary>
        /// クリック
        /// </summary>
        private void Event_lvwMain_Click(object sender, EventArgs e)
        {
            if (_lvwMain.SelectedItems.Count > 0)
            {
                this.SelectedPartsFilePath = (_lvwMain.SelectedItems[0].Tag as NodeScene)._filePath;

                this.Hide();
                this.OnSceneSelected(this, e);
            }
        }

        /// <summary>
        /// オーナー描画
        /// </summary>
        private void Event_lvwMain_DrawItem(object sender, DrawListViewItemEventArgs e)
        {
            NodeScene nodeState = e.Item.Tag as NodeScene;
            Point drawPos = e.Bounds.Location;

            // 背景色
            {
                if (e.Item.Index != 0)
                {
                    LinearGradientBrush newLinearGradientBrush = new LinearGradientBrush(
                        new Point(0, e.Bounds.Top), new Point(0, e.Bounds.Bottom),
                        e.Item.BackColor, GetNextItemColor_(e.Item.BackColor));

                    e.Graphics.FillRectangle(newLinearGradientBrush, e.Bounds);
                }
            }

            // 画像
            {
                Size imgSize = new Size(_ItemHight - _ImgMargin * 2, _ItemHight - _ImgMargin * 2);
                {
                    int itemMargin = (e.Bounds.Height - imgSize.Height) / 2;
                    drawPos.Offset(itemMargin, itemMargin);

                    Rectangle imgRect = new Rectangle(drawPos, imgSize);
                    if (e.Item.ImageIndex != -1)
                    {
                        Image img = e.Item.ListView.LargeImageList.Images[e.Item.ImageIndex];
                        e.Graphics.DrawImage(img, imgRect);
                        e.Graphics.DrawRectangle(SystemPens.ControlText, imgRect);
                    }
                    else
                    {
                        e.Graphics.FillRectangle(SystemBrushes.Control, imgRect);
                    }

                    // 更新マーク
                    if (nodeState._isChanged)
                    {
                        e.Graphics.DrawImage(_newIcon, e.Bounds.Right - 12 - _ImgMargin, drawPos.Y, 12, 12);
                    }

                    drawPos.X += imgRect.Width + _ImgMargin;
                }
            }

            // レイアウト名
            {
                e.Graphics.DrawString(e.Item.Text, e.Item.ListView.Font, SystemBrushes.ControlText, drawPos);
                var textSize = e.Graphics.MeasureString(e.Item.Text, e.Item.ListView.Font);
                drawPos.Y += (int)textSize.Height + _ImgMargin;
            }

            // レイアウト名の下に説明。
            {
                StringFormat stringFmt = new StringFormat(StringFormatFlags.LineLimit);
                stringFmt.Trimming = StringTrimming.EllipsisCharacter;
                e.Graphics.DrawString(
                    nodeState._descriptions,
                    this.Font,
                    SystemBrushes.ControlText,
                    Rectangle.FromLTRB(drawPos.X, drawPos.Y, e.Bounds.Right, e.Bounds.Bottom), stringFmt);
            }

            // 枠線
            {
                e.Graphics.DrawRectangle(e.Item.Index == 0 ? SystemPens.ControlText : SystemPens.ControlLight, e.Bounds);

                // 編集中のレイアウトを囲うワク
                if (nodeState._isCurrent)
                {
                    var bCursor = e.Bounds;
                    bCursor.Inflate(-3, -3);
                    e.Graphics.DrawRectangle(new Pen(SystemBrushes.ActiveCaption, 3), bCursor);
                }

                // マウスが乗っているノードを囲うワク
                if ((e.State & ListViewItemStates.Focused) != 0)
                {
                    e.Graphics.DrawRectangle(SystemPens.HotTrack, e.Bounds);
                }
            }
        }

        /// <summary>
        /// マウス移動でフォーカス移動
        /// </summary>
        private void Event_lvwMain_MouseMove(object sender, MouseEventArgs e)
        {
            Point _LvwMainPointToClient = _lvwMain.PointToClient(new Point(e.X, e.Y));
            var node = _lvwMain.GetItemAt(e.X, e.Y);
            if (node != null) { node.Focused = true; }

            _lvwMain.Invalidate();
        }

        /// <summary>
        /// ドラッグ
        /// </summary>
        private void Event_lvwMain_ItemDrag(object sender, ItemDragEventArgs e)
        {
            string filePath = ((e.Item as ListViewItem).Tag as NodeScene)._filePath;
            this.DoDragDrop(filePath, DragDropEffects.Move);
        }

        /// <summary>
        /// フォーカスを失った場合。
        /// </summary>
        private void Event_OpenRelatedPartsDlg_Deactivate(object sender, EventArgs e)
        {
            this.Hide();
        }
    }
}
