﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using LayoutEditor.Forms.ToolWindows.common;
using LECore.Structures;
using LECore.Util;
using CoreManipulator = LECore.Manipulator;
using LECore;

namespace LayoutEditor.Forms.ToolWindows.LayoutWindow
{
    public partial class MainView
    {
        /// <summary>
        /// メインウインドウのマウス状態と、マウスの振る舞いを定義するクラスです。
        ///
        /// Gof State パターンで実装されています。
        /// 本クラスは、Context 役を果たすクラスです。
        ///
        /// 各種マウスイベント
        /// は本クラスに処理が委譲されます。
        ///
        /// また、各マウスモードはキーUP、DOWNイベントについても、
        /// メッセージをハンドリングします
        ///
        /// </summary>
        internal class MainViewMouseState
        {
            #region 定数
            // マウスカーソル
            static readonly Cursor _handOpenCursor = ImageResMgr.GetManifestResourceCursor("hand_open.cur");
            static readonly Cursor _handCloseCursor = ImageResMgr.GetManifestResourceCursor("hand_close.cur");
            static readonly Cursor _setParentParentCursor = ImageResMgr.GetManifestResourceCursor("layout_parent_set_parent.cur");
            static readonly Cursor _setParentChildCursor = ImageResMgr.GetManifestResourceCursor("layout_parent_set_child.cur");
            static readonly Cursor _resetParentParentCursor = ImageResMgr.GetManifestResourceCursor("layout_parent_reset_parent.cur");
            static readonly Cursor _resetParentChildCursor = ImageResMgr.GetManifestResourceCursor("layout_parent_reset_child.cur");
            static readonly Cursor _magnityCursor = ImageResMgr.GetManifestResourceCursor("Magnify.cur");
            static readonly Cursor _magnityMinusCursor = ImageResMgr.GetManifestResourceCursor("MagnifyMinus.cur");
            static readonly Cursor _magnityPlusCursor = ImageResMgr.GetManifestResourceCursor("MagnifyPlus.cur");
            static readonly Cursor _viewScaleCursor = ImageResMgr.GetManifestResourceCursor("ViewScale.cur");
            static readonly Cursor _viewScrollCursor = ImageResMgr.GetManifestResourceCursor("ViewScroll.cur");

            // TODO：カーソル追加、切り替え処理の記述
            #endregion 定数

            /// <summary>
            /// メインビューへの参照
            /// </summary>
            MainView _mainView = null;

            // 現在のステート
            MouseState _curerntMoouseState = MouseState._Idling;
            MouseState _CurerntMoouseState
            {
                set
                {
                    _curerntMoouseState.LeaveState(this);
                    value.EnterState(this);

                    _curerntMoouseState = value;

                    // 変更イベントハンドラを呼びます
                    if (_mainView.MouseStateChange != null)
                    {
                        _mainView.MouseStateChange(_curerntMoouseState, null);
                    }
                }
                get { return _curerntMoouseState; }
            }


            // マウスダウン時の位置
            Point _mouseDownPos;
            PointF _mouseDownPosInScene;


            // マウスダウン時のスクリーン原点
            PointF _screenOrigAtMouseDown;


            /// <summary>
            /// 階層構築中か？
            /// </summary>
            public bool IsMakingHierarchy
            {
                get { return _CurerntMoouseState == MouseState._MakeHierarchy; }
            }

            /// <summary>
            /// 階層破壊中か？
            /// </summary>
            public bool IsBreakingHierarchy
            {
                get { return _CurerntMoouseState == MouseState._BreakHierarchy; }
            }

            /// <summary>
            /// マウスでシーンを操作中か？
            /// </summary>
            public bool IsTweaking
            {
                get { return _CurerntMoouseState == MouseState._SelectedSetTweaking; }
            }

            /// <summary>
            /// 階層構造の構築を開始します。
            /// </summary>
            public void StartHierarchyMaking()
            {
                // 待機状態の場合にのみ...
                if (_CurerntMoouseState == MouseState._Idling)
                {
                    // MakeHierarchyへ移行
                    _CurerntMoouseState = MouseState._MakeHierarchy;
                }
            }

            /// <summary>
            /// 階層構造の構築を開始します。
            /// </summary>
            public void StartHierarchyBreaking()
            {
                // 待機状態の場合にのみ...
                if (_CurerntMoouseState == MouseState._Idling)
                {
                    // MakeHierarchyへ移行
                    _CurerntMoouseState = MouseState._BreakHierarchy;
                }
            }

            /// <summary>
            /// ビュー移動モードに設定します。
            /// </summary>
            public void SetViewTranslatingMode()
            {
                _CurerntMoouseState = MouseState._ViewTranslating;
                MouseState.TryToChangeMouseCursor_(_mainView, MainViewMouseState._handOpenCursor);
            }

            /// <summary>
            /// 待機モードに設定します。
            /// </summary>
            public void SetIdlingMode()
            {
                _CurerntMoouseState = MouseState._Idling;
                MouseState.TryToChangeMouseCursor_(_mainView, Cursors.Arrow);
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="mainView"></param>
            public MainViewMouseState(MainView mainView)
            {
                Debug.Assert(mainView != null);
                _mainView = mainView;
            }

            #region --------- マウスイベントハンドラ ---------
            /// <summary>
            /// ステートパターンによる実装。
            /// 各ステートクラスに処理が委譲されます。
            /// </summary>
            /// <param name="e"></param>
            public void OnMouseDown(MouseEventArgs e)
            {
                Point posMouse = new Point(e.X, e.Y);

                _mouseDownPos = posMouse;

                _screenOrigAtMouseDown = _mainView.ScreenOrigInScene;
                _mouseDownPosInScene = _mainView.ScreenToScene_(_mouseDownPos);

                //DbgConsole.WriteLine("_mouseDownPosInScene => ({0},{1})", _mouseDownPosInScene.X, _mouseDownPosInScene.Y);

                _curerntMoouseState.OnMouseDown(e, this);
            }

            public void OnMouseMove(MouseEventArgs e)
            {
                _curerntMoouseState.OnMouseMove(e, this);
            }

            public void OnMouseUp(MouseEventArgs e)
            {
                _curerntMoouseState.OnMouseUp(e, this);
            }

            #endregion --------- マウスイベントハンドラ ---------

            #region --------- キーダウン等のイベント ---------
            /// <summary>
            /// キーダウン等のイベントもマウスステートクラスに処理が委譲されます
            /// </summary>
            public void OnKeyDown(KeyEventArgs e)
            {
                _curerntMoouseState.OnKeyDown(e, this);
            }

            public void OnKeyUp(KeyEventArgs e)
            {
                _curerntMoouseState.OnKeyUp(e, this);
            }
            #endregion --------- キーダウン等のイベント ---------

            #region MouseState クラスと、その派生クラス
            /// <summary>
            /// マウス状態クラス。
            /// 仮想基底クラスです。
            /// </summary>
            abstract public class MouseState
            {
                // ステートクラスインスタンス。
                public static readonly MouseState _Idling = new MouseState_Idling();
                public static readonly MouseState _AreaSelecting = new MouseState_AreaSelecting();
                public static readonly MouseState _SelectedSetTweaking = new MouseState_SelectedSetTweaking();
                public static readonly MouseState _ViewTranslating = new MouseState_ViewTranslating();
                public static readonly MouseState _MakeHierarchy = new MouseState_MakeHierarchy();
                public static readonly MouseState _BreakHierarchy = new MouseState_BreakHierarchy();
                public static readonly MouseState _Duplicate = new MouseState_Duplicate();

                /// <summary>
                /// コンストラクタ
                /// </summary>
                public MouseState()
                {

                }

                /// <summary>
                /// ステートの名称を取得します。
                /// </summary>
                public abstract string StateName
                {
                    get;
                }

                /// <summary>
                /// ペイン選択処理を中断するかを判定します。中断する場合 false を返してください。
                /// </summary>
                protected bool PaneSelectTerminateCondition(IPane pane, LECore.Structures.Core.DrawableOption drawableOption)
                {
                    // 非表示状態にしている、Capture ペインは子階層含め選択しません。
                    if (pane.PaneKind == PaneKind.Capture && drawableOption.CheckActive(LECore.Structures.Core.DrawableOptionFlag.IgnoreCapturePaneDrawing))
                    {
                        return false;
                    }

                    return true;
                }

                /// <summary>
                /// MouseDownハンドラ
                /// </summary>
                protected void OnMouseDown_(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    PaneDragTargetAdapter paneAdapter = null;
                    MainView mainView = mouseState._mainView;
                    PointF scenePos = mouseState._mouseDownPosInScene;
                    bool alreadySendClickedEvent = false;

                    ISubScene currentScene = mainView._TargetActiveSubScene;

                    // 選択のリセット
                    mainView._subSceneMnp.BindTarget(currentScene);
                    mainView._subSceneMnp.BeginSelectSetChange();

                    Func<IPane, bool> selectTerminateCondition = (pane) => PaneSelectTerminateCondition(pane, mainView.DrawableOption);

                    // shiftがおされていれば
                    if (Control.ModifierKeys == Keys.Shift)
                    {
                        // 衝突したペインを取得します。
                        IPane hitPane = SubSceneHelper.GetPanesByPoint(mainView._subSceneMnp.ISubScene, new FVec2(scenePos), selectTerminateCondition);
                        if (hitPane != null)
                        {
                            // 衝突したペインがふくまれていれば取り除く、ふくまれていなけば追加する。
                            mainView._subSceneMnp.InvertSelection(hitPane);
                        }
                        else
                        {
                            mouseState._CurerntMoouseState = MouseState._AreaSelecting;
                        }
                    }
                    else
                    {
                        //PaneDragTargetAdapter paneAdapter;
                        if (mainView._draggedPane.CheckClicked(mainView._magnify, scenePos, out paneAdapter))
                        {
                            // クリックイベントを送信する。
                            if (paneAdapter != null)
                            {
                                Matrix mtx = PaneHelper.CalcLeftTopCoordinateWorldMtx(paneAdapter.IPane);
                                if (mtx.IsInvertible)
                                {
                                    mtx.Invert();
                                }
                                scenePos = MathUtil.MtxTransformPoint(mtx, scenePos);
                                mainView._TargetActiveSubScene.ISelectedSet.NotifySelectedSetClickedEvent(paneAdapter.IPane, scenePos);

                                // クリックイベントが一度、送られたことを記録する。
                                alreadySendClickedEvent = true;
                            }

                            // Altによる回転操作と複製操作のバッティングを抑止するため、
                            // ドラッグ時に変更モードをチェックし、Rotateなら調整モードに遷移させる
                            DragModifierHelper.Contact modifierContact;
                            IPane[] paneSet = mainView._targetActiveSubScene.ISelectedSet.IPaneArray;
                            var modifyMode = mainView._draggedPane.GetModifyMode(
                                PaneDragTargetAdapter.FromPaneSet(paneSet),
                                mainView._magnify,
                                scenePos,
                                DragModifierHelper.OptionFlag.IgnoreShortDrag,
                                out modifierContact);

                            if (modifyMode != PaneDragModifier.Mode.Rotate && (Control.ModifierKeys & Keys.Alt) != 0)
                            {
                                // 複製モードに移行。
                                mouseState._CurerntMoouseState = MouseState._Duplicate;
                            }
                            else
                            {
                                // 調整モードに移行。
                                mouseState._CurerntMoouseState = MouseState._SelectedSetTweaking;
                            }
                        }
                        else
                        {
                            // 衝突したペインを取得します。
                            IPane hitPane = SubSceneHelper.GetPanesByPoint(mainView._subSceneMnp.ISubScene, new FVec2(scenePos), selectTerminateCondition);
                            // ペインが存在すれば...
                            if (hitPane != null)
                            {
                                // おされていなければ
                                // 選択セットをリセットし、衝突したペインを選択セットに登録する。
                                if (!hitPane.IsSelected)
                                {
                                    mainView._subSceneMnp.ResetSelectedSet();
                                }
                                mainView._subSceneMnp.SelectPanesByPaneRef(hitPane);

                                if ((Control.ModifierKeys & Keys.Alt) != 0)
                                {
                                    // 複製モードに移行。
                                    mouseState._CurerntMoouseState = MouseState._Duplicate;
                                }
                                else
                                {
                                    // 調整モードに移行。
                                    mouseState._CurerntMoouseState = MouseState._SelectedSetTweaking;
                                }
                            }
                            else
                            {
                                // ペインが存在しなければ、範囲選択モードに移行します。
                                mouseState._CurerntMoouseState = MouseState._AreaSelecting;
                            }
                        }
                    }

                    mainView._subSceneMnp.EndSelectSetChange();

                    // 既に一度、クリックイベントが送られていたら処理しない。
                    if (alreadySendClickedEvent == false)
                    {
                        ProcessClickedEvent(mainView, mouseState._mouseDownPosInScene);
                    }
                }

                /// <summary>
                /// クリックイベントを送信する。
                /// </summary>
                private void ProcessClickedEvent(MainView mainView, PointF posInScene)
                {
                    PaneDragTargetAdapter paneAdapter = null;

                    if (mainView._draggedPane.CheckClicked(mainView._magnify, posInScene, out paneAdapter))
                    {
                        if (paneAdapter != null)
                        {
                            Matrix mtx = PaneHelper.CalcLeftTopCoordinateWorldMtx(paneAdapter.IPane);
                            if (mtx.IsInvertible)
                            {
                                mtx.Invert();
                            }
                            posInScene = MathUtil.MtxTransformPoint(mtx, posInScene);
                            ISelectedSet selectedSet = mainView._TargetActiveSubScene.ISelectedSet;
                            selectedSet.NotifySelectedSetClickedEvent(paneAdapter.IPane, posInScene);
                        }
                    }
                }

                /// <summary>
                /// マウスカーソルの変更
                /// </summary>
                static public void TryToChangeMouseCursor_(MainView mainView, Cursor cursor)
                {
                    if (mainView.Cursor == Cursors.WaitCursor)
                    {
                        return;
                    }

                    mainView.Cursor = cursor;
                }

                /// <summary MouseDownハンドラ </summary>
                public virtual void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState) { }

                /// <summary MouseMoveハンドラ </summary>
                public virtual void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState) { }

                /// <summary MouseUpハンドラ </summary>
                public virtual void OnMouseUp(MouseEventArgs e, MainViewMouseState mouseState) { }

                /// <summary> KeyDownハンドラ </summary>
                public virtual void OnKeyDown(KeyEventArgs e, MainViewMouseState mouseState)
                {
                }

                /// <summary> KeyUpハンドラ </summary>
                public virtual void OnKeyUp(KeyEventArgs e, MainViewMouseState mouseState)
                {
                    // ビュー移動モード時に、スペースキーアップが検知されたら、
                    // マウスモードをリセットします。
                    if (mouseState._CurerntMoouseState == MouseState._ViewTranslating)
                    {
                        if (e.KeyCode == Keys.Space)
                        {
                            mouseState._CurerntMoouseState = MouseState._Idling;
                        }
                    }
                }

                /// <summary>
                /// ステート開始時の処理を記述します。
                /// </summary>
                public virtual void EnterState(MainViewMouseState mouseState) { }

                /// <summary>
                /// ステート終了時の処理を記述します。
                /// </summary>
                public virtual void LeaveState(MainViewMouseState mouseState) { }
            }

            ///
            class MouseState_Idling : MouseState
            {
                public override string StateName { get { return ""; } }

                public override void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    base.OnMouseDown_(e, mouseState);
                }

                /// <summary>
                /// キーダウン
                /// </summary>
                public override void OnKeyDown(KeyEventArgs e, MainViewMouseState mouseState)
                {
                    // スペースが押されたらView移動モードに移行します。
                    if ((e.KeyData & Keys.KeyCode) == Keys.Space)
                    {
                        mouseState._CurerntMoouseState = MouseState._ViewTranslating;
                    }

                    // カーソル更新
                    UpdateKeyCursor_(mouseState._mainView.PointToClient(Control.MousePosition), mouseState);

                    base.OnKeyDown(e, mouseState);
                }

                /// <summary>
                /// KeyUpハンドラ
                /// </summary>
                public override void OnKeyUp(KeyEventArgs e, MainViewMouseState mouseState)
                {
                    // カーソル更新
                    UpdateKeyCursor_(mouseState._mainView.PointToClient(Control.MousePosition), mouseState);

                    base.OnKeyUp(e, mouseState);
                }

                /// <summary>
                /// マウス移動
                /// </summary>
                public override void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // ホイールが動いたらビュー変更モードへ
                    if (e.Delta != 0)
                    {
                        mouseState._CurerntMoouseState = MouseState._ViewTranslating;
                        mouseState.OnMouseMove(e);
                    }
                    else
                    {
                        // カーソル更新
                        UpdateKeyCursor_(e.Location, mouseState);
                    }
                }

                /// <summary>
                /// カーソルを更新します。
                /// </summary>
                private void UpdateKeyCursor_(Point mousePos, MainViewMouseState mouseState)
                {
                    // Alt押しのときは、回転操作をします。
                    if ((Control.ModifierKeys & Keys.Alt) != 0)
                    {
                        mouseState._mainView._draggedPane.AddOptionFlag(PaneDragModifier.Option.TryToRotate);
                    }
                    else
                    {
                        mouseState._mainView._draggedPane.ResetOptionFlag(PaneDragModifier.Option.TryToRotate);
                    }

                    var newCursor = mouseState._mainView._draggedPane.GetCursor(
                        PaneDragTargetAdapter.FromPaneSet(mouseState._mainView._targetActiveSubScene.ISelectedSet.IPaneArray),
                        mouseState._mainView.Magnify,
                        mouseState._mainView.ScreenToScene_(mousePos),
                        DragModifierHelper.OptionFlag.IgnoreShortDrag);

                    TryToChangeMouseCursor_(mouseState._mainView, newCursor);
                }

                public override void EnterState(MainViewMouseState mouseState)
                {
                    TryToChangeMouseCursor_(mouseState._mainView, Cursors.Arrow);
                }
            }

            /// <summary>
            ///
            /// </summary>
            class MouseState_AreaSelecting : MouseState
            {
                public override string StateName { get { return "Selecting"; } }

                /// <summary>
                /// 座標変換
                /// </summary>
                /// <param name="pos"></param>
                /// <returns></returns>
                PointF ConvertPosFromScrnToScene_(MainViewMouseState mouseState, PointF pos)
                {
                    // ウインドウ中心からの座標に変換
                    PointF posCtr = MathUtil.ScaleVec(new Point(mouseState._mainView.Size), 0.5f);
                    pos = MathUtil.SubVec(pos, posCtr);
                    pos = new PointF(pos.X, -pos.Y);

                    return pos;
                }

                /// <summary>
                /// モードエントリ
                /// </summary>
                public override void EnterState(MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    mainView._selectRect.BeginSelecting(ConvertPosFromScrnToScene_(mouseState, mouseState._mouseDownPos));
                    mainView.Invalidate();

                    base.EnterState(mouseState);
                }


                public override void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    base.OnMouseDown_(e, mouseState);
                }

                public override void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // ビューの位置を変更します
                    MainView mainView = mouseState._mainView;
                    PointF posMouse = new PointF(e.X, e.Y);
                    posMouse = ConvertPosFromScrnToScene_(mouseState, posMouse);

                    mainView._selectRect.UpDataSize(posMouse);
                    mainView.Invalidate();
                }

                public override void OnMouseUp(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    PointF posMouse = new PointF(e.X, e.Y);
                    posMouse = ConvertPosFromScrnToScene_(mouseState, posMouse);

                    mainView._selectRect.UpDataSize(posMouse);
                    RectangleF selRect = mainView._selectRect.EndSelecting();

                    // 左上原点、Y軸下向き座標系に変換して判定します。
                    selRect.X = selRect.X + mouseState._mainView.Size.Width * 0.5f;
                    selRect.Y = mouseState._mainView.Size.Height * 0.5f - (selRect.Y + selRect.Height);

                    selRect = mainView.ScreenRectToSceneRect_(selRect);

                    Func<IPane, bool> selectTerminateCondition = (pane) => PaneSelectTerminateCondition(pane, mainView.DrawableOption);

                    ArrayList hitPanes = SubSceneHelper.GetPanesByRect(mainView._subSceneMnp.ISubScene, selRect, selectTerminateCondition);

                    mainView._subSceneMnp.BeginSelectSetChange();
                    // Shiftが押されていない
                    if (Control.ModifierKeys != Keys.Shift)
                    {
                        // 選択範囲をリセット後、シーンを選択します。
                        mainView._subSceneMnp.ResetSelectedSet();

                        foreach (IPane pane in hitPanes)
                        {
                            mainView._subSceneMnp.SelectPanesByPaneRef(pane);
                        }
                    }
                    else
                    {
                        // 選択状態を反転します。
                        foreach (IPane pane in hitPanes)
                        {
                            mainView._subSceneMnp.InvertSelection(pane);
                        }
                    }


                    mainView._subSceneMnp.EndSelectSetChange();
                    mainView.Invalidate();

                    // マウスモードをリセットします。
                    mouseState._CurerntMoouseState = MouseState._Idling;
                }
            }

            /// <summary>
            ///
            /// </summary>
            class MouseState_SelectedSetTweaking : MouseState
            {
                public override string StateName { get { return "Tweaking"; } }


                /// <summary>
                ///
                /// </summary>
                void ResetDragging_(MainView mainView, PointF scenePos)
                {
                    if (mainView.PaneDragModifierChangesSize)
                    {
                        mainView._draggedPane.AddOptionFlag(PaneDragModifier.Option.ModifySize);
                    }

                    mainView._draggedPane.ResetDragging(mainView._magnify, scenePos);

                    // グリッド吸着が有効ならば、グリッドサイズを設定します。
                    if (mainView.BackGround.GridSnapEnalbed)
                    {
                        mainView._draggedPane.SetGridSnapEnable(mainView._GridSize);
                    }
                    else
                    {
                        mainView._draggedPane.SetGridSnapEnable(1.0f);
                    }
                }

                /// <summary>
                /// モードエントリ
                /// </summary>
                public override void EnterState(MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    PointF scenePos = mouseState._mouseDownPosInScene;

                    if (!mainView._draggedPane.Empty)
                    {
                        // Alt押しのときは、回転操作をします。
                        if ((Control.ModifierKeys & Keys.Alt) != 0)
                        {
                            // mainView._draggedPane.AddOptionFlag(PaneDragModifier.Option.TryToRotate);
                        }
                        else
                        {
                            // "T"が押されていた時には移動操作を行います。
                            if ((GetAsyncKeyState(Keys.T) & 0x8000) != 0)
                            {
                                mainView._draggedPane.AddOptionFlag
                                    (PaneDragModifier.Option.ForceToTranslate);
                            }
                        }

                        ResetDragging_(mainView, scenePos);
                    }

                    base.EnterState(mouseState);
                }

                /// <summary>
                /// モード終了
                /// </summary>
                public override void LeaveState(MainViewMouseState mouseState)
                {
                    TryToChangeMouseCursor_(mouseState._mainView, Cursors.Arrow);
                    base.LeaveState(mouseState);
                }

                /// <summary>
                /// MouseDown
                /// </summary>
                public override void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // ドラッグ始点をリセットします。
                    MainView mainView = mouseState._mainView;
                    Point posMouse = new Point(e.X, e.Y);

                    ResetDragging_(mainView, mainView.ScreenToScene_(posMouse));

                    base.OnMouseDown_(e, mouseState);
                }

                /// <summary>
                /// MouseMove
                /// </summary>
                public override void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // ビューの位置を変更します
                    MainView mainView = mouseState._mainView;
                    PaneDragModifier draggedPane = mainView._draggedPane;

                    if (mainView._draggedPane.Empty)
                    {
                        // マウスカーソルを変更する。
                        return;
                    }

                    // 移動モード
                    if(mainView._draggedPane.ModifyMode == PaneDragModifier.Mode.Scale||
                        mainView._draggedPane.ModifyMode == PaneDragModifier.Mode.Size)
                    {
                        TryToChangeMouseCursor_(mouseState._mainView, Cursors.Cross);
                    }

                    Point posMouse = new Point(e.X, e.Y);

                    //------------------------------------------------------------------
                    // 選択グループの位置を更新します。
                    // Shift が押されている場合には、移動方向が限定されます。
                    if ((Control.ModifierKeys & Keys.Shift) != 0)
                    {
                        draggedPane.AddOptionFlag(PaneDragModifier.Option.SnapDirections);
                    }

                    PointF posInScene = mainView.ScreenToScene_(posMouse);

                    // 調整処理をキャンセルします。
                    draggedPane.UpdateDragging(mainView._magnify, posInScene);

                    mainView.Invalidate();
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseUp(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;

                    if (mainView._draggedPane.Empty)
                    {
                        return;
                    }

                    PointF posLastStart = mainView._draggedPane.DragStartPointF;

                    // ドラッグを終了し、移動量を取得します。
                    mainView._draggedPane.EndDragging();

                    // 再度選択します。
                    IPane[] paneSet = mainView._targetActiveSubScene.ISelectedSet.IPaneArray;

                    mainView._draggedPane.BeginDragging(
                        PaneDragTargetAdapter.FromPaneSet(paneSet),
                        mainView._magnify,
                        posLastStart,
                        DragModifierHelper.OptionFlag.IgnoreShortDrag);

                    // マウスモードをリセットします。
                    mouseState._CurerntMoouseState = MouseState._Idling;
                }
            }

            /// <summary>
            ///
            /// </summary>
            class MouseState_ViewTranslating : MouseState
            {
                /// <summary>
                /// ズームの種類
                /// </summary>
                private enum ZoomMode
                {
                    /// <summary>
                    /// 非ズーム
                    /// </summary>
                    None,
                    /// <summary>
                    /// ズーム処理準備中
                    /// </summary>
                    Prepare,
                    /// <summary>
                    /// ボタン押しズーム
                    /// </summary>
                    LPress,
                    /// <summary>
                    /// ドラッグズーム
                    /// </summary>
                    LDrag,
                    /// <summary>
                    /// ホイールズーム
                    /// </summary>
                    Wheel,
                };

                //----------------------------------------------------------
                Timer _keyStateChecker = null;

                bool _buttonPush = false;
                bool _wheelMove = false;
                int _buttonPushFrameCount = 0;
                ZoomMode _zoomMode = ZoomMode.None;
                float _originalMagnify = 1.0f;
                MainViewMouseState _targetMouseState;

                const float OnShiftTransMagnify = 5.0f;
                const float OnShiftZoomMagnify = 2.0f;

                //----------------------------------------------------------

                /// <summary>
                /// ステート名文字列
                /// </summary>
                public override string StateName
                {
                    get { return _ZoomMode != ZoomMode.None ? "Zoom" : "Move"; }
                }

                /// <summary>
                /// ズームの種類
                /// </summary>
                ZoomMode _ZoomMode
                {
                    get { return _zoomMode; }
                    set
                    {
                        // DbgConsole.WriteLine("{0} => {1}", _zoomMode, value);
                        _zoomMode = value;
                        UpdateCursor_();
                    }
                }

                /// <summary>
                /// マウスボタンが押されている。
                /// </summary>
                bool _MousebuttonPush
                {
                    get { return _buttonPush; }
                    set
                    {
                        _buttonPush = value;
                        UpdateCursor_();
                    }
                }

                /// <summary>
                /// マウスボタンが押されている。
                /// </summary>
                bool _WheelMove
                {
                    get { return _wheelMove; }
                    set
                    {
                        _wheelMove = value;
                        UpdateCursor_();
                    }
                }

                MainView _OwnerMainView
                {
                    get { return _targetMouseState != null ? _targetMouseState._mainView : null; }
                }

                private bool _IsSpacePush { get { return LECore.Win32.User32.GetKeyState(LECore.Win32.VK.VK_SPACE) < 0; } }
                private bool _IsAltPush { get { return LECore.Win32.User32.GetKeyState(LECore.Win32.VK.VK_MENU) < 0; } }
                private bool _IsCtrPush { get { return LECore.Win32.User32.GetKeyState(LECore.Win32.VK.VK_CONTROL) < 0; } }

                //----------------------------------------------------------

                /// <summary>
                /// 状態を更新します。
                /// </summary>
                private void UpdateState_(object sender, EventArgs e)
                {

                    // 現在のカーソルが、自分のウインドウ上に無い時は処理しない
                    {
                        if (_OwnerMainView == null)
                        {
                            return;
                        }

                        if (!_OwnerMainView.IsHandleCreated)
                        {
                            return;
                        }

                        if (!_OwnerMainView.Bounds.Contains(_OwnerMainView.PointToClient(Control.MousePosition)))
                        {
                            return;
                        }
                    }

                    // Lボタン状態更新。
                    _MousebuttonPush = Control.MouseButtons == MouseButtons.Left;

                    // _WheelMove は MouseMove で更新します。

                    // 押しっぱなしの判定
                    if (_MousebuttonPush && (_IsCtrPush || _IsAltPush))
                    {
                        if (_ZoomMode != ZoomMode.LDrag)
                        {
                            _buttonPushFrameCount++;
                            if (_buttonPushFrameCount > 10)
                            {
                                if (_IsCtrPush)
                                {
                                    _ZoomMode = ZoomMode.LPress;
                                    _targetMouseState._mainView.Magnify *= 1.1f;
                                }
                                else if (_IsAltPush)
                                {
                                    _ZoomMode = ZoomMode.LPress;
                                    _targetMouseState._mainView.Magnify *= 0.9f;
                                }

                                return;
                            }
                        }
                        else
                        {
                            _buttonPushFrameCount = 0;
                        }
                    }

                    // それ以外
                    ZoomMode nextMode = _ZoomMode;
                    if(_IsSpacePush)
                    {
                        // スペースを押している時
                        if (_IsAltPush)
                        {
                            if (_ZoomMode == ZoomMode.None)
                            {
                                nextMode = ZoomMode.Prepare;
                            }
                        }
                        else if (_IsCtrPush)
                        {
                            if (_ZoomMode == ZoomMode.None)
                            {
                                nextMode = ZoomMode.Prepare;
                            }
                        }else
                        {
                            nextMode = ZoomMode.None;
                        }
                    }
                    else
                    {
                        // ホイール操作の時
                        if (_IsAltPush)
                        {
                            if (_ZoomMode == ZoomMode.None)
                            {
                                nextMode = ZoomMode.Wheel;
                            }
                        }
                        else
                        {
                            nextMode = ZoomMode.None;
                        }
                    }

                    // ズーム状態に変更があれば設定して、UI状態を変更する。
                    if (_ZoomMode != nextMode)
                    {
                        _ZoomMode = nextMode;

                        _targetMouseState._mainView.MouseStateChange(_targetMouseState._CurerntMoouseState, null);
                    }
                }

                /// <summary>
                /// マウスカーソルの更新
                /// </summary>
                private void UpdateCursor_()
                {
                    Cursor newCursor = _targetMouseState._mainView.Cursor;
                    switch (_ZoomMode)
                    {
                        case ZoomMode.None:
                            {
                                if (_IsSpacePush)
                                {
                                    newCursor = (_MousebuttonPush) ?
                                        MainViewMouseState._handCloseCursor : MainViewMouseState._handOpenCursor;
                                }
                                else if(_WheelMove)
                                {
                                    newCursor = Cursors.Arrow;
                                }

                                break;
                            }
                        case ZoomMode.Prepare:
                            {
                                newCursor = (_IsCtrPush) ?
                                    MainViewMouseState._magnityPlusCursor : MainViewMouseState._magnityMinusCursor;
                                break;
                            }
                        case ZoomMode.LPress:
                            {
                                newCursor = (_IsCtrPush) ?
                                    MainViewMouseState._magnityPlusCursor : MainViewMouseState._magnityMinusCursor;
                                break;
                            }
                        case ZoomMode.LDrag:
                            {
                                newCursor = MainViewMouseState._magnityCursor;
                                break;
                            }
                        case ZoomMode.Wheel:
                            {
                                newCursor = MainViewMouseState._magnityCursor;
                                break;
                            }
                    }

                    TryToChangeMouseCursor_(_targetMouseState._mainView, newCursor);
                }

                //----------------------------------------------------------

                public override void EnterState(MainViewMouseState mouseState)
                {
                    if (_keyStateChecker == null)
                    {
                        _keyStateChecker = new Timer();
                        _keyStateChecker.Interval = 100; // ms
                        _keyStateChecker.Tick += UpdateState_;
                    }

                    _targetMouseState = mouseState;
                    ResetState_();

                    _keyStateChecker.Start();
                }

                /// <summary>
                /// 状態をリセットします。
                /// </summary>
                private void ResetState_()
                {
                    _originalMagnify = 1.0f;
                    _MousebuttonPush = false;
                    _WheelMove = false;
                    _buttonPushFrameCount = 0;
                    _ZoomMode = ZoomMode.None;
                }

                public override void LeaveState(MainViewMouseState mouseState)
                {
                    _keyStateChecker.Stop();
                    ResetState_();
                    TryToChangeMouseCursor_(mouseState._mainView, Cursors.Arrow);
                }

                public override void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    _MousebuttonPush = true;
                }

                private void TranslateView_(MainViewMouseState mouseState, PointF diff, PointF origine)
                {
                    MainView mainView = mouseState._mainView;

                    diff = MathUtil.ScaleVec(diff, 1.0f / mainView.Magnify);
                    diff.Y *= -1.0f;

                    diff.X *= mainView._WidthRate;
                    diff.Y *= mainView._HeightRate;

                    mainView.ScreenOrigInScene = MathUtil.SubVec(origine, diff);
                }

                public override void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    Point posMouse = new Point(e.X, e.Y);

                    _WheelMove = e.Delta != 0;

                    if (_WheelMove && (Control.ModifierKeys & Keys.Alt) != 0)
                    {
                        _ZoomMode = ZoomMode.Wheel;
                    }

                    if (_ZoomMode == ZoomMode.None)
                    {
                        if (_WheelMove)
                        {
                            // ホイール操作での平行移動
                            float value = e.Delta > 0 ? 20.0f : -20.0f;

                            PointF diff = ((Control.ModifierKeys & Keys.Control) != 0) ? new PointF(value, 0.0f) : new PointF(0.0f, value);

                            if ((Control.ModifierKeys & Keys.Shift) != 0)
                            {
                                diff = MathUtil.ScaleVec(diff, OnShiftTransMagnify);
                            }

                            TranslateView_(mouseState, diff, mainView.ScreenOrigInScene);
                            mainView.Invalidate();
                        }
                        else
                        {
                            if (!_IsSpacePush)
                            {
                                return;
                            }

                            // 平行移動
                            if (_MousebuttonPush && Control.ModifierKeys == Keys.None)
                            {
                                PointF diff = MathUtil.PointToPointF(MathUtil.SubVec(posMouse, mouseState._mouseDownPos));
                                TranslateView_(mouseState, diff, mouseState._screenOrigAtMouseDown);
                                mainView.Invalidate();
                            }
                        }
                    }
                    else
                    {
                        // ビューの倍率を変更します
                        if (_WheelMove)
                        {
                            // ホイールが動いた
                            _ZoomMode = ZoomMode.Wheel;

                            float delta = e.Delta;

                            if ((Control.ModifierKeys & Keys.Shift) != 0)
                            {
                                delta *= OnShiftZoomMagnify;
                            }

                            if (_IsSpacePush)
                            {
                                if ((Control.ModifierKeys & Keys.Alt) != 0)
                                {
                                    mainView.ZoomForm(mainView.FindForm(), e.Delta > 0);
                                }
                                else if ((Control.ModifierKeys & Keys.Control) != 0)
                                {
                                    mainView.Magnify += delta * 0.0007f;
                                    _originalMagnify = mainView.Magnify;

                                    mainView.Invalidate();
                                }
                            }
                            else
                            {
                                if ((Control.ModifierKeys & Keys.Alt) != 0)
                                {
                                    mainView.Magnify += delta * 0.0007f;
                                    _originalMagnify = mainView.Magnify;

                                    mainView.Invalidate();
                                }
                            }
                        }
                        else
                        {
                            if (!_IsSpacePush)
                            {
                                return;
                            }

                            // ドラッグ
                            if (_MousebuttonPush)
                            {
                                PointF diff = MathUtil.PointToPointF(MathUtil.SubVec(posMouse, mouseState._mouseDownPos));

                                // ちょっとドラッグしただけなら処理しない
                                if (MathUtil.GetVecLength(diff) > 10.0f)
                                {
                                    _ZoomMode = ZoomMode.LDrag;

                                    mainView.Magnify = _originalMagnify + diff.X * 0.01f;
                                    mainView.Invalidate();
                                }
                            }
                        }
                    }
                }

                public override void OnMouseUp(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    _MousebuttonPush = false;
                    _buttonPushFrameCount = 0;

                    // スペースがおされていれば...
                    if (LECore.Win32.User32.GetKeyState(LECore.Win32.VK.VK_SPACE) < 0)
                    {
                        if (_ZoomMode == ZoomMode.None || _ZoomMode == ZoomMode.Prepare)
                        {
                            if (_IsAltPush)
                            {
                                mouseState._mainView.ZoomOut();
                            }
                            else if (_IsCtrPush)
                            {
                                mouseState._mainView.ZoomIn();
                            }
                        }

                        _ZoomMode = ZoomMode.None;
                        _originalMagnify = mouseState._mainView.Magnify;
                    }
                    else
                    {
                        mouseState._CurerntMoouseState = MouseState._Idling;
                    }
                }

                public override void OnKeyUp(KeyEventArgs e, MainViewMouseState mouseState)
                {
                    // スペースが離されたかつ、Lボタンがクリックされていない
                    if (((e.KeyData & Keys.Space) != 0 && !_MousebuttonPush))
                    {
                        // Altは押されていない
                        if (!e.Alt)
                        {
                            // 通常モードに移行します。
                            mouseState._CurerntMoouseState = MouseState._Idling;
                        }
                    }
                    else if (!e.Alt)
                    {
                        // （おそらく）Altが離された。
                        // Spaceが押されていない
                        if (LECore.Win32.User32.GetKeyState(LECore.Win32.VK.VK_SPACE) >= 0)
                        {
                            // 通常モードに移行します。
                            mouseState._CurerntMoouseState = MouseState._Idling;
                        }
                    }

                    base.OnKeyDown(e, mouseState);
                }
            }

            /// <summary>
            ///
            /// </summary>
            class MouseState_MakeHierarchy : MouseState
            {
                public override string StateName { get { return "MakeHierarchy"; } }

                /// <summary>
                ///
                /// </summary>
                public override void EnterState(MainViewMouseState mouseState)
                {
                    TryToChangeMouseCursor_(mouseState._mainView, MainViewMouseState._setParentParentCursor);
                }

                /// <summary>
                ///
                /// </summary>
                public override void LeaveState(MainViewMouseState mouseState)
                {
                    TryToChangeMouseCursor_(mouseState._mainView, Cursors.Default);
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // クリック点にペインが存在するなら
                    MainView mainView = mouseState._mainView;


                    Point posInScene = MathUtil.PointFToPoint(mouseState._mouseDownPosInScene);

                    if (mainView._hierarchyMnp.BeginMakingHierarchy(posInScene))
                    {
                        mainView.Invalidate();
                        TryToChangeMouseCursor_(mouseState._mainView, MainViewMouseState._setParentChildCursor);
                    }
                    else
                    {
                        // 階層生成モードを解除
                        // _mouseOpMode = MouseOpMode.None;
                        mouseState._CurerntMoouseState = MouseState._Idling;
                    }


                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    CoreManipulator.HierarchyManipulator hierarchyBuilder = mainView._hierarchyMnp;

                    // 階層構築中ならば...
                    if (hierarchyBuilder.IsBuilding || hierarchyBuilder.IsBreaking)
                    {
                        // マウス離しが検出されたら、モードを階層生成を解除します
                        if (e.Button != MouseButtons.Left)
                        {
                            hierarchyBuilder.EndMakingHierarchy();
                            mouseState._CurerntMoouseState = MouseState._Idling;
                        }
                        else
                        {
                            // 現在位置に存在するペインが子ノード候補として妥当か調査します。
                            Point posInScene = MathUtil.PointFToPoint(mainView.ScreenToScene_(new Point(e.X, e.Y)));
                            hierarchyBuilder.SetCurrentMousePos(posInScene);
                            mainView.Invalidate();
                        }
                    }
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseUp(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    // 必ず構築中であるはず...
                    Debug.Assert(mainView._hierarchyMnp.IsBuilding);
                    mainView._hierarchyMnp.EndMakingHierarchy();
                    mainView.Invalidate();

                    // マウスモードをリセットします。
                    mouseState._CurerntMoouseState = MouseState._Idling;
                }
            }

            /// <summary>
            ///
            /// </summary>
            class MouseState_BreakHierarchy : MouseState
            {
                public override string StateName { get { return "BreakHierarchy"; } }

                /// <summary>
                ///
                /// </summary>
                public override void EnterState(MainViewMouseState mouseState)
                {
                    TryToChangeMouseCursor_(mouseState._mainView, MainViewMouseState._resetParentParentCursor);
                }

                /// <summary>
                ///
                /// </summary>
                public override void LeaveState(MainViewMouseState mouseState)
                {
                    TryToChangeMouseCursor_(mouseState._mainView, Cursors.Default);
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;

                    // Event_MainView_MouseDown_BreakHierarchy_( sender, e );
                    // クリック点にペインが存在するなら
                    Point posInScene = MathUtil.PointFToPoint(mouseState._mouseDownPosInScene);
                    if (mainView._hierarchyMnp.BeginBreakingHierarchy(posInScene))
                    {
                        mainView.Invalidate();
                        TryToChangeMouseCursor_(mouseState._mainView, MainViewMouseState._resetParentChildCursor);
                    }
                    else
                    {
                        // 階層生成モードを解除
                        // _mouseOpMode = MouseOpMode.None;
                        mouseState._CurerntMoouseState = MouseState._Idling;
                    }
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    CoreManipulator.HierarchyManipulator hierarchyBuilder = mainView._hierarchyMnp;

                    // 階層構築中ならば...
                    if (hierarchyBuilder.IsBuilding || hierarchyBuilder.IsBreaking)
                    {
                        // マウス離しが検出されたら、モードを階層生成を解除します
                        if (e.Button != MouseButtons.Left)
                        {
                            hierarchyBuilder.EndBreakingHierarchy();
                            mouseState._CurerntMoouseState = MouseState._Idling;
                        }
                        else
                        {
                            // 現在位置に存在するペインが子ノード候補として妥当か調査します。
                            Point posInScene = MathUtil.PointFToPoint(mainView.ScreenToScene_(new Point(e.X, e.Y)));
                            hierarchyBuilder.SetCurrentMousePos(posInScene);
                            mainView.Invalidate();
                        }
                    }
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseUp(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // 必ず構築中であるはず...
                    MainView mainView = mouseState._mainView;

                    Debug.Assert(mainView._hierarchyMnp.IsBreaking);
                    mainView._hierarchyMnp.EndBreakingHierarchy();
                    mainView.Invalidate();

                    // マウスモードをリセットします。
                    mouseState._CurerntMoouseState = MouseState._Idling;

                }
            }

            /// <summary>
            ///
            /// </summary>
            class MouseState_Duplicate : MouseState
            {
                public override string StateName { get { return "Duplicate"; } }

                private List<bool> orderList = new List<bool>();

                /// <summary>
                ///
                /// </summary>
                void BeginDragging_(MainView mainView, PointF scenePos)
                {
                    IPane[] paneSet = mainView._targetActiveSubScene.ISelectedSet.IPaneArray;
                    mainView._draggedPane.BeginDragging(
                        PaneDragTargetAdapter.FromPaneSet(paneSet),
                        mainView._magnify,
                        scenePos,
                        DragModifierHelper.OptionFlag.IgnoreShortDrag);

                    // グリッド吸着が有効ならば、グリッドサイズを設定します。
                    if (mainView.BackGround.GridSnapEnalbed)
                    {
                        mainView._draggedPane.SetGridSnapEnable(mainView._GridSize);
                    }
                    else
                    {
                        mainView._draggedPane.SetGridSnapEnable(1.0f);
                    }
                }

                /// <summary>
                /// モードエントリ
                /// </summary>
                public override void EnterState(MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    PointF scenePos = mouseState._mouseDownPosInScene;

                    mainView._TargetActiveSubScene.BeginMassiveModify();

                    // 現在位置にペインを複製します(複製したペインが選択状態になります)
                    CoreManipulator.SubSceneManipulator subSceneMnp = mainView._subSceneMnp;
                    subSceneMnp.BindTarget(mainView._TargetActiveSubScene);
                    orderList.Clear();
                    orderList.AddRange(subSceneMnp.CopySelectedPanesRecursive());
                    subSceneMnp.PastePanes(mainView.GetAppSetting().PasteAnimation, mainView.IsPaneValidToPaste);

                    // 複製モードは平行移動のみ可能とします
                    mainView._draggedPane.AddOptionFlag(PaneDragModifier.Option.ForceToTranslate);

                    BeginDragging_(mainView, scenePos);

                    base.EnterState(mouseState);
                }

                /// <summary>
                /// モード終了
                /// </summary>
                public override void LeaveState(MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;
                    mainView._TargetActiveSubScene.EndMassiveModify();

                    TryToChangeMouseCursor_(mouseState._mainView, Cursors.Arrow);
                    base.LeaveState(mouseState);
                }

                /// <summary>
                /// MouseDown
                /// </summary>
                public override void OnMouseDown(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // ドラッグ始点をリセットします。
                    MainView mainView = mouseState._mainView;
                    Point posMouse = new Point(e.X, e.Y);

                    base.OnMouseDown_(e, mouseState);
                }

                /// <summary>
                /// MouseMove
                /// </summary>
                public override void OnMouseMove(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    // ビューの位置を変更します
                    MainView mainView = mouseState._mainView;
                    PaneDragModifier draggedPane = mainView._draggedPane;

                    if (mainView._draggedPane.Empty)
                    {
                        // マウスカーソルを変更する。
                        return;
                    }
                    // TODO:
                    // カーソルは複製用のものを作ってもらう予定
                    //TryToChangeMouseCursor_(mouseState._mainView, Cursors.Cross);

                    Point posMouse = new Point(e.X, e.Y);

                    //------------------------------------------------------------------
                    // 選択グループの位置を更新します。
                    // Shift が押されている場合には、移動方向が限定されます。
                    if ((Control.ModifierKeys & Keys.Shift) != 0)
                    {
                        draggedPane.AddOptionFlag(PaneDragModifier.Option.SnapDirections);
                    }

                    PointF posInScene = mainView.ScreenToScene_(posMouse);

                    // 調整処理をキャンセルします。
                    draggedPane.UpdateDragging(mainView._magnify, posInScene);

                    mainView.Invalidate();
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseUp(MouseEventArgs e, MainViewMouseState mouseState)
                {
                    MainView mainView = mouseState._mainView;

                    if (mainView._draggedPane.Empty)
                    {
                        return;
                    }

                    PointF posLastStart = mainView._draggedPane.DragStartPointF;

                    // ドラッグを終了し、移動量を取得します。
                    mainView._draggedPane.EndDragging();

                    // 再度選択します。
                    CoreManipulator.SubSceneManipulator subSceneMnp = mainView._subSceneMnp;
                    subSceneMnp.SelectedChangeByOrder(mainView._targetActiveSubScene.ISelectedSet.IPaneArray, orderList.ToArray());
                    orderList.Clear();
                    IPane[] paneSet = mainView._targetActiveSubScene.ISelectedSet.IPaneArray;

                    mainView._draggedPane.BeginDragging(
                        PaneDragTargetAdapter.FromPaneSet(paneSet),
                        mainView._magnify,
                        posLastStart,
                        DragModifierHelper.OptionFlag.IgnoreShortDrag);

                    // マウスモードをリセットします。
                    mouseState._CurerntMoouseState = MouseState._Idling;
                }
            }

            #endregion MouseState クラスと、その派生クラス
        }
    }
}
