﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.DebugWindow
{
    using LECore.Structures;
    using LECore.Structures.Core.Command;

    /// <summary>
    /// Undo/Redo用コマンドキューの状態をデバック表示する
    /// ウインドウです。
    ///
    /// デバックビルド時のみ動作します。
    /// </summary>
    public class CmdQueueDebugWindow :
        LEToolWindow
    {
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;
        private System.Windows.Forms.TextBox   _tbxLog;

        #region フィールド

        ISubSceneCmdQueue                  _targetSubSceneCmdQueue = null;
        private System.Windows.Forms.TextBox _tbxNumUndoCmd;
        private System.Windows.Forms.TextBox _tbxNumRedoCmd;
        private System.Windows.Forms.Label _lblNumUndoCmd;
        private System.Windows.Forms.Label _lblNumRedoCmd;
        readonly QueueStateChangedHandler  _CmdQueueStateChangedHandler = null;

        #endregion フィールド

        #region プロパティ
        /// <summary>
        /// 表示対象サブシーンを設定します。
        /// </summary>
        public ISubSceneCmdQueue TargetSubSceneCmdQueue
        {
            get{ return _targetSubSceneCmdQueue;}
            set
            {
                // 以前のターゲットからイベントハンドラを解除
                if( _targetSubSceneCmdQueue != null )
                {
                    _targetSubSceneCmdQueue.QueueStateChangedEvent -= _CmdQueueStateChangedHandler;
                }

                _targetSubSceneCmdQueue = value;
                _tbxLog.AppendText( "-------- change SubScene -------- \r\n" );
                UpdateUndoRedoCmdTextBox_( _targetSubSceneCmdQueue );

                if( _targetSubSceneCmdQueue != null )
                {
                    _targetSubSceneCmdQueue.QueueStateChangedEvent += _CmdQueueStateChangedHandler;
                }

            }
        }
        #endregion プロパティ


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CmdQueueDebugWindow()
        {
            InitializeComponent();

            _CmdQueueStateChangedHandler =
                new QueueStateChangedHandler( QueueStateChangedHandler_ );
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region Windows フォーム デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this._tbxLog = new System.Windows.Forms.TextBox();
            this._tbxNumUndoCmd = new System.Windows.Forms.TextBox();
            this._tbxNumRedoCmd = new System.Windows.Forms.TextBox();
            this._lblNumUndoCmd = new System.Windows.Forms.Label();
            this._lblNumRedoCmd = new System.Windows.Forms.Label();
            this.SuspendLayout();
            //
            // _tbxLog
            //
            this._tbxLog.Dock = System.Windows.Forms.DockStyle.Bottom;
            this._tbxLog.Font = new System.Drawing.Font( "MS UI Gothic", 5.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ( (byte)( 128 ) ) );
            this._tbxLog.Location = new System.Drawing.Point( 0, 20 );
            this._tbxLog.Multiline = true;
            this._tbxLog.Name = "_tbxLog";
            this._tbxLog.ReadOnly = true;
            this._tbxLog.ScrollBars = System.Windows.Forms.ScrollBars.Vertical;
            this._tbxLog.Size = new System.Drawing.Size( 394, 110 );
            this._tbxLog.TabIndex = 4;
            //
            // _tbxNumUndoCmd
            //
            this._tbxNumUndoCmd.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this._tbxNumUndoCmd.Location = new System.Drawing.Point( 76, 0 );
            this._tbxNumUndoCmd.Name = "_tbxNumUndoCmd";
            this._tbxNumUndoCmd.ReadOnly = true;
            this._tbxNumUndoCmd.Size = new System.Drawing.Size( 56, 17 );
            this._tbxNumUndoCmd.TabIndex = 1;
            //
            // _tbxNumRedoCmd
            //
            this._tbxNumRedoCmd.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this._tbxNumRedoCmd.Location = new System.Drawing.Point( 208, 0 );
            this._tbxNumRedoCmd.Name = "_tbxNumRedoCmd";
            this._tbxNumRedoCmd.ReadOnly = true;
            this._tbxNumRedoCmd.Size = new System.Drawing.Size( 56, 17 );
            this._tbxNumRedoCmd.TabIndex = 3;
            //
            // _lblNumUndoCmd
            //
            this._lblNumUndoCmd.Location = new System.Drawing.Point( 8, 3 );
            this._lblNumUndoCmd.Name = "_lblNumUndoCmd";
            this._lblNumUndoCmd.Size = new System.Drawing.Size( 64, 13 );
            this._lblNumUndoCmd.TabIndex = 0;
            this._lblNumUndoCmd.Text = "UndoCmd :";
            //
            // _lblNumRedoCmd
            //
            this._lblNumRedoCmd.Location = new System.Drawing.Point( 144, 2 );
            this._lblNumRedoCmd.Name = "_lblNumRedoCmd";
            this._lblNumRedoCmd.Size = new System.Drawing.Size( 64, 14 );
            this._lblNumRedoCmd.TabIndex = 2;
            this._lblNumRedoCmd.Text = "RedoCmd :";
            //
            // CmdQueueDebugWindow
            //
            this.AutoScaleBaseSize = new System.Drawing.Size( 5, 12 );
            this.ClientSize = new System.Drawing.Size( 394, 130 );
            this.Controls.Add( this._lblNumRedoCmd );
            this.Controls.Add( this._lblNumUndoCmd );
            this.Controls.Add( this._tbxLog );
            this.Controls.Add( this._tbxNumUndoCmd );
            this.Controls.Add( this._tbxNumRedoCmd );
            this.Name = "CmdQueueDebugWindow";
            this.Text = "CmdQueueDebugWindow";
            this.ResumeLayout( false );
            this.PerformLayout();

        }
        #endregion

        #region LEToolWindow メンバ

        #region ISceneModifyListener メンバ

        /// <summary>
        /// シーン更新ハンドラ
        /// デバックビルド時のみ動作します。
        /// </summary>
        public override void OnSceneModifyHandler(
            object                  sender,
            SceneModifyEventArgs    e )
        {
#if DEBUG
            // 操作対象サブシーンが変更されたなら...
            if( e.Target == SceneModifyEventArgs.Kind.CurrentSubSceneChanged )
            {
                // カレントサブシーンの更新
                // コマンドキューデバックウインドウの対象を更新
                ISubScene newCurrentScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
                if( newCurrentScene != null )
                {
                    // 対象が変更されていれば...
                    ISubSceneCmdQueue cmdQueue = newCurrentScene.ISubSceneCmdQueue;
                    if( cmdQueue != this.TargetSubSceneCmdQueue )
                    {
                        // 新たに対象に設定
                        this.TargetSubSceneCmdQueue = cmdQueue;
                    }
                }
                else
                {
                    this.TargetSubSceneCmdQueue = null;
                }
            }
#endif
            base.OnSceneModifyHandler( sender, e );
        }


        #endregion ISceneModifyListener メンバ

        #endregion LEToolWindow メンバ

        #region コマンドキューの状態変更イベントハンドラ
        /// <summary>
        /// コマンドログ文字列を生成します。
        /// </summary>
        /// <param name="kind"></param>
        /// <param name="cmd"></param>
        /// <returns></returns>
        string MakeEventLogString_( CmdQueueChangedEventKind kind, ICommand cmd )
        {
            string cmdStr = cmd != null ? cmd.ToString() : "N/A";
            return String.Format( "Kind => {0} : Params => [ {1} ]\r\n", kind.ToString(), cmdStr ) ;
        }

        /// <summary>
        /// Undo/Redoコマンド数を表示するテキストボックスを更新します。
        /// </summary>
        void UpdateUndoRedoCmdTextBox_( ISubSceneCmdQueue sender )
        {
            if( sender != null )
            {
                _tbxNumUndoCmd.Text = sender.NumUndoCommands.ToString();
                _tbxNumRedoCmd.Text = sender.NumRedoCommands.ToString();
            }
            else
            {
                _tbxNumUndoCmd.Text = 0.ToString();
                _tbxNumRedoCmd.Text = 0.ToString();
            }
        }

        /// <summary>
        /// キュー追加
        /// </summary>
        void QueueStateChangedHandler_( ISubSceneCmdQueue sender, CmdQueueChangedEventKind kind, ICommand cmd )
        {
            UpdateUndoRedoCmdTextBox_( sender );
            // ログを書き出します。
            string eventLogStr = MakeEventLogString_( kind, cmd );

            if (_tbxLog.TextLength + eventLogStr.Length > _tbxLog.MaxLength)
            {
                _tbxLog.Clear();
            }
            _tbxLog.AppendText( eventLogStr );
        }
        #endregion コマンドキューの状態変更イベントハンドラ

        #region オーバーライドイベント
        #region Load
        /// <summary>
        /// オーバーライドイベント。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            // 位置、サイズをデスクトップ隅に設定
            Rectangle desktop = Screen.PrimaryScreen.WorkingArea;
            this.Location = new Point(desktop.X, desktop.Bottom - this.Size.Height);
        }
        #endregion
        #endregion

    }
}
