﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using LECore;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    /// <summary>
    /// 目盛りを描画するクラス。
    /// </summary>
    public class NumberLineDrawer
    {
        // 目盛りイメージの全体サイズ
        int       _pixSize;
        // 値開始位置
        int       _valueStart;
        // 値終了位置
        int       _valueEmd;
        // 目盛り間隔の最小ピクセルサイズ
        int       _minNumberLinePix = 10;
        // ユーザが自由に使用する、描画パラメータ
        object    _userDrawParams = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public NumberLineDrawer( int pixSize, int valueStart, int valueEmd )
        {
            _pixSize     = pixSize;
            _valueStart  = valueStart;
            _valueEmd    = valueEmd;
        }


        // 目盛りを描画すべきタイミングでコールバックされるハンドラ
        public delegate bool OnNumberLineDrawHandler( NumberLineDrawer sender, int diffPix, int val );



        /// <summary>
        /// 目盛りサイズ増加分を計算します。
        /// </summary>
        /// <param name="val"></param>
        /// <param name="pixPerValue"></param>
        /// <returns></returns>
        int CatcValueDiiference_( float pixPerValue )
        {
            int diffVal = (int)CalcValueDiiference_( pixPerValue, _minNumberLinePix );

            // 最低でも1以上になるようにする
            diffVal = Math.Max( diffVal, 1 );

            return diffVal;
        }

        static float CalcValueDiiference_( float pixPerValue, int minNumberLinePix )
        {
            // 最小ピクセルに相当する 目盛り間隔を計算します。
            double minNumberLineVal = (float)minNumberLinePix / pixPerValue;

            // { 1 or 2 or 5 } * 10^N のなかから、最も近い値を探します。
            double   N    = Math.Pow( 10, Math.Floor( Math.Log10( minNumberLineVal ) ) );
            int n = (int)(minNumberLineVal / N);
            if( n < 2 )
            {
                n = 1;
            }
            else if( n < 5 )
            {
                n = 2;
            }
            else
            {
                n = 5;
            }

            return (float)(n * N);
        }

        static public float GetNumberLineBlank( float pixSize, float valueStart, float valueEmd, int minNumberLinePix )
        {
              float pixPerVal = GetPixelPerVal_( pixSize, valueStart, valueEmd );
              return CalcValueDiiference_( pixPerVal, minNumberLinePix );
        }

        static float GetPixelPerVal_( float pixSize, float valueStart, float valueEmd )
        {
            return pixSize / (valueEmd - valueStart);
        }

        public float PixelPerVal
        {
            get{ return GetPixelPerVal_((float)_pixSize,(float)(_valueStart) ,(float)(_valueEmd) );}
        }

        /// <summary>
        /// 描画処理を行います。
        /// 実際の描画処理を行う、コールバックハンドラを引数に与えます。
        /// </summary>
        /// <param name="onNumberLineDrawHandler">描画処理を行うハンドラ</param>
        public void DoDraw( OnNumberLineDrawHandler onNumberLineDrawHandler )
        {
            // 値1に対して、使用されるピクセル値を計算します。
            float pixPerValue = PixelPerVal;

            // 開始値（目盛りサイズで整列）からスタートして、終了値に到達するまで...
            int minNumberLineVal = CatcValueDiiference_( pixPerValue );
            Debug.Assert( minNumberLineVal != 0 );

            int currentVal  = _valueStart - (_valueStart % minNumberLineVal );
            while( currentVal < _valueEmd )
            {
                // 条件を満たす場合は、描画処理ハンドラを呼びます
                if( currentVal >= _valueStart )
                {
                    if( false == onNumberLineDrawHandler( this, (int)(pixPerValue * (currentVal-_valueStart)), currentVal ) )
                    {
                        // ハンドラが false を返した場合には、処理を中断します。
                        break;
                    }
                }

                // 目盛りの最小値を下回らないように、目盛り増分を設定します。
                int dVal = CatcValueDiiference_( pixPerValue );

                // 現在位置に、目盛り増分を足します。
                currentVal += dVal;
            }
        }

        /// <summary>
        /// ピクセル位置から、目盛り値を計算します。
        /// </summary>
        /// <param name="pix"></param>
        /// <returns></returns>
        public int CalcNumberLineValFromPixPosition( int pix )
        {
            return (int)( pix / PixelPerVal ) + _valueStart;
        }

        /// <summary>
        /// 目盛り値から、ピクセル値を取得します。
        /// </summary>
        /// <param name="val"></param>
        /// <returns></returns>
        public int CalcPixPosFromNumberLineVal( int val )
        {
            return (int)( PixelPerVal * (val - _valueStart) );
        }


        #region --------------- プロパティ ---------------
        /// <summary>
        /// 目盛り間隔の最小値を設定します。
        /// </summary>
        public int MinNumberLineSize
        {
            set{ _minNumberLinePix = value; }
        }

        /// <summary>
        /// 描画処理の期間に、
        /// ユーザが自由に使用する、描画パラメータを設定、取得します。
        /// </summary>
        public object UserDrawParams
        {
            get{ return _userDrawParams;}
            set{ _userDrawParams = value;}
        }
        #endregion // --------------- プロパティ ---------------
    }
}
