﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using Dialogs;
    using LECore.Structures;
    using LECore.Structures.Core;

    /// <summary>
    /// アトリビュートフィルタ作成ダイアログ
    /// </summary>
    partial class CreateAttributeFilterDlg : LEBaseDlg
    {
        TargetAnmAttributeSet		_targetAnmAttributeSet;
        AttributeFilterManager	_attributeFilterManager;

        /// <summary>
        /// 表示対象となるノードを集めます。
        /// </summary>
        void CollectNotCheckedNode_( List<IAnmAttribute> attrSet, TreeNode node )
        {
            // if( !node.Checked )
            if( node.Checked )
            {
                attrSet.Add( node.Tag as IAnmAttribute );
            }

            foreach( TreeNode child in node.Nodes )
            {
                CollectNotCheckedNode_( attrSet, child );
            }
        }

        /// <summary>
        /// フィルタの名前を取得します。
        /// </summary>
        public string FiltereName
        {
            get { return _tbxFilterName.Text; }
        }

        /// <summary>
        /// フィルタアトリビュートセットを取得します。
        /// （すべてのチェックされていないノード）
        /// </summary>
        public IAnmAttribute[] FilteredAttributeSet
        {
            get
            {
                List<IAnmAttribute> set = new List<IAnmAttribute>();
                foreach( TreeNode node in _tvwAttributeTree.Nodes )
                {
                    CollectNotCheckedNode_( set, node );
                }
                return set.ToArray();
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateAttributeFilterDlg()
        {
            InitializeComponent();
        }

        #region 初期化関連

        /// <summary>
        ///
        /// </summary>
        void UpdateOLButtonState_()
        {
            if( _attributeFilterManager != null && _targetAnmAttributeSet != null )
            {
                _btnOK.Enabled = _attributeFilterManager.CheckValid(
                    FiltereName,
                    FilteredAttributeSet );
            }else{
                _btnOK.Enabled = false;
            }
        }

        /// <summary>
        /// アトリビュートからノードを生成します。
        /// </summary>
        TreeNode BuildAttributeNode_( List<string> attributePathStrSet, IAnmAttribute attr )
        {

            TreeNode node = null;
            string fullPath = AttributeFilterHelper.GetFilterPath(attr);

            if( !attributePathStrSet.Contains( fullPath ) )
            {
                node = new TreeNode( attr.Name );
                node.ToolTipText = fullPath;
                node.Checked = true;
                node.Tag = attr;

                attributePathStrSet.Add( fullPath );

                for( int i = 0 ; i < attr.NumSubAttribute ; i++ )
                {
                    TreeNode childNode = BuildAttributeNode_(
                        attributePathStrSet,
                        attr.FindSubAttributeByIdx( i ) );

                    if( childNode != null )
                    {
                        node.Nodes.Add( childNode );
                    }
                }
            }

            return node;
        }

        /// <summary>
        /// GUIを初期化します。
        /// 表示前に実行します。
        /// </summary>
        public void Setup( AttributeFilterManager attributeFilterManager, TargetAnmAttributeSet targetAnmAttributeSet )
        {
            _attributeFilterManager = attributeFilterManager;
            _targetAnmAttributeSet = targetAnmAttributeSet;

            _tvwAttributeTree.Nodes.Clear();

            List<string> attributePathStrSet = new List<string>();
            foreach( IAnmAttribute attr in targetAnmAttributeSet._TargetAttributeSet )
            {
                TreeNode node = BuildAttributeNode_( attributePathStrSet, attr );
                if( node != null )
                {
                    _tvwAttributeTree.Nodes.Add( node );
                }
            }

            _tvwAttributeTree.ExpandAll();


            UpdateOLButtonState_();
        }
        #endregion 初期化関連

        #region イベントハンドラ

        #region 再帰的チェック状態更新関連
        /// <summary>
        /// ノードのチェック状態を取得する関数
        /// </summary>
        delegate bool IsTreeNodeChecked_( TreeNode node );

        /// <summary>
        /// _chkCheckAllのチェック状態に伴ってチェック
        /// </summary>
        bool CheckAll_( TreeNode node )
        {
            return _chkCheckAll.Checked;
        }

        /// <summary>
        /// 選択カーブをチェック
        /// </summary>
        bool CheckCurveSelected_( TreeNode node )
        {
            IAnmCurve[] curveSet = _targetAnmAttributeSet._TargetCurveSet;
            return Array.IndexOf( curveSet, ( node.Tag as IAnmAttribute ).ICurrentAnimationCurve) != -1;
        }

        /// <summary>
        /// 再帰的にチェック状態を更新します。
        /// </summary>
        void UpdateNodeCheckStateRecursive_(TreeNode node, Predicate<TreeNode> condition, bool checkState)
        {
            if (condition(node))
            {
                node.Checked = checkState;
            }

            foreach( TreeNode child in node.Nodes )
            {
                UpdateNodeCheckStateRecursive_(child, condition, checkState);
            }
        }
        #endregion 再帰的チェック状態更新関連

        /// <summary>
        /// フィルタ名：値更新
        /// </summary>
        private void Event_TbxFilterName_ValueChanged( object sender, EventArgs e )
        {
            // 有効な名前か判定して、OKボタンの状態を更新
            UpdateOLButtonState_();
        }

        /// <summary>
        /// ツリーノード、チェック変更後
        /// </summary>
        private void Event_TvwAttributeTree_AfterCheck( object sender, TreeViewEventArgs e )
        {
            UpdateOLButtonState_();
        }

        /// <summary>
        /// 選択カーブをチェック
        /// </summary>
        private void Event_BtnCheckTargetCurve_Click( object sender, EventArgs e )
        {
            if( _targetAnmAttributeSet == null )
            {
                return;
            }

            foreach( TreeNode node in _tvwAttributeTree.Nodes )
            {
                UpdateNodeCheckStateRecursive_(node, CheckCurveSelected_, _chkCheckSelected.Checked);
            }
        }

        /// <summary>
        /// すべてをチェック
        /// </summary>
        private void Event_ChkCheckAll_CheckedChanged( object sender, EventArgs e )
        {
            foreach( TreeNode node in _tvwAttributeTree.Nodes )
            {
                UpdateNodeCheckStateRecursive_( node, (subNode) => true, _chkCheckAll.Checked );
            }
        }

        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOK_Click( object sender, EventArgs e )
        {
            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click( object sender, EventArgs e )
        {
            DialogResult = DialogResult.Cancel;
        }

        #endregion イベントハンドラ





    }
}
