﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;



namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LECore.Structures.Core;
    using common;
    using LECore.Structures;
    using LayoutEditor.Controls;

    /// <summary>
    /// アトリビュートツリーの外見を更新するクラスです。
    /// クラス内部に状態を保存し、不必要な更新をできるだけ回避するようにします。
    /// </summary>
    static class AttributeTreeAppearanceUpdater
    {
        /// <summary>
        /// ツリーノードが利用する、アイコン番号
        /// </summary>
        const int _ImageIndexAttr_NotActive = 0;
        const int _ImageIndexAttrSet_NotActive = 1;
        const int _ImageIndexAttr_Active = 2;
        const int _ImageIndexAttrSet_Active = 3;

        const int _ImageIndexNull_NotActive = 4;
        const int _ImageIndexNull_Active = 5;
        const int _ImageIndexPict_NotActive = 6;
        const int _ImageIndexPict_Active = 7;
        const int _ImageIndexText_NotActive = 8;
        const int _ImageIndexText_Active = 9;
        const int _ImageIndexWind_NotActive = 10;
        const int _ImageIndexWind_Active = 11;
        const int _ImageIndexBond_NotActive = 12;
        const int _ImageIndexBond_Active = 13;
        const int _ImageIndexPart_NotActive = 14;
        const int _ImageIndexPart_Active = 15;
        const int _ImageIndexCapture_NotActive = 16;
        const int _ImageIndexCapture_Active = 17;
        const int _ImageIndexAlignment_NotActive = 18;
        const int _ImageIndexAlignment_Active = 19;
        const int _ImageIndexScissor_NotActive = 20;
        const int _ImageIndexScissor_Active = 21;

        const int _ImageIndexNullP_NotActive = 22;
        const int _ImageIndexNullP_Active = 23;
        const int _ImageIndexPictP_NotActive = 24;
        const int _ImageIndexPictP_Active = 25;
        const int _ImageIndexTextP_NotActive = 26;
        const int _ImageIndexTextP_Active = 27;
        const int _ImageIndexWindP_NotActive = 28;
        const int _ImageIndexWindP_Active = 29;
        const int _ImageIndexBondP_NotActive = 30;
        const int _ImageIndexBondP_Active = 31;
        const int _ImageIndexPartP_NotActive = 32;
        const int _ImageIndexPartP_Active = 33;
        const int _ImageIndexCaptP_NotActive = 34;
        const int _ImageIndexCaptP_Active = 35;
        const int _ImageIndexAligP_NotActive = 36;
        const int _ImageIndexAligP_Active = 37;
        const int _ImageIndexScissorP_NotActive = 38;
        const int _ImageIndexScissorP_Active = 39;

        const int _ImageIndexAttrSet_Active_ReadOnly = 40;
        const int _ImageIndexAttrSet_NoActive_ReadOnly = 41;

        public const int AnimationOverrayIconIdx = 1;

        /// <summary>
        /// 末端のアトリビュートからから、対応するツリーノードを検索する処理。
        /// </summary>
        public delegate List<TreeNode> NodeFindFunction( IAnmAttribute curve );

        /// <summary>
        /// ノードが読み取り専用か？
        /// </summary>
        static bool IsNodeReadOnly_(TreeNode node)
        {
            return
                node.ImageIndex == _ImageIndexAttrSet_Active_ReadOnly ||
                node.ImageIndex == _ImageIndexAttrSet_NoActive_ReadOnly;
        }

        /// <summary>
        /// キー状態に関連する表示を更新します。
        /// </summary>
        static public void UpdateNodeKeyState(
            NodeFindFunction  nodeFindFuction,
            IAnmAttribute[]        currentCurve )
        {

            List<TreeNode> nodeSet = null;

            // 全てのノードの文字色をリセットします。
            // TODO:前回色が変更された部分のみを更新すれば十分なはずである。
            foreach (IAnmAttribute curve in currentCurve)
            {
                nodeSet = nodeFindFuction(curve);
                foreach (TreeNode node in nodeSet)
                {
                    TreeNode n = node;
                    while (n != null)
                    {
                        var bc = n.TreeView != null ? n.TreeView.BackColor : n.BackColor;
                        n.BackColor = bc;
                        n = n.Parent;
                    }
                }
            }

            foreach( IAnmAttribute curve in currentCurve )
            {
                if (curve.ICurrentAnimationCurve.NumSelectedKey != 0)
                {
                    nodeSet = nodeFindFuction(curve);
                    foreach (TreeNode node in nodeSet)
                    {
                        // キー選択アトリビュートの背景色
                        // 描画時に、背景色を参考に、アニメーション有無をオーバーレイアイコンを設定する。
                        TreeNode n = node;
                        while (n != null)
                        {
                            var bc = n.TreeView != null ? GetScaledColor_(n.TreeView.BackColor, 0.9f) : n.BackColor;
                            n.BackColor =  bc;
                            n = n.Parent;
                        }
                    }
                }
            }
        }

        static Color GetScaledColor_(Color c, float scale)
        {
            return Color.FromArgb((byte)((float)c.R * scale), (byte)((float)c.G * scale), (byte)((float)c.B * scale));
        }

       #region UpdateNodeSelectedState

        static int GetPaneKindIndex_(TreeNode treeNode, bool isActive)
        {
            PaneKind paneKind;

            if (treeNode.Tag is IPane)
            {
                paneKind =(treeNode.Tag as IPane).PaneKind;
                switch (paneKind)
                {
                    case PaneKind.Null: return isActive ? _ImageIndexNullP_Active : _ImageIndexNullP_NotActive;
                    case PaneKind.Picture: return isActive ? _ImageIndexPictP_Active : _ImageIndexPictP_NotActive;
                    case PaneKind.Textbox: return isActive ? _ImageIndexTextP_Active : _ImageIndexTextP_NotActive;
                    case PaneKind.Window: return isActive ? _ImageIndexWindP_Active : _ImageIndexWindP_NotActive;
                    case PaneKind.Bounding: return isActive ? _ImageIndexBondP_Active : _ImageIndexBondP_NotActive;
                    case PaneKind.Parts: return isActive ? _ImageIndexPartP_Active : _ImageIndexPartP_NotActive;
                    case PaneKind.Capture: return isActive ? _ImageIndexCaptP_Active : _ImageIndexCaptP_NotActive;
                    case PaneKind.Alignment: return isActive ? _ImageIndexAligP_Active : _ImageIndexAligP_NotActive;
                    case PaneKind.Scissor: return isActive ? _ImageIndexScissorP_Active : _ImageIndexScissorP_NotActive;
                    default: return isActive ? _ImageIndexAttrSet_Active : _ImageIndexAttrSet_NotActive;
                }
            }
            else if (treeNode.Parent != null && treeNode.Parent.Tag is IPane)
            {
                paneKind = (treeNode.Parent.Tag as IPane).PaneKind;
                switch (paneKind)
                {
                    case PaneKind.Null: return isActive ? _ImageIndexNull_Active : _ImageIndexNull_NotActive;
                    case PaneKind.Picture: return isActive ? _ImageIndexPict_Active : _ImageIndexPict_NotActive;
                    case PaneKind.Textbox: return isActive ? _ImageIndexText_Active : _ImageIndexText_NotActive;
                    case PaneKind.Window: return isActive ? _ImageIndexWind_Active : _ImageIndexWind_NotActive;
                    case PaneKind.Bounding: return isActive ? _ImageIndexBond_Active : _ImageIndexBond_NotActive;
                    case PaneKind.Parts: return isActive ? _ImageIndexPart_Active : _ImageIndexPart_NotActive;
                    case PaneKind.Capture: return isActive ? _ImageIndexCaptP_Active : _ImageIndexCaptP_NotActive;
                    case PaneKind.Alignment: return isActive ? _ImageIndexAligP_Active : _ImageIndexAligP_NotActive;
                    case PaneKind.Scissor: return isActive ? _ImageIndexScissorP_Active : _ImageIndexScissorP_NotActive;
                    default: return isActive ? _ImageIndexAttrSet_Active : _ImageIndexAttrSet_NotActive;
                }
            }
            else
            {
                return -1;
            }
        }

        /// <summary>
        /// ノードの表示を選択状態にします。
        /// </summary>
        static void SetSelected_( TreeNode node )
        {
            TreeNode treeNode = node;

            // 変更がなかったら更新しようようにする
            while( treeNode != null )
            {
                int newIndex = 0;
                if(treeNode.Nodes.Count != 0)
                {
                    // 種類に応じて変える。
                    newIndex = GetPaneKindIndex_(treeNode, true);
                    if (newIndex == -1)
                    {
                        newIndex = _ImageIndexAttrSet_Active;
                    }

                }else{
                    newIndex = ((treeNode.Tag as IAnmAttribute).ICurrentAnimationCurve.IsReadOnlyLocked) ? _ImageIndexAttrSet_Active_ReadOnly : _ImageIndexAttr_Active;
                }

                if (newIndex != treeNode.ImageIndex)
                {
                    treeNode.ImageIndex = newIndex;
                    treeNode.SelectedImageIndex = newIndex;
                }

                treeNode = treeNode.Parent;
            }
        }

        /// <summary>
        /// ノードの状態を選択状態からリセットします。
        /// </summary>
        static void ResetSelected_( TreeNode node )
        {
            TreeNode treeNode = node;
            while( treeNode != null )
            {
                int newIndex = 0;
                if (treeNode.Nodes.Count != 0)
                {
                    // 種類に応じて変える。
                    newIndex = GetPaneKindIndex_(treeNode, false);
                    if (newIndex == -1)
                    {
                        newIndex = _ImageIndexAttrSet_NotActive;
                    }
                }
                else
                {
                    if (treeNode.Tag is IAnmAttribute &&
                        (treeNode.Tag as IAnmAttribute).ICurrentAnimationCurve != null &&
                        (treeNode.Tag as IAnmAttribute).ICurrentAnimationCurve.IsReadOnlyLocked)
                    {
                        newIndex = _ImageIndexAttrSet_NoActive_ReadOnly;
                    }
                    else
                    {
                        newIndex = _ImageIndexAttr_NotActive;
                    }
                }

                if (newIndex != treeNode.ImageIndex)
                {
                    treeNode.ImageIndex = newIndex;
                    treeNode.SelectedImageIndex = newIndex;
                }

                treeNode = treeNode.Parent;
            }
        }


        /// <summary>
        /// ノード以下の、リーフを再帰的に走査し、全て選択します。
        /// </summary>
        static public void SetNodeAppearenceSelectedRecursive( TreeNode node )
        {
            // リーフノードのみ処理します。
            if( node.Tag is IAnmAttribute )
            {
                if( !( node.Tag as IAnmAttribute ).HasSubAttribute )
                {
                    SetSelected_( node );
                }
            }

            foreach( TreeNode child in node.Nodes )
            {
                SetNodeAppearenceSelectedRecursive( child );
            }
        }

        /// <summary>
        /// ノードの表示を選択状態にします。
        /// </summary>
        static public void SetNodeAppearenceSelected( List<TreeNode> nodeSet )
        {
            foreach( TreeNode node in nodeSet )
            {
                // リーフノードのみ処理します。
                if( node.Tag is IAnmAttribute )
                {
                    if( !( node.Tag as IAnmAttribute ).HasSubAttribute )
                    {
                        SetSelected_( node );
                    }
                }
            }
        }

        /// <summary>
        /// ノードの表示を非選択状態にします。
        /// </summary>
        static public void ResetNodeAppearenceSelected(TreeNode node)
        {
            if (node.Tag is IAnmAttribute)
            {
                if (!(node.Tag as IAnmAttribute).HasSubAttribute)
                {
                    ResetSelected_(node);
                }
            }
        }

        /// <summary>
        /// ノードの表示を非選択状態にします。
        /// </summary>
        static public void ResetNodeAppearenceSelected( List<TreeNode> nodeSet )
        {
            foreach( TreeNode node in nodeSet )
            {
                ResetNodeAppearenceSelected(node);
            }
        }

        /// <summary>
        ///
        /// </summary>
        static public bool GetNodeSelected(TreeNode node)
        {
            switch (node.ImageIndex)
            {
                default:
                case _ImageIndexAttr_NotActive:
                case _ImageIndexAttrSet_NotActive:
                case _ImageIndexPict_NotActive:
                case _ImageIndexAttrSet_NoActive_ReadOnly:
                    return false;
                case _ImageIndexAttr_Active:
                case _ImageIndexAttrSet_Active:
                case _ImageIndexPict_Active:
                case _ImageIndexAttrSet_Active_ReadOnly:
                    return true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        static public Color GetNodeElmentBGColor(TreeNode node)
        {
            if (node.IsExpanded)
            {
                return node.BackColor;
            }

            return node.ForeColor;
        }

        /// <summary>
        ///
        /// </summary>
        static public Color GetNodeTargetCurveBGColor(TreeNode node)
        {
            return
                node.BackColor != node.TreeView.BackColor &&
                node.Tag is IAnmAttribute &&
                (node.Tag as IAnmAttribute).HasAnimationCurve ?
                GetScaledColor_(GetNodeStateBGColor(node), 0.5f) : node.BackColor;
        }

        /// <summary>
        ///
        /// </summary>
        static public Color GetNodeKeyBGColor(TreeNode node)
        {
            if (node.IsExpanded)
            {
                return node.BackColor;
            }

            IAnmAttribute anmAttr = node.Tag as IAnmAttribute;
            if (anmAttr != null && (anmAttr.CheckHaveKeyRecursive() || anmAttr.CheckParameterizedAnimRecursive()))
            {
                return Color.FromArgb(230, 0, 0);
            }

            switch (node.ImageIndex)
            {
                default:
                    return node.BackColor;
                case _ImageIndexAttrSet_Active_ReadOnly:
                case _ImageIndexAttrSet_NoActive_ReadOnly:
                    return Color.FromArgb(230,0,0);
            }
        }

        /// <summary>
        ///
        /// </summary>
        static public Color GetNodeStateBGColor(TreeNode node)
        {
            if (node.IsExpanded)
            {
                return node.BackColor;
            }

            return GetNodeSelected(node) ? Color.LightGray : node.BackColor;
        }

        /// <summary>
        ///
        /// </summary>
        static public Color GetNodeReadonlyBGColor(TreeNode node)
        {
            return IsNodeReadOnly_(node) ? Color.DimGray : node.BackColor;
        }
       #endregion UpdateNodeSelectedState




    }
}
