﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Text.RegularExpressions;
using System.Linq;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LECore.Structures;
    using LECore.Structures.Core;

    #region フィルタの設定関連クラス


    //----------------------------------------------------------------
    #region AttributeFilter

    /// <summary>
    /// アトリビュートフィルタのヘルパクラス
    /// </summary>
    public class AttributeFilterHelper
    {
        /// <summary>
        /// フィルタに適したパス文字列を取得します。
        /// </summary>
        static public string GetFilterPath(IAnmAttribute attribute)
        {
            // 親までたどりながら、パスを作る
            string path = string.Empty;
            {
                StringBuilder fullPath = new StringBuilder();
                IAnmAttribute node = attribute;

                while (node != null)
                {
                    fullPath.Insert(0, '/');
                    fullPath.Insert(0, node.Name != node.DescriptionName ? string.Format("{0}_{1}", node.Name, node.DescriptionName) : node.Name);
                    node = node.OwnerNode;
                }

                path = fullPath.ToString();
            }

            // パスの先頭にはペイン名が書かれているはずなので
            // 適当な文字列（Root）で置換します。
            if (path.StartsWith("_ ["))
            {
                int index = path.IndexOf('/');
                if (index != -1)
                {
                    path = "Root" + path.Substring(index, path.Length - index);
                }
            }

            return path;
        }
    }

    /// <summary>
    /// フィルタ公開インタフェース
    /// </summary>
    public interface IAttributeFilter
    {
        /// <summary>
        /// 名前
        /// </summary>
        string Name { get; }

        /// <summary>
        /// 指定パス文字列が、フィルタ対象か調査します。
        /// </summary>
        bool Contains(IAnmAttribute attribute);

        /// <summary>
        /// システムフィルタか
        /// </summary>
        bool IsSystemDefault { get; }

        /// <summary>
        /// 分割モードで区間タグの切り替えが行なわれた場合にツリービューの更新を行うか
        /// </summary>
        bool IsUpdateTagChange { get; }
    }

    /// <summary>
    /// フィルタしないフィルタ
    /// </summary>
    public class NoneAttributeFilter : IAttributeFilter
    {
        /// <summary>
        /// 名前
        /// UI には表示されない
        /// </summary>
        public string Name { get { return StringResMgr.Get("CURVEEDITOR_FILTER_NONE"); } }

        /// <summary>
        /// フィルタをパスするか調査します。
        /// </summary>
        public bool Contains(IAnmAttribute attribute)
        {
            return false;
        }

        /// <summary>
        /// システムフィルタか
        /// </summary>
        public bool IsSystemDefault { get { return true; } }

        /// <summary>
        /// 分割モードで区間タグの切り替えが行なわれた場合にツリービューの更新を行うか
        /// </summary>
        public bool IsUpdateTagChange { get { return false; } }
    }

    /// <summary>
    /// アニメーションしていない項目をフィルタ
    /// </summary>
    public class AnimatedAttributeFilter : IAttributeFilter
    {
        /// <summary>
        /// 名前
        /// UI には表示されない
        /// </summary>
        public string Name { get { return StringResMgr.Get("CURVEEDITOR_FILTER_ANIMATION"); } }

        /// <summary>
        /// フィルタをパスするか調査します。
        /// </summary>
        public bool Contains(IAnmAttribute attribute)
        {
            return
                attribute.ICurrentAnimationCurve == null ||
                attribute.ICurrentAnimationCurve.NumKeyFrame <= 0;
        }

        /// <summary>
        /// システムフィルタか
        /// </summary>
        public bool IsSystemDefault { get { return true; } }

        /// <summary>
        /// 分割モードで区間タグの切り替えが行なわれた場合にツリービューの更新を行うか
        /// </summary>
        public bool IsUpdateTagChange { get { return true; } }
    }

    /// <summary>
    /// 基本的な項目以外をフィルタ
    /// </summary>
    public class BasicPaneAttributeFilter : IAttributeFilter
    {
        /// <summary>
        /// 名前
        /// UI には表示されない
        /// </summary>
        public string Name { get { return StringResMgr.Get("CURVEEDITOR_FILTER_PANEBASIC"); } }

        /// <summary>
        /// フィルタをパスするか調査します。
        /// </summary>
        public bool Contains(IAnmAttribute attribute)
        {
            if (attribute.ICurrentAnimationCurve == null)
            {
                return true;
            }

            string path = AttributeFilterHelper.GetFilterPath(attribute);
            return !path.Contains("pane") && !path.Contains("WhiteColor") && !path.Contains("BalckColor");
        }

        /// <summary>
        /// システムフィルタか
        /// </summary>
        public bool IsSystemDefault { get { return true; } }

        /// <summary>
        /// 分割モードで区間タグの切り替えが行なわれた場合にツリービューの更新を行うか
        /// </summary>
        public bool IsUpdateTagChange { get { return false; } }
    }

    /// <summary>
    /// アトリビュートフィルタを表現するクラス。
    ///
    /// IAnmAttributeセットから、フィルタ情報を生成します。
    /// IAnmAttributeがフィルタ対象か判定します。
    /// フィルタ合格する(表示対象となる)項目を含んでいます。
    /// </summary>
    public class CustomAttributeFilter : IAttributeFilter
    {
        string _name;
        string[] _filteredPathSet;

        /// <summary>
        /// 名前
        /// <remarks>
        /// シリアライズ用にsetも提供しています。
        /// </remarks>
        /// </summary>
        public string Name
        {
            get { return _name; }
            set { _name = value; }
        }

        /// <summary>
        /// システムフィルタか
        /// </summary>
        public bool IsSystemDefault { get { return false; } }

        /// <summary>
        /// 分割モードで区間タグの切り替えが行なわれた場合にツリービューの更新を行うか
        /// </summary>
        public bool IsUpdateTagChange { get { return true; } }

        /// <summary>
        /// フィルタパスセット
        /// <remarks>
        /// シリアライズ用にsetも提供しています。
        /// </remarks>
        /// </summary>
        public string[] FilteredPathSet
        {
            get { return _filteredPathSet; }
            set { _filteredPathSet = value; }
        }

        /// <summary>
        /// デフォルトコンストラクタ(シリアライズ用)
        /// </summary>
        public CustomAttributeFilter()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CustomAttributeFilter(string name, IAnmAttribute[] filteredAttrSet)
            : this(name, AttrToStrArray_(filteredAttrSet))
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        private static string[] AttrToStrArray_(IAnmAttribute[] filteredAttrSet)
        {
            List<string> pathSet = new List<string>();
            foreach (IAnmAttribute attr in filteredAttrSet)
            {
                string pathStr = AttributeFilterHelper.GetFilterPath(attr);
                if (!pathSet.Contains(pathStr))
                {
                    pathSet.Add(pathStr);
                }
            }
            return pathSet.ToArray();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CustomAttributeFilter( string name, string[] filteredPathSet )
        {
            Debug.Assert( name != null && name.Length > 0 );
            Debug.Assert( filteredPathSet != null && filteredPathSet.Length > 0 );

            _name = name;
            _filteredPathSet = filteredPathSet;
        }

        /// <summary>
        /// 指定パス文字列が、フィルタ対象か調査します。
        /// </summary>
        public bool Contains( IAnmAttribute attribute )
        {
            // 表示対象に含まれないものは、フィルタします。
            return Array.IndexOf(
                _filteredPathSet,
                AttributeFilterHelper.GetFilterPath(attribute)) == -1;
        }
    }
    #endregion AttributeFilter

    //----------------------------------------------------------------
    #region AttributeFilterManager

    /// <summary>
    /// AttributeFilterのセット
    ///
    /// AttributeTreeFactoryがインスタンスを持ち、
    /// フィルタ関連のステート管理、処理をすべて本クラスに委譲します。
    /// </summary>
    public class AttributeFilterManager
    {
        //----------------------------------------
        // イベント
        /// <summary>
        /// イベント種類
        /// </summary>
        public enum EventKind
        {
            None,
            FilterSetModified,
            CurrentFilterChanged,
        }

        public delegate void CurrentFilterStateChangedHandler( EventKind kind );
        public event CurrentFilterStateChangedHandler CurrentFilterStateChanged = null;

        //----------------------------------------
        #region フィールド
        // フィルタセット
        List<IAttributeFilter> _filterSet = new List<IAttributeFilter>();
        // 現在のフィルタ
        IAttributeFilter _currentFilter = null;

        #endregion フィールド

        //----------------------------------------
        #region プロパティ
        /// <summary>
        /// 一覧の取得(AttributeFilter):GUIの初期化に利用します。
        /// </summary>
        public IAttributeFilter[] AttributeFilterSet
        {
            get { return _filterSet.ToArray(); }
        }

        /// <summary>
        /// 現在のフィルタ
        /// </summary>
        public IAttributeFilter CurrentFilter
        {
            get { return _currentFilter; }
            set
            {
                if( _currentFilter != value )
                {
                    _currentFilter = value;
                    NotifyCurrentFilterStateChanged_( EventKind.CurrentFilterChanged );
                }
            }
        }

        #endregion プロパティ

        //----------------------------------------
        #region private

        /// <summary>
        /// 現在のフィルタが更新されたことを通知します。
        /// </summary>
        void NotifyCurrentFilterStateChanged_( EventKind kind )
        {
            if( CurrentFilterStateChanged != null )
            {
                CurrentFilterStateChanged( kind );
            }
        }

        #endregion private

        //----------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AttributeFilterManager()
        {
        }

        /// <summary>
        /// 登録
        /// </summary>
        public void Add( string name, IAnmAttribute[] filteredAttrSet )
        {
            if( CheckValid( name, filteredAttrSet ) )
            {
                _filterSet.Add( new CustomAttributeFilter( name, filteredAttrSet ) );
                NotifyCurrentFilterStateChanged_( EventKind.FilterSetModified );
            }
            else
            {
                Debug.Assert( false );
            }
        }

        /// <summary>
        /// 登録
        /// </summary>
        public void Add( string name, string[] filteredPathSet )
        {
            if( FindUserFilterByName( name ) == null )
            {
                _filterSet.Add( new CustomAttributeFilter( name, filteredPathSet ) );
                NotifyCurrentFilterStateChanged_( EventKind.FilterSetModified );
            }
            else
            {
                Debug.Assert( false );
            }
        }

        /// <summary>
        /// 削除
        /// </summary>
        public void Remove( string name )
        {
            IAttributeFilter filter = FindUserFilterByName(name);
            if (filter == null)
            {
                return;
            }

            _filterSet.Remove( filter );
            NotifyCurrentFilterStateChanged_( EventKind.FilterSetModified );

            if( filter == CurrentFilter )
            {
                CurrentFilter = null;
            }
        }

        /// <summary>
        /// Clear
        /// </summary>
        public void Clear()
        {
            _filterSet.Clear();

            this._filterSet.Add(new NoneAttributeFilter());
            this._filterSet.Add(new AnimatedAttributeFilter());
            this._filterSet.Add(new BasicPaneAttributeFilter());
            CurrentFilter = this._filterSet[0];

            NotifyCurrentFilterStateChanged_( EventKind.FilterSetModified );
        }

        /// <summary>
        /// 変更(削除＋登録)
        /// </summary>
        public void EditName(
            string targetName,
            string newName )
        {
            CustomAttributeFilter target = FindUserFilterByName(targetName);
            if (target != null)
            {
                if( FindUserFilterByName( newName ) == null )
                {
                    int index = _filterSet.IndexOf( target );
                    _filterSet.Remove( target );

                    CustomAttributeFilter newFilter = new CustomAttributeFilter(newName, target.FilteredPathSet);
                    _filterSet.Insert( index, newFilter );
                    NotifyCurrentFilterStateChanged_( EventKind.FilterSetModified );

                    if( target == CurrentFilter )
                    {
                        CurrentFilter = newFilter;
                    }
                }
                else
                {
                    Debug.Assert( false );
                }
            }
            else
            {
                Debug.Assert( false );
            }
        }

        /// <summary>
        /// 妥当性確認
        /// </summary>
        public bool CheckValid( string name, IAnmAttribute[] filteredAttrSet )
        {
            return
                name != null && name.Length > 0 &&
                FindUserFilterByName( name ) == null &&
                filteredAttrSet != null &&
                filteredAttrSet.Length > 0;
        }

        /// <summary>
        /// 検索(名前)
        /// </summary>
        public CustomAttributeFilter FindUserFilterByName(string name)
        {
            return _filterSet.Find( delegate( IAttributeFilter filter )
            {
                return filter is CustomAttributeFilter && filter.Name == name;
            }) as CustomAttributeFilter;
        }

        /// <summary>
        /// 検索(名前)
        /// </summary>
        public IAttributeFilter FindFilterByName(string name)
        {
            return _filterSet.Find((filter) => filter.Name == name);
        }
    }
    #endregion AttributeFilterManager

    #endregion フィルタの設定

}
