﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.AnimShareWindow
{
    using System.Linq;
    using Structures.SerializableObject;
    using LayoutEditor.Controls;
    using LECore.Structures;
    using LECore.Manipulator;
    using common;

    public partial class AnimShareWindow : LEToolWindow
    {
        //--------------------------------------------------
        #region フィールド
        SubScenePaneHiddenState _subSceneHiddenState = new SubScenePaneHiddenState();
        readonly AppSetting _appSetting = null;
        #endregion フィールド

        //--------------------------------------------------
        #region プロパティ
        ISubScene _ISubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }


        IAnimShareInfomationChunk _IAnimShareInfomationChunk
        {
            get { return _ISubScene.IAnimShareInfomationManager.IAnimShareInfomationChunkSet.ElementAt(0); }
        }

        IAnimShareInfomation _SelectedIAnimShareInfomation
        {
            get
            {
                if (_lvwAnimShareInfo.SelectedItems.Count <= 0)
                {
                    return null;
                }
                return ((_lvwAnimShareInfo.SelectedItems[0] as ListViewItem).Tag as IAnimShareInfomation);
            }
        }

        #endregion プロパティ

        //--------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimShareWindow(AppSetting appSetting)
        {
            InitializeComponent();

            _cmhSrcPane.Text = StringResMgr.Get("ANIMSHARE_LISTVIEW_HEADER_SRCPANE");
            _cmhTargetGroup.Text = StringResMgr.Get("ANIMSHARE_LISTVIEW_HEADER_GROUP");
            _cmhComment.Text = StringResMgr.Get("TAG_COMMENT");
            _appSetting = appSetting;

            // メッセージフィルタの設定
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_tspMain);

            UpdateProperty_();
        }

        //--------------------------------------------------
        #region GUI状態更新
        /// <summary>
        /// ツールストリップボタンの状態を更新します。
        /// </summary>
        void UpdateButtonState_()
        {
            bool bSelected = _ISubScene != null && _lvwAnimShareInfo.SelectedItems.Count > 0;

            _tsbEdit.Enabled = bSelected;
            _tsbRemove.Enabled = bSelected;
            _tsbHilightGroup.Enabled = bSelected;
        }

        /// <summary>
        /// GUIの状態を更新します。
        /// </summary>
        void UpdateProperty_()
        {
            // フォームの状態を更新します。
            if (_ISubScene == null)
            {
                _tspMain.Enabled = false;
                _lvwAnimShareInfo.Items.Clear();
                return;
            }
            else
            {
                _tspMain.Enabled = true;
            }

            IAnimShareInfomation[] animShareInfoSet = _IAnimShareInfomationChunk.IAnimShareInfomationSet;
            // リストビューを更新します。

            int diffNum = animShareInfoSet.Length - _lvwAnimShareInfo.Items.Count;
            if (diffNum > 0)
            {
                for (int i = 0; i < diffNum; i++)
                {
                    ListViewItem newItem = new ListViewItem("_dummy");
                    newItem.SubItems.Add("_summy_sub0");
                    newItem.SubItems.Add("_summy_sub1");

                    _lvwAnimShareInfo.Items.Add(newItem);
                }
            }
            else if (diffNum < 0)
            {
                for (int numDelete = 0; numDelete < -diffNum; numDelete++)
                {
                    _lvwAnimShareInfo.Items.RemoveAt(_lvwAnimShareInfo.Items.Count - 1);
                }
            }

            for (int i = 0; i < animShareInfoSet.Length; i++)
            {
                IAnimShareInfomation asi = animShareInfoSet[i];
                ListViewItem lvi = _lvwAnimShareInfo.Items[i];

                lvi.Tag = asi;

                if (lvi.Text != asi.SrcPaneName)
                {
                    lvi.Text = asi.SrcPaneName;
                }

                if (lvi.SubItems[1].Text != asi.TargetGroupName)
                {
                    lvi.SubItems[1].Text = asi.TargetGroupName;
                }

                if (lvi.SubItems[2].Text != asi.Comment)
                {
                    lvi.SubItems[2].Text = asi.Comment;
                }

                if (asi.Valid)
                {
                    IPane pane = SubSceneHelper.FindPaneByName(_ISubScene, asi.SrcPaneName);
                    lvi.ForeColor = PaneAnimationHelper.CheckPaneHasPaneAnmKey(pane) ? SystemColors.WindowText : SystemColors.GrayText;
                }
                else
                {
                    lvi.ForeColor = SystemColors.GrayText;
                }
            }

            UpdateButtonState_();
        }
        #endregion GUI状態更新

        //--------------------------------------------------
        #region イベントハンドラ

        /// <summary>
        /// 新規クリック
        /// </summary>
        private void Event_sbNew_Click_(object sender, EventArgs e)
        {
            AnimShareInfomationRegisterDlg dlg = new AnimShareInfomationRegisterDlg();
            dlg.Owner = this;
            dlg.StartPosition = FormStartPosition.CenterParent;

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                AnimShareInfomationChunkManupulator animShareInfomationChunkMnp = new AnimShareInfomationChunkManupulator();

                animShareInfomationChunkMnp.BindTarget(_IAnimShareInfomationChunk);
                animShareInfomationChunkMnp.Register(dlg.SrcPane, dlg.TargetGroup, dlg.Comment);

                UpdateProperty_();
            }
        }

        /// <summary>
        /// 削除クリック
        /// </summary>
        private void Event_TsbRemove_Click_(object sender, EventArgs e)
        {
            if (_lvwAnimShareInfo.SelectedIndices.Count > 0)
            {
                IAnimShareInfomation asi = ((_lvwAnimShareInfo.SelectedItems[0] as ListViewItem).Tag as IAnimShareInfomation);

                AnimShareInfomationChunkManupulator animShareInfomationChunkMnp = new AnimShareInfomationChunkManupulator();

                animShareInfomationChunkMnp.BindTarget(_IAnimShareInfomationChunk);
                animShareInfomationChunkMnp.Remove(asi);
            }
        }

        /// <summary>
        /// ダイアログを出して、IAnimShareInfomationを編集します。
        /// </summary>
        void EditAnimShareInfomationByDlg_(IAnimShareInfomation asi)
        {
            AnimShareInfomationRegisterDlg dlg = new AnimShareInfomationRegisterDlg();

            dlg.Owner = this;
            dlg.StartPosition = FormStartPosition.CenterParent;

            dlg.Set(asi.SrcPaneName, asi.TargetGroupName, asi.Comment);
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                AnimShareInfomationChunkManupulator animShareInfomationChunkMnp = new AnimShareInfomationChunkManupulator();

                animShareInfomationChunkMnp.BindTarget(_IAnimShareInfomationChunk);
                animShareInfomationChunkMnp.Modify(asi, dlg.SrcPane, dlg.TargetGroup, dlg.Comment);
            }
        }

        /// <summary>
        /// 編集
        /// </summary>
        private void Event_TsbEdit_Click_(object sender, EventArgs e)
        {
            IAnimShareInfomation asi = _SelectedIAnimShareInfomation;
            if (asi != null)
            {
                EditAnimShareInfomationByDlg_(asi);
            }
        }

        /// <summary>
        /// グループ作成
        /// </summary>
        private void Event_BtnMakeGroup_Click_(object sender, EventArgs e)
        {
            AnimShareMakeTargetGroupDlg dlg = new AnimShareMakeTargetGroupDlg(_appSetting);
            dlg.Owner = this;
            dlg.StartPosition = FormStartPosition.CenterParent;

            if (dlg.ShowDialog() == DialogResult.OK)
            {
            }
        }

        /// <summary>
        /// ダブルクリック
        /// </summary>
        private void Event_LvwAnimShareInfo_DoubleClick_(object sender, EventArgs e)
        {
            if (_lvwAnimShareInfo.SelectedItems.Count > 0)
            {
                Debug.Assert(_lvwAnimShareInfo.SelectedItems.Count == 1);
                IAnimShareInfomation asi = ((_lvwAnimShareInfo.SelectedItems[0] as ListViewItem).Tag as IAnimShareInfomation);

                EditAnimShareInfomationByDlg_(asi);
            }
        }

        /// <summary>
        /// 選択アイテム変更
        /// </summary>
        private void Event_LvwAnimShareInfo_SelectedIndexChanged_(object sender, EventArgs e)
        {
            _tsbHilightGroup.CheckState = CheckState.Unchecked;
            UpdateButtonState_();
        }

        /// <summary>
        /// 共有対象をハイライト表示
        /// </summary>
        private void Event_TsbHilightGroup_CheckStateChanged_(object sender, EventArgs e)
        {
            // 保存していた状態を戻す。
            if (!_subSceneHiddenState.Empty)
            {
                _subSceneHiddenState.RestoreState();
            }

            if (_tsbHilightGroup.CheckState == CheckState.Checked)
            {
                _subSceneHiddenState.StoreState(_ISubScene);
                IAnimShareInfomation asi = _SelectedIAnimShareInfomation;
                if (asi != null && asi.Valid)
                {
                    PaneManipulator paneMnp = new PaneManipulator();

                    IPane shareSrcPane = SubSceneHelper.FindPaneByName(_ISubScene, asi.SrcPaneName);
                    ILEGroup group = _ISubScene.ILEGroupMgr.FindGroupByName(asi.TargetGroupName);
                    IPane[] groupMemberPane = group.Member;
                    Debug.Assert(shareSrcPane != null);


                    _ISubScene.BeginMassiveModify();

                    // グループ以外を隠す
                    foreach (IPane pane in _ISubScene.IPaneArray)
                    {
                        paneMnp.BindTarget(pane);
                        // グループに含まれてなければ隠す。
                        paneMnp.Hidden = Array.IndexOf(groupMemberPane, pane) == -1 && pane != shareSrcPane;
                    }

                    // 共有元ペインを選択する
                    SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                    subSceneMnp.BindTarget(_ISubScene);

                    subSceneMnp.BeginSelectSetChange();
                    subSceneMnp.ResetSelectedSet();
                    subSceneMnp.SelectPanesByPaneRef(shareSrcPane);
                    subSceneMnp.EndSelectSetChange();

                    _ISubScene.EndMassiveModify();
                }
                else
                {
                    _tsbHilightGroup.CheckState = CheckState.Unchecked;
                }
            }
        }

        /// <summary>
        /// アクティブ化ハンドラ
        /// </summary>
        private void Event_AnimShareWindow_Activated_(object sender, EventArgs e)
        {
            if (_ISubScene == null)
            {
                return;
            }

            AnimShareInfomationChunkManupulator animShareInfomationChunkMnp = new AnimShareInfomationChunkManupulator();

            animShareInfomationChunkMnp.BindTarget(_IAnimShareInfomationChunk);
            animShareInfomationChunkMnp.UpdateValidityState(_ISubScene);

            foreach (ListViewItem lvi in _lvwAnimShareInfo.Items)
            {
                IAnimShareInfomation asi = lvi.Tag as IAnimShareInfomation;
                if (asi.Valid)
                {
                    IPane pane = SubSceneHelper.FindPaneByName(_ISubScene, asi.SrcPaneName);
                    lvi.ForeColor = PaneAnimationHelper.CheckPaneHasPaneAnmKey(pane) ? SystemColors.WindowText : SystemColors.GrayText;
                }
                else
                {
                    lvi.ForeColor = SystemColors.GrayText;
                }
            }
        }
        #endregion イベントハンドラ

        //--------------------------------------------------
        #region GUI状態永続化：ILEDataSerializebleWindow オーバーライド
        public override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            setting.UserDataSet.Add(new LEControlUserDataChunk("columnPaneName_W", _cmhSrcPane.Width.ToString()));
            setting.UserDataSet.Add(new LEControlUserDataChunk("columnGroup_W", _cmhTargetGroup.Width.ToString()));
            setting.UserDataSet.Add(new LEControlUserDataChunk("columnComment_W", _cmhComment.Width.ToString()));

            base.SaveSetting(setting, option);
        }

        public override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            // リスト行の幅の読み込み
            int value = 0;

            if (setting.TryToFindUserIntDataByName("columnPaneName_W", out value))
            {
                _cmhSrcPane.Width = value;
            }
            if (setting.TryToFindUserIntDataByName("columnGroup_W", out value))
            {
                _cmhTargetGroup.Width = value;
            }
            if (setting.TryToFindUserIntDataByName("columnComment_W", out value))
            {
                _cmhComment.Width = value;
            }
            base.LoadSetting(setting, option);
        }

        #endregion ILEDataSerializebleWindow オーバーライド

        //--------------------------------------------------
        #region ショートカット　オーバーライド
        public override Keys CustomShortcut { get { return Keys.Alt | Keys.F8; } }
        #endregion ショートカット　オーバーライド
        //--------------------------------------------------
        #region ILEToolWindow.ISceneModifyListener

        /// <summary>
        /// シーン更新イベントハンドラ
        /// </summary>
        public override void OnSceneModifyHandler(object sender, SceneModifyEventArgs e)
        {
            switch (e.Target)
            {
                case SceneModifyEventArgs.Kind.GroupModify:
                case SceneModifyEventArgs.Kind.GroupMake:
                case SceneModifyEventArgs.Kind.GroupRemove:
                case SceneModifyEventArgs.Kind.GroupReorder:
                    {
                        if (_ISubScene != null)
                        {
                            AnimShareInfomationChunkManupulator animShareInfomationChunkMnp = null;
                            animShareInfomationChunkMnp = new AnimShareInfomationChunkManupulator();
                            animShareInfomationChunkMnp.BindTarget(_IAnimShareInfomationChunk);
                            animShareInfomationChunkMnp.UpdateValidityState(_ISubScene);
                            UpdateProperty_();
                        }
                        break;
                    }

                // カレントサブシーンの更新
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                case SceneModifyEventArgs.Kind.AnimShareInfomarionManager:
                    {
                        UpdateProperty_();
                        break;
                    }
                default:
                    {
                        // 何も反応しません。
                        break;
                    }
            }

            base.OnSceneModifyHandler(sender, e);
        }

        #endregion ILEToolWindow.ISceneModifyListener

        /// <summary>
        /// キーメッセージを処理します。
        /// 登録されたイベントハンドラを実行します。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // "元に戻す"、"やり直し"を行う
            if (SubSceneUndoRedoHelper.CanUndoRedo(this, keyData) != false)
            {
                // オーナーが ILECmdKeyAccepter ならメッセージを伝播します。
                if (base.NotifyCmdKeyMessageToOwner(ref msg, keyData))
                {
                    return true;
                }
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }
    }
}
