﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.AnimShareWindow
{
    using LECore.Structures;
    using LECore.Manipulator;
    using LayoutEditor.Controls;
    using ToolWindows.LayoutWindow;
    using Structures.SerializableObject;

    /// <summary>
    ///
    /// </summary>
    public partial class AnimShareMakeTargetGroupDlg : Form
    {
        SubScenePaneHiddenState _subScenePaneHiddenState = new SubScenePaneHiddenState();
        //-------------------------------------------------
        #region フィールド

        readonly AppSetting _appSetting = null;

        #endregion フィールド

        #region プロパティ
        ISubScene _ISubScene
        {
            get{ return LECore.LayoutEditorCore.Scene.CurrentISubScene;}
        }

        ILEGroupMgr _ILEGroupMgr
        {
            get { return _ISubScene.ILEGroupMgr; }
        }
        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimShareMakeTargetGroupDlg(AppSetting appSetting)
        {

            InitializeComponent();

            _appSetting = appSetting;

            TextBoxValidator tbxValidator = new TextBoxValidator(
                _tbxTargetGroupName,
                delegate( string text ){
                    return GroupHelper.CheckGroupNameValid( text );
                } );

            UpdateProperty_();
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateProperty_PaneHiddenState_()
        {
            if( !_subScenePaneHiddenState.Empty )
            {
                _subScenePaneHiddenState.RestoreState();
            }

            if( _chkHideOthers.Checked )
            {
                _subScenePaneHiddenState.StoreState( _ISubScene );

                PaneManipulator paneMnp = new PaneManipulator();

                IPane[] targetPaneSet = _lvwPaneName.Tag as IPane[];

                _ISubScene.BeginMassiveModify();

                foreach( IPane pane in _ISubScene.IPaneArray )
                {
                    paneMnp.BindTarget( pane );
                    paneMnp.Hidden = Array.IndexOf( targetPaneSet, pane ) == -1;
                }

                _ISubScene.EndMassiveModify();
            }
        }

        /// <summary>
        ///
        /// </summary>
        void UpdateProperty_()
        {
            _btnDoSearch.Enabled = _cmbSrcPane.SelectedItem != null;

            // グループメンバペインが検索されているか？
            bool bGroupMemberValid = _lvwPaneName.Items.Count > 0;
            _grpTargetGroup.Enabled = bGroupMemberValid;

            // グループ名は適切か？
            bool bGroupNameValid = GroupHelper.CheckGroupNameValid( _tbxTargetGroupName.Text );

            _btnCreate.Enabled = bGroupNameValid & bGroupMemberValid;
        }

        /// <summary>
        ///
        /// </summary>
        void SetSelectedPaneToCombobox_()
        {
            if( !_ISubScene.ISelectedSet.Empty )
            {
                IPane selectedPane = _ISubScene.ISelectedSet.IPaneArray[0];
                UIListControlItem selectedItem = null;
                foreach( UIListControlItem item in _cmbSrcPane.Items )
                {
                    if( item.Data == selectedPane )
                    {
                        selectedItem = item;
                    }
                }


                if( selectedItem != null )
                {
                    _cmbSrcPane.SelectedItem = selectedItem;
                }
            }
        }

        //--------------------------------------------------------
        #region イベントハンドラ
        private void AnimShareMakeTargetGroupDlg_Shown( object sender, EventArgs e )
        {
            // シーンに存在するペイン名前をコンボボックスに登録します。
            foreach( IPane pane in _ISubScene.IPaneArray )
            {
                UIListControlItem item = new UIListControlItem( pane.PaneName, pane );
                _cmbSrcPane.Items.Add( item );
            }

            // 選択ペインがあれば、対象に設定します。
            SetSelectedPaneToCombobox_();
        }

        /// <summary>
        /// 選択ペインを設定
        /// </summary>
        private void Event_BtnSetSelectedPane_Click( object sender, EventArgs e )
        {
            // 選択ペインがあれば、対象に設定します。
            SetSelectedPaneToCombobox_();
        }

        /// <summary>
        /// 対象を検索ハンドラ
        /// </summary>
        private void Event_BtnDoSearch_Click( object sender, EventArgs e )
        {
            //-----------------------------------
            // リストを消去する。
            _lvwPaneName.Items.Clear();

            //-----------------------------------
            // 条件に合わせて検索を実行する。
            Debug.Assert( _cmbSrcPane.SelectedItem != null );

            IPane srcPane = (_cmbSrcPane.SelectedItem as UIListControlItem).Data as IPane;
            Debug.Assert( srcPane != null );
            // 検索基点ペインを計算します。
            IPane serchOriginPane = srcPane;

            int levelFromSrcToOrigin = (int)_nudLevelFromSrcToOrigin.Value;
            Debug.Assert( levelFromSrcToOrigin >= 1 );



            IPane[] paneSet = AnimShareInfomationHelper.CollectTargetPane( srcPane, levelFromSrcToOrigin, _chkSearchOnlySameKind.Checked );


            //-----------------------------------
            // 検索結果をリストに登録する。
            foreach( IPane pane in paneSet )
            {
                ListViewItem lvi = new ListViewItem( pane.PaneName );
                lvi.Tag = pane;

                _lvwPaneName.Items.Add( lvi );
            }
            _lvwPaneName.Tag = paneSet;


            //-----------------------------------
            // 既定のグループ名を設定する。
            _tbxTargetGroupName.Text =
                AnimShareInfomationHelper.GetDefaultGroupNameForSrcPane( _ILEGroupMgr, srcPane );


            //-----------------------------------
            // ユーザインタフェースの状態を更新する。
            _chkHideOthers.Checked = false;
            UpdateProperty_();
            _tbxTargetGroupName.Focus();
        }

        /// <summary>
        /// 作成ボタンハンドラ
        /// </summary>
        private void Event_BtnCreate_Click_( object sender, EventArgs e )
        {
            IPane[] paneSet = _lvwPaneName.Tag as IPane[];

            string groupName = GroupHelper.GetUniqueGroupName( _ILEGroupMgr, _tbxTargetGroupName.Text );
            if( paneSet != null && GroupHelper.CheckGroupNameValid( groupName ) )
            {
                GroupMgrManipulator groupMgrManipulator = new GroupMgrManipulator();
                groupMgrManipulator.BindTarget( _ILEGroupMgr );
                ILEGroup newGroup = groupMgrManipulator.MakeNewGroupFromPaneSet(
                    paneSet,
                    _appSetting?.ProjectSettings?.GetNewGroupPaneName());
                if( newGroup != null )
                {
                    GroupManipulator groupMnp = new GroupManipulator();
                    groupMnp.BindTarget( newGroup );
                    groupMnp.Name = groupName;
                }
            }

            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタンハンドラ
        /// </summary>
        private void Evnet_BtnCancel_Click_( object sender, EventArgs e )
        {
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// 対象以外を隠すチェックボックス
        /// </summary>
        private void Event_ChkHideOthers_CheckedChanged_( object sender, EventArgs e )
        {
            UpdateProperty_PaneHiddenState_();
        }

        /// <summary>
        /// フォーム閉じる
        /// </summary>
        private void Event_AnimShareMakeTargetGroupDlg_FormClosed_( object sender, FormClosedEventArgs e )
        {
            // 隠しているペインを元に戻してから終了する。
            if( !_subScenePaneHiddenState.Empty )
            {
                _subScenePaneHiddenState.RestoreState();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_CmbSrcPane_SelectedIndexChanged_( object sender, EventArgs e )
        {
            UpdateProperty_();
        }
        #endregion イベントハンドラ






    }

    //-------------------------------------------------
    #region AnimShareInfomationHelper
    public static class AnimShareInfomationHelper
    {
        const string _DefaultGroupNamePreFix = "AS_";

        //-----------------------------------
        #region CollectTaskParamaters
        /// <summary>
        /// 共有対象を検索する際の、状態保持クラス
        /// </summary>
        class CollectTaskParamaters
        {
            public readonly IPane _animShareSrcPane;
            public readonly bool _bCollectSameKindOnly;
            public readonly List<IPane> _targetPaneSet = new List<IPane>();

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public CollectTaskParamaters( IPane animShareSrcPane, bool bCollectSameKindOnly )
            {
                _animShareSrcPane = animShareSrcPane;
                _bCollectSameKindOnly = bCollectSameKindOnly;
            }
        }
        #endregion CollectTaskParamaters

        /// <summary>
        /// ペインに対応した、既定のアニメーション共有用グループ名を取得します。
        /// </summary>
        public static string GetDefaultGroupNameForSrcPane( ILEGroupMgr groupMgr, IPane srcPane )
        {
            //-----------------------------------
            // 既定のグループ名を設定する。
            string groupName = _DefaultGroupNamePreFix + srcPane.PaneName;

            if( groupName.Length >= LECore.AppConstants.MaxGroupNameLength )
            {
                groupName = groupName.Substring( 0, LECore.AppConstants.MaxGroupNameLength );
            }

            return GroupHelper.GetUniqueGroupName( groupMgr, groupName );
        }


        /// <summary>
        /// 再帰的にペインを走査して、条件にあったペインを集めます。
        /// </summary>
        static void CollectPaneRecursive_( IPane pane, int depth, CollectTaskParamaters paramaters )
        {
            Debug.Assert( pane != null );
            Debug.Assert( paramaters != null );

            if( depth <= 0 )
            {
                // 同一種類のペインだけ収集する必要があればチェックする。
                if( paramaters._bCollectSameKindOnly &&
                    paramaters._animShareSrcPane.PaneKind != pane.PaneKind )
                {
                    // 不適合
                    return;
                }
                paramaters._targetPaneSet.Add( pane );
            }
            else
            {
                foreach( IPane childPane in pane.Children )
                {
                    CollectPaneRecursive_( childPane, depth - 1, paramaters );
                }
            }
        }

        /// <summary>
        /// searchRootPane を 基点に、srcPaneと階層的に、
        /// 同一距離に存在するペインを集めます。
        ///
        /// アニメーション共有対象
        /// </summary>
        public static IPane[] CollectTargetPane( IPane animSrcPane, int levelFromSrcToOrigin, bool bCollectSameKindOnly )
        {
            Debug.Assert( animSrcPane != null );
            Debug.Assert( levelFromSrcToOrigin > 0 );


            //------------------------------------------------------
            // srcPane から searchRootPane を探しならが
            // 何番目の子供かを記録しながらさかのぼる
            int maxDepthToTargetCandidate = levelFromSrcToOrigin;

            IPane serchOriginPane = animSrcPane;
            while( serchOriginPane.Parent != null && levelFromSrcToOrigin > 0 )
            {
                serchOriginPane = serchOriginPane.Parent as IPane;
                levelFromSrcToOrigin--;
            }

            //------------------------------------------------------
            // searchRootPaneのすべての子供について...
            CollectTaskParamaters paramaters = new CollectTaskParamaters( animSrcPane, bCollectSameKindOnly );
            CollectPaneRecursive_( serchOriginPane, maxDepthToTargetCandidate, paramaters );

            return paramaters._targetPaneSet.ToArray();
        }
    }

    #endregion AnimShareInfomationHelper

    //-------------------------------------------------
    #region TextBoxValidator
    /// <summary>
    /// テキストボックスへの無効な入力をキャンセルする
    /// 補助モジュール。
    /// </summary>
    public class TextBoxValidator
    {
        //-------------------------------------------------------
        #region フィールド
        /// <summary>
        /// 文字列の有効性を判定するデリゲータ
        /// </summary>
        readonly CheckTextValidityHandler _CheckTextValidityHandler;

        /// <summary>
        /// 有効性を検証する対象となる、TextBox
        /// </summary>
        TextBox _textBox = null;

        /// <summary>
        /// 最後に有効だった文字列
        /// 無効な文字列変更があった場合に、_textBoxに対して設定します。
        /// </summary>
        string _validTextBackup = string.Empty;

        /// <summary>
        ///
        /// </summary>
        public delegate bool CheckTextValidityHandler( string text );
        #endregion フィールド

        //-------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextBoxValidator( TextBox textBox, CheckTextValidityHandler checkTextValidityHandler )
        {
            Debug.Assert( textBox != null );
            Debug.Assert( checkTextValidityHandler != null );

            _textBox = textBox;
            _textBox.TextChanged += new EventHandler( Event_TextBox_TextChanged_ );

            _CheckTextValidityHandler = checkTextValidityHandler;

            CheckAndUpdateTextValidity_();
        }

        /// <summary>
        /// 文字列の有効性を検証します。
        /// </summary>
        void CheckAndUpdateTextValidity_()
        {
            if( _textBox.Text == null || _textBox.Text == string.Empty )
            {
                return;
            }

            // 変わっているなら
            if( _validTextBackup != _textBox.Text )
            {
                // 有効かどうかを判定し...
                if( _CheckTextValidityHandler( _textBox.Text ) )
                {
                    // 最後に設定された有効な文字列を更新します。
                    _validTextBackup = _textBox.Text;
                }
                else
                {
                    // 無効なら元に戻します。
                    _textBox.Text = _validTextBackup;
                }
            }
        }

        /// <summary>
        /// 文字列変更ハンドラ
        /// </summary>
        void Event_TextBox_TextChanged_( object sender, EventArgs e )
        {
            Debug.Assert( sender == _textBox );
            CheckAndUpdateTextValidity_();
        }
    }
    #endregion TextBoxValidator


}
