﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Diagnostics;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.AnimShareWindow
{
    using LECore.Structures;
    using LayoutEditor.Controls;

    public partial class AnimShareInfomationRegisterDlg : Form
    {
        /// <summary>
        ///
        /// </summary>
        class PaneData
        {
            public readonly IPane _pane;
            public readonly bool _bPaneHasAnimation;

            public PaneData( IPane pane )
            {
                _pane = pane;
                _bPaneHasAnimation = PaneAnimationHelper.CheckPaneHasPaneAnmKey(_pane);
            }
        }

        //-------------------------------------------------
        #region プロパティ
        ISubScene _ISubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        ILEGroupMgr _ILEGroupMgr
        {
            get { return _ISubScene.ILEGroupMgr; }
        }

        public IPane SrcPane
        {
            get
            {
                if( _cmbSrcPane.SelectedItem != null )
                {
                    return ((_cmbSrcPane.SelectedItem as UIListControlItem).Data as PaneData)._pane;
                }else{
                    return null;
                }
            }


        }

        public ILEGroup TargetGroup
        {
            get
            {
                if( _cmbTargetGroup.SelectedItem != null )
                {
                    return ((_cmbTargetGroup.SelectedItem as UIListControlItem).Data as ILEGroup);
                }else{
                    return null;
                }
            }
        }

        public string Comment
        {
            get{return _tbxComment.Text;}
        }
        #endregion プロパティ


        SolidBrush			_ownerDrawBrush = new SolidBrush( Color.Empty );


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimShareInfomationRegisterDlg()
        {
            InitializeComponent();

            // シーンに存在するペイン名前をコンボボックスに登録します。
            foreach( IPane pane in _ISubScene.IPaneArray )
            {
                UIListControlItem item = new UIListControlItem( pane.PaneName, new PaneData( pane ) );
                _cmbSrcPane.Items.Add( item );
            }

            // シーンに存在するグループ名前をコンボボックスに登録します。
            foreach( ILEGroup group in _ISubScene.ILEGroupMgr.GroupSet )
            {
                UIListControlItem item = new UIListControlItem( group.GrouprName, group );
                _cmbTargetGroup.Items.Add( item );
            }
            // 互換性を維持するためには、アニメーション共有機能を利用しないで下さい。
            //_capAnimationShare.ToolTipText = StringResMgr.Get( "ANIMSHARE_COMPATIVILITY_TOOLTIP" );
        }

        /// <summary>
        ///
        /// </summary>
        public void Set( string paneName, string targetGroupName, string commnetStr )
        {
            foreach( UIListControlItem item in _cmbSrcPane.Items )
            {
                if( item.Text == paneName )
                {
                    _cmbSrcPane.SelectedItem = item;
                    break;
                }
            }

            foreach( UIListControlItem item in _cmbTargetGroup.Items )
            {
                if( item.Text == targetGroupName )
                {
                    _cmbTargetGroup.SelectedItem = item;
                    break;
                }
            }

            _tbxComment.Text = commnetStr;
        }

        /// <summary>
        ///
        /// </summary>
        void SetSelectedPaneToCombobox_()
        {
            if( !_ISubScene.ISelectedSet.Empty )
            {
                IPane selectedPane = _ISubScene.ISelectedSet.IPaneArray[0];
                UIListControlItem selectedItem = null;
                foreach( UIListControlItem item in _cmbSrcPane.Items )
                {
                    if( (item.Data as PaneData)._pane == selectedPane )
                    {
                        selectedItem = item;
                    }
                }


                if( selectedItem != null )
                {
                    _cmbSrcPane.SelectedItem = selectedItem;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        bool _ReadyToRegister
        {
            get { return _cmbSrcPane.SelectedItem != null && _cmbTargetGroup.SelectedItem != null; }
        }

        /// <summary>
        /// ボタンの状態を更新します。
        /// </summary>
        void UpdateButtonState_()
        {
            _btnRegister.Enabled = _ReadyToRegister;
        }

        #region イベントハンドラ

        int GetComboItemIndexByStartString_( ComboBox cbx, string startStr )
        {
            for( int i = 0 ; i < cbx.Items.Count ; i++ )
            {
                string itemText = ( cbx.Items[i] as UIListControlItem ).Text;
                if( itemText.StartsWith( startStr ) )
                {
                    return i;
                }
            }
            return -1;
        }

        /// <summary>
        ///
        /// </summary>
        private void AnimShareInfomationRegisterDlg_Shown( object sender, EventArgs e )
        {
            // 選択ペインがあれば、対象に設定します。
            if( _cmbSrcPane.SelectedItem == null )
            {
                SetSelectedPaneToCombobox_();
            }

            // 選択ペインから類推して、グループを設定する。
            if( _cmbSrcPane.SelectedItem != null )
            {
                IPane srcPane = ( ( _cmbSrcPane.SelectedItem as UIListControlItem ).Data as PaneData )._pane;

                string groupName = AnimShareInfomationHelper.GetDefaultGroupNameForSrcPane( _ILEGroupMgr, srcPane );

                // グループ名で開始されているアイテムの選択を試みる。
                int index = GetComboItemIndexByStartString_( _cmbTargetGroup, groupName );
                if( index == -1 )
                {
                    // 行末の数値を取り除き、再度試みる。
                    groupName = Regex.Replace( groupName, "[0-9]+$", "" );
                    index = GetComboItemIndexByStartString_( _cmbTargetGroup, groupName );
                }

                if( index != -1 )
                {
                    _cmbTargetGroup.SelectedIndex = index;
                }
            }

            UpdateButtonState_();
        }

        /// <summary>
        /// 選択ペインを設定
        /// </summary>
        private void Event_BtnSetSelectedPane_Click( object sender, EventArgs e )
        {
            // 選択ペインがあれば、対象に設定します。
            SetSelectedPaneToCombobox_();
        }

        /// <summary>
        /// 抽出元ペインを設定
        /// </summary>
        private void Event_CmbSrcPane_SelectedIndexChanged_( object sender, EventArgs e )
        {
            UpdateButtonState_();
        }

        /// <summary>
        /// 対象グループを設定
        /// </summary>
        private void Event_CmbTargetGroup_SelectedIndexChanged_( object sender, EventArgs e )
        {
            UpdateButtonState_();
        }

        /// <summary>
        /// 登録ボタン
        /// </summary>
        private void Event_BtnRegister_Click_( object sender, EventArgs e )
        {
            Debug.Assert( _ReadyToRegister );

            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click_( object sender, EventArgs e )
        {
            DialogResult = DialogResult.Cancel;
        }
        #endregion イベントハンドラ

        private void Event_CmbSrcPane_DrawItem_( object sender, DrawItemEventArgs e )
        {
            if( e.Index != -1 )
            {
                UIListControlItem item = _cmbSrcPane.Items[e.Index] as UIListControlItem;

                string text = ( item.Data as PaneData )._bPaneHasAnimation ?
                    item.Text : "    " + item.Text + "    ( NO Animation. )";

                if( (e.State & DrawItemState.Focus) != 0  )
                {
                    _ownerDrawBrush.Color = SystemColors.HighlightText;
                }
                else
                {
                    _ownerDrawBrush.Color = ( item.Data as PaneData )._bPaneHasAnimation ? SystemColors.WindowText : SystemColors.GrayText;
                }

                e.DrawBackground();
                e.Graphics.DrawString( text, _cmbSrcPane.Font, _ownerDrawBrush, new RectangleF( e.Bounds.X, e.Bounds.Y, e.Bounds.Width, e.Bounds.Height ) );
                e.DrawFocusRectangle();
            }
            else
            {
                e.DrawBackground();
                e.DrawFocusRectangle();
            }
        }


    }
}
