﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Linq;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.AnimSectionTagWindow
{
    using LECore.Structures;
    using LECore.Manipulator;
    using LayoutEditor.Controls;

    /// <summary>
    /// フレーム区間 編集ダイアログ
    /// </summary>
    public partial class AnimSectionEditDlg : Form
    {
        IAnimFrameSectionSet _targetAnimFrameSectionSet = null;

        int _frameLength = 1;
        int _frameOffset = 0;

        /// <summary>
        ///
        /// </summary>
        int _FrameLength
        {
            get { return _frameLength; }
            set { _frameLength = value; }
        }

        /// <summary>
        ///
        /// </summary>
        int _FrameOffset
        {
            get { return _frameOffset; }
            set { _frameOffset = value; }
        }

       #region プロパティ
        /// <summary>
        /// フレーム区間セット
        /// </summary>
        public IAnimFrameSectionSet IAnimFrameSectionSet
        {
            set { _targetAnimFrameSectionSet = value; }
        }

        /// <summary>
        /// ターゲットフレーム区間セット
        /// </summary>
        IEnumerable<IAnimFrameSection> _TargetframeSectionSet
        {
            get
            {
                if( _targetAnimFrameSectionSet != null )
                {
                    return _targetAnimFrameSectionSet.IAnimFrameSectionSet;
                }
                else
                {
                    return new IAnimFrameSection[0];
                }
            }
        }

        /// <summary>
        /// フレームにオフセット値を加算する機能 有効/無効
        /// </summary>
        public bool ChkAddFrameOffsetEnable
        {
            set { _chkAddFrameOffset.Enabled = value; }
        }
       #endregion プロパティ



        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimSectionEditDlg()
        {
            InitializeComponent();
        }

       #region イベントハンドラ

        /// <summary>
        /// OKボタンの状態
        /// </summary>
        void UpdateOKButtonEnabled_()
        {
            _btnOK.Enabled = _chkSetFrameLength.Checked || _chkAddFrameOffset.Checked;
        }

        /// <summary>
        /// チェックボックス
        /// </summary>
        private void Event_ChkSetFrameLength_CheckedChanged( object sender, EventArgs e )
        {
            CheckBox cb = sender as CheckBox;

            _lblFrameLength.Enabled = cb.Checked;
            _tbxFrameLength.Enabled = cb.Checked;

            UpdateOKButtonEnabled_();
        }

        /// <summary>
        /// チェックボックス
        /// </summary>
        private void Event_ChkAddFrameOffset_CheckedChanged( object sender, EventArgs e )
        {
            CheckBox cb = sender as CheckBox;

            _lblFrameOffset.Enabled = cb.Checked;
            _tbxFrameOffset.Enabled = cb.Checked;

            UpdateOKButtonEnabled_();
        }

        /// <summary>
        /// 可視状態変更
        /// 表示時にGUIの初期化を行います。
        /// </summary>
        private void Event_AnimSectionEditDlg_VisibleChanged( object sender, EventArgs e )
        {
            Debug.Assert( _TargetframeSectionSet.Count() > 0 );

            if( this.Visible == true )
            {
                _lblFrameLength.Enabled = _chkSetFrameLength.Checked;
                _tbxFrameLength.Enabled = _chkSetFrameLength.Checked;
                _lblFrameOffset.Enabled = _chkAddFrameOffset.Checked;
                _tbxFrameOffset.Enabled = _chkAddFrameOffset.Checked;

                _tbxFrameLength.SetValue( _frameLength );
                _tbxFrameOffset.SetValue( _frameOffset );

                foreach( IAnimFrameSection section in _TargetframeSectionSet )
                {
                    var newItem = new ListViewItem(section.Name);
                    newItem.Checked = true;
                    newItem.ForeColor = section.IsReadOnlyLocked ? LECore.AppConstants.InactiveForeColor : LECore.AppConstants.ActiveForeColor;
                    newItem.BackColor = section.IsReadOnlyLocked ? LECore.AppConstants.InactiveBackColor : LECore.AppConstants.ActiveBackColor;
                    _clbTargetSection.Items.Add(newItem);
                }

                UpdateOKButtonEnabled_();
            }
        }

        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOK_Click( object sender, EventArgs e )
        {
            int offset = ( _chkAddFrameOffset.Checked ) ? _tbxFrameOffset.Value : 0;
            int length = _tbxFrameLength.Value;
            foreach( ListViewItem checkedItem in _clbTargetSection.CheckedItems )
            {
                string sectionTagName = checkedItem.Text;
                foreach( IAnimFrameSection section in _TargetframeSectionSet )
                {
                    if( section.Name == sectionTagName )
                    {
                        int endFrame = ( _chkSetFrameLength.Checked ) ?
                            section.StartFrame + offset + length :
                            section.EndFrame + offset;

                        /// <summary>
                        /// アニメーション区間タグ編集クラス。
                        /// </summary>
                        AnimFrameSectionSetManipulator _animFrameSectionSetMnp = new AnimFrameSectionSetManipulator();
                        _animFrameSectionSetMnp.BindTarget( _targetAnimFrameSectionSet );

                        _animFrameSectionSetMnp.SetTagBasic(
                            section,
                            section.Name,
                            section.Comment,
                            section.StartFrame + offset,
                            endFrame );
                    }
                }
            }
            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click( object sender, EventArgs e )
        {
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// すべてをチェック
        /// </summary>
        private void Event_ChkCheckAll_CheckedChanged( object sender, EventArgs e )
        {
            foreach (ListViewItem item in _clbTargetSection.Items)
            {
                item.Checked = _chkCheckAll.Checked;
            }
        }

        /// <summary>
        /// フレーム長さ 値変更
        /// </summary>
        private void Event_TbxFrameLength_ValueChanged( object sender, EventArgs e )
        {
            IntTextBox itb = sender as IntTextBox;
            if( itb.Value > 0 )
            {
                _frameLength = itb.Value;
            }
            else
            {
                itb.SetValue( _frameLength );
            }
        }

        /// <summary>
        /// 有効なオフセット値か確認します。
        /// </summary>
        bool IsFrameOffsetValid_( int offset )
        {
            foreach( IAnimFrameSection section in _TargetframeSectionSet )
            {
                if( section.StartFrame + offset < 0 )
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// フレームオフセット 値変更
        /// </summary>
        private void Event_TbxFrameOffset_ValueChanged( object sender, EventArgs e )
        {
            IntTextBox itb = sender as IntTextBox;
            if( IsFrameOffsetValid_( itb.Value ) )
            {
                _frameOffset = itb.Value;
            }
            else
            {
                itb.SetValue( _frameOffset );
            }
        }

        /// <summary>
        /// チェック更新
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Event_ClbTargetSection_ItemCheck(object sender, ItemCheckEventArgs e)
        {
            var targetframeSection = _TargetframeSectionSet.ElementAt(e.Index);
            if (targetframeSection != null)
            {
                // ロック状態なら変更をキャンセルします。
                if (targetframeSection.IsReadOnlyLocked)
                {
                    e.NewValue = e.CurrentValue;
                }
            }
        }
       #endregion イベントハンドラ







    }
}
