﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;
using LECore.Structures;

namespace LayoutEditor.Forms.Dialogs
{
    /// <summary>
    /// 部品設定ダイアログ
    /// </summary>
    public partial class SetPartsLayoutDialog : LEBaseDlg
    {
        private const int NormalLayoutIndex = 0;

        /// <summary>
        /// コンボボックス用アイテムです。
        /// </summary>
        private class PartsControlSettingItem
        {
            private readonly IPartsControlSetting _controlSetting;

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public PartsControlSettingItem(IPartsControlSetting controlSetting)
            {
                _controlSetting = controlSetting;
            }

            /// <summary>
            /// 文字列に変換します。
            /// </summary>
            public override string  ToString()
            {
                return string.Format("{0} [{1}]", this._controlSetting.UIName, this._controlSetting.Name);
            }

            /// <summary>
            /// 部品コントロール定義です。
            /// </summary>
            public IPartsControlSetting PartsControlSetting
            {
                get { return this._controlSetting; }
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SetPartsLayoutDialog()
        {
            InitializeComponent();
        }

        //----------------------------------------------------------
        // プロパティ
        //----------------------------------------------------------

        /// <summary>
        /// ユーザによって選択された部品設定ひな形です。
        /// </summary>
        public bool IsPartsLayoutEnabled
        {
            get { return _chkPartsLayout.Checked; }
        }

        /// <summary>
        /// 説明名称です。
        /// </summary>
        public string PartsDescriptionName
        {
            get { return _tbxPartsDescriptionName.Text; }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 初期化します。
        /// </summary>
        public void Initialize(bool isPartsLayoutEnabled, string descriptionName)
        {
            // 現在の状態
            {
                _chkPartsLayout.SetChecked(isPartsLayoutEnabled);
                _tbxPartsDescriptionName.Text = descriptionName;
            }

            // UI の更新。
            this.UpdateUIState();
        }

        //----------------------------------------------------------

        /// <summary>
        /// UIを更新します。
        /// </summary>
        private void UpdateUIState()
        {
            _btnOK.Enabled =
                !IsPartsLayoutEnabled ||
                !string.IsNullOrEmpty(this._tbxPartsDescriptionName.Text);
        }

        //----------------------------------------------------------
        // イベントハンドラです。
        //----------------------------------------------------------

        /// <summary>
        /// 部品名変更。
        /// </summary>
        private void Event_TbxPartsDescriptionName_TextChanged(object sender, EventArgs e)
        {
            this.UpdateUIState();
        }

        /// <summary>
        /// 部品チェック
        /// </summary>
        private void Event_chkPartsLayout_CheckedChanged(object sender, EventArgs e)
        {
            this.UpdateUIState();
        }

        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOK_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Cancel;
        }
    }
}
