﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using LECore.Structures;
using LayoutEditor.Controls;

namespace LayoutEditor.Forms.Dialogs
{
    /// <summary>
    /// 部品レイアウトを選択するダイアログです。
    /// プロパティウインドウの部品ペインページから呼ばれます。
    /// </summary>
    public partial class SelectPartsLayoutFiles : LEBaseDlg
    {
        // 比較の順番
        static private UIListView.ComponentOfComparer[] _columns = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SelectPartsLayoutFiles()
        {
            InitializeComponent();

            _lvwPartsLayout.Columns[0].Text = StringResMgr.Get("PARTS_LISTVIEW_NAME");
            _lvwPartsLayout.Columns[1].Text = StringResMgr.Get("PARTS_LISTVIEW_FILENAME");
            _lvwPartsLayout.Columns[2].Text = StringResMgr.Get("PARTS_LISTVIEW_KIND");
            _lvwPartsLayout.Columns[3].Text = StringResMgr.Get("PARTS_LISTVIEW_DESC");
        }

        //----------------------------------------------------------
        // プロパティ
        //----------------------------------------------------------

        /// <summary>
        /// 区間タグ生成オプションです。
        /// </summary>
        public string SelectedPartsLayoutFileName
        {
            get
            {
                return _lvwPartsLayout.SelectedItems.Count > 0 ?
                    _lvwPartsLayout.SelectedItems[0].SubItems[1].Text : string.Empty;
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 初期化します。
        /// </summary>
        public void Initialize(string currentPartsLayoutName)
        {
            _lvwPartsLayout.Items.Clear();
            foreach (var partsSubScenes in LECore.LayoutEditorCore.Scene.PartsSubScenes)
            {
                if (!partsSubScenes.IsValidPartsSubScene)
                {
                    continue;
                }

                ListViewItem newItem = new ListViewItem(
                    new string[]{
                        partsSubScenes.DescriptionName,
                        Path.GetFileName(partsSubScenes.FilePath),
                        partsSubScenes.PartsKind,
                        partsSubScenes.Description });

                newItem.Tag = partsSubScenes.SubScene;

                if (partsSubScenes.IsLoaded)
                {
                    if (partsSubScenes.SubScene.IControlSettings.Any((ctrlSettings) => ctrlSettings.ContainsInvalidAnimationTag()) ||
                        partsSubScenes.SubScene.IControlSettings.Any((ctrlSettings) => ctrlSettings.ContainsInvalidPane()))
                    {
                        newItem.ForeColor = Color.Red;
                        newItem.SubItems[0].Text = "[NG] " + newItem.SubItems[0].Text;
                    }
                }

                _lvwPartsLayout.Items.Add(newItem);
            }

            // ソートを行ないます。
            if (_columns != null)
            {
                _lvwPartsLayout.ColumnOrder = _columns;
            }

            // 現在のノードを選択し、文字色を変えます。
            foreach(ListViewItem item in _lvwPartsLayout.Items)
            {
                if (item.SubItems[1].Text == currentPartsLayoutName)
                {
                    item.Selected = true;
                    item.ForeColor = Color.Blue;
                    item.SubItems[0].Text += "[*]";
                }
            }

            // UI の更新。
            this.UpdateUIState();
        }

        /// <summary>
        /// ソートの状態を保持します。
        /// </summary>
        private void SaveSortOrder()
        {
            _columns = _lvwPartsLayout.ColumnOrder;
        }

        //----------------------------------------------------------

        /// <summary>
        /// UIを更新します。
        /// </summary>
        private void UpdateUIState()
        {
            _btnOK.Enabled = !string.IsNullOrEmpty(SelectedPartsLayoutFileName);
        }

        //----------------------------------------------------------
        // イベントハンドラです。
        //----------------------------------------------------------

        /// <summary>
        /// リスト選択変更
        /// </summary>
        private void Event_LvwPartsLayout_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateUIState();
        }

        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOK_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.OK;
            SaveSortOrder();
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Cancel;
            SaveSortOrder();
        }
    }
}
