﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.Dialogs
{
    using System.Text.RegularExpressions;
    using LECore;
    using LECore.Save_Load;
    using LECore.Structures;

    public partial class ResourcePathSettingDlg : Form
    {
        FolderBrowserUility _folderBrowserUility = new FolderBrowserUility();

        readonly string _layoutFileName;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ResourcePathSettingDlg( string layoutFileName )
        {
            InitializeComponent();

            _layoutFileName = layoutFileName;
            _tbxNewPath.Text = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);

            UpdateProperty_();
        }

        /// <summary>
        /// 出力パスを取得します。
        /// </summary>
        string GetExportPath_()
        {
            return ( _chkMakeSubFolderForLayout.Checked ) ?
                _tbxNewPath.Text +"\\"+ _layoutFileName : _tbxNewPath.Text;
        }

        /// <summary>
        /// ボタンの状態を更新します。
        /// </summary>
        void UpdateButtonState_()
        {
            _btnExecute.Enabled = Directory.Exists( _tbxNewPath.Text );
        }

        /// <summary>
        /// GUIの状態を更新します。
        /// </summary>
        void UpdateProperty_()
        {
            _tbxFinalExportPath.Text = GetExportPath_();
            UpdateButtonState_();
        }

        /// <summary>
        /// ダイアログ表示
        /// </summary>
        private void Event_ResourcePathSettingDlg_Shown_( object sender, EventArgs e )
        {
            _tbxNewPath.Focus();
        }

        /// <summary>
        /// パス文字列更新ハンドラ
        /// </summary>
        private void Event_TbxNewPath_TextChanged_( object sender, EventArgs e )
        {
            _tbxFinalExportPath.Text = GetExportPath_();
            UpdateButtonState_();
        }

        /// <summary>
        /// 新しいパス指定ボタン
        /// </summary>
        private void Event_BtnNewPath_Click_( object sender, EventArgs e )
        {
            string resultPath = _folderBrowserUility.SelectFolder(string.Empty, _tbxNewPath.Text, this.Handle);
            if (resultPath != null)
            {
                _tbxNewPath.Text = resultPath;
            }

            UpdateProperty_();
        }

        /// <summary>
        /// 実行ボタン
        /// </summary>
        private void Event_BtnExecute_Click_( object sender, EventArgs e )
        {
            string exportPath = GetExportPath_();

            LECore.LayoutAggregationHelper.CreateDirIfNotExist(
                exportPath,
                (errMsg) => MessageBox.Show(errMsg, this.Text, MessageBoxButtons.OK));

            // ファイル保存先をユーザに確認する
            string exportLytPath;
            string filePath = Path.Combine(exportPath, Path.GetFileNameWithoutExtension(_layoutFileName) + "_NEW" + Path.GetExtension(_layoutFileName));
            if (!SaveLoadUIHelper.AskOutputFilePath(LECore.Save_Load.ImportExportFileKind.All, filePath, out exportLytPath))
            {
                DialogResult = DialogResult.Cancel;
                return;
            }

            // 変換します。
            var copiedFiles = LECore.LayoutAggregationHelper.Aggregate(
                LECore.LayoutEditorCore.Scene.CurrentISubScene,
                exportPath,
                exportLytPath,
                _chkMakeTexSubDir.Checked,
                (errMsg) => MessageBox.Show(errMsg, this.Text, MessageBoxButtons.OK));

            // 結果をユーザに報告します。
            {
                string reportMsg = LECore.LayoutAggregationHelper.GetReportString(copiedFiles);
                MessageReportDlg messageReportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                messageReportDlg.Title = this.Text;
                messageReportDlg.Message = reportMsg;
                messageReportDlg.ShowDialog();
            }

            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// チェック状態の更新
        /// </summary>
        private void Event_ChkMakeSubFolderForLayout_CheckedChanged_( object sender, EventArgs e )
        {
            UpdateProperty_();
        }
    }
}
