﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

namespace LayoutEditor.Forms.Dialogs
{
    using LECore.Structures;
    using LECore.Manipulator;

    /// <summary>
    /// グループを編集するダイアログです。
    /// モーダルダイアログです。
    /// </summary>
    public partial class PaneGroupEditDlg : LEBaseDlg
    {
        /// <summary>
        /// 定数
        /// </summary>
        const string _NO_COMMENT_STRING = " --No Comment-- ";
        const int _MAX_GROUP_LENGTH = LECore.AppConstants.MaxGroupNameLength + 4;

        /// <summary>
        /// フィールド
        /// </summary>
        ISubScene   _subScene = null;
        List<IPane> _paneIndexTable = new List<IPane>();
        List<ILEGroup> _groupIndexTable = new List<ILEGroup>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneGroupEditDlg( ISubScene subScene )
        {
            InitializeComponent();

            Debug.Assert( subScene != null );
            _subScene = subScene;

            InitializeProperty_();
        }

        /// <summary>
        /// グループメンバが操作対象か調べます
        /// </summary>
        static bool CheckGroupContainsTargetPane_( ILEGroup group, IPane[] paneSet )
        {
            foreach( IPane targetPane in paneSet )
            {
                if( group.Contains( targetPane ) )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 書式指定されたグループ項目文字列を取得します。
        /// </summary>
        static string GetGroupFormatedString_( ILEGroup group )
        {
            Debug.Assert( group != null );

            string commentStr =
                group.UserCommentString.Length != 0 ? group.UserCommentString : _NO_COMMENT_STRING;

            string baseString = "{0, -" + _MAX_GROUP_LENGTH.ToString() + "}({1})";
            return string.Format( baseString, group.GrouprName, commentStr );
        }

        /// <summary>
        /// GUI初期化
        /// </summary>
        void InitializeProperty_()
        {
            if( _subScene == null )
            {
                return;
            }

            // ペインリストの初期化
            IPane[] paneSet = _subScene.ISelectedSet.IPaneArray;
            _paneIndexTable.Capacity = paneSet.Length;
            foreach( IPane pane in paneSet )
            {
                int index = _clbTargetPane.Items.Add( pane.PaneName, true );
                _paneIndexTable.Add( pane );
            }

            // グループリストの初期化
            ILEGroup[] groupSet = _subScene.ILEGroupMgr.GroupSet;
            _groupIndexTable.Capacity = groupSet.Length;
            foreach (ILEGroup group in groupSet)
            {
                var newItem = new ListViewItem(GetGroupFormatedString_(group));
                newItem.Tag = group;
                newItem.Checked = CheckGroupContainsTargetPane_(group, paneSet);
                newItem.ForeColor = group.IsReadOnlyLocked ? LECore.AppConstants.InactiveForeColor : LECore.AppConstants.ActiveForeColor;
                newItem.BackColor = group.IsReadOnlyLocked ? LECore.AppConstants.InactiveBackColor : LECore.AppConstants.ActiveBackColor;
                _clbGroup.Items.Add(newItem);

                _groupIndexTable.Add(group);
            }
        }

       #region イベントハンドラ
        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOk_Click( object sender, EventArgs e )
        {
            Debug.Assert( _subScene != null );
            Debug.Assert( _clbTargetPane.CheckedIndices.Count > 0 );


            // チェックされているペインを列挙する
            List<IPane>        targetPaneSet = new List<IPane>();
            foreach( int index in _clbTargetPane.CheckedIndices )
            {
                targetPaneSet.Add( _paneIndexTable[index] );
            }

            // チェックされているペイン全てについて
            GroupMgrManipulator groupMngMnp = new GroupMgrManipulator();
            groupMngMnp.BindTarget( _subScene.ILEGroupMgr );
            groupMngMnp.BeginEditGroup();

            foreach( IPane pane in targetPaneSet )
            {
                // 全てのグループについて
                for( int i = 0 ; i < _groupIndexTable.Count ; i++ )
                {
                    ILEGroup group = _groupIndexTable[i];
                    bool bChecked = _clbGroup.CheckedIndices.IndexOf( i ) != -1;
                    // チェックされていれば新規に登録する
                    if( bChecked )
                    {
                        if( !group.Contains( pane ) )
                        {
                            // 登録
                            groupMngMnp.AddMember( group, pane );
                        }
                    }
                    else
                    {
                        // チェックされていなければグループから削除する
                        if( group.Contains( pane ) )
                        {
                            groupMngMnp.RemoveMember( group, pane );
                        }
                    }
                }
            }

            // 空のグループを削除します。
            //
            groupMngMnp.RemoveEmptyGroup();
            // 区間タグのグループ登録を整理
            AnimFrameSectionSetHelper.RemoveInvalidTargetGroupAll(_subScene.IAnimFrameSectionSet, _subScene.ILEGroupMgr);
            groupMngMnp.EndEditGroup();

            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click( object sender, EventArgs e )
        {
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// 対象ペインチェック項目変更
        /// </summary>
        private void Event_ClbTargetPane_ItemCheck( object sender, ItemCheckEventArgs e )
        {
            // 対象ペインの、最後のチェックがはずされる場合
            if( _clbTargetPane.CheckedIndices.Count == 1 )
            {
                if( e.NewValue != CheckState.Checked )
                {
                    // 対象が存在しない場合は、OKボタンを押せない
                    _btnOk.Enabled = false;
                    return;
                }
            }
            _btnOk.Enabled = true;
        }

        /// <summary>
        /// チェック変更
        /// </summary>
        private void Event_ClbGroup_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            var group = e.Item.Tag as ILEGroup;
            if (group != null && group.IsReadOnlyLocked)
            {
                e.Item.Checked = false;
            }
        }
       #endregion イベントハンドラ


    }
}
