﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
//
// 本コードは、3D エディタからコピーしたコードです。
// $Id: UITrackBar.cs,v 1.4 2007/03/23 07:09:41 kitani_toshikazu Exp $
//

using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Windows.Forms.Design;


namespace LayoutEditor.Controls
{
    using Win32 = LECore.Win32;

    /// <summary>
    /// ＵＩトラックバークラス。
    /// </summary>
    [ToolboxBitmap(typeof(TrackBar))]
    public class UITrackBar : TrackBar
    {
        private static bool _invokeEvent = true;

        /// <summary>スクロール変更イベント。</summary>
        public event ScrollChangedEventHandler ScrollChanged = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UITrackBar()
        {
            this.AutoSize  = false;
            this.TickStyle = TickStyle.Both;
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool AutoSize
        {
            get { return base.AutoSize;  }
            set { base.AutoSize = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(TickStyle.Both)]
        public new TickStyle TickStyle
        {
            get { return base.TickStyle;  }
            set { base.TickStyle = value; }
        }

        /// <summary>
        /// Valueプロパティ設定。
        /// </summary>
        public void SetValue(int val)
        {
            _invokeEvent = false;
            {
                this.Value = val;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams cp = base.CreateParams;
                cp.Style |= Win32.TBS.TBS_NOTICKS;
                return cp;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
            case Win32.WM.WM_REFLECT|Win32.WM.WM_HSCROLL:
            case Win32.WM.WM_REFLECT|Win32.WM.WM_VSCROLL:
            {
                // スクロール終了イベント
                if (m.WParam == (IntPtr)Win32.SB.SB_ENDSCROLL)
                {
                    OnScrollChanged(new ScrollChangedEventArgs(true));
                }
                break;
            }
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// スクロール変更ハンドラ。
        /// </summary>
        protected virtual void OnScrollChanged(ScrollChangedEventArgs e)
        {
            if (this.ScrollChanged != null)
            {
                ScrollChanged(this, e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnValueChanged(EventArgs e)
        {
            if (_invokeEvent)
            {
                // スクロール変更イベント
                OnScrollChanged(new ScrollChangedEventArgs(false));

                base.OnValueChanged(e);
            }
        }

        private void InitializeComponent()
        {
            ( (System.ComponentModel.ISupportInitialize)( this ) ).BeginInit();
            this.SuspendLayout();
            ( (System.ComponentModel.ISupportInitialize)( this ) ).EndInit();
            this.ResumeLayout( false );

        }
    }

    /// <summary>
    /// 整数値トラックバークラス。
    /// </summary>
    [ToolboxBitmap(typeof(TrackBar))]
    public class IntTrackBar : UITrackBar
    {
        internal const int _defSmallChange = 1;
        internal const int _defLargeChange = 5;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntTrackBar() {}
    }

    /// <summary>
    /// 実数値トラックバークラス。
    /// </summary>
    [ToolboxBitmap(typeof(TrackBar))]
    public class FloatTrackBar : UITrackBar
    {
        internal const int   _defPrecision   = 10;
        internal const float _defValue       = 0.0f;
        internal const float _defMinimum     = 0.0f;
        internal const float _defMaximum     = 1.0f;
        internal const float _defSmallChange = 0.1f;
        internal const float _defLargeChange = 0.5f;

        private int _precision = _defPrecision;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatTrackBar()
        {
        }

        /// <summary>
        /// 精度。
        /// </summary>
        [DefaultValue(_defPrecision)]
        [Description("精度。")]
        public int Precision
        {
            get { return _precision;  }
            set { _precision = value; }
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(_defValue)]
        public new float Value
        {
            get { return ToFloat(base.Value); }
            set { base.Value = ToInt(value);  }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(_defMinimum)]
        public new float Minimum
        {
            get { return ToFloat(base.Minimum); }
            set { base.Minimum = ToInt(value);  }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(_defMaximum)]
        public new float Maximum
        {
            get { return ToFloat(base.Maximum); }
            set { base.Maximum = ToInt(value);  }
        }

        /// <summary>
        /// 小変化量。
        /// </summary>
        [DefaultValue(_defSmallChange)]
        public new float SmallChange
        {
            get { return ToFloat(base.SmallChange); }
            set { base.SmallChange = ToInt(value);  }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        [DefaultValue(_defLargeChange)]
        public new float LargeChange
        {
            get { return ToFloat(base.LargeChange); }
            set { base.LargeChange = ToInt(value);  }
        }

        /// <summary>
        /// Valueプロパティ設定。
        /// 値は範囲内に収まるように変換されるので SetValue 直後の Value が val と異なることがありえる。
        /// </summary>
        public void SetValue(float val)
        {
            base.SetValue(Math.Min(Math.Max(ToInt(val), base.Minimum), base.Maximum));
        }

        /// <summary>
        /// 実数変換。
        /// </summary>
        private float ToFloat(int val)
        {
            return (float)val / (float)_precision;
        }

        /// <summary>
        /// 整数変換。
        /// </summary>
        private int ToInt(float val)
        {
            return (int)(val * _precision);
        }
    }

    #region ScrollChangedEvent
    /// <summary>
    /// スクロール変更イベントハンドラデリゲート。
    /// </summary>
    public delegate void ScrollChangedEventHandler(object sender, ScrollChangedEventArgs e);

    /// <summary>
    /// スクロール変更イベントデータクラス。
    /// </summary>
    public sealed class ScrollChangedEventArgs : EventArgs
    {
        private readonly bool _endScroll;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ScrollChangedEventArgs(bool endScroll)
        {
            _endScroll = endScroll;
        }

        /// <summary>
        /// スクロール終了。
        /// </summary>
        public bool EndScroll
        {
            get { return _endScroll; }
        }
    }
    #endregion
}
