﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;

namespace LayoutEditor.Controls
{
    using System.Runtime.InteropServices;
    using LayoutEditor.Utility;
    using LECore.Util;
    using Win32 = LECore.Win32;
    using LayoutEditor.Controls.UI;

    /// <summary>
    /// ＵＩリストビュークラス。
    ///
    /// サブアイテムのラベル編集を拡張しています。
    ///
    /// サブアイテムのラベル編集時には、既定の
    /// BeforeLabelEdit
    /// AfterLabelEdit
    /// イベントが機能しなくなります。
    ///
    /// </summary>
    [ToolboxBitmap( typeof( UIListView ) )]
    public class UIListView : ListView
    {

        //----------------------------------------------
        #region フィールド
        // カスタム描画フラグ（詳細表示用）
        private bool _customDraw = false;
        // カスタム描画時の状態保存値
        private readonly CustomDrawState _customDrawState = new CustomDrawState();

        // 列サイズ変更をロックする
        private bool _lockColumnResize = false;

        // 項目比較オブジェクト
        private readonly ItemComparer _itemComparer = null;

        #endregion フィールド

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(true)]
        [ReadOnly(true)]
        public new bool FullRowSelect
        {
            get { return base.FullRowSelect; }
            set { base.FullRowSelect = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        [ReadOnly(true)]
        public new bool HideSelection
        {
            get { return base.HideSelection; }
            set { base.HideSelection = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool MultiSelect
        {
            get { return base.MultiSelect; }
            set { base.MultiSelect = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(true)]
        public new bool Scrollable
        {
            get { return base.Scrollable; }
            set { base.Scrollable = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(true)]
        public new bool DoubleBuffered
        {
            get { return base.DoubleBuffered; }
            set { base.DoubleBuffered = value; }
        }

        /// <summary>
        /// 列サイズ変更のロックフラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("列サイズの変更をロックするかどうかを示します。")]
        public bool LockColumnResize
        {
            get { return _lockColumnResize; }
            set { _lockColumnResize = value; }
        }

        /// <summary>
        /// カスタム描画フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("カスタム描画を行うかどうかを示します。OwnerDrawプロパティがtrueの場合のみ有効です。")]
        public bool CustomDraw
        {
            get { return _customDraw; }
            set { _customDraw = value; }
        }

        /// <summary>
        /// 比較の順番。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ComponentOfComparer[] ColumnOrder
        {
            get { return _itemComparer.Columns; }
            set {
                _itemComparer.Columns = value;
                DoSort();
            }
        }

        /// <summary>
        /// ソートされている列番号か実施されているか？
        /// </summary>
        public int SortedColumnIndex
        {
            get { return _itemComparer.ColumnIndex; }
        }

        /// <summary>
        /// ソートされている列番号か昇順か？
        /// </summary>
        public bool SortedColumnAscent
        {
            get { return _itemComparer.Ascent; }
        }

        //---------------------------------------------------------------------
        // 構築
        //---------------------------------------------------------------------

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListView()
        {
            // 標準は最も頻繁に使用する詳細ビューにする
            base.View = View.Details;

            base.FullRowSelect = true;
            base.HideSelection = false;
            base.MultiSelect = false;
            base.Scrollable = true;
            base.DoubleBuffered = true;

            // 項目比較用
            _itemComparer = new ItemComparer(this.OnCompareItem);
        }

        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // UIListView
            //
            this.ResumeLayout(false);
        }

        //---------------------------------------------------------------------
        // ソート
        //---------------------------------------------------------------------

        //---------------------------------------------------------------------
        // 項目の並べ替え
        /// <summary>
        /// 項目の並べ替え。
        /// メイン項目のみで並べ替えます。
        /// </summary>
        public void Sort(bool ascent)
        {
            // メイン項目で並べ替える
            Sort(0, ascent);
        }

        /// <summary>
        /// 項目の並べ替え。
        /// サブ項目（列項目）を指定して並べ替えます。
        /// サブ項目がない場合は状態は変更されません。
        /// </summary>
        public void Sort(int column, bool ascent)
        {
            // 比較条件設定
            _itemComparer.ColumnIndex = column;
            _itemComparer.ColumnHeader = this.Columns[column];
            _itemComparer.Ascent = ascent;

            DoSort();
        }

        /// <summary>
        /// ソートを実行します。
        /// </summary>
        public void DoSort()
        {
            // ソート開始
            if (this.ListViewItemSorter != _itemComparer)
            {
                // 内部で Sort() を呼んでくれる
                this.ListViewItemSorter = _itemComparer;
            }
            else
            {
                // 手動で開始させる
                this.Sort();
            }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_COMPAREITEM = new object();

        /// <summary>
        /// 項目比較イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("列項目をクリックして項目を並べ替える時に発生します。")]
        public event CompareListViewItemEventHandler CompareItem
        {
            add { base.Events.AddHandler(EVENT_COMPAREITEM, value); }
            remove { base.Events.RemoveHandler(EVENT_COMPAREITEM, value); }
        }

        /// <summary>
        /// 項目比較ハンドラ。
        /// </summary>
        protected virtual int OnCompareItem(CompareListViewItemEventArgs e)
        {
            CompareListViewItemEventHandler handler = (CompareListViewItemEventHandler)base.Events[EVENT_COMPAREITEM];
            if (handler != null) { return handler(this, e); }

            // ハンドラがなければデフォルト比較
            return e.CompareDefault();
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_COLUMNDIVIDERDOUBLECLICK = new object();

        /// <summary>
        /// 列分割線ダブルクリックハンドラ。
        /// </summary>
        protected virtual void OnColumnDividerDoubleClick(ColumnDividerDoubleClickEventArgs e)
        {
            ColumnDividerDoubleClickEventHandler handler = (ColumnDividerDoubleClickEventHandler)base.Events[EVENT_COLUMNDIVIDERDOUBLECLICK];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_CUSTOMDRAWITEM = new object();

        /// <summary>
        /// 項目カスタム描画イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("項目をカスタム描画する時に発生します。")]
        public event CustomDrawListViewItemEventHandler CustomDrawItem
        {
            add { base.Events.AddHandler(EVENT_CUSTOMDRAWITEM, value); }
            remove { base.Events.RemoveHandler(EVENT_CUSTOMDRAWITEM, value); }
        }

        /// <summary>
        /// 項目カスタム描画ハンドラ。
        /// </summary>
        protected virtual void OnCustomDrawItem(CustomDrawListViewItemEventArgs e)
        {
            CustomDrawListViewItemEventHandler handler = (CustomDrawListViewItemEventHandler)base.Events[EVENT_CUSTOMDRAWITEM];
            if (handler != null)
            {
                handler(this, e);
            }
            else
            {
                e.DrawDefault();
            }
        }

        //---------------------------------------------------------------------
        // オーバーライド
        //---------------------------------------------------------------------

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnColumnClick(ColumnClickEventArgs e)
        {
            // 項目があれば並べ替えを行う
            if (this.Items.Count > 1)
            {
                // 比較条件
                bool ascent = false;
                if (_itemComparer.ColumnIndex == e.Column)
                {
                    // 同じ列なら順序を逆にする
                    ascent = !_itemComparer.Ascent;
                }

                // 並べ替え開始
                Sort(e.Column, ascent);
                SortIconHelper.SetSortIcon(this, e.Column, ascent ? SortOrder.Ascending : SortOrder.Descending);
            }

            base.OnColumnClick(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // ヘッダ通知メッセージ
                case Win32.WM.WM_NOTIFY:
                    {
                        Win32.NMHDR nmhdr = (Win32.NMHDR)m.GetLParam(typeof(Win32.NMHDR));
                        switch (nmhdr.code)
                        {
                            // 分割線トラッキング開始
                            case Win32.HDN.HDN_BEGINTRACKA:
                            case Win32.HDN.HDN_BEGINTRACKW:
                                if (_lockColumnResize)
                                {
                                    // Returns FALSE to allow tracking of the divider, or TRUE to prevent tracking.
                                    m.Result = (IntPtr)1;
                                    return;
                                }
                                break;

                            // 分割線ダブルクリック
                            case Win32.HDN.HDN_DIVIDERDBLCLICKA:
                            case Win32.HDN.HDN_DIVIDERDBLCLICKW:
                                {
                                    if (_lockColumnResize)
                                    {
                                        return;
                                    }

                                    // 列番号
                                    Win32.NMHEADER nmheader = (Win32.NMHEADER)m.GetLParam(typeof(Win32.NMHEADER));
                                    int column = nmheader.iItem;

                                    // イベント発行
                                    ColumnDividerDoubleClickEventArgs e = new ColumnDividerDoubleClickEventArgs(this.Columns[column]);
                                    OnColumnDividerDoubleClick(e);
                                    if (e.Handled) { return; }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;

                // マウスダウン
                case Win32.WM.WM_LBUTTONDOWN:
                case Win32.WM.WM_MBUTTONDOWN:
                case Win32.WM.WM_RBUTTONDOWN:
                    // アクティベート
                    if (!this.Focused)
                    {
                        Focus();
                    }
                    break;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        //---------------------------------------------------------------------
        // オーナー描画
        //---------------------------------------------------------------------

        /// <summary>
        /// サブ項目領域を取得。
        /// </summary>
        private Rectangle GetSubItemRect(int item, int subItem)
        {
            // 領域取得
            Rectangle rcItem = ListViewHelper.GetSubItemRect(this, item, subItem, Win32.LVIR.LVIR_BOUNDS);

            // メイン項目の領域補正
            // item == 0 の時は行全体が取得される為
            if (subItem == 0)
            {
                // 右位置補正
                Rectangle rcLabel = ListViewHelper.GetSubItemRect(this, item, subItem, Win32.LVIR.LVIR_LABEL);
                RectangleUtil.SetRight(ref rcItem, rcLabel.Right);

                // 左端じゃなければ左位置補正
                int[] order = ListViewHelper.GetColumnOrderArray(this);
                if (order[0] != 0)
                {
                    // 列番号
                    int column = 1;
                    for (int i = 1; i < order.Length; i++)
                    {
                        if (order[i] == 0)
                        {
                            column = i;
                            break;
                        }
                    }

                    // 左位置を左隣列の右位置に合わせる
                    Rectangle rcLeft = ListViewHelper.GetSubItemRect(this, item, order[column - 1], Win32.LVIR.LVIR_BOUNDS);
                    RectangleUtil.SetLeft(ref rcItem, rcLeft.Right);
                }
            }
            return rcItem;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDrawItem(DrawListViewItemEventArgs e)
        {
            // 無効な状態フラグの時は何もしない
            if (e.State == 0) { return; }

            // カスタム描画（詳細表示のみ）
            if (_customDraw && this.View == View.Details)
            {
                // 領域補正
                Rectangle bounds = e.Bounds;
                if (this.GridLines)
                {
                    bounds.Height -= 1;
                }

                // 選択背景
                if (e.Item.Selected)
                {
                    if (this.Focused)
                    {
                        e.Graphics.FillRectangle(SystemBrushes.Highlight, bounds);
                    }
                    else
                    {
                        e.Graphics.FillRectangle(SystemBrushes.InactiveBorder, bounds);
                    }
                }

                // フォーカス枠
                if ((e.State & ListViewItemStates.Focused) != 0)
                {
                    GraphicsUtil.DrawFocusRectangle(e.Graphics, bounds);
                }

                // 状態をサブ項目描画用に保持しておく
                // 選択状態：e.State の ListViewItemStates.Selected は常に立っているので使えない（バグ？）
                // 列０の左：常に行の左端として取得されるため、そのままではサブ項目描画時に使えない
                _customDrawState.Selected = e.Item.Selected;
                _customDrawState.Column0Left = GetSubItemRect(e.ItemIndex, 0).Left;
            }

            base.OnDrawItem(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDrawSubItem(DrawListViewSubItemEventArgs e)
        {
            // 無効な状態フラグの時は何もしない
            if (e.ItemState == 0) { return; }

            // カスタム描画（詳細表示のみ）
            if (_customDraw && this.View == View.Details)
            {
                // 領域補正
                Rectangle bounds = e.Bounds;
                {
                    // 列０は順序変更された時でも左位置が０のままになってしまう
                    if (e.ColumnIndex == 0)
                    {
                        bounds.X = _customDrawState.Column0Left;
                    }
                }
                if (this.GridLines)
                {
                    bounds.Height -= 1;
                }

                // イベント発行
                CustomDrawListViewItemEventArgs cde = new CustomDrawListViewItemEventArgs(this, e, bounds, _customDrawState.Selected);
                OnCustomDrawItem(cde);
                if (!cde.Handled)
                {
                    cde.DrawDefault();
                }
            }
            base.OnDrawSubItem(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDrawColumnHeader(DrawListViewColumnHeaderEventArgs e)
        {
            // OwnerDrawを有効にすると列ヘッダもオーナー描画になる
            e.DrawDefault = true;
            base.OnDrawColumnHeader(e);
        }

        //---------------------------------------------------------------------
        // 内部クラス
        //---------------------------------------------------------------------

        /// <summary>
        /// SortIcon 設定
        /// </summary>
        private static class SortIconHelper
        {
            [StructLayout(LayoutKind.Sequential)]
            private struct LVCOLUMN
            {
                public Int32 mask;
                public Int32 cx;
                [MarshalAs(UnmanagedType.LPTStr)]
                public string pszText;
                public IntPtr hbm;
                public Int32 cchTextMax;
                public Int32 fmt;
                public Int32 iSubItem;
                public Int32 iImage;
                public Int32 iOrder;
            }

            [DllImport("user32.dll", EntryPoint = "SendMessage")]
            private static extern IntPtr SendMessage(IntPtr hWnd,Int32 Msg,IntPtr wParam,ref LVCOLUMN lPLVCOLUMN);

            public static void SetSortIcon(ListView ListViewControl,int ColumnIndex,SortOrder Order)
            {
                IntPtr ColumnHeader = Win32.User32.SendMessage(ListViewControl.Handle, Win32.LVM.LVM_GETHEADER, IntPtr.Zero, IntPtr.Zero);

                for (int ColumnNumber = 0; ColumnNumber <= ListViewControl.Columns.Count - 1; ColumnNumber++)
                {
                    IntPtr ColumnPtr = new IntPtr(ColumnNumber);
                    LVCOLUMN lvColumn = new LVCOLUMN();
                    lvColumn.mask = Win32.HDI.HDI_FORMAT;
                    SendMessage(ColumnHeader, Win32.HDM.HDM_GETITEMW, ColumnPtr, ref lvColumn);

                    if (!(Order == SortOrder.None) && ColumnNumber == ColumnIndex)
                    {
                        switch (Order)
                        {
                            case SortOrder.Ascending:
                                lvColumn.fmt &= ~Win32.HDF.HDF_SORTDOWN;
                                lvColumn.fmt |= Win32.HDF.HDF_SORTUP;
                                break;
                            case SortOrder.Descending:
                                lvColumn.fmt &= ~Win32.HDF.HDF_SORTUP;
                                lvColumn.fmt |= Win32.HDF.HDF_SORTDOWN;
                                break;
                        }
                    }
                    else
                    {
                        lvColumn.fmt &= ~Win32.HDF.HDF_SORTDOWN & ~Win32.HDF.HDF_SORTUP;
                    }

                    SendMessage(ColumnHeader, Win32.HDM.HDM_SETITEMW, ColumnPtr, ref lvColumn);
                }
            }
        }

        #region ItemComparer
        /// <summary>項目比較ハンドラデリゲート。</summary>
        private delegate int CompareItemHandler(CompareListViewItemEventArgs e);

        public class ComponentOfComparer
        {
            public int index { get; set; }
            public ColumnHeader header { get; set; }
            public bool ascent { get; set; }
        }

        /// <summary>
        /// 項目比較クラス。
        /// </summary>
        private sealed class ItemComparer : System.Collections.IComparer
        {
            // 比較の順番
            private ComponentOfComparer[] _columns = new[] {new ComponentOfComparer()};

            // 項目比較ハンドラ
            private readonly CompareItemHandler _handler = null;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public ItemComparer(CompareItemHandler handler)
            {
                _handler = handler;
            }

            /// <summary>
            /// カラム。
            /// </summary>
            public ComponentOfComparer[] Columns
            {
                get { return _columns; }
                set { _columns = value; }
            }

            /// <summary>
            /// 列番号。
            /// </summary>
            public int ColumnIndex
            {
                get { return _columns[0].index; }
                set {
                    _columns =
                        Enumerable.Repeat(new ComponentOfComparer() { index = value }, 1).Concat(
                        _columns.Where(x => x.index != value)).ToArray();
                }
            }

            /// <summary>
            /// 列ヘッダ。
            /// </summary>
            public ColumnHeader ColumnHeader
            {
                set { _columns[0].header = value; }
            }

            /// <summary>
            /// 昇順比較か。
            /// </summary>
            public bool Ascent
            {
                get { return _columns[0].ascent; }
                set { _columns[0].ascent = value; }
            }

            /// <summary>
            /// 比較。
            /// </summary>
            public int Compare(object x, object y)
            {
                ListViewItem item1 = (ListViewItem)x;
                ListViewItem item2 = (ListViewItem)y;

                foreach (var column in _columns)
                {
                    int result;
                    // 昇順比較
                    if (column.ascent)
                    {
                        result = _handler(new CompareListViewItemEventArgs(column.index, column.header, item2, item1));
                    }
                    // 降順比較
                    else
                    {
                        result = _handler(new CompareListViewItemEventArgs(column.index, column.header, item1, item2));
                    }

                    if (result != 0)
                    {
                        return result;
                    }
                }

                return 0;
            }
        }
        #endregion

        #region CustomDrawState
        /// <summary>
        /// カスタム描画状態保持クラス。
        /// </summary>
        private sealed class CustomDrawState
        {
            /// <summary>選択状態。</summary>
            public bool Selected = false;
            /// <summary>列項目０の左位置。</summary>
            public int Column0Left = 0;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public CustomDrawState()
            {
            }

            /// <summary>
            /// リセット。
            /// </summary>
            public void Reset()
            {
                this.Selected = false;
                this.Column0Left = 0;
            }
        }
        #endregion
    }
}
