﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Windows.Forms.Design;

namespace LayoutEditor.Controls
{
    using Win32 = LECore.Win32;

    /// <summary>
    /// ヘルパ
    /// </summary>
    public static class ComboBoxHelper
    {
        /// <summary>
        /// 値で選択する。
        /// 要素はUIListControlItemであることを想定します。
        /// </summary>
        public static void SelectByData(this ToolStripComboBox cbm, object data)
        {
            UIListControlItem currentItem = null;
            if (data != null && data.GetType().IsEnum)
            {
                currentItem = cbm.Items.Cast<UIListControlItem>().FirstOrDefault((item) => item.Data.ToString() == data.ToString());
            }
            else {
                currentItem = cbm.Items.Cast<UIListControlItem>().FirstOrDefault((item) => item.Data == data);
            }

            cbm.SelectedItem = currentItem;
        }
    }

    /// <summary>
    /// ＵＩコンボボックスクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ComboBox))]
    public class UIComboBox : ComboBox
    {
        private static bool _invokeEvent      = true;

        /// <summary>
        /// 直前の選択状態
        /// </summary>
        /// <remarks>
        /// 項目を削除した後や SelectedIndex が更新されて、OnSelectedIndexChanged が呼ばれるまでの間は
        /// SelectedIndex と異なる可能性がある
        /// </remarks>
        private int _preSelectedIndex = -1;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIComboBox()
        {
            this.DropDownStyle    = ComboBoxStyle.DropDownList;
            this.MaxDropDownItems = 30; // それなりの適当値
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(ComboBoxStyle.DropDownList)]
        public new ComboBoxStyle DropDownStyle
        {
            get { return base.DropDownStyle;  }
            set { base.DropDownStyle = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(30)]
        public new int MaxDropDownItems
        {
            get { return base.MaxDropDownItems;  }
            set { base.MaxDropDownItems = value; }
        }

        /// <summary>
        /// 選択項目データ。
        /// </summary>
        [Browsable(false)]
        public object SelectedItemData
        {
            get
            {
                UIListControlItem item = this.SelectedItem as UIListControlItem;
                if (item != null)
                {
                    return item.Data;
                }
                return null;
            }
            set
            {
                if (value != null)
                {
                    foreach (object item in this.Items)
                    {
                        UIListControlItem listItem = item as UIListControlItem;
                        if (listItem != null && listItem.Data.Equals(value))
                        {
                            this.SelectedItem = item;
                            return;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data)
        {
            this.Items.Add(new UIListControlItem(text, data, false));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data, bool textIsStringID)
        {
            this.Items.Add(new UIListControlItem(text, data, textIsStringID));
        }

        /// <summary>
        /// タグデータで選択します。
        /// </summary>
        public void SetSelectedByCondition(object data, Func<object, object, bool> predicate)
        {
            int index = 0;
            bool hit = false;
            foreach (UIListControlItem item in this.Items)
            {
                if (item.Data == null)
                {
                    if (data == null)
                    {
                        this.SetSelectedIndex(index);
                        this.SetSelectedItem(item);
                        this.SetSelectedItemData(data);
                        hit = true;
                        break;
                    }
                }
                else
                {
                    if (predicate(item.Data, data))
                    {
                        this.SetSelectedIndex(index);
                        this.SetSelectedItem(item);
                        this.SetSelectedItemData(data);
                        hit = true;
                        break;
                    }
                }

                index++;
            }

            if (!hit)
            {
                this.SetSelectedIndex(-1);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void SetSelectedByText(string text)
        {
            foreach (var item in this.Items)
            {
                if (item.ToString() == text)
                {
                    this.SelectedItem = item;
                    break;
                }
            }
        }

        /// <summary>
        /// タグデータで選択します。
        /// </summary>
        public void SetSelectedByData(object data)
        {
            SetSelectedByCondition(data, (objL, objR) => objL.Equals(objR));
        }

        /// <summary>
        /// SelectedIndex プロパティ
        /// </summary>
        public override int SelectedIndex
        {
            get
            {
                return base.SelectedIndex;
            }

            set
            {
                base.SelectedIndex = value;
                _preSelectedIndex = SelectedIndex;
            }
        }


        /// <summary>
        /// SelectedIndexプロパティ設定。
        /// </summary>
        public void SetSelectedIndex(int val)
        {
            _invokeEvent = false;
            {
                this.SelectedIndex = val;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// SelectedItemプロパティ設定。
        /// </summary>
        public void SetSelectedItem(object val)
        {
            _invokeEvent = false;
            {
                this.SelectedItem = val;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// SelectedItemDataプロパティ設定。
        /// </summary>
        public void SetSelectedItemData(object val)
        {
            _invokeEvent = false;
            {
                this.SelectedItemData = val;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// ドロップダウン幅を調整します。
        /// </summary>
        static public void ExpandDropDownWidth(ComboBox cmb)
        {
            // コンボドロップダウン幅の調整
            int maxWidth = cmb.DropDownWidth;
            using (var gfx = Graphics.FromHwnd(cmb.Handle))
            {
                foreach (var item in cmb.Items)
                {
                    // 一文字分の余白こみの文字サイズを計算
                    int size = (int)Math.Ceiling(gfx.MeasureString(item.ToString() + "M", cmb.Font).Width);
                    maxWidth = Math.Max(size, maxWidth);
                }
            }

            cmb.DropDownWidth = maxWidth;
        }

        /// <summary>
        /// ドロップダウン幅を調整します。
        /// </summary>
        public void ExpandDropDownWidth()
        {
            ExpandDropDownWidth(this);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            // マウスホイールで選択を変えさせない
            if (m.Msg == Win32.WM.WM_MOUSEWHEEL)
            {
                if (!this.DroppedDown)
                {
                    // TODO: SendMessageで親に投げる？
                    m.Msg = Win32.WM.WM_NULL;
                    return;
                }
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSelectedIndexChanged(EventArgs e)
        {
            // アイテムを削除して選択しなおしたときは _preSelectedIndex と SelectedIndex が一致する
            if (_invokeEvent && _preSelectedIndex != SelectedIndex)
            {
                _preSelectedIndex = SelectedIndex;
                base.OnSelectedIndexChanged(e);
            }
        }

        private void InitializeComponent()
        {
            this.SuspendLayout();
            this.ResumeLayout( false );
        }
    }

    /// <summary>
    ///
    /// </summary>
    public sealed class UIToolStripComboBox : ToolStripComboBox
    {
    }

    /// <summary>
    /// ＵＩリストコントロール項目クラス。
    /// </summary>
    public sealed class UIListControlItem
    {
        private readonly string _text;
        private readonly object _data;
        private readonly bool   _textIsStringID;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListControlItem(string text, object data)
            : this(text, data, false)
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListControlItem(string text, object data, bool textIsStringID)
        {
            _text           = text;
            _data           = data;
            _textIsStringID = textIsStringID;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            // ComboBox.ObjectCollectionからテキスト表示用に呼ばれる
            return this.Text;
        }

        /// <summary>
        /// テキスト。
        /// </summary>
        public string Text
        {
            get
            {
                if (_textIsStringID)
                {
                    return StringResMgr.Get(_text);
                }
                else
                {
                    return _text;
                }
            }
        }

        /// <summary>
        /// データ。
        /// </summary>
        public object Data
        {
            get { return _data; }
        }
    }
}
