﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Windows.Forms.Design;



namespace LayoutEditor.Controls
{
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
    using LECore.Util;
    using System;

    /// <summary>
    /// サムネイルリストボックスクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ListBox))]
    [Designer(typeof(ThumbnailListBox.CustomDesigner))]
    public class ThumbnailListBox : UIListBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ThumbnailListBox()
        {
            this.DrawMode       = DrawMode.OwnerDrawFixed;
            this.ItemHeight     = 32;
            this.IntegralHeight = false;
        }

        /// <summary>
        /// アイテムの背景色を取得する関数
        /// </summary>
        public Func<UIListControlItem, Color> GetItemBackGroundColor;

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(IThumbnail item)
        {
            base.AddItem(item.Name, item);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDrawItem(DrawItemEventArgs e)
        {
            // 背景
            if (this.Enabled)
            {
                e.DrawBackground();
            }
            else
            {
                e.Graphics.FillRectangle(SystemBrushes.Window, e.Bounds);
            }

            // 項目
            if (e.Index >= 0)
            {
                UIListControlItem item     = (UIListControlItem)this.Items[e.Index];
                IThumbnail        thumnail = (IThumbnail)item.Data;

                if(GetItemBackGroundColor != null)
                {
                    Color color = GetItemBackGroundColor.Invoke(item);
                    if (color != null && color != Color.Empty && (e.State & DrawItemState.Selected) == 0)
                    {
                        e.Graphics.FillRectangle(new SolidBrush(color), e.Bounds);
                    }
                }

                // サムネイル枠
                Rectangle rcFrame = e.Bounds;
                rcFrame.Inflate(-1, -1);
                rcFrame.Width = rcFrame.Height;
                if (this.Enabled)
                {
                    GraphicsUtil.DrawRectangle(e.Graphics, Pens.Black, rcFrame);
                }
                else
                {
                    GraphicsUtil.DrawRectangle(e.Graphics, Pens.Gray, rcFrame);
                }

                // サムネイル
                Rectangle rcImage = rcFrame;
                rcImage.Inflate(-1, -1);
                thumnail.DrawColorImage(e.Graphics, rcImage, this.Enabled);

                // テキスト
                Rectangle rcText = e.Bounds;
                RectangleUtil.OffsetLeft(ref rcText, rcText.Height);
                using (StringFormat sf = new StringFormat())
                {
                    sf.Alignment     = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.Trimming      = StringTrimming.EllipsisCharacter;
                    sf.FormatFlags  |= StringFormatFlags.NoWrap;

                    if (this.Enabled)
                    {
                        if ((e.State & DrawItemState.Selected) != 0)
                        {
                            e.Graphics.DrawString(thumnail.Name, e.Font, SystemBrushes.HighlightText, rcText, sf);
                        }
                        else
                        {
                            e.Graphics.DrawString(thumnail.Name, e.Font, SystemBrushes.WindowText, rcText, sf);
                        }
                    }
                    else
                    {
                        e.Graphics.DrawString(thumnail.Name, e.Font, SystemBrushes.InactiveCaption, rcText, sf);
                    }
                }
            }

            // フォーカス枠
            e.DrawFocusRectangle();
            base.OnDrawItem(e);
        }

        #region CustomDesigner
        /// <summary>
        /// カスタムデザイナクラス。
        /// </summary>
        internal class CustomDesigner : ControlDesigner
        {
            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void PostFilterProperties(IDictionary properties)
            {
                // TODO: 不要なプロパティを外す
                base.PostFilterProperties(properties);
            }
        }
        #endregion
    }
}
