﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Drawing.Drawing2D;

using System.ComponentModel;
using System.Diagnostics;


namespace LayoutEditor.src.Controls
{
    using GraphicUtil = LECore.Util.GraphicsUtil;

    /// <summary>
    /// 分割パネルフォルダクラス。
    /// </summary>
    public abstract class SplitPanelFolder : Control
    {
        // ターゲットコンテナ
        private SplitContainer _targetContainer = null;

        // ターゲットパネル
        Panel _targetPanel = null;

        // マウスポイント中フラグ
        private bool _mouseEnter = false;

        /// <summary>パネル開閉状態変更イベント。</summary>
        public event EventHandler PanelCollapsedChanged = null;

        /// <summary>デフォルト長さ。</summary>
        protected const int _defaultLength = 120;
        /// <summary>デフォルト厚み。</summary>
        protected const int _defaultThickness = 6;

        /// <summary>
        /// ターゲットコンテナ。
        /// </summary>
        [DefaultValue( null )]
        [Description( "ターゲットコンテナ。" )]
        protected SplitContainer _TargetContainer
        {
            get { return _targetContainer; }
        }

        /// <summary>
        ///
        /// </summary>
        public Panel TargetPanel
        {
            get { return _targetPanel; }
            set
            {
                _targetPanel = value;
                InitializeTargetContainer_();
            }
        }

        /// <summary>
        /// _targetContainer を再設定します。
        /// </summary>
        void InitializeTargetContainer_()
        {
            _targetContainer = null;

            Control ctrl = _targetPanel;
            while( ctrl != null )
            {

                if( ctrl is SplitContainer )
                {
                    _targetContainer = ctrl as SplitContainer;
                    break;
                }
                ctrl = ctrl.Parent;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SplitPanelFolder()
        {
            base.SetStyle(ControlStyles.Opaque, true);
            base.SetStyle(ControlStyles.ResizeRedraw, true);
            base.SetStyle(ControlStyles.Selectable, false);
            base.DoubleBuffered = true;
            base.TabStop = false;
        }

        /// <summary>
        /// パネルの開閉状態を切り替える。
        /// </summary>
        public void TogglePanelCollapsed()
        {
            // パネル開閉
            if( _targetContainer != null )
            {
                // パネル１
                if( _targetPanel == _targetContainer.Panel1 )
                {
                    _targetContainer.Panel1Collapsed = !_targetContainer.Panel1Collapsed;
                }
                // パネル２
                else if( _targetPanel == _targetContainer.Panel2 )
                {
                    _targetContainer.Panel2Collapsed = !_targetContainer.Panel2Collapsed;
                }

                // イベント発行
                if( this.PanelCollapsedChanged != null )
                {
                    this.PanelCollapsedChanged( this, EventArgs.Empty );
                }
            }
            else if (_targetPanel != null)
            {
                if (_targetPanel.Dock == DockStyle.None)
                {
                    _targetPanel.Dock = Dock;
                    _targetPanel.Show();
                }
                else
                {
                    _targetPanel.Hide();
                    _targetPanel.Dock = DockStyle.None;
                }
            }
        }

        /// <summary>
        /// トリガーマーク描画。
        /// </summary>
        protected abstract void DrawTrigger( Graphics g, Color color );

        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // SplitPanelFolder
            //
            this.ResumeLayout( false );

        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint( PaintEventArgs e )
        {
            // 背景
            if( _mouseEnter )
            {
                e.Graphics.Clear( GraphicUtil.BlendColors( this.BackColor, SystemColors.Highlight, 0.25f ) );
                GraphicUtil.DrawRectangle( e.Graphics, SystemPens.Highlight, this.ClientRectangle );
                ControlPaint.DrawBorder(e.Graphics, this.ClientRectangle, SystemColors.ActiveBorder, ButtonBorderStyle.Solid);
            }
            else
            {
                e.Graphics.Clear( this.BackColor );
                ControlPaint.DrawBorder(e.Graphics, this.ClientRectangle, SystemColors.InactiveBorder, ButtonBorderStyle.Solid);
            }

            // トリガーマーク
            DrawTrigger( e.Graphics, SystemColors.ControlText );
            base.OnPaint( e );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseEnter( EventArgs e )
        {
            _mouseEnter = true;
            Invalidate();
            base.OnMouseEnter( e );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseLeave( EventArgs e )
        {
            _mouseEnter = false;
            Invalidate();
            base.OnMouseLeave( e );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClick( EventArgs e )
        {
            _mouseEnter = false;
            TogglePanelCollapsed();
            base.OnClick( e );
        }
        #endregion オーバーライド

    }

    #region HSplitPanelFolder
    /// <summary>
    /// 水平分割パネルフォルダクラス。
    /// </summary>
    [ToolboxItem( true )]
    public sealed class HSplitPanelFolder : SplitPanelFolder
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public HSplitPanelFolder()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size( _defaultLength, _defaultThickness ); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void DrawTrigger( Graphics g, Color color )
        {
            int center = this.Width / 2;

            // 方向を決める
            // TODO:
            bool up = true;
            if( this._TargetContainer != null )
            {
                if( TargetPanel == _TargetContainer.Panel1 )
                {
                    up = !_TargetContainer.Panel1Collapsed;
                }
                else
                {
                    up = _TargetContainer.Panel2Collapsed;
                }
            }
            else if (TargetPanel != null)
            {
                up = (TargetPanel.Dock == Dock) == (Dock == DockStyle.Top);
            }

            // 描画
            using ( Pen pen = new Pen( color ) )
            {
                if( up )
                {
                    g.DrawLine( pen, center - 0, 1, center + 1, 1 );
                    g.DrawLine( pen, center - 1, 2, center + 2, 2 );
                    g.DrawLine( pen, center - 2, 3, center + 3, 3 );
                    g.DrawLine( pen, center - 3, 4, center + 4, 4 );
                }
                else
                {
                    g.DrawLine( pen, center - 3, 1, center + 4, 1 );
                    g.DrawLine( pen, center - 2, 2, center + 3, 2 );
                    g.DrawLine( pen, center - 1, 3, center + 2, 3 );
                    g.DrawLine( pen, center - 0, 4, center + 1, 4 );
                }
            }
        }
    }
    #endregion

    #region VSplitPanelFolder
    /// <summary>
    /// 垂直分割パネルフォルダクラス。
    /// </summary>
    [ToolboxItem( true )]
    public sealed class VSplitPanelFolder : SplitPanelFolder
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public VSplitPanelFolder()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size( _defaultThickness, _defaultLength ); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void DrawTrigger( Graphics g, Color color )
        {
            int center = this.Height / 2;

            // 方向を決める
            // TODO:
            bool left = true;
            if( this._TargetContainer != null )
            {
                if( this.TargetPanel == _TargetContainer.Panel1 )
                {
                    left = !_TargetContainer.Panel1Collapsed;
                }
                else
                {
                    left = _TargetContainer.Panel2Collapsed;
                }
            }
            else if (TargetPanel != null)
            {
                left = (TargetPanel.Dock == Dock) == (Dock == DockStyle.Left);
            }

            // 描画
            using( Pen pen = new Pen( color ) )
            {
                if( left )
                {
                    g.DrawLine( pen, 1, center - 0, 1, center + 1 );
                    g.DrawLine( pen, 2, center - 1, 2, center + 2 );
                    g.DrawLine( pen, 3, center - 2, 3, center + 3 );
                    g.DrawLine( pen, 4, center - 3, 4, center + 4 );
                }
                else
                {
                    g.DrawLine( pen, 1, center - 3, 1, center + 4 );
                    g.DrawLine( pen, 2, center - 2, 2, center + 3 );
                    g.DrawLine( pen, 3, center - 1, 3, center + 2 );
                    g.DrawLine( pen, 4, center - 0, 4, center + 1 );
                }
            }
        }
    }
    #endregion

    ///// <summary>
    ///// スプリッタパネルフォルダ用ターゲットパネル。
    ///// </summary>
    //public enum SplitterPanelFolderTargetPanel
    //{
    //    /// <summary>パネル１。</summary>
    //    Panel1,
    //    /// <summary>パネル２。</summary>
    //    Panel2,
    //}
}
