﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;
using System.Drawing;
using System.Drawing.Drawing2D;


namespace LayoutEditor.Controls
{
    /// <summary>
    /// ConnectionImagePanel の概要の説明です。
    /// </summary>
    public class ConnectionImagePanel : Panel
    {

        /// <summary>
        /// 水平位置種類
        /// </summary>
        public enum PositionTypeH
        {
            Left,
            Center,
            Right
        }

        /// <summary>
        /// 垂直位置種類
        /// </summary>
        public enum PositionTypeV
        {
            Top,
            Center,
            Bottom
        }

        /// <summary>
        /// 直線書き順
        /// </summary>
        public enum LineDrawOrder
        {
           Normal,
           HorizontalFast,
           VerticalFast
        }


        PositionTypeH          _startPointH    = PositionTypeH.Left;
        PositionTypeV          _startPointV    = PositionTypeV.Center;

        PositionTypeH          _endPointH      = PositionTypeH.Right;
        PositionTypeV          _endPointV      = PositionTypeV.Center;

        Pen                    _pen            = new Pen( Color.Black, 20.0f );
        LineDrawOrder          _drawOrder      = LineDrawOrder.Normal;
        bool                   _lineActive     = true;

        /// <summary>
        /// 始点種類
        /// </summary>
        public PositionTypeH StartPointH
        {
            get{ return _startPointH;}
            set{ _startPointH = value; Invalidate();}
        }

        /// <summary>
        /// 始点種類
        /// </summary>
        public PositionTypeV StartPointV
        {
            get{ return _startPointV;}
            set{ _startPointV = value; Invalidate();}
        }

        /// <summary>
        /// 終点種類
        /// </summary>
        public PositionTypeH EndPointH
        {
            get{ return _endPointH;}
            set{ _endPointH = value; Invalidate();}
        }

        /// <summary>
        /// 終点種類
        /// </summary>
        public PositionTypeV EndPointV
        {
            get{ return _endPointV;}
            set{ _endPointV = value; Invalidate();}
        }

        /// <summary>
        /// 直線で描画、折れ線で描画
        /// </summary>
        public LineDrawOrder  DrawOrder
        {
            get{ return _drawOrder;}
            set{ _drawOrder = value; Invalidate();}
        }

        /// <summary>
        /// ラインの有効状態
        /// </summary>
        public bool LineActive
        {
            get{ return _lineActive;}
            set{ _lineActive = value; Invalidate();}
        }

        /// <summary>
        /// 始点キャップ
        /// </summary>
        public LineCap   StartCap
        {
            get{ return _pen.StartCap;}
            set{ _pen.StartCap = value; Invalidate();}
        }

        /// <summary>
        /// 終点キャップ
        /// </summary>
        public LineCap   EndCap
        {
            get{ return _pen.EndCap;}
            set{ _pen.EndCap = value; Invalidate();}
        }

        /// <summary>
        /// 接続線の太さ
        /// </summary>
        public float    LineWidth
        {
            get{ return _pen.Width;}
            set{ _pen.Width = value; Invalidate();}
        }

        /// <summary>
        /// 接続線の色
        /// </summary>
        public Color    LineColor
        {
            get{ return _pen.Color;}
            set{ _pen.Color = value; Invalidate();}
        }

        /// <summary>
        /// 位置型から実際の点を算出します。
        /// </summary>
        static Point GetPoint_( PositionTypeH typeH, PositionTypeV typeV, Size size, int lineWidth )
        {
            int x = 0;
            switch( typeH )
            {
                case PositionTypeH.Left: x = lineWidth;break;
                case PositionTypeH.Center: x = size.Width /2;break;
                case PositionTypeH.Right: x = size.Width - lineWidth;break;
            }

            int y = 0;
            switch( typeV )
            {
                case PositionTypeV.Top: y = lineWidth;break;
                case PositionTypeV.Center: y = size.Height /2;break;
                case PositionTypeV.Bottom: y = size.Height - lineWidth;break;
            }
            return new Point( x, y );
        }

        /// <summary>
        /// 接続 始点
        /// </summary>
        public Point          StartPoint
        {
            get{ return GetPoint_( _startPointH, _startPointV, this.Size, (int)_pen.Width );}
        }

        /// <summary>
        /// 接続 終点
        /// </summary>
        public Point          EndPoint
        {
            get{ return GetPoint_( _endPointH, _endPointV, this.Size, (int)_pen.Width );}
        }

       /// <summary>
       /// コンストラクタ
       /// </summary>
        public ConnectionImagePanel()
        {
            InitializeComponent();
        }

        #region デザイナ生成コード
        private void InitializeComponent()
        {
            //
            // ConnectionImagePanel
            //
            this.BackColor = System.Drawing.Color.Transparent;
            this.Paint += new System.Windows.Forms.PaintEventHandler(this.Event_ConnectionImagePanel_Paint);
            SetStyle(ControlStyles.SupportsTransparentBackColor, true);
        }
        #endregion デザイナ生成コード


        #region イベントハンドラ
        /// <summary>
        /// 描画イベントハンドラ
        /// </summary>
        private void Event_ConnectionImagePanel_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
        {
           Graphics g = e.Graphics;

            Color     oldColor = _pen.Color;

            // 非アクティブ時には色を薄くします。
            if( !LineActive )
            {
                _pen.Color = Color.FromArgb( 32,  _pen.Color );
            }

            Point pDiff = new Point( EndPoint.X - StartPoint.X, EndPoint.Y - StartPoint.Y );

            // 矢印の描画順
            switch( _drawOrder )
            {
                // 通常の描画
                case LineDrawOrder.Normal:
                    g.DrawLine( _pen, StartPoint, EndPoint);break;

                // 水平方向から
                case LineDrawOrder.HorizontalFast:
                {
                    Point pM    = StartPoint;
                    pM.Offset( pDiff.X, 0 );

                    Point[]  points = new Point[3];
                    points[0] = StartPoint;
                    points[1] = pM;
                    points[2] = EndPoint;

                    g.DrawLines( _pen, points );
                    break;
                }
                // 垂直方向から
                case LineDrawOrder.VerticalFast:
                {
                    Point pM    = StartPoint;
                    pM.Offset( 0, pDiff.Y );

                    Point[]  points = new Point[3];
                    points[0] = StartPoint;
                    points[1] = pM;
                    points[2] = EndPoint;

                    g.DrawLines( _pen, points );

                    break;
                }
            }

            // 色を復元します。
            if( !LineActive )
            {
                _pen.Color = oldColor;
            }
        }

        #endregion イベントハンドラ
    }
}
