﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using LayoutEditor.Controls;

using LECore.Util;
using NW4R.ColorPicker;

namespace LayoutEditor.Controls
{
    /// <summary>
    /// カラーピッカーアダプタクラス。
    /// </summary>
    public sealed class ColorPickerAdapter
    {
        // トリガーボタン
        private readonly ColorButton _trigger;
        // 関連コントロールリスト
        private readonly ArrayList _controls = new ArrayList();
        // ロストフォーカス処理フラグ
        private bool _handleLostFocus = true;
        // カラーモード
        ColorMode _colorMode = ColorMode.RGB;

        /// <summary>カラー編集イベント。</summary>
        public event ColorEditEventHandler ColorEdit = null;

        /// <summary> </summary>
        public event EventHandler ColorPickerClosed
        {
            add { ColorPickerDialog.DialogClose += value; }
            remove { ColorPickerDialog.DialogClose -= value; }
        }


        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorPickerAdapter(ColorButton trigger)
        {
            // トリガー登録
            _trigger = trigger;
            _trigger.Click += new EventHandler(Event_Trigger_Click);

            // トリガーも関連コントロールと同様なイベント処理をさせる
            AddControl(_trigger);
        }

        #region アダプタ利用者アクセス用
        /// <summary>
        /// コントロール追加。
        /// </summary>
        public void AddControl(Control control)
        {
            if (!_controls.Contains(control))
            {
                _controls.Add(control);

                // イベント登録
                control.GotFocus        += new EventHandler(Event_Controls_GotFocus);
                control.LostFocus       += new EventHandler(Event_Controls_LostFocus);
                control.HandleDestroyed += new EventHandler(Event_Controls_HandleDestroyed);
            }
        }

        /// <summary>
        /// 更新通知。
        /// </summary>
        public void NotifyUpdate()
        {
            // 接続更新
            ColorPickerDialog.UpdateConnection(this);
        }
        #endregion

        #region カラーピッカーアクセス用
        /// <summary>
        /// カラー。
        /// </summary>
        public Color Color
        {
            get { return _trigger.Color.ToSystemColor(); }
        }

        /// <summary>
        /// カラーモード。
        /// </summary>
        public ColorMode ColorMode
        {
            get
            {
                if (_trigger.EnableAlpha)
                {
                    switch (_colorMode)
                    {
                        case ColorMode.HSV: return ColorMode.HSVA;
                        case ColorMode.RGB: return ColorMode.RGBA;
                        default:return _colorMode;
                    }
                }else{
                    switch (_colorMode)
                    {
                        case ColorMode.HSVA: return ColorMode.HSV;
                        case ColorMode.RGBA: return ColorMode.RGB;
                        default: return _colorMode;
                    }
                }
            }

            set
            {
                _colorMode = value;
            }
        }

        /// <summary>
        /// ロストフォーカス処理をするかどうか。
        /// </summary>
        public bool HandleLostFocus
        {
            get { return _handleLostFocus;  }
            set { _handleLostFocus = value; }
        }

        /// <summary>
        /// カラー編集イベント発行。
        /// </summary>
        public void InvokeColorEdit(ColorEditEventArgs e)
        {
            // 利用者にイベント発行
            if (this.ColorEdit != null)
            {
                this.ColorEdit(this, e);
            }
        }
        #endregion

        #region イベントハンドラ
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_Trigger_Click(object sender, EventArgs e)
        {
            // 接続開始
            ColorPickerDialog.StartConnection( this );
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_Controls_GotFocus(object sender, EventArgs e)
        {
            // 接続変更
            ColorPickerDialog.ChangeConnection(this);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_Controls_LostFocus(object sender, EventArgs e)
        {
            if (_handleLostFocus)
            {
                // 接続終了
                ColorPickerDialog.EndConnection(this);
            }
            // フラグを元に戻す
            _handleLostFocus = true;
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_Controls_HandleDestroyed(object sender, EventArgs e)
        {
            // 接続終了
            ColorPickerDialog.EndConnection(this);
        }
        #endregion
    }
}
